{Copyright (C) [Benito van der Zander]
All rights reserved.

How you can contact me:
  Mail: BeniBela@aol.com
  WWW:  http://www.benibela.de

This program is distributed under the terms of the MPL license:

"The contents of this file are subject to the Mozilla Public License
Version 1.1 (the "License"); you may not use this file except in
compliance with the License. You may obtain a copy of the License at
http://www.mozilla.org/MPL/

Software distributed under the License is distributed on an "AS IS"
basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
License for the specific language governing rights and limitations
under the License.

The Original Code is NVCanvas.pas, nvHashClass.pas and NVResMan.pas.

The Initial Developer of the Original Code is Benito van der Zander.
All Rights Reserved.

Alternatively, the contents of this file may be used under the terms
of the Gnu Public License Version 2 or higher license (the  "GPL 2 License"), in which case the
provisions of GPL 2 License are applicable instead of those
above. If you wish to allow use of your version of this file only
under the terms of the GPL 2 License and not to allow others to use
your version of this file under the MPL, indicate your decision by
deleting the provisions above and replace them with the notice and
other provisions required by the GPL 2 License. If you do not delete
the provisions above, a recipient may use your version of this file
under either the MPL or the GPL 2 License."
}
{$DEFINE UseResManager} //Ohne den Ressourcenmanager gehen die Fonts nicht
unit NVCanvas;

interface
uses
  Windows {$IFDEF UseResManager}, NVResMan{$ENDIF} ;

type
  long=dword;
  int=integer;
  {$IFDEF UseResManager}
  TChangeEvent=procedure (sender:TObject) of object;
  TNVGraphicObject=class
   protected
    OnChange,OnChanging:TChangeEvent;
   public
    constructor create;
  end;
  {$ENDIF}
  TNVBrush=class {$IFDEF UseResManager}(TNVGraphicObject){$ENDIF}
  protected
    f_handle: HBRUSH;
    brushInfo:TLogBrush;
    procedure ReCreate;
    procedure SetStyle(const newStyle:UINT);
    function GetStyle():UINT;
    procedure SetColor(const newcolor:COLORREF);
    function GetColor():COLORREF;
    procedure SetHatch(const newHatch:long);
    function GetHatch():long;
  public
    {$IFDEF UseResManager}
    procedure Assign(newBrush:TNVBrush);
    {$ENDIF}

    property Style:UINT read GetStyle write SetStyle;
    property Color:COLORREF read GetColor write SetColor;
    property Hatch:long read GetHatch write SetHatch;
    property Handle:Hbrush read F_handle;

    constructor create;
    destructor destroy;override;
  end;

  TNVPen=class {$IFDEF UseResManager}(TNVGraphicObject){$ENDIF}
  protected
    f_handle: HPen;
    penInfo:TLogPen;
    procedure ReCreate;
    procedure SetStyle(const newStyle:INT);
    function GetStyle():INT;
    procedure SetWidth(const newWidth:int);
    function GetWidth():int;
    procedure SetColor(const newcolor:COLORREF);
    function GetColor():COLORREF;
  public
    {$IFDEF UseResManager}
    procedure Assign(newPen:TNVPen);
    {$ENDIF}

    property Style:INT read GetStyle write SetStyle;
    property Width:int read GetWidth write SetWidth;
    property Color:COLORREF read GetColor write SetColor;
    property Handle:HPen read F_handle;

    constructor create;
    destructor destroy;override;
  end;

  {$IFDEF UseResManager}
  TString32=string[32];
  TNVFontStyle=set of (nvfsItalic,nvfsStrikeOut,nvfsUnderline);
  TNVFont=class (TNVGraphicObject)
  protected
    f_handle: HFONT;
    fontInfo:TLogFont;
    f_color:COLORREF;
    procedure ReCreate;
    function GetHeight:long;
    procedure SetHeight(const value:long);
    function GetWidth:long;
    procedure SetWidth(const value:long);
    function GetAngle:long;
    procedure SetAngle(const value:long);
    function GetWeight:long;
    procedure SetWeight(const value:long);
    function GetStyle:TNVFontStyle;
    procedure SetStyle(const value:TNVFontStyle);
    function GetCharSet:byte;
    procedure SetCharSet(const value:byte);
    function GetOutPrecision:byte;
    procedure SetOutPrecision(const value:byte);
    function GetClipPrecision:byte;
    procedure SetClipPrecision(const value:byte);
    function GetQuality:byte;
    procedure SetQuality(const value:byte);
    function GetPitchAndFamily:byte;
    procedure SetPitchAndFamily(const value:byte);
    function GetFaceName:Tstring32;
    procedure SetFaceName(const value:TString32);

    procedure SetColor(const value:COLORREF);
  public
    procedure Assign(newFont:TNVFont);

    property height:long read GetHeight write SetHeight;
    property Width:long read GetWidth write SetWidth;
    property Angle:long read GetAngle write SetAngle;
    property Weight:long read GetWeight write SetWeight;
    property Style:TNVFontStyle read GetStyle write SetStyle;
    property CharSet:byte read GetCharSet write SetCharSet;
    property OutPrecision:byte read GetOutPrecision write SetOutPrecision;
    property ClipPrecision:byte read GetClipPrecision write SetClipPrecision;
    property Quality:byte read GetQuality write SetQuality;
    property PitchAndFamily:byte read GetPitchAndFamily write SetPitchAndFamily;
    property FaceName:TString32 read GetFaceName write SetFaceName;
    property Color:COLORREF read F_color write SetColor;

    property Handle:HFONT read F_handle;
    constructor create;
    destructor destroy;override;
  end;
  {$ENDIF}
type TNVCanvas=class
  protected
    {$IFDEF UseResManager}
      F_Brush:TNVBrush;
      F_Pen:TNVPen;
      F_Font:TNVFont;
      startfonthandle:HFONT;
    {$ELSE}
      startpen:TNVPen;
      startbrush:TNVBrush;
      aktbrush:TNVBrush;
    {$ENDIF}
    startbrushhandle:HBRUSH;
    startpenhandle:HPEN;
    F_Handle:HDC;
    BgTransparent:boolean;
    procedure SetHandle(const newHandle:HDC);

    procedure SetBgTransparent(const new:boolean);

    procedure SetPenPos(const pos:TPoint);
    function GetPenPos():TPoint;

    {$IFDEF UseResManager}
      procedure OnBrushChange(sender:TObject);
      procedure OnFontChange(sender:TObject);
      procedure OnPenChange(sender:TObject);
      procedure OnBrushChanging(sender:TObject);
      procedure OnFontChanging(sender:TObject);
      procedure OnPenChanging(sender:TObject);

      procedure SetBrush(const newBrush:TNVBrush);
      procedure SetPen(const newPen:TNVPen);
      procedure SetFont(const newFont:TNVFont);
      procedure DeSelectFont();
      procedure DeSelectPen();
      procedure DeSelectBrush();
//      procedure SetBrush(newBrush:TNVBrush);
    {$ENDIF}
  public
    {$IFDEF UseResManager}
    property Brush:TNVBrush read F_Brush write SetBrush;
    property pen:TNVpen read F_pen write Setpen;
    property Font:TNVFont read F_Font write SetFont;
    {$ELSE}
    procedure SelectPen(const pen:TNVPen);   //Selectiert einen Stift
    procedure DeSelectPen(); //Whlt den Standardstift
    procedure SelectBrush(const brush:TNVBrush); //Selectiert einen Hintergrund
    procedure DeSelectBrush(); //Whlt den Standardhintergrund
    {$ENDIF}

    procedure Arc(const X1, Y1, X2, Y2, X3, Y3, X4, Y4: Integer);
    procedure ArcTo(const X1, Y1, X2, Y2, X3, Y3, X4, Y4: Integer);
    procedure Chord(const X1, Y1, X2, Y2, X3, Y3, X4, Y4: Integer);
    procedure Circel(const X, Y, Radius: integer);
    procedure CopyRect(const dest:TRect;const sourcedc:HDC;const source:TRect;const mode:DWORD); //right=width!!!!!!!!!!
    procedure DrawFocusRect(const Rect: TRect);
    procedure DrawText(const str:string;const lpRect:TRect;const uFormat:UINT);
    procedure DrawTextEx(const str:string;const lpRect:TRect;const uFormat:UINT;var lpDTParams:TDrawTextParams);
    procedure Ellipse(const X1, Y1, X2, Y2: integer);
    procedure FillRect(const Rect: TRect);
    procedure FloodFill(const X, Y: Integer; const Color: ColorRef; const FillTyp: integer);
    procedure FrameRect(const Rect: TRect);
    procedure LineTo(const X, Y: Integer);
    procedure MoveTo(const X, Y: Integer);
    procedure Pie(const X1, Y1, X2, Y2, X3, Y3, X4, Y4: Longint);
    procedure PolyBezier(const Points: array of TPoint);
    procedure PolyBezierTo(const Points: array of TPoint);
    procedure PolyDraw(const Points:array of TPoint;const Types:array of byte);
    procedure Polygon(const Points: array of TPoint);
    procedure Polyline(const Points: array of TPoint);
    procedure Rectangle(const x1,y1,x2,y2:integer);
    procedure RoundRect(const X1, Y1, X2, Y2, X3, Y3: Integer);
    function TextExtent(const Text: string): TSize;
    procedure TextOut(const X, Y: Integer; const Text: string);
    procedure TextRect(const Rect: TRect; const X, Y: Integer; const Text: string);
    function TextHeight(const Text: string): Integer;
    function TextWidth(const Text: string): Integer;

    constructor create; virtual;
    destructor destroy;override;

    property Handle:HDC read F_Handle write SetHandle;
    property PenPos:TPoint read GetPenPos write SetPenPos;
    property BackgroundTransparent:boolean read BgTransparent write SetBgTransparent;
end;
implementation
{$IFDEF UseResManager}
  var penResMan,brushResMan,fontResMan:TNVRessourceManager;

constructor TNVGraphicObject.create;
begin
  inherited;
  OnChange:=nil;
  OnChanging:=nil;
end;
{$ENDIF}
{============================== TNVBrush =======================================}
procedure TNVBrush.ReCreate;
begin
  {$IFDEF UseResManager}
    brushResMan.DeleteRes(f_handle);
    f_handle:=CreateBrushIndirect(brushInfo);
    brushResMan.GetRes(f_handle);
  {$ELSE}
    DeleteObject(f_handle);
    f_handle:=CreateBrushIndirect(brushInfo);
  {$ENDIF}
end;

procedure TNVBrush.SetStyle(const newStyle:UINT);
begin
  {$IFDEF UseResManager}if Assigned(OnChanging) then OnChanging(self);{$ENDIF}
  brushInfo.lbStyle:=newStyle;
  ReCreate;
  {$IFDEF UseResManager}if Assigned(OnChange) then OnChange(self);{$ENDIF}
end;
function TNVBrush.GetStyle():UINT; begin  Result:=brushInfo.lbStyle; end;

procedure TNVBrush.SetColor(const newcolor:COLORREF);
begin
  {$IFDEF UseResManager}if Assigned(OnChanging) then OnChanging(self);{$ENDIF}
  brushInfo.lbColor:=newcolor;
  ReCreate;
  {$IFDEF UseResManager}if Assigned(OnChange) then OnChange(self);{$ENDIF}
end;
function TNVBrush.GetColor():COLORREF; begin  Result:=brushInfo.lbColor; end;

procedure TNVBrush.SetHatch(const newHatch:long);
begin
  {$IFDEF UseResManager}if Assigned(OnChanging) then OnChanging(self);{$ENDIF}
  brushInfo.lbColor:=newHatch;
  ReCreate;
  {$IFDEF UseResManager}if Assigned(OnChange) then OnChange(self);{$ENDIF}
end;
function TNVBrush.GetHatch():long;begin Result:=brushInfo.lbHatch; end;

{$IFDEF UseResManager}
procedure TNVBrush.Assign(newBrush:TNVBrush);
begin
  {$IFDEF UseResManager}if Assigned(OnChanging) then OnChanging(self);{$ENDIF}
  brushResMan.DeleteRes(self.f_handle);
  brushResMan.GetRes(newBrush.f_handle);
  self.f_handle:=newBrush.f_handle;
  self.brushInfo:=newBrush.brushInfo;
  {$IFDEF UseResManager}if Assigned(OnChange) then OnChange(self);{$ENDIF}
end;
{$ENDIF}

constructor TNVBrush.create;
begin
  inherited;
  brushInfo.lbStyle:=BS_SOLID;
  brushInfo.lbColor:=0;
  brushInfo.lbHatch:=0;
  f_handle:=CreateBrushIndirect(brushInfo);
  {$IFDEF UseResManager}
    brushResMan.GetRes(f_handle);
  {$ENDIF}
end;
destructor TNVBrush.destroy;
begin
  {$IFDEF UseResManager}
    brushResMan.DeleteRes(f_handle);
  {$ELSE}
    DeleteObject(f_handle);
  {$ENDIF}
  inherited;
end;
//----------------------------- TNVBrush -------------------------------------\\


{============================== TNVPen =======================================}
procedure TNVPen.ReCreate;
begin
  {$IFDEF UseResManager}
    penResMan.DeleteRes(f_handle);
    f_handle:=CreatePenIndirect(penInfo);
    penResMan.GetRes(f_handle);
  {$ELSE}
    DeleteObject(f_handle);
    f_handle:=CreatePenIndirect(PenInfo);
  {$ENDIF}
end;

procedure TNVPen.SetStyle(const newStyle:INT);
begin
  {$IFDEF UseResManager}if Assigned(OnChanging) then OnChanging(self);{$ENDIF}
  PenInfo.lopnStyle:=newStyle;
  ReCreate;
  {$IFDEF UseResManager}if Assigned(OnChange) then OnChange(self);{$ENDIF}
end;
function TNVPen.GetStyle():INT; begin  Result:=PenInfo.lopnStyle; end;

procedure TNVPen.Setwidth(const newwidth:int);
begin
  {$IFDEF UseResManager}if Assigned(OnChanging) then OnChanging(self);{$ENDIF}
  PenInfo.lopnWidth.x:=newwidth;
  ReCreate;
  {$IFDEF UseResManager}if Assigned(OnChange) then OnChange(self);{$ENDIF}
end;
function TNVPen.Getwidth():int;begin Result:=PenInfo.lopnWidth.x; end;

procedure TNVPen.SetColor(const newcolor:COLORREF);
begin
  {$IFDEF UseResManager}if Assigned(OnChanging) then OnChanging(self);{$ENDIF}
  PenInfo.lopnColor:=newcolor;
  ReCreate;
  {$IFDEF UseResManager}if Assigned(OnChange) then OnChange(self);{$ENDIF}
end;
function TNVPen.GetColor():COLORREF; begin  Result:=PenInfo.lopnColor; end;

{$IFDEF UseResManager}
procedure TNVpen.Assign(newpen:TNVpen);
begin
  {$IFDEF UseResManager}if Assigned(OnChanging) then OnChanging(self);{$ENDIF}
  penResMan.DeleteRes(self.f_handle);
  penResMan.GetRes(newpen.f_handle);
  self.f_handle:=newpen.f_handle;
  self.penInfo:=newpen.penInfo;
  {$IFDEF UseResManager}if Assigned(OnChange) then OnChange(self);{$ENDIF}
end;
{$ENDIF}

constructor TNVPen.create;
begin
  inherited;
  PenInfo.lopnStyle:=BS_SOLID;
  PenInfo.lopnWidth.x:=0;
  PenInfo.lopnColor:=0;
  f_handle:=CreatePenIndirect(PenInfo);
  {$IFDEF UseResManager}
    penResMan.GetRes(f_handle);
  {$ENDIF}
end;
destructor TNVPen.destroy;
begin
  {$IFDEF UseResManager}
    penResMan.DeleteRes(f_handle);
  {$ELSE}
    DeleteObject(f_handle);
  {$ENDIF}
  inherited;
end;
//------------------------------- TNVPen -------------------------------------\\

{================================= TNVFont =====================================}
{$IFDEF UseResManager}
procedure TNVFont.ReCreate;
begin
    fontResMan.DeleteRes(f_handle);
    f_handle:=CreateFontIndirect(FontInfo);
    FontResMan.GetRes(f_handle);
end;
function TNVFont.GetHeight:long;
begin
  result:=fontInfo.lfHeight;
end;
procedure TNVFont.SetHeight(const value:long);
begin
  if Assigned(OnChanging) then OnChanging(self);
  fontInfo.lfHeight:=value;
  ReCreate;
  if Assigned(OnChange) then OnChange(self);
end;
function TNVFont.GetWidth:long;
begin
  result:=fontInfo.lfWidth;
end;
procedure TNVFont.SetWidth(const value:long);
begin
  if Assigned(OnChanging) then OnChanging(self);
  fontInfo.lfWidth:=value;
  ReCreate;
  if Assigned(OnChange) then OnChange(self);
end;
function TNVFont.GetAngle:long;
begin
  result:=fontInfo.lfEscapement;
end;
procedure TNVFont.SetAngle(const value:long);
begin
  if Assigned(OnChanging) then OnChanging(self);
  fontInfo.lfEscapement:=value;
  fontInfo.lfOrientation:=value;
  ReCreate;
  if Assigned(OnChange) then OnChange(self);
end;
function TNVFont.GetWeight:long;
begin
  result:=fontInfo.lfWeight;
end;
procedure TNVFont.SetWeight(const value:long);
begin
  if Assigned(OnChanging) then OnChanging(self);
  fontInfo.lfWeight:=value;
  ReCreate;
  if Assigned(OnChange) then OnChange(self);
end;
function TNVFont.GetStyle:TNVFontStyle;
begin
  result:=[];
  if fontInfo.lfItalic<>0 then include(result,nvfsItalic);
  if fontInfo.lfUnderline<>0 then include(result,nvfsUnderline);
  if fontInfo.lfStrikeOut<>0 then include(result,nvfsStrikeOut);
end;
procedure TNVFont.SetStyle(const value:TNVFontStyle);
begin
  if Assigned(OnChanging) then OnChanging(self);

  fontInfo.lfItalic:=byte(nvfsItalic in value);
  fontInfo.lfUnderline:=byte(nvfsUnderline in value);
  fontInfo.lfStrikeOut:=byte(nvfsStrikeOut in value);
  ReCreate;
  if Assigned(OnChange) then OnChange(self);
end;
function TNVFont.GetCharSet:byte;
begin
  result:=fontInfo.lfCharSet;
end;
procedure TNVFont.SetCharSet(const value:byte);
begin
  if Assigned(OnChanging) then OnChanging(self);
  fontInfo.lfCharSet:=value;
  ReCreate;
  if Assigned(OnChange) then OnChange(self);
end;
function TNVFont.GetOutPrecision:byte;
begin
  result:=fontInfo.lfOutPrecision;
end;
procedure TNVFont.SetOutPrecision(const value:byte);
begin
  if Assigned(OnChanging) then OnChanging(self);
  fontInfo.lfOutPrecision:=value;
  ReCreate;
  if Assigned(OnChange) then OnChange(self);
end;
function TNVFont.GetClipPrecision:byte;
begin
  result:=fontInfo.lfClipPrecision;
end;
procedure TNVFont.SetClipPrecision(const value:byte);
begin
  if Assigned(OnChanging) then OnChanging(self);
  fontInfo.lfClipPrecision:=value;
  ReCreate;
  if Assigned(OnChange) then OnChange(self);
end;
function TNVFont.GetQuality:byte;
begin
  result:=fontInfo.lfQuality;
end;
procedure TNVFont.SetQuality(const value:byte);
begin
  if Assigned(OnChanging) then OnChanging(self);
  fontInfo.lfQuality:=value;
  ReCreate;
  if Assigned(OnChange) then OnChange(self);
end;
function TNVFont.GetPitchAndFamily:byte;
begin
  result:=fontInfo.lfPitchAndFamily;
end;
procedure TNVFont.SetPitchAndFamily(const value:byte);
begin
  if Assigned(OnChanging) then OnChanging(self);
  fontInfo.lfPitchAndFamily:=value;
  ReCreate;
  if Assigned(OnChange) then OnChange(self);
end;
function TNVFont.GetFaceName:Tstring32;
begin
  result:=fontInfo.lfFaceName;
end;
procedure TNVFont.SetFaceName(const value:TString32);
begin
  if Assigned(OnChanging) then OnChanging(self);
  move(value[1],fontInfo.lfFaceName[0],32);
  ReCreate;
  if Assigned(OnChange) then OnChange(self);
end;
procedure TNVFont.SetColor(const value:COLORREF);
begin
  if Assigned(OnChanging) then OnChanging(self);
  f_color:=value;
  if Assigned(OnChange) then OnChange(self);
end;


procedure TNVFont.Assign(newFont:TNVFont);
begin
  if Assigned(OnChanging) then OnChanging(self);
  fontResMan.DeleteRes(self.f_handle);
  self.f_handle:=newFont.f_handle;
  self.fontInfo:=newFont.fontInfo;
  self.f_color:=newFont.f_color;
  fontResMan.GetRes(f_handle);
  if Assigned(OnChange) then OnChange(self);
end;


constructor TNVFont.create;
begin
  inherited;
  //handle:=GetStockObject(SYSTEM_FONT);
  GetObject(GetStockObject(SYSTEM_FONT){handle},sizeof(fontInfo),@fontInfo);
  fontInfo.lfFaceName:='Arial'#0;
  f_handle:=CreateFontIndirect(fontInfo);
  FontResMan.GetRes(f_handle);

end;
destructor TNVFont.destroy;
begin
  fontResMan.DeleteRes(f_handle);
  inherited;
end;
{$ENDIF}
//------------------------------- TNVFont ------------------------------------\\

{=============================== TNVCanvas ====================================}

procedure TNVCanvas.SetBgTransparent(const new:boolean);
begin
  if BgTransparent<>new then begin
    BgTransparent:=new;
    if BgTransparent then SetBkMode(handle,TRANSPARENT)
    else SetBkMode(handle,OPAQUE);
  end;
end;

procedure TNVCanvas.SetPenPos(const pos:TPoint);
begin
  windows.MoveToEx(handle,pos.x,pos.y,nil);
end;
function TNVCanvas.GetPenPos():TPoint;
begin
  GetCurrentPositionEx(handle,@result);
end;

procedure TNVCanvas.SetHandle(const newHandle:HDC);
begin
  {$IFDEF UseResManager}
    if handle<>0 then begin
      SelectObject(Handle,startFontHandle);
      SelectObject(Handle,startBrushHandle);
      SelectObject(Handle,startPenHandle);
    end;
    f_handle:=newHandle;
    startpenhandle:=SelectObject(Handle,F_pen.f_handle);
    startbrushhandle:=SelectObject(Handle,F_brush.f_handle);
    startfonthandle:=SelectObject(Handle,F_font.f_handle);
  {$ELSE}
    if handle<>0 then begin
      SelectObject(Handle,startBrushHandle);
      SelectObject(Handle,startPenHandle);
    end;
    f_handle:=newHandle;
    startpenhandle:=SelectObject(Handle,startPen.f_handle);
    startbrushhandle:=SelectObject(Handle,startBrush.f_handle);

  {$ENDIF}
end;

{$IFDEF UseResManager}
procedure TNVCanvas.OnBrushChange(sender:TObject);
begin
  SelectObject(handle,brush.f_handle);
  SetBkColor(handle,brush.Color);
end;
procedure TNVCanvas.OnFontChange(sender:TObject);
begin
  SelectObject(handle,f_font.f_handle);
  SetTextColor(handle,f_font.f_color);
end;
procedure TNVCanvas.OnPenChange(sender:TObject);
begin
  SelectObject(handle,pen.f_handle);
end;
procedure TNVCanvas.OnBrushChanging(sender:TObject);
begin
  DeSelectBrush;
end;
procedure TNVCanvas.OnFontChanging(sender:TObject);
begin
  DeSelectFont;
end;
procedure TNVCanvas.OnPenChanging(sender:TObject);
begin
  DeSelectPen;
end;
procedure TNVCanvas.SetBrush(const newBrush:TNVBrush);
begin
  F_Brush.Assign(newBrush);
end;
procedure TNVCanvas.SetPen(const newPen:TNVPen);
begin
  F_Pen.Assign(newPen);
end;
procedure TNVCanvas.SetFont(const newFont:TNVFont);
begin
  F_Font.Assign(newFont);
end;
procedure TNVCanvas.DeSelectFont();
begin
  SelectObject(Handle,startfonthandle);
end;
{$ELSE}
procedure TNVCanvas.SelectPen(const pen:TNVPen);
begin
  SelectObject(Handle,pen.f_handle);
  setTextColor(pen.color);
end;
procedure TNVCanvas.SelectBrush(const brush:TNVBrush);
begin
  aktbrush:=brush;
  SelectObject(Handle,brush.f_handle);
end;
{$ENDIF}
procedure TNVCanvas.DeSelectPen();
begin
  {$IFDEF UseResManager}
  SelectObject(Handle,startpenhandle);
  {$ELSE}
  SelectObject(Handle,startpen.f_handle);
  {$ENDIF}
end;
procedure TNVCanvas.DeSelectBrush();
begin
  {$IFDEF UseResManager}
  SelectObject(Handle,startbrushhandle);
  {$ELSE}
  aktbrush:=nil;
  SelectObject(Handle,startbrush.f_handle);
  {$ENDIF}
end;


procedure TNVCanvas.Arc(const X1, Y1, X2, Y2, X3, Y3, X4, Y4: Integer);
begin
  windows.Arc(handle,x1,y1,x2,y2,x3,y3,x4,y4);
end;
procedure TNVCanvas.ArcTo(const X1, Y1, X2, Y2, X3, Y3, X4, Y4: Integer);
begin
  windows.ArcTo(handle,x1,y1,x2,y2,x3,y3,x4,y4);
end;
procedure TNVCanvas.Chord(const X1, Y1, X2, Y2, X3, Y3, X4, Y4: Integer);
begin
  windows.chord(handle,x1,y1,x2,y2,x3,y3,x4,y4);
end;
procedure TNVCanvas.Circel(const X, Y, Radius: integer);
begin
  windows.ellipse(handle,x-Radius,y-Radius,x+Radius,y+Radius);
end;
procedure TNVCanvas.CopyRect(const dest:TRect;const sourcedc:HDC;const source:TRect;const mode:DWORD);
begin
  windows.StretchBlt(handle,dest.left,dest.top,dest.Right,dest.bottom,sourcedc,source.left,source.top,source.right,source.bottom,mode);
end;
procedure TNVCanvas.DrawFocusRect(const Rect: TRect);
begin
 windows.drawFocusRect(handle,rect);
end;
procedure TNVCanvas.DrawText(const str:string;const lpRect:TRect;const uFormat:UINT);
begin
 windows.drawtext(handle,pchar(str),length(str),PRect(@lpRect)^,uFormat); //const -> var (sehr gefhrlich, aber drawText verndert das rect ohne entsprechenden Parameter nicht)
end;
procedure TNVCanvas.DrawTextEx(const str:string;const lpRect:TRect;const uFormat:UINT;var lpDTParams:TDrawTextParams);
begin
 lpDTParams.cbSize:=sizeof(lpDTParams);
 windows.drawtextEx(handle,pchar(str),lpDTParams.uiLengthDrawn,PRect(@lpRect)^,uFormat,@lpDTParams); //const -> var (sehr gefhrlich, aber drawText verndert es anscheinend nicht)
end;
procedure TNVCanvas.Ellipse(const X1, Y1, X2, Y2: integer);
begin
 windows.Ellipse(handle,x1,y1,x2,y2);
end;
procedure TNVCanvas.FillRect(const Rect: TRect);
begin
{$IFDEF UseResManager}
 windows.FillRect(handle,rect,f_brush.f_handle);
{$ELSE}
 windows.FillRect(handle,rect,aktbrush.f_handle);
{$ENDIF}
end;
procedure TNVCanvas.FloodFill(const X, Y: Integer; const Color: ColorRef; const FillTyp: integer);
begin
 windows.ExtFloodFill(handle,x,y,color,fillTyp);
end;
procedure TNVCanvas.FrameRect(const Rect: TRect);
begin
{$IFDEF UseResManager}
 windows.FrameRect(handle,rect,f_brush.f_handle);
{$ELSE}
 windows.FrameRect(handle,rect,aktbrush.f_handle);
{$ENDIF}
end;
procedure TNVCanvas.LineTo(const X, Y: Integer);
begin
 windows.LineTo(handle,x,y);
end;
procedure TNVCanvas.MoveTo(const X, Y: Integer);
begin
 windows.MoveToEx(handle,x,y,nil);
end;
procedure TNVCanvas.Pie(const X1, Y1, X2, Y2, X3, Y3, X4, Y4: Longint);
begin
 windows.pie(handle,X1, Y1, X2, Y2, X3, Y3, X4, Y4);
end;
procedure TNVCanvas.PolyBezier(const Points: array of TPoint);
begin
 windows.PolyBezier(handle,points[0],length(points));
end;
procedure TNVCanvas.PolyBezierTo(const Points: array of TPoint);
begin
 windows.PolyBezierTo(handle,points[0],length(points));
end;
procedure TNVCanvas.PolyDraw(const Points:array of TPoint;const Types:array of byte);
begin
 windows.PolyDraw(handle,points,Types,length(points));
end;
procedure TNVCanvas.Polygon(const Points: array of TPoint);
begin
 windows.Polygon(handle,PPoint(@points[0])^,length(points)); //gefhrlich, ist aber laut PSDK const
end;
procedure TNVCanvas.Polyline(const Points: array of TPoint);
begin
 windows.PolyLine(handle,PPoint(@points[0])^,length(points)); //gefhrlich, ist aber laut PSDK const
end;
procedure TNVCanvas.Rectangle(const x1,y1,x2,y2:integer);
begin
 windows.Rectangle(handle,x1,y1,x2,y2);
end;
procedure TNVCanvas.RoundRect(const X1, Y1, X2, Y2, X3, Y3: Integer);
begin
 windows.RoundRect(handle,x1,y1,x2,y2,x3,y3);
end;
function TNVCanvas.TextExtent(const Text: string): TSize;
begin
 windows.GetTextExtentPoint(handle,pchar(text),length(text),result);
end;
procedure TNVCanvas.TextOut(const X, Y: Integer; const Text: string);
begin
 windows.TextOut(handle,x,y,pchar(text),length(text));
end;
procedure TNVCanvas.TextRect(const Rect: TRect; const X, Y: Integer; const Text: string);
begin
 windows.DrawText(handle,pchar(text),length(text),PRect(@Rect)^,DT_LEFT);
end;
function TNVCanvas.TextHeight(const Text: string): Integer;
begin
  result:=Textextent(text).cy;
end;
function TNVCanvas.TextWidth(const Text: string): Integer;
begin
  result:=Textextent(text).cx;
end;

constructor TNVCanvas.create;
begin
{$IFDEF UseResManager}
  f_pen:=TNVPen.create;
  F_Pen.OnChange:=OnPenChange;
  F_Pen.OnChanging:=OnPenChanging;
  F_brush:=TNVBrush.create;
  f_brush.OnChange:=OnBrushChange;
  f_brush.OnChanging:=OnBrushChanging;
  F_Font:=TNVFont.create;
  F_Font.OnChange:=OnFontChange;
  F_Font.OnChanging:=OnFontChanging;
{$ELSE}
  startpen:=TNVPen.create;
  startbrush:=TNVBrush.create;
  aktbrush:=startbrush;
{$ENDIF}
handle:=0;
end;
destructor TNVCanvas.destroy;
begin
  if handle<>0 then begin
    {$IFDEF UseResManager}
    SelectObject(Handle,startfontHandle);
    {$ENDIF}
    SelectObject(Handle,startBrushHandle);
    SelectObject(Handle,startPenHandle);
  end;
{$IFDEF UseResManager}
  f_pen.free;
  f_brush.free;
  F_Font.free;
{$ELSE}
  startbrush.free;
  startpen.free;
{$ENDIF}
end;
//----------------------------- TNVCanvas ------------------------------------\\


{$IFDEF UseResManager}
  type TGraphicResVerwaltung = class
    class procedure OnDeleteObject(const id:THandle);
  end;

  class procedure TGraphicResVerwaltung.OnDeleteObject(const id:THandle);
  begin
    DeleteObject(id);
  end;
{$ENDIF}
initialization
{$IFDEF UseResManager}
  penResMan:=TNVRessourceManager.create;
  penResMan.OnDelete:=TGraphicResverwaltung.OnDeleteObject;
  brushResMan:=TNVRessourceManager.create;
  brushResMan.OnDelete:=TGraphicResverwaltung.OnDeleteObject;
  fontResMan:=TNVRessourceManager.create;
  fontResMan.OnDelete:=TGraphicResverwaltung.OnDeleteObject;
{$ENDIF}
finalization
{$IFDEF UseResManager}
  fontResMan.free;
  brushResMan.free;
  penResMan.free;
{$ENDIF}
end.





