/*
 * Copyright (C) 1997 Red Hat Software, Inc.
 *
 * Modify:  2015/11/26  Tommy Scheunemann <net@arrishq.net>
 *
 * Modify:  2015/04/16  Tommy Scheunemann <net@arrishq.net>
 *
 * Modify:  2015/01/01  Tommy Scheunemann <net@arrishq.net>
 *
 * Modify:  2014/07/29  Tommy Scheunemann <net@arrishq.net>
 *
 * Modify:  2014/05/04  Tommy Scheunemann <net@arrishq.net>
 *
 * Modify:  2014/01/07  Tommy Scheunemann <net@arrishq.net>
 *
 * Modify:  2013/12/02  Tommy Scheunemann <net@arrishq.net>
 *
 * Modify:  2013/10/30  Tommy Scheunemann <net@arrishq.net>
 *
 * Modify:  2013/09/24  Tommy Scheunemann <net@arrishq.net>
 *
 * Modify:  2013/08/07  Tommy Scheunemann <net@arrishq.net>
 *
 * Modify:  2013/03/19  Tommy Scheunemann <net@arrishq.net>
 *
 * Modify:  2013/06/01  Tommy Scheunemann <net@arrishq.net>
 *
 * Modify:  2012/12/18  Tommy Scheunemann <net@arrishq.net>
 *
 * Modify:  2010/12/27  Tommy Scheunemann <net@arrishq.net>
 *
 * Modify:  2008/21/01  Tommy Scheunemann <net@arrishq.net>
 *
 * Modify:  2006/24/10  Tommy Scheunemann <net@arrishq.net>
 *
 * Modify:  2006/19/10  Tommy Scheunemann <net@arrishq.net>
 *
 * Modify:  2005/09/11  Tommy Scheunemann <net@arrishq.net>
 *
 * Modify:  2005/01/11  Tommy Scheunemann <net@arrishq.net>
 *
 * Modify:  2005/28/01  Tommy Scheunemann <net@arrishq.net>
 *
 * Modify:  2004/10/03  Tommy Scheunemann <net@arrishq.net>
 *
 * Modify:  2004/10/02  Tommy Scheunemann <net@arrishq.net>
 *
 * Modify:  2004/09/15  Tommy Scheunemann <net@arrishq.net>
 *
 * Modify:  2004/08/10  Tommy Scheunemann <net@arrishq.net>
 *
 * Modify:  2004/08/09  Tommy Scheunemann <net@arrishq.net>
 *
 * Modify:  2004/08/07  Tommy Scheunemann <net@arrishq.net>
 *
 * Modify:  2004/08/04  Romildo <romildo@uber.com.br>
 *
 * Modify:  2004/07/16  Tommy Scheunemann <net@arrishq.net>
 *
 * Modify:  2004/07/10  Tommy Scheunemann <net@arrishq.net>
 *
 * Modify:  2004/06/27  Tommy Scheunemann <net@arrishq.net>
 *
 * Modify:  2004/06/23  Tommy Scheunemann <net@arrishq.net>
 *
 * Modify:  2004/06/03  Tommy Scheunemann <net@arrishq.net>
 *
 * Modify:  2004/05/16  Tommy Scheunemann <net@arrishq.net>
 *
 * Modify:  2003/09/14  Tommy Scheunemann <net@arrishq.org>
 *
 * Modify:  2003/07/12  Tommy Scheunemann <net@arrishq.org>
 *
 * Modify:  2003/06/25  Tommy Scheunemann <net@arrishq.org>
 *
 * Modify:  2003/04/02  Tommy Scheunemann <net@arrishq.org>
 *
 * Modify:  2002/11/10  Tommy Scheunemann <net@arrishq.org>
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * WMCONFIG.C
 *
 * Core of Wmconfig
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <getopt.h>
#include <string.h>
#include <strings.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <ctype.h>
#include <dirent.h>
#include <pwd.h>
#include <locale.h>
#include <config.h>

/* This is required for BSD and Cygwin */

#if (defined(__unix__) || defined(unix)) && !defined(USG)
    #include <limits.h>
    #include <sys/param.h>
#endif

static const char *ProgName = NULL;

/* Define which Window managers we support */

#define DO_FVWM2 1
#define DO_AFTERSTEP 2
#define DO_MWM 3
#define DO_ICEWM 4
#define DO_WMAKER 5
#define DO_KDE 6
#define DO_DEBUG 7
#define DO_BLACKBOX 8
#define DO_TWM 9
#define DO_OLWM 10
#define DO_FVWM 11
#define DO_MLVWM 12
#define DO_KDE2 13
#define DO_QVWM 14
#define DO_PEKWM 15
#define DO_GOLEM 16
#define DO_OPENBOX 17
#define DO_AEWM 18
#define DO_AMIWM 19
#define DO_KAHAKAI 20
#define DO_PWM 21
#define DO_SAPPHIRE 22
#define DO_ENLIGHTENMENT 23
#define DO_EPIWM 24
#define DO_EQUINOX 25
#define DO_UDE 26
#define DO_WMX 27
#define DO_FREEDESKTOP 28
#define DO_JWM 29
#define DO_SAWFISH 30

static int output_type = 0;
unsigned int flags = 0;

#include "wmconfig.h"

struct group *root_group = NULL;
extern struct package *packages;
extern int num_packages;
static const char *sys_confdir = NULL;
static const char *user_confdir = NULL;
const char *root_menu_name = NULL;
const char *output_directory = NULL;
static const char *use_term = NULL;
int freedesktop_type = 0;
int twm_type = 0;
int aewm_type = 0;

/* i18n support, if requested */

static void init_i18n (void)
{
#ifdef ENABLE_NLS
    setlocale (LC_ALL, "");
    bindtextdomain (PACKAGE, LOCALEDIR);
    textdomain (PACKAGE);
#endif
}

/*
Generic help function
*/

static void usage(void)
{
    /* the gettext output may contain translated strings */
    printf (gettext(	"Usage: \t%s [options] [file ...]\nwhere options can be:\n"), ProgName);
    printf (		"  --output\tdebug | fvwm95 | fvwm2 | afterstep | mwm |\n"
			"\t\ticewm | blackbox | wmaker | twm | olwm | olvwm | fvwm |\n"
			"\t\tmlvwm | kde1 | kde2 | kde3 | kde4 | qvwm | pekwm | golem\n"
			"\t\tfluxbox | ctwm | gnome | openbox | aewm | amiwm\n"
			"\t\tkahakai | pwm | sapphire | e16 | e17 | e19 | enlightenment\n"
			"\t\tepiwm | ede | equinox | ude | xfce | wmx | flwm | tvtwm\n"
			"\t\tvtwm | piewm | freedesktop | lxde | jwm | sithwm\n"
			"\t\tsawfish\n");
    printf (gettext(	"\t\t\t(gives the type of output; default is debug)\n"));
    printf (gettext(	"  --sysdir\tsys_config_dir\t\t(default: \"%s\")\n"),		DEFAULT_SYS_CONFDIR?DEFAULT_SYS_CONFDIR:"");
    printf (gettext(	"  --userdir\tuser_config_dir\t\t(default: \"$HOME/%s\")\n"),	DEFAULT_USER_CONFDIR?DEFAULT_USER_CONFDIR:"");
    printf (gettext(	"  --rootmenu\troot_menu_name\t\t(default: \"%s\")\n"),		DEFAULT_ROOT_NAME?DEFAULT_ROOT_NAME:"");
    printf (gettext(	"  --outputdir\toutput_directory\t(default: \"%s\")\n"),	DEFAULT_OUTPUTDIR?"$HOME/"DEFAULT_OUTPUTDIR:"");
    printf (		"  --flag\tno-icons,no-mini-icons,promote,\n");
    printf (gettext(	"\t\tdirectories,newstyle-directories\t(no defaults)\n"));
    printf (		"  --no-check-existence\n");
    printf (		"  --no-icons\n");
    printf (		"  --no-mini-icons\n");
    printf (		"  --directories\n");
    printf (		"  --newstyle-directories\n");
    printf (		"  --promote\n");
    printf (		"  --no-sysdir\n");
    printf (		"  --no-userdir\n");
    printf (gettext(	"  --terminal\t(default: %s)\n"), TERMINAL?TERMINAL:"");
    printf (		"  --help\n");
    printf (		"  --version\n");
    exit(-1);
}

/* Version output function */
static void version(void)
{
    printf("%s version %s\n", ProgName, VERSION);
    exit(0);
}

/* Read and parse given options */
static int parse_options(int argc, char **argv)
{
    int option_index = 0;
    int c;
    static int val = 0;
    /* which options we have, which are required, which are optional */
    static struct option options[] = {
	{ "output",		required_argument, 0, 'o' },
	{ "sysdir",		required_argument, 0, 's' },
	{ "userdir",		required_argument, 0, 'u' },
	{ "rootmenu",		required_argument, 0, 'r' },
	{ "flags",		required_argument, 0, 'f' },
	{ "help",		no_argument, 0, 'h' },
	{ "version",		no_argument, 0, 'v' },
	{ "promote",		no_argument, &val, FLAG_PROMOTE },
	{ "directories",	no_argument, &val, FLAG_DIRECTORIES },
	{ "newstyle-directories", no_argument, &val, FLAG_NEWSTYLE_DIRECTORIES },
	{ "no-icons",		no_argument, &val, FLAG_NO_ICONS },
	{ "no-check-existence",	no_argument, &val, FLAG_NO_CHECK_EXISTENCE },
	{ "no-mini-icons",	no_argument, &val, FLAG_NO_MINI_ICONS },
	{ "no-sysdir",		no_argument, &val, FLAG_NO_SYSDIR },
	{ "no-userdir",		no_argument, &val, FLAG_NO_USERDIR },
	{ "outputdir",		required_argument, 0, 'd'},
	{ "terminal",		required_argument, 0, 't' },
	{0, 0, 0, 0 },
    };

    /* parse the given options */
    while ((c = getopt_long(argc, argv, "o:s:u:hr:f:d:t:", options, &option_index)) != EOF) {
	switch (c) {
	    case 0:
		flags |= val;
		val = 0;
	    break;
	    case 'o':
		if (optarg == NULL) {
		    usage();
		    break;
		}

		/* The output options is case insensitive
		   Basically just define the output_type that
		   gets read later and calls the specific output */

		/* Fvwm output requested */
		if (strncasecmp(optarg, "fvwm", 4) == 0) {
		    output_type = DO_FVWM;
		}

		/* Fvwm2 or Fvwm95 output requested */
		if ((strncasecmp(optarg, "fvwm2", 5) == 0) || (strncasecmp(optarg, "fvwm95", 6) == 0)) {
		    output_type = DO_FVWM2;
		}

		/* Afterstep output requested */
		if (strncasecmp(optarg, "afterstep", 9) == 0) {
		    output_type = DO_AFTERSTEP;
		}

		/* Mwm output requested */
		if (strncasecmp(optarg, "mwm", 3) == 0) {
		    output_type = DO_MWM;
		}

		/* Windowmaker output requested */
		if (strncasecmp(optarg, "wmaker", 6) == 0) {
		    output_type = DO_WMAKER;
		}

		/* Icewm output requested */
		if (strncasecmp(optarg, "icewm", 5) == 0) {
		    output_type = DO_ICEWM;
		}

		/* Fluxbox or Blackbox output requested */
		if ((strncasecmp(optarg, "fluxbox", 7) == 0) || (strncasecmp(optarg, "blackbox", 8) == 0)) {
		    output_type = DO_BLACKBOX;
		}

		/* KDE1 output requested */
		if (strncasecmp(optarg, "kde1", 4) == 0) {
		    output_type = DO_KDE;
		}

		/* Debug output - something special */
		if (strncasecmp(optarg, "debug", 5) == 0) {
		    output_type = DO_DEBUG;
		}

		/* Twm output requested */
		if (strncasecmp(optarg, "twm", 3) == 0) {
		    twm_type = 0;
		    output_type = DO_TWM;
		}

		/* Tvtwm or Vtwm output requested, sub-output of Twm */
		if ((strncasecmp(optarg, "tvtwm", 5) == 0) || (strncasecmp(optarg, "vtwm", 4) == 0)) {
		    twm_type = 1;
		    output_type = DO_TWM;
		}

		/* Piewm output requested, sub-output of Twm */
		if (strncasecmp(optarg, "piewm", 5) == 0) {
		    twm_type = 2;
		    output_type = DO_TWM;
		}

		/* Ctwm output requested, sub-output of Twm */
		if (strncasecmp(optarg, "ctwm", 4) == 0) {
		    twm_type = 3;
		    output_type = DO_TWM;
		}

		/* Olwm or Olvwm output requested */
		if ((strncasecmp(optarg, "olwm", 4) == 0) || (strncasecmp(optarg, "olvwm", 5) == 0)) {
		    output_type = DO_OLWM;
		}

		/* Mlvwm output requested */
		if (strncasecmp(optarg, "mlvwm", 5) == 0) {
		    output_type = DO_MLVWM;
		}

		/* KDE2 or KDE3 output requested */
		if ((strncasecmp(optarg, "kde2", 4) == 0) || (strncasecmp(optarg, "kde3", 4) == 0)) {
		    output_type = DO_KDE2;
		}

		/* KDE4 (freedesktop like) output requested */
		if (strncasecmp(optarg, "kde4", 4) == 0) {
		    /* Freedesktop Output - define Type for showin parameter */
		    freedesktop_type = 1;
		    output_type = DO_FREEDESKTOP;
		}

		/* Qvwm output requested */
		if (strncasecmp(optarg, "qvwm", 4) == 0) {
		    output_type = DO_QVWM;
		}

		/* Pekwm output requested */
		if (strncasecmp(optarg, "pekwm", 5) == 0) {
		    output_type = DO_PEKWM;
		}

		/* Golem output requested */
		if (strncasecmp(optarg, "golem", 5) == 0) {
		    output_type = DO_GOLEM;
		}

		/* Gnome (freedesktop like) output requested */
		if (strncasecmp(optarg, "gnome", 5) == 0) {
		    /* Freedesktop Output - define Type for showin parameter */
		    freedesktop_type = 2;
		    output_type = DO_FREEDESKTOP;
		}

		/* Openbox output requested */
		if (strncasecmp(optarg, "openbox", 7) == 0) {
		    output_type = DO_OPENBOX;
		}

		/* Aewm output requested */
		if (strncasecmp(optarg, "aewm", 4) == 0) {
		    aewm_type = 0;
		    output_type = DO_AEWM;
		}

		/* Sithwm output requested, similar to Aewm without a quit function */
		if (strncasecmp(optarg, "sithwm", 6) == 0) {
		    aewm_type = 1;
		    output_type = DO_AEWM;
		}

		/* Amiwm output requested */
		if (strncasecmp(optarg, "amiwm", 5) == 0) {
		    output_type = DO_AMIWM;
		}

		/* Kahakai output requested */
		if (strncasecmp(optarg, "kahakai", 7) == 0) {
		    output_type = DO_KAHAKAI;
		}

		/* Pwm output requested */
		if (strncasecmp(optarg, "pwm", 3) == 0) {
		    output_type = DO_PWM;
		}

		/* Sapphire output requested */
		if (strncasecmp(optarg, "sapphire", 8) == 0) {
		    output_type = DO_SAPPHIRE;
		}

		/* Enlightenment DR16 output requested */
		if (strncasecmp(optarg, "e16", 3) == 0) {
		    output_type = DO_ENLIGHTENMENT;
		}

		/* Enlightenment DR17 and newer (freedesktop like) output requested */
		if ((strncasecmp(optarg, "enlightenment", 13) == 0) || (strncasecmp(optarg, "e17", 3) == 0) || (strncasecmp(optarg, "e19", 3) == 0)) {
		    /* Freedesktop Output - define Type for showin parameter */
		    freedesktop_type = 5;
		    output_type = DO_FREEDESKTOP;
		}

		/* Epiwm output requested */
		if (strncasecmp(optarg, "epiwm", 6) == 0) {
		    output_type = DO_EPIWM;
		}

		/* Ede aka Equinox desktop environment output requested */
		if ((strncasecmp(optarg, "ede", 3) == 0) || (strncasecmp(optarg, "equinox", 7) == 0)) {
		    output_type = DO_EQUINOX;
		}

		/* Ude aka Unix desktop environment aka Uwm output requested */
		if ((strncasecmp(optarg, "ude", 3) == 0) || (strncasecmp(optarg, "uwm", 3) == 0)) {
		    output_type = DO_UDE;
		}

		/* Xfce output requested */
		if (strncasecmp(optarg, "xfce", 4) == 0) {
		    /* Freedesktop Output - define Type for showin parameter */
		    freedesktop_type = 3;
		    output_type = DO_FREEDESKTOP;
		}

		/* Wmx output requested, also works for Flwm */
		if ((strncasecmp(optarg, "wmx", 3) == 0) || (strncasecmp(optarg, "flwm", 4) == 0)) {
		    output_type = DO_WMX;
		}

		/* Lxde output requested */
		if (strncasecmp(optarg, "lxde", 4) == 0) {
		    /* Freedesktop Output - define Type for showin parameter */
		    freedesktop_type = 4;
		    output_type = DO_FREEDESKTOP;
		}

		/* Generic Freedesktop output */
		if (strncasecmp(optarg, "freedesktop", 11) == 0) {
		    output_type = DO_FREEDESKTOP;
		}

		/* Jwm output */
		if (strncasecmp(optarg, "jwm", 3) == 0) {
		    output_type = DO_JWM;
		}

		/* Sawfish output */
		if (strncasecmp(optarg, "sawfish", 7) == 0) {
		    output_type = DO_SAWFISH;
		}
	    break;

	    /* Display help if options are missing */
	    case 's':
		if (optarg == NULL) {
		    usage();
		}
		sys_confdir = optarg;
	    break;
	    case 'u':
		if (optarg == NULL) {
		    usage();
		}
		user_confdir = optarg;
	    break;
	    case 'r':
		if (optarg == NULL) {
		    usage();
		}
		root_menu_name = optarg;
	    break;
	    case 't':
		if (optarg == NULL) {
		    usage();
		}
		use_term = optarg;
	    break;

	    /* Parse the options for icons, menus ... */
	    case 'f':
		if (strcmp(optarg, "no-icons") != 0) {
		    flags |= FLAG_NO_ICONS;
		}
		if (strcmp(optarg, "no-check-existence") != 0) {
		    flags |= FLAG_NO_CHECK_EXISTENCE;
		}
		if (strcmp(optarg, "no-mini-icons") != 0) {
		    flags |= FLAG_NO_MINI_ICONS;
		}
		if (strcmp(optarg, "directories") != 0) {
		    flags |= FLAG_DIRECTORIES;
		}
		if (strcmp(optarg, "newstyle-directories") != 0) {
		    flags |= FLAG_NEWSTYLE_DIRECTORIES;
		}
		if (strcmp(optarg, "promote") != 0) {
		    flags |= FLAG_PROMOTE;
		}
	    break;
	    case 'd':
		if (optarg == NULL) {
		    usage();
		}
		output_directory = optarg;
	    break;
	    case 'v':
		version();
	    break;
	    case 'h':
	    default:
		usage();
	    break;
	}
    }
    return 0;
}

/*
Read the users home directory - required for input,
if $HOME/.wmconfig exists and for output (KDE ...)
*/

static char *get_home_dir(void)
{
/*    char *user = NULL; */
    char *home = NULL;
    struct passwd *pwd;
    struct stat st;
    int ret;
    pwd = getpwuid(getuid());

    /* No - do not define /dev/null as home directory */
    if (pwd == (struct passwd *)NULL) {
	fprintf(stderr, gettext("Can not identify the user home directory !\n"));
	return NULL;
    }
    if (pwd->pw_dir != NULL) {
	int maxlen;
	maxlen = strlen(pwd->pw_dir)+strlen(user_confdir)+5;
	if (maxlen > PATH_MAX) {
	    fprintf(stderr, gettext("user home directory %s/%s is too long !\n"), home, user_confdir);
	    return NULL;
	}
	home = malloc(maxlen);
	if (home == NULL) {
	    fprintf(stderr, gettext("%d@%s: out of memory !\n"), __LINE__-1, __FUNCTION__);
	    return NULL;
	}
	snprintf(home, maxlen, "%s/%s",
	pwd->pw_dir?pwd->pw_dir:"", user_confdir);
    } else {
	return NULL;
    }
    ret = stat(home, &st);
    if (ret != 0) {
	return NULL;
    }
    if (!S_ISDIR(st.st_mode)) {
	fprintf(stderr, gettext("%d@%s: %s is not a directory\n"), __LINE__-1, __FUNCTION__, home);
	return NULL;
    };
    return home;
}

/* Main function */
int main (int argc, char **argv)
{
    int ret;
    char *home;
    int have_files = 0;

    /* Init international language support, if set */
    init_i18n ();

    ProgName = argv[0];
    if (parse_options(argc, argv) != 0) {
	fprintf(stderr, gettext("Error parsing program options\n"));
	exit(-2);
    }

    /* Output option is required */
    if (!output_type) {
	fprintf(stderr, gettext("Missing output option...\n"));
	usage();
	exit (99);
    }

    /* If other options fail, try the default values */
    if (sys_confdir == NULL) {
	sys_confdir = DEFAULT_SYS_CONFDIR;
    }
    if (user_confdir == NULL) {
	user_confdir = DEFAULT_USER_CONFDIR;
    }
    if (root_menu_name == NULL) {
	root_menu_name = DEFAULT_ROOT_NAME;
    }
    init_packages();
    if (!is_set(NO_SYSDIR)) {
	ret = parse_files(sys_confdir);
	if (ret != PARSE_OK) {
	    fprintf(stderr, gettext("Parsing system files error: %s\n"), parse_errstr(ret));
	}
    }

    /* No terminal given. Use default value */
    if(use_term == NULL) {
	use_term = TERMINAL;
    }

    /* Check if everything is ok with the user files */
    if (!is_set(NO_USERDIR)) {
	home = get_home_dir();
	if (home != NULL) {
	    ret = parse_files(home);
	    if (ret != PARSE_OK) {
		fprintf(stderr, gettext("Parsing user files error: %s\n"), parse_errstr(ret));
	    }
	    free(home);
	}
    }

    /* File failed - permissions or anything else is wrong */
    while (argc > optind) {
	have_files ++;
	ret = read_file(argv[optind]);
	if (ret != 0) {
	    fprintf(stderr, gettext("\tReading file \"%s\" failed !\n"), argv[optind]);
	}
	optind++;
    }

    /* Cant find any input files in home directory and the system directory */
    if (is_set(NO_SYSDIR) && is_set(NO_USERDIR) && !have_files) {
	fprintf(stderr, gettext("No input files were parsed.\n"));
	exit(-1);
    }

    setbuf(stdout, NULL);
    if (build_root() != 0) {
	fprintf(stderr, gettext("package configuration error.\n"));
	exit(-3);
    }

    /* the output functions - every function calls the $whatever$ file */
    switch (output_type) {
	/* Fvwm2 and Fvwm95 output */
	case DO_FVWM2:
	    output_fvwm2(root_group, use_term);
	break;

	/* Afterstep directory output requested or menu output */
	case DO_AFTERSTEP:
	    if (is_set(DIRECTORIES) || is_set(NEWSTYLE_DIRECTORIES)) {
		output_afterstep_dir(root_group, use_term);
	    } else {
		output_afterstep(root_group, use_term);
	    }
	break;

	/* Other window managers */
	case DO_MWM:
	    output_mwm(root_group, use_term);
	break;
	case DO_WMAKER:
	    output_wmaker(root_group, use_term);
	break;
	case DO_ICEWM:
	    output_icewm(root_group, 0, use_term);
	break;
	case DO_BLACKBOX:
	    output_blackbox(root_group, 0, use_term);
	break;
	case DO_KDE:
	    output_kde(root_group);
	break;
	case DO_TWM:
	    output_twm(root_group, use_term, twm_type);
	break;
	case DO_OLWM:
	    output_olwm(root_group, use_term);
	break;
	case DO_FVWM:
	    output_fvwm(root_group, use_term);
	break;
	case DO_MLVWM:
	    output_mlvwm(root_group, use_term);
	break;
	case DO_KDE2:
	    output_kde2(root_group);
	break;
	case DO_QVWM:
	    output_qvwm(root_group, use_term);
	break;
	case DO_PEKWM:
	    output_pekwm(root_group, use_term);
	break;
	case DO_GOLEM:
	    output_golem(root_group, use_term);
	break;
	case DO_OPENBOX:
	    output_openbox(root_group, use_term);
	break;
	case DO_AEWM:
	    output_aewm(root_group, use_term, aewm_type);
	break;
	case DO_AMIWM:
	    output_amiwm(root_group, use_term);
	break;
	case DO_KAHAKAI:
	    output_kahakai(root_group, 0, use_term);
	break;
	case DO_PWM:
	    output_pwm(root_group, use_term);
	break;
	case DO_SAPPHIRE:
	    output_sapphire(root_group, 0, use_term);
	break;
	case DO_ENLIGHTENMENT:
	    output_enlightenment(root_group, use_term);
	break;
	case DO_EPIWM:
	    output_epiwm(root_group, use_term);
	break;
	case DO_EQUINOX:
	    output_ede(root_group);
	break;
	case DO_UDE:
	    output_ude(root_group, 0, use_term);
	break;
	case DO_WMX:
	    output_wmx(root_group, use_term);
	break;
	case DO_FREEDESKTOP:
	    output_freedesktop(root_group, freedesktop_type);
	break;
	case DO_JWM:
	    output_jwm(root_group, 0, use_term);
	break;
	case DO_SAWFISH:
	    output_sawfish(root_group, use_term);
	break;
	case DO_DEBUG:
	default:
	    output_debug(root_group);
    }
    return 0;
}
