/*
 * trexec.c - trexec main program
 * Copyright (C) 2005 - 2010 Michael Riepe
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif

static const char rcsid[] = "@(#) $Id: trexec.c,v 1.15 2010/03/20 13:30:06 michael Exp $";

#if STDC_HEADERS
#include <stddef.h>
#include <stdlib.h>
#include <string.h>
#else
char *getenv();
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#else
extern int optind;
extern char *optarg;
#endif

#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include <sys/ptrace.h>
#include <sys/stat.h>
#include <sys/wait.h>

#include <stdio.h>
#include <locale.h>

#include <assert.h>

#include <error.h>
#include <xmalloc.h>
#include <myptrace.h>
#include <process.h>
#include <action.h>

#ifndef __linux__
#error "This program is for Linux only. I'm sorry."
#endif

FILE *trexec_fp = NULL;
int report_failed = 0;

#ifndef NDEBUG

FILE *debug_fp = NULL;

#endif

static const char valid_options[] =
#if HAVE_FNMATCH
	"I:"
#endif
	"l:fh";

void
usage(int status) {
	fprintf(stderr,
		"usage: trexec [option...] command [arg...]\n"
		"options:\n"
		"  -l <file>      log to <file>\n"
#if HAVE_FNMATCH
		"  -I <pattern>   ignore path names that match <pattern>\n"
#endif
		"  -f             report failed reads\n"
		"  -h             display this message\n"
		);
	exit(status);
}

int
main(int argc, char **argv) {
	char *s;
	int res;
	int i;

	setlocale(LC_ALL, "");
	setprogname(*argv);
	if (chdir("/proc/self/cwd") == -1) {
		file_error("/proc/self/cwd", "chdir: %s", strerror(errno));
		fprintf(stderr, "Maybe the /proc filesystem isn't mounted?\n");
		fprintf(stderr, "Can't continue, sorry.\n");
		exit(1);
	}
	/*
	 * Parent may exec us with SIGCHLD ignored.
	 * Revert to default.
	 */
	signal(SIGCHLD, SIG_DFL);
#ifndef NDEBUG
	if ((s = getenv("TRACKFS_DEBUG")) != NULL) {
		debug_fp = fopen(s, "w");
		if (debug_fp == NULL) {
			file_error(s, "%s", strerror(errno));
			exit(1);
		}
		/* no buffering please */
		if (setvbuf(debug_fp, NULL, _IONBF, 0) != 0) {
			warn("setvbuf failed");
		}
		fcntl(fileno(debug_fp), F_SETFD, FD_CLOEXEC);
	}
#endif
	while ((i = getopt(argc, argv, valid_options)) != -1) {
		switch (i) {
			case 'l':
				if (trexec_fp != NULL) {
					error("option \"-l\" specified repeatedly");
					usage(1);
				}
				trexec_fp = fopen(optarg, "w");
				if (trexec_fp == NULL) {
					file_error(optarg, "%s", strerror(errno));
					exit(1);
				}
				fcntl(fileno(trexec_fp), F_SETFD, FD_CLOEXEC);
				break;
			case 'I':
				ignore_path(optarg);
				break;
			case 'f':
				report_failed = 1;
				break;
			case 'h':
				usage(0);
				break;
			case '?':
				usage(1);
				break;
			default:
				break;
		}
	}
	if (optind >= argc) {
		error("missing command");
		usage(1);
	}
	res = run_first_process(argv + optind);
	exit(res);
}
