/*
 * trexec-x32.c - trexec x32 syscall handling
 * Copyright (C) 2005 - 2013 Michael Riepe
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif

static const char rcsid[] = "@(#) $Id: trexec-x32.c,v 1.2 2013/03/04 21:11:56 michael Exp $";

#if STDC_HEADERS
#include <stddef.h>
#include <stdlib.h>
#include <string.h>
#else
#define offsetof(t,m)	((size_t)&((t*)0)->m)	/* BEWARE! */
#endif

#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include <sys/stat.h>
#include <sys/ptrace.h>

#include <stdio.h>

#include <assert.h>

#include <error.h>
#include <xmalloc.h>
#include <myptrace.h>
#include <process.h>
#include <action.h>

extern FILE *trexec_fp;
extern int report_failed;

#if defined(__x86_64__) && HAVE_X32_ABI

#include <x32.h>
#include <generic-x32.h>

int
app_handler_x32(struct pstate *p, struct user_regs_struct regs, int enter, int error) {
	long a1 = regs.__a1;
	long a2 = regs.__a2;
	long b1 = -1L;

	switch (regs.__nr) {
		case __NR_execve:
			if (enter) {
				const char *s;

				assert(a1);
				s = peek_string(p->pid, a1);
				if (s == NULL) {
					warn("%u: execve(%08lx) doesn't make sense",
						(int)p->pid, a1);
				}
				else {
					p->path = add_path(s);
					assert(p->path);
				}
			}
			else if (p->path) {
				if (!error) {
					if (trexec_fp
					 && !(p->path->flags & PATH_IGNORE)) {
						fprintf(trexec_fp, "e %s\n", p->path->name);
					}
					p->name = p->path;
				}
				p->path = NULL;
			}
			break;
#ifdef __NR_openat
		case __NR_openat:
			if (enter) {
				b1 = a1, a1 = a2, a2 = regs.__a3;
			}
			/* flow through */
#endif
		case __NR_open:
			if (enter) {
				if ((a2 & O_ACCMODE) != O_RDONLY) {
					/* ignore write access */
					p->path = NULL;
					break;
				}
				p->path = translate_path(p, b1, a1, 0);
			}
			else if (p->path) {
				if ((report_failed || !error)
				 && trexec_fp
				 && !(p->path->flags & PATH_IGNORE)) {
					char flag = error ? 'f' : 'r';

					if (p->name) {
						fprintf(trexec_fp, "%c [%s] %s\n",
							flag, p->name->name, p->path->name);
					}
					else {
						fprintf(trexec_fp, "%c [%d] %s\n",
							flag, (int)p->pid, p->path->name);
					}
				}
				p->path = NULL;
			}
			break;
		default:
			break;
	}
	return 0;
}

#endif /* defined(__x86_64__) && HAVE_X32_ABI */
