/*
 * trackfs-x86_64.c - trackfs x86_64 syscall handling
 * Copyright (C) 2005 - 2010 Michael Riepe
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif

static const char rcsid[] = "@(#) $Id: trackfs-x86_64.c,v 1.25 2010/03/19 20:02:28 michael Exp $";

#if STDC_HEADERS
#include <stddef.h>
#include <stdlib.h>
#include <string.h>
#else
#define offsetof(t,m)	((size_t)&((t*)0)->m)	/* BEWARE! */
#endif

#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include <sys/stat.h>
#include <sys/ptrace.h>

#include <stdio.h>

#include <assert.h>

#include <error.h>
#include <xmalloc.h>
#include <cpio.h>
#include <myptrace.h>
#include <process.h>
#include <action.h>

#if defined(__x86_64__)

#include <x86_64.h>
#include <generic-x86_64.h>

int
app_handler_x86_64(struct pstate *p, struct user_regs_struct regs, int enter, int error) {
	int follow_link = 0;
	long a1 = regs.__a1;
	long a2 = regs.__a2;
	long b1 = -1L;
	long b2 = -1L;
	const char *s;

	switch (regs.__nr) {
		case __NR_chdir:
			/*
			 * check for trackfs executing itself
			 */
			if (!enter || !regs.__a1) {
				break;
			}
			s = peek_string(p->pid, regs.__a1);
			if (s == NULL) {
				warn("%u: chdir(%08lx) doesn't make sense",
					(int)p->pid, regs.__a1);
				break;
			}
			if (strcmp(s, "/de/mr511/trackfs") != 0) {
				break;
			}
			warn("called recursively; detaching my alter ego");
			/* detach this process */
			return 1;
		case __NR_execve:
			if (enter) {
				assert(a1);
				s = peek_string(p->pid, a1);
				if (s == NULL) {
					warn("%u: execve(%08lx) doesn't make sense",
						(int)p->pid, a1);
				}
				else {
					p->path = add_path(s);
					assert(p->path);
				}
			}
			else if (p->path) {
				if (!error) {
					p->name = p->path;
				}
				p->path = NULL;
			}
			break;
#ifdef __NR_openat
		case __NR_openat:
			if (enter) {
				b1 = a1, a1 = a2, a2 = regs.__a3;
			}
			/* flow through */
#endif
		case __NR_open:
			if (enter) {
				if ((a2 & O_ACCMODE) == O_RDONLY) {
					/* ignore read-only access */
					p->path = NULL;
					break;
				}
			}
			/* flow through */
#ifdef __NR_creat
		case __NR_creat:
#endif
		case __NR_truncate:
#ifdef __NR_truncate64
		case __NR_truncate64:
#endif
			if (enter) {
				p->path = translate_path(p, b1, a1, 0);
				if (p->path != NULL) {
					backup_file(p, p->path);
				}
			}
			else if (p->path) {
				if (!error) {
					report_file(p, p->path);
				}
				p->path = NULL;
			}
			break;
#if defined(__NR_mkdirat) || defined(__NR_mknodat)
#ifdef __NR_mkdirat
		case __NR_mkdirat:
#endif
#ifdef __NR_mknodat
		case __NR_mknodat:
#endif
			if (enter) {
				b1 = a1, a1 = a2;
			}
			/* flow through */
#endif
		case __NR_mkdir:
		case __NR_mknod:
			if (enter) {
				p->path = translate_path(p, b1, a1, 0);
			}
			else if (p->path) {
				if (!error) {
					report_node(p, p->path);
				}
				p->path = NULL;
			}
			break;
#ifdef __NR_linkat
		case __NR_linkat:
			if (enter) {
				b1 = a1, a1 = a2, b2 = regs.__a3, a2 = regs.__a4;
			}
			goto link_common;
#endif
#ifdef __NR_symlinkat
		case __NR_symlinkat:
			if (enter) {
				b2 = a2, a2 = regs.__a3;
			}
			goto link_common;
#endif
		link_common:
		case __NR_link:
		case __NR_symlink:
			if (enter) {
				p->path = translate_path(p, b2, a2, 0);
			}
			else if (p->path) {
				if (!error) {
					report_node(p, p->path);
				}
				p->path = NULL;
			}
			break;
#ifdef __NR_unlinkat
		case __NR_unlinkat:
			if (enter) {
				b1 = a1, a1 = a2;
			}
			/* flow through */
#endif
		case __NR_rmdir:
		case __NR_unlink:
			if (enter) {
				p->path = translate_path(p, b1, a1, 0);
				if (p->path != NULL) {
					backup_node(p, p->path);
				}
			}
			else if (p->path) {
				if (!error) {
					report_remove(p, p->path);
				}
				p->path = NULL;
			}
			break;
#ifdef __NR_renameat
		case __NR_renameat:
			if (enter) {
				b1 = a1, a1 = a2, b2 = regs.__a3, a2 = regs.__a4;
			}
			/* flow through */
#endif
		case __NR_rename:
			if (enter) {
				p->path = translate_path(p, b1, a1, 0);
				if (p->path == NULL) {
					p->path2 = NULL;
					break;
				}
				p->path2 = translate_path(p, b2, a2, 0);
				if (p->path2 == NULL) {
					p->path = NULL;
					break;
				}
				backup_node(p, p->path2);
			}
			else if (p->path2) {
				assert(p->path);
				if (!error) {
					report_move(p, p->path, p->path2);
				}
				p->path2 = NULL;
				p->path = NULL;
			}
			break;
#if defined(__NR_fchmodat) || defined(__NR_fchownat)
#ifdef __NR_fchmodat
		case __NR_fchmodat:
#endif
#ifdef __NR_fchownat
		case __NR_fchownat:
#endif
			if (enter) {
				b1 = a1, a1 = a2;
			}
			/* flow through */
#endif
		case __NR_chmod:
#ifdef __NR_chown
		case __NR_chown:
#endif
#ifdef __NR_chown32
		case __NR_chown32:
#endif
			follow_link = 1;
			/* flow through */
#ifdef __NR_lchown
		case __NR_lchown:
#endif
#ifdef __NR_lchown32
		case __NR_lchown32:
#endif
			if (enter) {
				p->path = translate_path(p, b1, a1, follow_link);
				if (p->path != NULL) {
					backup_node(p, p->path);
				}
			}
			else if (p->path) {
				if (!error) {
					action(ACTION_UPDATE, p->path);
				}
				p->path = NULL;
			}
			break;
		case __NR_fchmod:
#ifdef __NR_fchown
		case __NR_fchown:
#endif
#ifdef __NR_fchown32
		case __NR_fchown32:
#endif
			if (enter) {
				p->path = translate_fd(p, a1);
				if (p->path == NULL) {
					break;
				}
				backup_node(p, p->path);
			}
			else if (p->path) {
				if (!error) {
					action(ACTION_UPDATE, p->path);
				}
				p->path = NULL;
			}
			break;
		default:
			break;
	}
	return 0;
}

#endif /* defined(__x86_64__) */
