/*
 * process.c - track processes
 * Copyright (C) 2005 - 2013 Michael Riepe
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif

static const char rcsid[] = "@(#) $Id: process.c,v 1.31 2013/02/21 18:51:45 michael Exp $";

#if STDC_HEADERS
#include <stddef.h>
#include <stdlib.h>
#include <string.h>
#else
extern int strcmp();
extern size_t strlen();
extern char *strcpy();
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#else
extern int optind;
extern char *optarg;
#endif

#ifndef NULL
#define NULL ((void*)0)
#endif

#if HAVE_FNMATCH_H
#include <fnmatch.h>
#else
extern int fnmatch();
#endif

#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include <sys/ptrace.h>
#include <sys/stat.h>
#include <sys/wait.h>

#include <stdio.h>

#include <assert.h>

#include <error.h>
#include <xmalloc.h>
#include <myptrace.h>
#include <process.h>

#ifndef NDEBUG

extern FILE *debug_fp;

#endif /* NDEBUG */

#define PROC_HASH	251u

static struct pstate *proc_list[PROC_HASH];

static unsigned proc_nprocs = 0;

static struct path *program_name(pid_t pid);

struct pstate*
find_process(pid_t pid, int attach) {
	unsigned hash = pid % PROC_HASH;
	struct pstate *p;

	for (p = proc_list[hash]; p; p = p->next) {
		if (p->pid == pid) {
			return p;
		}
	}
	if (attach) {
		p = xmalloc(sizeof(struct pstate)); assert(p);
		p->pid = pid;
		p->flags = 0;
		p->name = program_name(pid);
		p->path = NULL;
		p->path2 = NULL;
		p->next = proc_list[hash];
		proc_list[hash] = p;
		++proc_nprocs;
	}
	return p;
}

void
remove_process(struct pstate *p) {
	unsigned hash = p->pid % PROC_HASH;
	struct pstate **q = &proc_list[hash];

	while (*q && *q != p) {
		q = &(*q)->next;
	}
	assert(*q == p);
	*q = p->next;
	--proc_nprocs;
}

#define PATH_HASH	4093u

static struct path *path_list[PATH_HASH];

static const char **ignore_list = NULL;
static size_t num_ignores = 0;

static unsigned
path_hash(const char *s, size_t len) {
	const char *const end = s + len;
	unsigned hash = len;
	unsigned tmp;

	while (s < end) {
		tmp = (unsigned char)*s++;
		hash = (hash << 4) + tmp;
		if ((tmp = hash & 0xf0000000)) {
			hash ^= tmp | (tmp >> 24);
		}
	}
	return hash;
}

static struct path*
add_path_2(const char *path, size_t len) {
	unsigned hash = path_hash(path, len);
	struct path *l;
	size_t i;

	for (l = path_list[hash % PATH_HASH]; l; l = l->next) {
		if (hash == l->hash && memcmp(path, l->name, len) == 0) {
			return l;
		}
	}
	l = xmalloc(sizeof(struct path) + len + 1); assert(l);
	l->name = memcpy(l + 1, path, len);
	l->name[len] = '\0';
	l->hash = hash;
	l->flags = 0;
	l->pdir = NULL;
	while (len > 0 && path[len - 1] != '/') {
		--len;
	}
	while (len > 0 && path[len - 1] == '/') {
		--len;
	}
	if (len > 0) {
		l->pdir = add_path_2(path, len);
		l->flags = l->pdir->flags & PATH_IGNORE;
	}
#if HAVE_FNMATCH
	if (!(l->flags & PATH_IGNORE)) {
		for (i = 0; i < num_ignores; i++) {
			if (fnmatch(ignore_list[i], l->name, FNM_PATHNAME | FNM_PERIOD) == 0) {
				l->flags |= PATH_IGNORE;
			}
		}
	}
#endif
	hash %= PATH_HASH;
	l->next = path_list[hash];
	path_list[hash] = l;
	return l;
}

struct path*
add_path(const char *path) {
	return add_path_2(path, strlen(path));
}

static struct path*
program_name(pid_t pid) {
	char path[6 + 3 * sizeof(int) + 5];
	char buf[65536];
	ssize_t n;

	sprintf(path, "/proc/%u/exe", (int)pid);
	if ((n = readlink(path, buf, sizeof(buf))) == -1) {
		file_warn(path, "readlink: %s", strerror(errno));
		return NULL;
	}
	if (n >= sizeof(buf)) {
		n = sizeof(buf) - 1;
	}
	buf[n] = '\0';
	return add_path_2(buf, n);
}

void
ignore_path(const char *path) {
	const size_t incr = 32u;

	if ((num_ignores % incr) == 0) {
		ignore_list = xrealloc(ignore_list,
			(num_ignores + incr) * sizeof(const char*));
	}
	ignore_list[num_ignores++] = xstrdup(path);
}

pid_t
start_child(char **argv, int traced) {
	pid_t pid;

	switch ((pid = fork())) {
		case 0: /* child */
			if (traced) {
				if (ptrace_traceme() == -1) {
					error("ptrace(PTRACE_TRACEME)");
					_exit(1);
				}
				kill(getpid(), SIGTRAP);	/* let parent notice me */
			}
			execvp(argv[0], argv);
			file_error(argv[0], "%s", strerror(errno));
			_exit(127);
		case -1: /* error */
			return -1;
		default: /* parent */
			break;
	}
	return pid;
}

int
run_first_process(char **argv) {
	int wait_flags = __WALL;
	int main_exited = 0;
	int main_exit = 0;
	int options_set = 0;
	struct pstate *p;
	pid_t mpid;
	pid_t pid;
	int status;

	mpid = start_child(argv, 1);
	if (mpid == -1) {
		error("fork: %s", strerror(errno));
		exit(1);
	}
	p = find_process(mpid, 1); assert(p);
	assert(proc_nprocs == 1);
	while (proc_nprocs) {
		switch (main_exited) {
			case 1:
				warn("command exited but %u children are still running", proc_nprocs);
				wait_flags |= WNOHANG;
				++main_exited;
				break;
		}
		if ((pid = waitpid(-1, &status, wait_flags)) == -1) {
			if (errno == ECHILD) {
				warn("nprocs = %u but no children left", proc_nprocs);
				break;
			}
			if (errno != EINTR) {
				error("wait: %s", strerror(errno));
				exit(1);
			}
		}
		else if (pid == 0) {
			assert(wait_flags & WNOHANG);
			if (++main_exited > 2 + 10) {
				warn("detaching remaining %u children", proc_nprocs);
				break;
			}
			sleep(1);
		}
		else if (WIFEXITED(status)) {
			if ((p = find_process(pid, 0))) {
				remove_process(p);
				xfree(p);
			}
			if (pid == mpid) {
				main_exited = 1;
				main_exit = WEXITSTATUS(status);
			}
		}
		else if (WIFSIGNALED(status)) {
#ifndef NDEBUG
			if (debug_fp) {
				fprintf(debug_fp, "[%d] process killed (%d)\n",
					(int)pid, WTERMSIG(status));
			}
#endif /* NDEBUG */
			if ((p = find_process(pid, 0))) {
				remove_process(p);
				xfree(p);
			}
			if (pid == mpid) {
				main_exited = 1;
				main_exit = 128 + WTERMSIG(status);
			}
		}
		else if (!WIFSTOPPED(status)) {
			fatal("child not stopped");
			exit(1);
		}
		else if (WSTOPSIG(status) == SIGSTOP) {
#ifndef NDEBUG
			if (debug_fp) {
				fprintf(debug_fp, "[%d] process stopped\n", (int)pid);
			}
#endif /* NDEBUG */
			p = find_process(pid, 1); assert(p);
			p->flags &= ~PROCESS_INSYSCALL;
			generic_syscall(p, 0);
		}
		else if (WSTOPSIG(status) == SIGTRAP) {
#ifndef NDEBUG
			if (debug_fp) {
				fprintf(debug_fp, "[%d] process trapped (%#x)\n",
					(int)pid, status);
			}
#endif /* NDEBUG */
			if (pid == mpid && !options_set) {
				myptrace_reason("setting ptrace options");
				myptrace(PTRACE_SETOPTIONS, pid, 0L,
					(long)(PTRACE_O_TRACEFORK | PTRACE_O_TRACEVFORK | PTRACE_O_TRACECLONE));
				options_set = 1;
			}
			if ((status >> 16) & 0xff) {
				/* trace event, continue process */
				if (ptrace(PTRACE_SYSCALL, pid, (void*)0, (void*)0) == -1) {
					warn("can't continue process %d: %s",
						(int)pid, strerror(errno));
				}
			}
			else {
				p = find_process(pid, 0); assert(p);
				generic_syscall(p, !(p->flags & PROCESS_INSYSCALL));
			}
		}
		else {
#ifndef NDEBUG
			if (debug_fp) {
				fprintf(debug_fp, "[%d] process got signal %d\n",
					(int)pid, WSTOPSIG(status));
			}
#endif /* NDEBUG */
			/* pass signal to child */
			p = find_process(pid, 0); assert(p);
			p->flags &= ~PROCESS_INSYSCALL;
			myptrace_reason("passing signal");
			myptrace(PTRACE_SYSCALL, pid, 0L, (long)WSTOPSIG(status));
		}
	}
	return main_exit;
}
