/*
 * myptrace.c - wrapper for the ptrace() syscall
 * Copyright (C) 2005 - 2010 Michael Riepe
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif

static const char rcsid[] = "@(#) $Id: myptrace.c,v 1.12 2011/03/04 23:15:09 michael Exp $";

#if STDC_HEADERS
#include <stddef.h>
#include <stdlib.h>
#include <string.h>
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#include <errno.h>
#include <sys/ptrace.h>

#include <stdio.h>

#include <assert.h>

#include <error.h>
#include <xmalloc.h>
#include <myptrace.h>

static const char *reason = NULL;

void
myptrace_reason(const char *r) {
	reason = r;
}

void
myptrace(int what, pid_t pid, long addr, long data) {
	if (ptrace(what, pid, (void*)addr, (void*)data) == -1) {
		error("ptrace(%d, %d, %08lx): %s [%s]",
			what, (int)pid, addr, strerror(errno), reason);
		exit(1);
	}
	reason = NULL;
}

int
peek_user(pid_t pid, long addr, long *data) {
	int save;

	errno = 0;
	*data = ptrace(PTRACE_PEEKUSER, pid, (void*)addr, NULL);
	if ((save = errno) != 0) {
		error("ptrace(PTRACE_PEEKUSER, %d, %08lx): %s",
			(int)pid, addr, strerror(errno));
	}
	return -save;
}

int
peek_data(pid_t pid, long addr, long *data) {
	int save;

	errno = 0;
	*data = ptrace(PTRACE_PEEKDATA, pid, (void*)addr, NULL);
	if ((save = errno) != 0) {
		error("ptrace(PTRACE_PEEKDATA, %d, %08lx): %s",
			(int)pid, addr, strerror(errno));
	}
	return -save;
}

int
peek_text(pid_t pid, long addr, long *data) {
	int save;

	errno = 0;
	*data = ptrace(PTRACE_PEEKTEXT, pid, (void*)addr, NULL);
	if ((save = errno) != 0) {
		error("ptrace(PTRACE_PEEKTEXT, %d, %08lx): %s",
			(int)pid, addr, strerror(errno));
	}
	return -save;
}

long
peek_user_safe(pid_t pid, long addr) {
	long res;

	if (peek_user(pid, addr, &res)) {
		fatal("can not access child memory, exiting");
	}
	return res;
}

long
peek_data_safe(pid_t pid, long addr) {
	long res;

	if (peek_data(pid, addr, &res)) {
		fatal("can not access child memory, exiting");
	}
	return res;
}

long
peek_text_safe(pid_t pid, long addr) {
	long res;

	if (peek_text(pid, addr, &res)) {
		fatal("can not access child memory, exiting");
	}
	return res;
}

void
poke_user(pid_t pid, long addr, long data) {
	if (ptrace(PTRACE_POKEUSER, pid, (void*)addr, (void*)data) == -1) {
		error("ptrace(PTRACE_POKEUSER, %d, %08lx): %s",
			(int)pid, addr, strerror(errno));
		exit(1);
	}
}

void
poke_data(pid_t pid, long addr, long data) {
	if (ptrace(PTRACE_POKEDATA, pid, (void*)addr, (void*)data) == -1) {
		error("ptrace(PTRACE_POKEDATA, %d, %08lx): %s",
			(int)pid, addr, strerror(errno));
		exit(1);
	}
}

void
poke_text(pid_t pid, long addr, long data) {
	if (ptrace(PTRACE_POKETEXT, pid, (void*)addr, (void*)data) == -1) {
		error("ptrace(PTRACE_POKETEXT, %d, %08lx): %s",
			(int)pid, addr, strerror(errno));
		exit(1);
	}
}

char*
peek_string(pid_t pid, long addr) {
	static char *buf = NULL;
	static size_t len = 0;
	size_t n = 0;
	size_t off;
	long v;
	int i;

	if ((off = addr % sizeof(long))) {
		addr -= off;
	}
	i = off;
	for (;;) {
		if (n + sizeof(long) > len) {
			buf = xrealloc(buf, len += 1024u); assert(buf);
		}
		if (peek_data(pid, addr + n, &v) != 0) {
			return NULL;
		}
		*(long*)(buf + n) = v;
		for (; i < sizeof(long); i++) {
			if (buf[n + i] == '\0') {
				return xstrdup(buf + off);
			}
		}
		n += sizeof(long);
		i = 0;
	}
}

int
ptrace_traceme(void) {
	return ptrace(PTRACE_TRACEME, 0, NULL, NULL);
}

int
getregs(pid_t pid, void *regs) {
	if (ptrace(PTRACE_GETREGS, pid, (void*)0, regs)) {
		return -errno;
	}
	return 0;
}

int
setregs(pid_t pid, const void *regs) {
	int save;

	if (ptrace(PTRACE_SETREGS, pid, (void*)0, regs)) {
		save = errno;
		error("setregs(%d): %s", (int)pid, strerror(errno));
		return -save;
	}
	return 0;
}
