/*
 * generic-i386.c - generic i386 syscall handling
 * Copyright (C) 2005 - 2013 Michael Riepe
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif

static const char rcsid[] = "@(#) $Id: generic-i386.c,v 1.12 2013/03/04 20:48:41 michael Exp $";

#if STDC_HEADERS
#include <stddef.h>
#include <stdlib.h>
#include <string.h>
#else
#define offsetof(t,m)	((size_t)&((t*)0)->m)	/* BEWARE! */
#endif

#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include <sys/stat.h>
#include <sys/ptrace.h>

#include <stdio.h>

#include <assert.h>

#include <error.h>
#include <xmalloc.h>
#include <myptrace.h>
#include <process.h>
#include <action.h>

#include <sched.h>		/* for CLONE_* flags */

#if defined(__i386__) || defined(__x86_64__)

#include <i386.h>

#include <generic-i386.h>

#ifndef NDEBUG

extern FILE *debug_fp;

static const char*
syscall_name(unsigned long nr) {
	static const char *table[] = {
#define __syscall__(x)	[__NR_##x] = #x,
#include <syscalls.h>
#undef __syscall__
	};
	static char buf[sizeof(nr) * 3 + 2];

	if (nr < sizeof(table) / sizeof(table[0]) && table[nr] != NULL) {
		return table[nr];
	}
	sprintf(buf, "#%lu", nr);
	return buf;
}

#endif /* NDEBUG */

#if defined(__i386__)

void
generic_syscall(struct pstate *p, int enter) {
	struct user_regs_struct regs;

	if (getregs(p->pid, &regs)) {
		/*
		 * process may have been killed, so don't print a
		 * message if it's gone for good.
		 */
		if (errno != ESRCH) {
			error("getregs(%d): %s", (int)p->pid, strerror(errno));
		}
		ptrace(PTRACE_DETACH, p->pid, NULL, NULL);
		return;
	}
	switch (regs.__cs) {
		case 0x33:	/* 64-bit binary */
			error("process %d is not a 32-bit process!", (int)p->pid);
			fprintf(stderr, "You need to use the 64-bit version of this program.\n");
			fprintf(stderr, "Terminating process for safety reasons.\n");
			myptrace_reason("killing 64-bit process");
			myptrace(PTRACE_KILL, p->pid, 0, 0);
			return;
		case 0x23:	/* 32-bit binary on 64-bit kernel */
		case 0x73:	/* 32-bit binary on 32-bit kernel */
			generic_i386_syscall(p, regs, enter);
			return;
		default:	/* unknown */
			/* don't let the process continue */
			error("KILLING PROCESS %d (unknown CS value 0x%02x)", (int)p->pid, (int)regs.__cs);
			myptrace_reason("killing unknown process");
			myptrace(PTRACE_KILL, p->pid, 0, 0);
			return;
	}
}

#endif /* defined(__i386__) */

void
generic_i386_syscall(struct pstate *p, struct user_regs_struct regs, int enter) {
	int error = 0;
	long res;
	struct pstate *q;

	/*
	 * sigreturn is a special case
	 */
#if defined(__NR_sigreturn) || defined(__NR_rt_sigreturn)
	if (p->flags & PROCESS_SIGRETURN) {
		p->flags &= ~PROCESS_SIGRETURN;
#ifndef NDEBUG
		if (debug_fp) {
			fprintf(debug_fp, "[%d]-sigreturn\n", (int)p->pid);
		}
#endif /* NDEBUG */
		goto out;
	}
	switch (regs.__nr) {
#ifdef __NR_sigreturn
		case __NR_sigreturn:
#endif
#ifdef __NR_rt_sigreturn
		case __NR_rt_sigreturn:
#endif
			p->flags |= PROCESS_SIGRETURN;
#ifndef NDEBUG
			if (debug_fp) {
				fprintf(debug_fp, "[%d]+sigreturn\n", (int)p->pid);
			}
#endif /* NDEBUG */
			goto out;
		default:
			break;
	}
#endif /* defined(__NR_sigreturn) || defined(__NR_rt_sigreturn) */
	/*
	 * enter/leave handling
	 */
	res = (int)regs.__res;
	if (p->flags & PROCESS_EXEC) {
		p->flags &= ~PROCESS_EXEC;
		if (regs.__nr != __NR_execve) {	/* may have changed due to 32/64 bit switch! */
			regs.__nr = __NR_execve;
			setregs(p->pid, &regs);
		}
		enter = 0;
	}
	else if (regs.__nr == __NR_execve) {
		if (regs.__a1) {
			p->flags |= PROCESS_EXEC;
			enter = 1;
		}
		else {
			/* don't know why this happens, but it does */
			enter = 0;
		}
	}
	if (enter) {
		p->flags |= PROCESS_INSYSCALL;
	}
	else {
		p->flags &= ~PROCESS_INSYSCALL;
		if ((unsigned)(res + 4096) < 4096) {
			error = -res;
			res = -1L;
		}
	}
#ifndef NDEBUG
	if (debug_fp) {
		const char *str = syscall_name(regs.__nr);

		if (enter) {
			fprintf(debug_fp, "[%d]+%s %lx %lx %lx ...\n",
				(int)p->pid, str, regs.__a1, regs.__a2, regs.__a3);
		}
		else if (error) {
			fprintf(debug_fp, "[%d]-%s %lx %lx %lx ... = %ld (%s)\n",
				(int)p->pid, str, regs.__a1, regs.__a2, regs.__a3, res, strerror(error));
		}
		else {
			fprintf(debug_fp, "[%d]-%s %lx %lx %lx ... = %lx\n",
				(int)p->pid, str, regs.__a1, regs.__a2, regs.__a3, res);
		}
	}
#endif /* NDEBUG */
	/*
	 * application specific handlers
	 */
	if (app_handler_i386(p, regs, enter, error)) {
		/* detach this process */
		remove_process(p);
		myptrace_reason("detach after app_handler_i386 error");
		myptrace(PTRACE_DETACH, p->pid, 0L, 0L);
#ifndef NDEBUG
		if (debug_fp) {
			fprintf(debug_fp, "[%d] detached process\n", (int)p->pid);
		}
#endif /* NDEBUG */
		xfree(p);
		return;	/* no longer traced */
	}
out:
	if (ptrace(PTRACE_SYSCALL, p->pid, (void*)0, (void*)0) == -1) {
		warn("can't continue process %d: %s", (int)p->pid, strerror(errno));
	}
}

#endif /* defined(__i386__) || defined(__x86_64__) */
