/*
 * cpio.c - backup files in cpio format
 * Copyright (C) 2005 - 2008 Michael Riepe
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif

static const char rcsid[] = "@(#) $Id: cpio.c,v 1.8 2008/05/27 14:33:12 michael Exp $";

#if STDC_HEADERS
#include <stddef.h>
#include <stdlib.h>
#include <string.h>
#else
size_t strlen();
void exit();
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#else
int close();
int read();
int readlink();
int write();
#endif

#include <fcntl.h>
#include <sys/stat.h>

#include <stdio.h>
#include <errno.h>

#include <assert.h>

#include <error.h>
#include <xmalloc.h>
#include <cpio.h>

int cpio_fd = -1;

void
cpio_write_header(struct stat *stp, const char *path) {
	size_t len = strlen(path);
	char header[77];
	long size;

	assert(path);
	assert(*path == '/');
	if (S_ISREG(stp->st_mode) || S_ISLNK(stp->st_mode)) {
		size = (long)stp->st_size;
	}
	else {
		size = 0;
	}
	sprintf(header, "070707%06o%06o%06o%06o%06o%06o%06o%011lo%06o%011lo",
		(int)stp->st_dev & 0777777, (int)stp->st_ino & 0777777,
		(int)stp->st_mode & 0777777, (int)stp->st_uid & 0777777,
		(int)stp->st_gid & 0777777, (int)stp->st_nlink & 0777777,
		(int)stp->st_rdev & 0777777,
		(unsigned long)(stp->st_mtime & 077777777777ULL),
		(int)len, (unsigned long)(size & 077777777777ULL));
	if (write(cpio_fd, header, 76) != 76
	 || write(cpio_fd, path + 1, len) != len) {
		error("cpio write error");
		exit(1);
	}
}

void
cpio_write_data(struct stat *stp, const char *path) {
	char buf[16384];
	size_t written = 0;
	size_t n;
	int fd;

	fd = open(path, O_RDONLY);
	if (fd == -1) {
		file_error(path, "%s", strerror(errno));
		exit(1);
	}
	while (written < stp->st_size) {
		n = stp->st_size - written;
		if (n > sizeof(buf)) {
			n = sizeof(buf);
		}
		n = read(fd, buf, n);
		if (n == -1) {
			file_error(path, "read: %s", strerror(errno));
			exit(1);
		}
		if (n == 0) {
			file_error(path, "short file");
			exit(1);
		}
		if (write(cpio_fd, buf, n) != n) {
			file_error(path, "cpio write error");
			exit(1);
		}
		written += n;
	}
	close(fd);
}

void
cpio_write_link(struct stat *stp, const char *path) {
	char *buf;

	if (stp->st_size == 0) {
		file_warn(path, "link has zero size");
		return;
	}
	buf = xmalloc(stp->st_size); assert(buf);
	if (readlink(path, buf, stp->st_size) == -1) {
		file_error(path, "readlink: %s", strerror(errno));
		exit(1);
	}
	if (write(cpio_fd, buf, stp->st_size) != stp->st_size) {
		error("cpio write error");
		exit(1);
	}
	xfree(buf);
}

void
cpio_write_node(struct stat *stp, const char *path) {
	cpio_write_header(stp, path);
	if (S_ISREG(stp->st_mode)) {
		cpio_write_data(stp, path);
	}
	else if (S_ISLNK(stp->st_mode)) {
		cpio_write_link(stp, path);
	}
}

void
cpio_write_trailer(void) {
	static struct stat st = { 0, };

	cpio_write_header(&st, "/TRAILER!!!");
}
