/*
 * action.c - trackfs action routines
 * Copyright (C) 2005 - 2009 Michael Riepe
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif

static const char rcsid[] = "@(#) $Id: action.c,v 1.13 2011/04/18 22:34:29 michael Exp $";

#include <stddef.h>
#include <stdlib.h>
#include <string.h>

#include <sys/stat.h>

#include <stdio.h>

#include <assert.h>

#include <xmalloc.h>
#include <myptrace.h>
#include <process.h>
#include <cpio.h>
#include <action.h>
#include <error.h>

#ifndef NDEBUG
extern FILE *debug_fp;
#endif

struct {
	struct action *head;
	struct action *tail;
} actions = { NULL, NULL };

void
action_print(FILE *fp, const struct action *a) {
	static const char table[] = {
		[ACTION_NONE] = 't',
		[ACTION_CREATE] = 'c',
		[ACTION_UPDATE] = 'u',
		[ACTION_DELETE] = 'd',
		[ACTION_EXEC] = 'e',
		[ACTION_READ] = 'r'
	};

	switch (a->type) {
		case ACTION_NONE:
		case ACTION_CREATE:
		case ACTION_UPDATE:
		case ACTION_DELETE:
		case ACTION_EXEC:
		case ACTION_READ:
			break;
		default:
			assert(0);
	}
	fprintf(fp, "%c %s\n", table[a->type], a->path->name);
}

void
action(int what, struct path *path) {
	struct action *a = xmalloc(sizeof(struct action));

	a->type = what;
	a->path = path;
	a->next = NULL;
	if (actions.tail) {
		actions.tail->next = a;
	}
	else {
		actions.head = a;
	}
	actions.tail = a;
#ifndef NDEBUG
	if (debug_fp) {
		action_print(debug_fp, a);
	}
#endif
}

void
action_report(FILE *fp) {
	struct action *a, *b, **bb;

	while ((a = actions.head)) {
		actions.head = a->next;
		if (a->path->flags & PATH_IGNORE) {
			continue;
		}
		if (a->type == ACTION_EXEC || a->type == ACTION_READ) {
			if (fp) {
				action_print(fp, a);
			}
			continue;
		}
		bb = &actions.head;
		while ((b = *bb)) {
			if (b->path != a->path) {
				bb = &b->next;
				continue;
			}
			switch ((a->type << 4) | b->type) {
				case (ACTION_NONE << 4) | ACTION_CREATE:
					a->type = ACTION_CREATE;
					break;
				case (ACTION_UPDATE << 4) | ACTION_DELETE:
					a->type = ACTION_DELETE;
					break;
				case (ACTION_CREATE << 4) | ACTION_DELETE:
					a->type = ACTION_NONE;
					break;
				case (ACTION_DELETE << 4) | ACTION_CREATE:
					a->type = ACTION_UPDATE;
					break;
				case (ACTION_CREATE << 4) | ACTION_UPDATE:
				case (ACTION_UPDATE << 4) | ACTION_UPDATE:
					/* keep a->type */
					break;
				case (ACTION_NONE << 4) | ACTION_UPDATE:
					fprintf(stderr, "trackfs: warning: missed create(\"%s\")\n",
						a->path->name);
					a->type = ACTION_CREATE;
					break;
				case (ACTION_DELETE << 4) | ACTION_UPDATE:
					fprintf(stderr, "trackfs: warning: missed create(\"%s\")\n",
						a->path->name);
					a->type = ACTION_UPDATE;
					break;
				case (ACTION_NONE << 4) | ACTION_DELETE:
				case (ACTION_DELETE << 4) | ACTION_DELETE:
					fprintf(stderr, "trackfs: warning: missed create(\"%s\")\n",
						a->path->name);
					/* keep a->type */
					break;
				case (ACTION_CREATE << 4) | ACTION_CREATE:
				case (ACTION_UPDATE << 4) | ACTION_CREATE:
					fprintf(stderr, "trackfs: warning: missed delete(\"%s\")\n",
						a->path->name);
					/* keep a->type */
					break;
				default:
					assert(0);
			}
			*bb = b->next;
		}
		if (fp) {
			action_print(fp, a);
		}
	}
}

struct path*
translate_path(struct pstate *p, int base, long addr, int followsymlinks) {
	char tmp[512];
	struct path *l;
	struct stat st;
	size_t len;
	char *path;
	char *buf;
	char *s;
	int n;

	if (addr == 0) {
		return NULL;
	}
	path = peek_string(p->pid, addr);
	if (path == NULL) {
		warn("%u: string argument %08lx doesn't make sense",
			(int)p->pid, addr);
		return NULL;
	}
	if (*path == '/') {
		sprintf(tmp, "/proc/%d/root", (int)p->pid);
	}
	else if (base >= 0) {
		sprintf(tmp, "/proc/%d/fd/%d/", (int)p->pid, base);
	}
	else {
		sprintf(tmp, "/proc/%d/cwd/", (int)p->pid);
	}
	n = strlen(tmp);
	len = strlen(path);
	buf = xmalloc(n + len + 1);
	memcpy(buf, tmp, n);
	memcpy(buf + n, path, len + 1);
	xfree(path);
#ifndef NDEBUG
	if (debug_fp) {
		fprintf(debug_fp, "> translate_path(\"%s\")\n", buf);
	}
#endif /* NDEBUG */
	/* remove trailing slashes */
	s = buf + n + len;
	while (s > buf && *--s == '/') {
		*s = '\0';
	}
	if (lstat(buf, &st) != -1 && (followsymlinks || !S_ISLNK(st.st_mode))) {
		path = realpath(buf, NULL);	/* GNU special */
		if (path == NULL) {
			xfree(buf);
			return NULL;
		}
	}
	else {
		/* doesn't exist or is a symlink */
		/* get parent directory */
		s = strrchr(buf, '/'); assert(s);
		*s = '\0';
		path = realpath(buf, NULL);	/* GNU special */
		if (path == NULL) {
			xfree(buf);
			return NULL;
		}
		*s = '/';
		n = strlen(s);
		len = strlen(path);
		path = xrealloc(path, len + n + 1);
		memcpy(path + len, s, n + 1);
	}
	xfree(buf);
#ifndef NDEBUG
	if (debug_fp) {
		fprintf(debug_fp, "< translate_path(\"%s\")\n", path);
	}
#endif /* NDEBUG */
	l = add_path(path); assert(l);
	xfree(path);
	return l;
}

struct path*
translate_fd(struct pstate *p, int fd) {
	char tmp[512];
	struct path *l;
	char *path;

	sprintf(tmp, "/proc/%d/fd/%d", (int)p->pid, fd);
	path = realpath(tmp, NULL);	/* GNU special */
	if (path == NULL) {
		return NULL;
	}
	l = add_path(path); assert(l);
	xfree(path);
	return l;
}

static void
backup_parents(struct path *path) {
	struct stat st;

	if (path == NULL
	 || (path->flags & (PATH_NOBACKUP | PATH_IGNORE))) {
		return;
	}
	backup_parents(path->pdir);
#ifndef NDEBUG
	if (debug_fp) {
		fprintf(debug_fp, "backing up parent dir \"%s\"\n", path->name);
	}
#endif
	if (stat(path->name, &st) == -1) {
		file_warn(path->name, "directory has been (re)moved");
	}
	else {
		cpio_write_node(&st, path->name);
	}
	path->flags |= PATH_NOBACKUP;
}

void
backup_file(struct pstate *p, struct path *path) {
	struct stat st;
	char *s;

	if (stat(path->name, &st) == -1) {
		p->flags |= PROCESS_NEWFILE;
	}
	else {
		p->flags &= ~PROCESS_NEWFILE;
		if (!S_ISREG(st.st_mode)) {
			return;
		}
		s = realpath(path->name, NULL);	/* GNU special */
		assert(s);
		path = add_path(s); assert(path);
		if (cpio_fd != -1
		 && !(path->flags & (PATH_NOBACKUP | PATH_IGNORE))) {
			backup_parents(path->pdir);
			cpio_write_node(&st, path->name);
		}
	}
	path->flags |= PATH_NOBACKUP;
}

void
backup_node(struct pstate *p, struct path *path) {
	struct stat st;

	if (lstat(path->name, &st) == -1) {
		p->flags |= PROCESS_NEWFILE;
	}
	else {
		p->flags &= ~PROCESS_NEWFILE;
		if (cpio_fd != -1
		 && !(path->flags & (PATH_NOBACKUP | PATH_IGNORE))) {
			backup_parents(path->pdir);
			cpio_write_node(&st, path->name);
		}
	}
	path->flags |= PATH_NOBACKUP;
}

void
report_file(struct pstate *p, struct path *path) {
	struct stat st;

	if (stat(path->name, &st) == -1) {
		fprintf(stderr, "trackfs: warning: stat(\"%s\") failed\n",
			path->name);
		return;
	}
	if (S_ISREG(st.st_mode)) {
		if (p->flags & PROCESS_NEWFILE) {
			action(ACTION_CREATE, path);
		}
		else {
			action(ACTION_UPDATE, path);
		}
	}
	p->flags &= ~PROCESS_NEWFILE;
}

void
report_move(struct pstate *p, struct path *path, struct path *path2) {
	if (p->flags & PROCESS_NEWFILE) {
		action(ACTION_CREATE, path2);
	}
	else {
		action(ACTION_UPDATE, path2);
	}
	p->flags &= ~PROCESS_NEWFILE;
	action(ACTION_DELETE, path);
}

void
report_node(struct pstate *p, struct path *path) {
	action(ACTION_CREATE, path);
}

void
report_read(struct pstate *p, struct path *path) {
	action(ACTION_READ, path);
}

void
report_remove(struct pstate *p, struct path *path) {
	action(ACTION_DELETE, path);
}
