#include <gtk/gtk.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <libintl.h>

#include "config.h"
#include "gencfs.h"
#include "encryptedfolders.h"


gboolean is_mounted (const gchar *mountpoint) {
	gboolean mounted;
	gchar *content, *search, **lines;
	gint i;
	
	g_file_get_contents ("/proc/mounts", &content, NULL, NULL);
	lines = g_strsplit (content, "\n", 0);
	g_free (content);
	search = g_strdup_printf("encfs %s/", mountpoint);
	mounted = FALSE;
	for (i=0; !mounted && i < g_strv_length(lines); i++) {
		if (g_ascii_strncasecmp(search, lines[i], strlen(search)) == 0) {
			mounted = TRUE;
		}
	}
	g_free (search);
	g_strfreev (lines);
	
	return mounted;
}


void load_encryptedfolders_wrapper () {
	GtkListStore *model;
	GtkTreeView *view;
	
	view = (GtkTreeView *) gtk_builder_get_object (widgetstree, "encryptedfolders");
	model = (GtkListStore *) gtk_tree_view_get_model (view);
	gtk_list_store_clear (model);
	load_encryptedfolders (g_getenv("HOME"), MAXDEPTH);
}


void load_encryptedfolders (const gchar *folderpath, gint maxdepth) {
	GDir *folder;
	const gchar *folderentry;
	gchar *entrypath;
	GtkTreeIter iter;
	GtkListStore *model;
	GtkTreeView *view;
	
	view = (GtkTreeView *) gtk_builder_get_object (widgetstree, "encryptedfolders");
	model = (GtkListStore *) gtk_tree_view_get_model (view);
	
	folder = g_dir_open (folderpath, 0, NULL);
	folderentry = g_dir_read_name (folder);
	while (folderentry != NULL) {
		if (g_ascii_strcasecmp(".encfs6.xml", folderentry) == 0) {
			gtk_list_store_append (model, &iter);
			gtk_list_store_set (model, &iter, ENCRYPTEDFOLDER, folderpath, MOUNTED, is_mounted (folderpath), -1);
			break;
		} else {
			entrypath = g_strdup_printf("%s/%s", folderpath, folderentry);
			if (maxdepth != 0 && g_file_test (entrypath, G_FILE_TEST_IS_DIR)) {
				load_encryptedfolders(entrypath, maxdepth - 1);
			}
			g_free(entrypath);
		}
		folderentry = g_dir_read_name (folder);
	}
	g_dir_close (folder);
}


void add_encryptedfolder (GtkWidget *widget, gpointer user_data) {
	GtkTreeIter iter;
	GtkTreeView *view;
	GtkListStore *model;
	GtkFileChooser *encryptedfolderwidget;
	gchar *encryptedfolder;
	
	encryptedfolderwidget = (GtkFileChooser *) gtk_builder_get_object(widgetstree, "encryptedfolder");
	encryptedfolder = gtk_file_chooser_get_filename(encryptedfolderwidget);
	mount_encryptedfolder (encryptedfolder);
	g_free(encryptedfolder);
}


void on_mounted_toggled (GtkWidget *widget, gpointer user_data) {
	GtkTreeIter iter;
	GtkTreeSelection *selection;
	GtkTreeView *view;
	GtkListStore *model;
	gchar *encryptedfolder;
	gboolean mounted;
	
	view = (GtkTreeView *) gtk_builder_get_object (widgetstree, "encryptedfolders");
	selection = gtk_tree_view_get_selection (view);
	gtk_tree_selection_get_selected (selection, (GtkTreeModel **) &model, &iter);
	gtk_tree_model_get ((GtkTreeModel *) model, &iter, ENCRYPTEDFOLDER, &encryptedfolder, MOUNTED, &mounted, -1);
	if (mounted) {
		umount_encryptedfolder (encryptedfolder);
	} else {
		mount_encryptedfolder (encryptedfolder);
	}
}


void mount_encryptedfolder (const gchar *encryptedfolder) {
	gchar *mountpoint, *commandline, **command;
	GPid encfspid;
	
	mountpoint = g_strdup_printf ("%s/%s", encryptedfolder, MOUNTPOINT);
	g_mkdir (mountpoint, 0700);
	commandline = g_strdup_printf ("setsid encfs --extpass gencfs-askpass '%s' '%s'", encryptedfolder, mountpoint);
	g_shell_parse_argv (commandline, NULL, &command, NULL);
	g_free (commandline);
	g_spawn_async (NULL, command, NULL, 
				G_SPAWN_SEARCH_PATH|G_SPAWN_DO_NOT_REAP_CHILD, 
				NULL, NULL, &encfspid, NULL);
	g_child_watch_add (encfspid, (GChildWatchFunc) askpassfinished, (gpointer) mountpoint);
	g_strfreev (command);
}


void umount_encryptedfolder (const gchar *encryptedfolder) {
	gchar *mountpoint, *command;
	
	mountpoint = g_strdup_printf ("%s/%s", encryptedfolder, MOUNTPOINT);
	command = g_strdup_printf ("fusermount -u '%s'", mountpoint);
	g_spawn_command_line_sync (command, NULL, NULL, NULL, NULL);
	g_free (command);
	g_rmdir (mountpoint);
	g_free(mountpoint);
	load_encryptedfolders_wrapper();
}


void askpassfinished (GPid pid, gint status, gpointer data) {
	gchar *command;
	gchar *mountpoint = (gchar *) data;
	load_encryptedfolders_wrapper ();
	if (status == 0) {
		command = g_strdup_printf ("exo-open 'file://%s'", mountpoint);
		g_spawn_command_line_sync (command, NULL, NULL, NULL, NULL);
		g_free (command);
	}
	g_free (mountpoint);
}
