/*
  Copyright (C) 2008 Ben Asselstine
  Written by Ben Asselstine

  This file is part of fileschanged.

  fileschanged is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  fileschanged is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with fileschanged; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor Boston, MA 02110-1301  USA
*/
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <time.h>
#include <argz.h>
#include "wl_priv.h"
#include "fileschanged_priv.h"

int
wl_init (wl_t *wl)
{
  struct wl_priv_t *w = malloc (sizeof (struct wl_priv_t));
  if (!w)
    return -1;
  memset (w, 0, sizeof (struct wl_priv_t));
  *wl = (wl_t) w;
  return 0;
}

static char *
argz_search (char *argz, size_t len, char *search)
{
  char *last_entry = NULL;
  char *entry;
  while ((entry = argz_next (argz, len, last_entry)))
    {
      if (strcmp (search, entry)==0)
	return entry;
      last_entry = entry;
    }
  return NULL;
}

void 
wl_add_file (wl_t wl, char *filename)
{
  struct wl_priv_t *w = (struct wl_priv_t *) wl;
  char *found;
  found = argz_search (w->argz, w->argz_len, filename);
  if (!found)
    argz_add (&w->argz, &w->argz_len, filename);
  return;
}

void 
wl_destroy(wl_t wl)
{
  struct wl_priv_t *w = (struct wl_priv_t *) wl;
  free (w->argz);
  free (w);
  return;
}

void 
wl_show_changed_files(wl_t wl, unsigned int filechangetimeout, void *hook, int (*handler)(char *filename, enum fileschanged_action_enum_t action, void *hook))
{
  struct wl_priv_t *w = (struct wl_priv_t *) wl;
  static time_t lastchecked;
  if ((lastchecked != time (NULL)) && (w->argz_len > 0)) //check every sec.
    {
      int retval;
      struct stat statbuf;
      char *filename;
      //printf ("searching tree for files that are old and written to...\n");
      char *last_filename = NULL;
      while ((filename = argz_next (w->argz, w->argz_len, last_filename)))
	{
	  w->index++;
	  //printf ("trying index %d with max %d\n", w->index, w->argz_len);
	  //printf ("checking %s...\n", filename);
	  retval = lstat (filename, &statbuf);
	  if ((retval == 0) && 
	      (statbuf.st_ctime < (time (NULL) - filechangetimeout)))
	    {
	      handler (filename, FC_CHANGED_FILE, hook);

	      //printf ("removing\n");
	      argz_delete (&(w->argz), &(w->argz_len), filename);
	      if (w->argz_len == 0)
		break;
	      w->index--;
	    }
	  else if (retval == -1) //can't stat file.  must be gone.
	    {
	      argz_delete (&(w->argz), &(w->argz_len), filename);
	      if (w->argz_len == 0)
		break;
	      w->index--;
	    }

	  last_filename = filename;
	}
      lastchecked = time (NULL);
    }
  return;
}
