/*
  Copyright (C) 2008 Ben Asselstine
  Written by Ben Asselstine

  This file is part of fileschanged.

  fileschanged is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  fileschanged is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with fileschanged; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor Boston, MA 02110-1301  USA
*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "fileschanged_priv.h"
#include "monitor.h"
#include "list.h"
#include "wl.h"
#include "filelist.h"

int 
fileschanged_init (fileschanged_t *fc, char *filelist_argz, 
		   size_t filelist_argz_len, unsigned int recurse,
		   unsigned int dereference_symlinks, int timeout)
{
  int err = 0;
  struct fileschanged_priv_t *f = malloc (sizeof (struct fileschanged_priv_t));
  if (!f)
    return -1;
  memset (f, 0, sizeof (struct fileschanged_priv_t));
  wl_init (&f->wl);
  f->timeout = timeout;
  err = FAMOpen (&f->c);
  if (err)
    {
      free (f);
      return err;
    }

  FAMNoExists (&f->c);
  list_init (&f->list_of_files_to_monitor);
  err = filelist_populate (f->list_of_files_to_monitor, filelist_argz, filelist_argz_len, recurse, dereference_symlinks);
  if (err)
    {
      free (f);
      return err;
    }

  *fc = (fileschanged_t) f;
  return err;
}

int 
fileschanged_monitor (fileschanged_t fc, void *hook, int (*handler)(char *filename, enum fileschanged_action_enum_t action, void *hook))
{
  int err = 0;
  struct fileschanged_priv_t *f = (struct fileschanged_priv_t*) fc;
  monitor_begin (f, hook, f->list_of_files_to_monitor);
  while (1)
    {
      err = monitor_handle_events (f, 0, -1, hook, handler);
      if (!err)
	{
	  if (f->timeout > 0)
	    {
	      wl_show_changed_files (f->wl, f->timeout, hook, handler);
	    }
	}
      else if (err == 1)
	break;
    }

  return err;
}

void 
fileschanged_destroy (fileschanged_t fc)
{
  struct fileschanged_priv_t *f = (struct fileschanged_priv_t*) fc;
  monitor_stop (f);
  FAMClose (&f->c);
  list_free (f->list_of_files_to_monitor);
  wl_destroy (f->wl);
  free (f);
  return;
}
