/* Copyright (c) 2006 Dirk Jagdmann <doj@cubic.org>

This software is provided 'as-is', without any express or implied
warranty. In no event will the authors be held liable for any damages
arising from the use of this software.

Permission is granted to anyone to use this software for any purpose,
including commercial applications, and to alter it and redistribute it
freely, subject to the following restrictions:

    1. The origin of this software must not be misrepresented; you
       must not claim that you wrote the original software. If you use
       this software in a product, an acknowledgment in the product
       documentation would be appreciated but is not required.

    2. Altered source versions must be plainly marked as such, and
       must not be misrepresented as being the original software.

    3. This notice may not be removed or altered from any source
       distribution. */

/* $Header: /code/cbmfs/d81.cpp,v 1.4 2006/07/21 15:05:58 doj Exp $ */

#include "d81.hpp"
#include <stdio.h>

D81::D81(BYTE *disk_) :
  CBMdisk(disk_)
{
  bam_set(40,0,0);
  bam_set(40,1,0);
  bam_set(40,2,0);
  bam_set(40,3,0);
}

int D81::track_max() { return 80; }
int D81::maxblocks() { return 3200; }
CBMdisk::TS D81::dir() { return TS(40,3); }

void D81::fix(bamentry_81 *e, const BYTE t)
{
  const int64_t b=e->bm0 | (e->bm1 << 8) | (e->bm2 << 16) | (e->bm3 << 24) | (static_cast<int64_t>(e->bm4) << 32);
  const int sectors=this->sectorsPerTrack()[t];
  e->freeblocks=0;
  for(int i=0; i<sectors; i++)
    if(b&(1<<i))
      e->freeblocks++;
}

D81::bamentry_81* D81::getbamentry_81(const BYTE t)
{
  if(t==0)
    return NULL;
  if(t<=40)
    {
      bam_81 *BAM=reinterpret_cast<bam_81*>(this->getsector(40,1));
      if(!BAM) return NULL;
      return &(BAM->entry[t-1]);
    }
  if(t<=80)
    {
      bam_81 *BAM=reinterpret_cast<bam_81*>(this->getsector(40,2));
      if(!BAM) return NULL;
      return &(BAM->entry[t-41]);
    }
  return NULL;
}

void D81::bam_set(const BYTE t, const BYTE s, const bool b)
{
  bamentry_81 *e=this->getbamentry_81(t);
  if(!e)
    return;

  if(s<8)
    set_bit(e->bm0, s, b);
  else if(s<16)
    set_bit(e->bm1, s-8, b);
  else if(s<24)
    set_bit(e->bm2, s-16, b);
  else if(s<32)
    set_bit(e->bm3, s-24, b);
  else if(s<40)
    set_bit(e->bm4, s-32, b);
  else
    return;

  this->fix(e, t);
}

bool D81::bam_get(const BYTE t, const BYTE s)
{
  bamentry_81 *e=this->getbamentry_81(t);
  if(!e) return false;

  if(s<8)
    return get_bit(e->bm0, s);
  if(s<16)
    return get_bit(e->bm1, s-8);
  if(s<24)
    return get_bit(e->bm2, s-16);
  if(s<32)
    return get_bit(e->bm3, s-24);
  if(s<40)
    return get_bit(e->bm4, s-32);

  return false;
}

const int* D81::sectorsPerTrack()
{
  static const int sectorsPerTrack_d81[] = {
    -1,
    40, 40, 40, 40, 40, 40, 40, 40, 40, 40,
    40, 40, 40, 40, 40, 40, 40, 40, 40, 40,
    40, 40, 40, 40, 40, 40, 40, 40, 40, 40,
    40, 40, 40, 40, 40, 40, 40, 40, 40, 40,
    40, 40, 40, 40, 40, 40, 40, 40, 40, 40,
    40, 40, 40, 40, 40, 40, 40, 40, 40, 40,
    40, 40, 40, 40, 40, 40, 40, 40, 40, 40,
    40, 40, 40, 40, 40, 40, 40, 40, 40, 40,
  };
  return sectorsPerTrack_d81;
}

CBMdisk::bamentry* D81::getbamentry(const BYTE t)
{
  return reinterpret_cast<bamentry*>(getbamentry_81(t));
}

void D81::fix(bamentry*, const BYTE t)
{
  syslog(LOG_ERR, "D81::fix(): this function should not be called!\n");
}

void D81::printbam()
{
  printf("BAM: D81 80 track image\n");
  this->printheaderinfo();
  this->printbamblocks();
}

void D81::printheaderinfo()
{
  const header_81 *HEADER=reinterpret_cast<header_81*>(getsector(40,0));
  fuse_assert(HEADER);
  printf("  dir=%i:%i version=%c unused1=0x%02X unused2=0x%02X,0x%02X id=%c%c unused3=0x%02X DOSversion=%i DiskVersion=%i\n",
	 HEADER->dir.track, HEADER->dir.sector,
	 HEADER->version,
	 HEADER->unused1,
	 HEADER->unused2[0], HEADER->unused2[1],
	 HEADER->id[0], HEADER->id[1],
	 HEADER->unused3,
	 HEADER->DOSversion,
	 HEADER->DiskVersion
	 );
}

bool D81::reserved_track(const BYTE t) { return t==40; }
