/* Copyright (c) 2006 Dirk Jagdmann <doj@cubic.org>

This software is provided 'as-is', without any express or implied
warranty. In no event will the authors be held liable for any damages
arising from the use of this software.

Permission is granted to anyone to use this software for any purpose,
including commercial applications, and to alter it and redistribute it
freely, subject to the following restrictions:

    1. The origin of this software must not be misrepresented; you
       must not claim that you wrote the original software. If you use
       this software in a product, an acknowledgment in the product
       documentation would be appreciated but is not required.

    2. Altered source versions must be plainly marked as such, and
       must not be misrepresented as being the original software.

    3. This notice may not be removed or altered from any source
       distribution. */

/* $Header: /code/cbmfs/d64.cpp,v 1.4 2006/07/21 14:46:04 doj Exp $ */

#include "d64.hpp"
#include <stdio.h>

D64::D64(BYTE *disk_) :
  CBMdisk(disk_)
{
  bam_set(18,0,0);
  bam_set(18,1,0);
}

int D64::track_max() { return 35; }
const int* D64::sectorsPerTrack()
{
  static const int sectorsPerTrack_d64[] = {
    -1,
    21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21,
    19, 19, 19, 19, 19, 19, 19,
    18, 18, 18, 18, 18, 18,
    17, 17, 17, 17, 17,
    /* extra tracks */
    17, 17, 17, 17, 17, 17, 17
  };
  return sectorsPerTrack_d64;
}

CBMdisk::TS D64::dir() { return TS(18,1); }
void D64::fix(bamentry *e, const BYTE t)
{
  fuse_assert(e);
  if(t==0 || t>this->track_max())
    return;

  const int sectors=this->sectorsPerTrack()[t];

  e->freeblocks=0;
  const int32_t b=e->bm0 | (e->bm1 << 8) | (e->bm2 << 16);
  for(int i=0; i<sectors; i++)
    if(b&(1<<i))
      e->freeblocks++;
}

CBMdisk::bamentry* D64::getbamentry(const BYTE t)
{
  if(t==0 || t>this->track_max()) return NULL;

  bam_64 *BAM=reinterpret_cast<bam_64*>(getsector(18,0));
  if(!BAM) return NULL;
  return &(BAM->entry[t-1]);
}

void D64::bam_set(const BYTE t, const BYTE s, const bool b)
{
  bamentry *e=this->getbamentry(t);
  if(!e)
    return;

  if(s<8)
    set_bit(e->bm0, s, b);
  else if(s<16)
    set_bit(e->bm1, s-8, b);
  else if(s<24)
    set_bit(e->bm2, s-16, b);
  else
    return;

  this->fix(e, t);
}

bool D64::bam_get(const BYTE t, const BYTE s)
{
  const bamentry *e=this->getbamentry(t);
  if(!e) return false;

  if(s<8)
    return get_bit(e->bm0, s);
  if(s<16)
    return get_bit(e->bm1, s-8);
  if(s<24)
    return get_bit(e->bm2, s-16);

  return false;
}

int D64::maxblocks() { return 683; }

void D64::printbaminfo()
{
  const bam_64 *BAM=reinterpret_cast<bam_64*>(getsector(18,0));
  fuse_assert(BAM);
  printf("  dir=%i:%i version=%c oneside=%i unused1=0x%02X,0x%02X id=%c%c unused2=0x%02X DOStype=%c%c unused3=0x%02X,0x%02X,0x%02X,0x%02X\n",
	 BAM->dir.track, BAM->dir.sector,
	 BAM->version,
	 BAM->onesided,
	 BAM->unused1[0], BAM->unused1[1],
	 BAM->id[0], BAM->id[1],
	 BAM->unused2,
	 BAM->DOStype[0], BAM->DOStype[1],
	 BAM->unused3[0], BAM->unused3[1], BAM->unused3[2], BAM->unused3[3]);
}

void D64::printbam()
{
  printf("BAM: D64 35 track image\n");
  this->printbaminfo();
  this->printbamblocks();
}

bool D64::reserved_track(const BYTE t) { return t==18; }
