/* Copyright (c) 2006 Dirk Jagdmann <doj@cubic.org>

This software is provided 'as-is', without any express or implied
warranty. In no event will the authors be held liable for any damages
arising from the use of this software.

Permission is granted to anyone to use this software for any purpose,
including commercial applications, and to alter it and redistribute it
freely, subject to the following restrictions:

    1. The origin of this software must not be misrepresented; you
       must not claim that you wrote the original software. If you use
       this software in a product, an acknowledgment in the product
       documentation would be appreciated but is not required.

    2. Altered source versions must be plainly marked as such, and
       must not be misrepresented as being the original software.

    3. This notice may not be removed or altered from any source
       distribution. */

/* $Header: /code/cbmfs/cbmdisk.cpp,v 1.6 2006/07/21 14:46:04 doj Exp $ */

#include "cbmdisk.hpp"
#include <stdio.h>
#include <ctype.h>

CBMdisk::CBMdisk(BYTE *disk_) :
  disk(disk_)
{
  fuse_assert(disk);
}

CBMdisk::filesector* CBMdisk::getsector(TS ts)
{
  if(ts.track==0 || ts.track > this->track_max())
    return NULL;

  if(ts.sector >= this->sectorsPerTrack()[ts.track])
    return NULL;

  int i=ts.sector;
  while(--ts.track)
    i+=this->sectorsPerTrack()[ts.track];
  return reinterpret_cast<filesector*>(disk + i*SECTOR_SIZE);
}

const int CBMdisk::fileLength(const TS& ts)
{
  //SYSLOG(LOG_DEBUG, "fileLengthTS arg track=%i sector=%i\n", ts.track, ts.sector);
  if(ts == this->dir())
    return 0;

  int l=0;
  const filesector *s=getsector(ts);
  while(s)
    {
      //SYSLOG(LOG_DEBUG, "fileLengthTS track=%i sector=%i\n", s->next.track, s->next.sector);
      l+=(s->next.track==0)?(s->next.sector):SECTOR_DATA_SIZE;
      if(s->next.track)
	s=getsector(s->next);
      else
	s=NULL;
    }

  //SYSLOG(LOG_DEBUG, "fileLengthTS return %i\n", l);
  return l;
}

CBMdisk::direntry* CBMdisk::get_direntry(const char *filename)
{
  fuse_assert(filename);
  fix_filename(filename);

  SYSLOG(LOG_INFO, "direntry(%s)\n", filename);

  TS dirts=this->dir();
  do
    {
      dirsector *sec=getdirsector(dirts);
      if(!sec)
	{
	  SYSLOG(LOG_DEBUG, "direntry(%s) not found: no dirsector\n", filename);
	  return NULL;
	}

      int i;
      for(i=0; i<8; i++)
	{
	  direntry *e=&(sec->entry[i]);

	  if(e->FileType==0)
	    continue;

	  if(!d64strcmp(e->filename, filename))
	    {
	      SYSLOG(LOG_DEBUG, "direntry(%s) found\n", filename);
	      return e;
	    }
	}
      dirts=sec->entry[0].next;
    }
  while(dirts.track!=0);

  SYSLOG(LOG_DEBUG, "direntry(%s) not found\n", filename);
  return NULL;
}

int CBMdisk::filenum()
{
  int z=0;
  TS dirts=this->dir();
  do
    {
      const dirsector *sec=getdirsector(dirts);
      if(!sec)
	return z;

      int i;
      for(i=0; i<8; i++)
	{
	  const struct direntry *e=&(sec->entry[i]);
	  if(e->FileType==0)
	    continue;
	  ++z;
	}
      dirts=sec->entry[0].next;
    }
  while(dirts.track!=0);

  return z;
}

int CBMdisk::freeblocks()
{
  int f=0, t;
  for(t=1; t<=this->track_max(); t++)
    {
      bamentry *e=this->getbamentry(t);
      if(!e)
	return f;
      this->fix(e, t);
      f+=e->freeblocks;
    }
  return f;
}

/**
   search for a free sector. If a free sector is found, mark is as
   allocated and return it. The track/sector numbers of the block are
   written to the "next" attribute. The data attribute is filled with
   0.
   @return an allocated sector. NULL if no sector was found
*/
CBMdisk::filesector* CBMdisk::newsector()
{
  for(BYTE t=1; t<=this->track_max(); ++t)
    {
      if(this->reserved_track(t))
	continue;
      const int sectors=this->sectorsPerTrack()[t];
      BYTE s=0;
      do {
	TS ts(t,s);
	if(this->bam_get(ts))
	  {
	    filesector *sec=getsector(ts);
	    if(sec)
	      {
		this->bam_set(ts,0); /* mark sector as allocated */
		sec->next=ts;
		memset(sec->data, 0, SECTOR_DATA_SIZE);
		SYSLOG(LOG_DEBUG, "cbmfs_newsector(): new(%i,%i)\n", sec->next.track, sec->next.sector);
		return sec;
	      }
	  }
      } while(++s < sectors);
    }
  return NULL;
}

/**
   search for a free directory entry.
   @return the new directory entry, properly linked. NULL is no directory entry could be allocated.
*/
CBMdisk::direntry* CBMdisk::new_direntry()
{
  int used_sec[40]; memset(used_sec, 0, sizeof(used_sec));
  /* search for end of directory */
  struct dirsector *sec=0;
  const TS dirstart=this->dir();
  TS dirts=dirstart, oldts=dirts;
  do
    {
      sec=getdirsector(dirts);
      if(!sec)
	return NULL;
      used_sec[dirts.sector]=1;
      int i;
      for(i=0; i<8; i++)
	{
	  struct direntry *e=&(sec->entry[i]);
	  if(e->FileType == 0x00)
	    {
	      //SYSLOG(LOG_DEBUG, "cbmfs_newdirentry(): sector(%i,%i) entry %i\n", dirts.track, dirts.sector, i);
	      return e;
	    }
	}

      oldts=dirts;
      dirts=sec->entry[0].next;
    }
  while(dirts.track!=0);

  /* check if sector in dir track is free */
  int free=0;
  for(unsigned i=1; i<sizeof(used_sec); ++i)
    if(used_sec[i] == 0)
      ++free;

  if(free)
    {
      /* search for a new directory sector */
      const int sectors=this->sectorsPerTrack()[dirstart.track];
      dirts=oldts;
      dirts.track=dirstart.track;
      while(used_sec[dirts.sector])
	{
	  dirts.sector++;
	  if(dirts.sector >= sectors) dirts.sector-=sectors;
	  if(dirts.sector == 0)  dirts.sector=1;

	  /* this is a guard, but should not happen */
	  if(dirts.sector == oldts.sector)
	    goto alloc_new;
	}

    }
  else
    {
    alloc_new: ;
      /* alloc new directory sector in free space */
      dirsector* newsec=reinterpret_cast<dirsector*>(newsector());
      if(!newsec)
	return NULL;
      dirts=newsec->entry[0].next;
    }

  SYSLOG(LOG_DEBUG, "cbmfs_newdirentry(): sector(%i,%i)\n", dirts.track, dirts.sector);

  dirsector* newsec=getdirsector(dirts);
  if(!newsec)
    return NULL;

  /* mark sector as allocated */
  this->bam_set(dirts,0);
  /* link to new sector */
  sec->entry[0].next=dirts;
  /* setup new sector */
  memset(newsec, 0, SECTOR_SIZE);
  direntry *e=&newsec->entry[0];
  e->next.sector=0xFF;
  return e;
}

void CBMdisk::printbamblocks()
{
  int z=0,f=0,f2=0;
  for(int t=1; t<=this->track_max(); ++t)
    {
      printf("  %02i: ", t);
      for(int s=0; s<this->sectorsPerTrack()[t]; ++s, ++z)
	{
	  const bool b=this->bam_get(t,s);
	  if(b) ++f;
	  printf("%i", b);
	  if((s%8)==7)
	    printf(" ");
	}

      bamentry *e=this->getbamentry(t);
      if(e)
	{
	  //this->fix(e, t);
	  printf("\t%i free blocks", e->freeblocks);
	  f2+=e->freeblocks;
	}
      printf("\n");
    }
  printf("total blocks=%i\tfree blocks=%i\n", z, f);
  if(f!=f2)
    {
      printf("freeblocks mismatch %i!=%i\n", f, f2);
    }
}

void CBMdisk::printdir()
{
  printf("DIR:\n");
  TS dirts=this->dir();
  do
    {
      printf("  TS(%i,%i)\n", dirts.track, dirts.sector);
      const dirsector *sec=this->getdirsector(dirts);
      if(!sec)
	return;
      for(int i=0; i<8; i++)
	{
	  printf("    Entry %i\n", i);
	  const struct direntry *e=&(sec->entry[i]);
	  printf("      next(%i,%i)\n", e->next.track, e->next.sector);

	  printf("      FileType 0x%02X ", e->FileType);
	  if(e->FileType & 0x20) putchar('@');
	  if(e->FileType & 0x40) putchar('>');
	  if(!(e->FileType&0x80))putchar('*');
	  switch(e->FileType & 0x0F)
	    {
	    case 0: printf("DEL"); break;
	    case 1: printf("SEQ"); break;
	    case 2: printf("PRG"); break;
	    case 3: printf("USR"); break;
	    case 4: printf("REL"); break;
	    case 5: printf("CBM"); break;
	    case 6: printf("DIR"); break;
	    default:printf("???");
	    }
	  putchar('\n');

	  printf("      file(%i,%i)\n", e->file.track, e->file.sector);
	  printf("      name:");
	  for(int j=0; j<16; ++j)
	    putchar(e->filename[j]);
	  printf("\n");

	  printf("      RELlength: %i RELsector:%02X,%02X\n", e->RELlength, e->REL.track, e->REL.sector);
	  printf("      date: %04i-%02i-%02i %02i:%02i\n", e->geos_year+1900, e->geos_month, e->geos_day, e->geos_hour, e->geos_minute);
	  printf("      blocks:%i\n", e->totalSectorsLo + e->totalSectorsHi*256);
	}

      dirts=sec->entry[0].next;
    }
  while(dirts.track!=0);
}

void CBMdisk::printsectors()
{
  for(int t=1; t<=this->track_max(); ++t)
    for(int s=0; s<this->sectorsPerTrack()[t]; ++s)
      {
	const filesector *sec=this->getsector(t,s);
	if(!sec)
	  {
	    printf("COULD NOT GET SECTOR %i:%i\n",t,s);
	    continue;
	  }
	printf("SECTOR %i:%i free=%i next=%i:%i\n", t, s, !!this->bam_get(t,s), sec->next.track, sec->next.sector);
	const BYTE *d=(const BYTE*)sec;
	for(int i=0; i<SECTOR_SIZE; i+=16)
	  {
	    printf("  0x%02X: ", i);
	    int j;
	    for(j=0; j<8; ++j)
	      printf("%02X ", d[i+j]);
	    putchar(' ');
	    for(; j<16; ++j)
	      printf("%02X ", d[i+j]);
	    putchar(' ');
	    for(j=0; j<8; ++j)
	      {
		BYTE b=d[i+j];
		putchar(isprint(b)?b:'.');
	      }
	    putchar(' ');
	    for(; j<16; ++j)
	      {
		BYTE b=d[i+j];
		putchar(isprint(b)?b:'.');
	      }
	    putchar('\n');
	  }
      }
}
