/*
 * Pidgin Authorization Blocker
 * Copyright (C) 2011 Konrad Gräfe
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02111-1301, USA.
 */

#include "authorization-blocker.h"

#include "prefs.h"

#include <debug.h>

#include <gtk/gtk.h>
#include <gtkblist.h>
#include <gtkutils.h>

extern PurplePlugin *plugin;

static GtkWidget *make_info_widget(const gchar *markup, gchar *stock_id, gboolean indent) {
	GtkWidget *infobox, *label, *img, *align;

	if(!markup) return NULL;

	infobox = gtk_hbox_new(FALSE, 5);

	if(indent) {
		label = gtk_label_new("");
		gtk_box_pack_start(GTK_BOX(infobox), label, FALSE, FALSE, 10);
	}

	if(stock_id) {
		align = gtk_alignment_new(0.5, 0, 0, 0); /* align img to the top of the space */
		gtk_box_pack_start(GTK_BOX(infobox), align, FALSE, FALSE, 0);

		img = gtk_image_new_from_stock(stock_id, GTK_ICON_SIZE_MENU);
		gtk_container_add(GTK_CONTAINER(align), img);
	}

	label = gtk_label_new(NULL);
	gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);
	gtk_label_set_markup(GTK_LABEL(label), markup);
	gtk_box_pack_start(GTK_BOX(infobox), label, FALSE, FALSE, 0);

	return infobox;
}

typedef struct _config_frame {
	GtkWidget *frame;
	GtkWidget *toggleBlockAuth;
	GtkWidget *textview;
	GtkWidget *toggleBlockDeny;
	GtkWidget *toggleDebug;
} ConfigFrame;
static ConfigFrame config_frame;

static void config_frame_destroy_cb(GtkWidget *w, gpointer user_data){
	GtkTextBuffer *buffer;
	GtkTextIter begin, end;
	gchar *newtext, *oldtext, *newtext_escaped;

	buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(config_frame.textview));
	gtk_text_buffer_get_start_iter(buffer, &begin);
	gtk_text_buffer_get_end_iter(buffer, &end);
	newtext = gtk_text_buffer_get_text(buffer, &begin, &end, TRUE);

	oldtext = prefs_get_deny_message();

	if(purple_utf8_strcasecmp(newtext, oldtext) != 0) {
		if(purple_utf8_strcasecmp(newtext, "") == 0) {
			purple_prefs_remove(PLUGIN_PREFS_PREFIX "/deny_message");
		} else {
			newtext_escaped = g_strescape(newtext, "");
			if(purple_prefs_exists(PLUGIN_PREFS_PREFIX "/deny_message")) {
				purple_prefs_set_string(PLUGIN_PREFS_PREFIX "/deny_message", newtext_escaped);
			} else {
				purple_prefs_add_string(PLUGIN_PREFS_PREFIX "/deny_message", newtext_escaped);
			}
			g_free(newtext_escaped);
		}
	}

}

static void toggle_cb(GtkWidget *widget, gpointer data) {
	gboolean value;
	gchar *pref;

	value = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget));
	pref = (gchar *) data;

	purple_prefs_set_bool(pref, value);

	gtk_widget_set_sensitive(GTK_WIDGET(config_frame.textview), purple_prefs_get_bool(PLUGIN_PREFS_PREFIX "/block_requests"));
}


GtkWidget *get_config_frame(PurplePlugin *plugin) {
	GtkWidget *frame, *textview, *vbox, *infobox, *toggle, *sw, *border;
	GtkTextBuffer *buffer;
	gchar *msg;

	frame = gtk_vbox_new(FALSE, 5);
	gtk_container_set_border_width(GTK_CONTAINER(frame), 12);
	config_frame.frame = frame;

	infobox = make_info_widget(_("Please note that the plugin won't work with other protocols than ICQ."), GTK_STOCK_INFO, FALSE);
	gtk_box_pack_start(GTK_BOX(frame), infobox, FALSE, FALSE, 0);

	toggle = gtk_check_button_new_with_mnemonic(_("Block authorization requests"));
	gtk_box_pack_start(GTK_BOX(frame), toggle, FALSE, FALSE, 0);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(toggle), purple_prefs_get_bool(PLUGIN_PREFS_PREFIX "/block_requests"));
	g_signal_connect(G_OBJECT(toggle), "toggled", G_CALLBACK(toggle_cb), PLUGIN_PREFS_PREFIX "/block_requests");
	config_frame.toggleBlockAuth = toggle;

	vbox = pidgin_make_frame(frame, _("Deny message:"));

	border = gtk_frame_new(NULL);
	gtk_box_pack_start(GTK_BOX(vbox), border, TRUE, TRUE, 0);

	sw = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_container_add(GTK_CONTAINER(border), sw);

	textview = gtk_text_view_new();
	gtk_widget_set_size_request(textview, 350, 150);
	gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(textview), GTK_WRAP_WORD_CHAR);
	gtk_container_add(GTK_CONTAINER(sw), textview);
	gtk_widget_set_sensitive(GTK_WIDGET(textview), purple_prefs_get_bool(PLUGIN_PREFS_PREFIX "/block_requests"));
	config_frame.textview = textview;

	msg = prefs_get_deny_message();
	buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(textview));
	gtk_text_buffer_set_text(buffer, msg, -1);
	g_free(msg);

	toggle = gtk_check_button_new_with_mnemonic(_("Block unrequested deny messages"));
	gtk_box_pack_start(GTK_BOX(frame), toggle, FALSE, FALSE, 0);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(toggle), purple_prefs_get_bool(PLUGIN_PREFS_PREFIX "/block_deny_msg"));
	g_signal_connect(G_OBJECT(toggle), "toggled", G_CALLBACK(toggle_cb), PLUGIN_PREFS_PREFIX "/block_deny_msg");
	config_frame.toggleBlockAuth = toggle;

	toggle = gtk_check_button_new_with_mnemonic(_("Notify about blocked authorization requests"));
	gtk_box_pack_start(GTK_BOX(frame), toggle, FALSE, FALSE, 0);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(toggle), purple_prefs_get_bool(PLUGIN_PREFS_PREFIX "/show_blocked_requests"));
	g_signal_connect(G_OBJECT(toggle), "toggled", G_CALLBACK(toggle_cb), PLUGIN_PREFS_PREFIX "/show_blocked_requests");
	config_frame.toggleDebug = toggle;

	g_signal_connect(G_OBJECT(frame), "destroy", G_CALLBACK(config_frame_destroy_cb), NULL);


	gtk_widget_show_all(frame);

	return frame;
}

gchar *prefs_get_deny_message(void) {
	gchar *msg;

	if(purple_prefs_exists(PLUGIN_PREFS_PREFIX "/deny_message")) {
		msg = g_strcompress(purple_prefs_get_string(PLUGIN_PREFS_PREFIX "/deny_message"));
	} else {
		msg = g_strdup_printf(_("Your authorization request has been denied automatically.\n\nPlease re-request the authorization to get it executed normally."));
	}

	return msg;

}

void init_prefs(void) {
	purple_prefs_add_none(PLUGIN_PREFS_PREFIX);

	purple_prefs_add_bool(PLUGIN_PREFS_PREFIX "/block_requests", TRUE);
	purple_prefs_add_bool(PLUGIN_PREFS_PREFIX "/block_deny_msg", TRUE);
	purple_prefs_add_bool(PLUGIN_PREFS_PREFIX "/show_blocked_requests", FALSE);

	/* TODO: import from "/plugins/gtk/authorization-greylisting" */
}

