/*
 * This file is part of libbluray
 * Copyright (C) 2015  Petri Hintukainen <phintuka@users.sourceforge.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 */

package org.videolan;

import jail.java.io.File;
import java.util.PropertyPermission;
import java.security.AccessControlContext;
import java.security.AccessController;
import java.security.Permission;
import java.security.PrivilegedAction;
import blues.Log;

final public class BDJSecurityManager extends SecurityManager {

    private String discRoot;
    private String cacheRoot;
    private String budaRoot;
    private String persistentRoot;
    private boolean usingUdf = false;
    private AccessControlContext libblurayContext;

    private static Class urlPermission = null;
    static {
        try {
            /* Java 8 */
            urlPermission = Class.forName("java.net.URLPermission");
        } catch (Exception e) {
        }
    }

    BDJSecurityManager(String discRoot, String persistentRoot, String budaRoot) {
        this.discRoot = discRoot;
        this.cacheRoot = null;
        this.budaRoot = budaRoot;
        this.persistentRoot = persistentRoot;
        if (discRoot == null) {
            usingUdf = true;
        }
        libblurayContext = AccessController.getContext();
    }

    public void setCacheRoot(String root) {
        if (cacheRoot != null) {
            // limit only
            if (!root.startsWith(cacheRoot)) {
                logger.error("setCacheRoot(" + root + ") denied\n" + Logger.dumpStack());
                throw new SecurityException("cache root already set");
            }
        }
        cacheRoot = root;
    }
    
    /*
     *
     */

    private void deny(Permission perm) {
        Log.log(2, Log.LOG_LIBBLURAY,"denied " , perm.toString() );
        throw new SecurityException("denied " + perm);
    }
    
    public void checkPermission(Permission perm) {
        
        AccessControlContext acc = AccessController.getContext();
        if (acc==null) return;
        if (acc.equals(libblurayContext)) return;
       
        if (perm instanceof RuntimePermission) {
            if (perm.implies(new RuntimePermission("createSecurityManager"))) {

                // allow initializing of javax.crypto.JceSecurityManager
                if (classDepth("javax.crypto.JceSecurityManager") < 3) {
                    return;
                }

                deny(perm);
            }
            if (perm.implies(new RuntimePermission("setSecurityManager"))) {
                if (classDepth("org.videolan.Libbluray") == 3) {
                    return;
                }
                deny(perm);
            }

            // work around bug in openjdk 7 / 8
            // sun.awt.AWTAutoShutdown.notifyThreadBusy is missing doPrivileged()
            // (fixed in jdk9 / http://hg.openjdk.java.net/jdk9/client/jdk/rev/5b613a3c04be )
            if (classDepth("sun.awt.AWTAutoShutdown") > 0) {
                return;
            }

            if (perm.implies(new RuntimePermission("modifyThreadGroup"))) {
                /* do check here (no need to log failures) */
                super.checkPermission(perm);
            }
        }

        else if (perm instanceof PropertyPermission) {
            // allow read
            if (perm.getActions().equals("read")) {
                String prop = perm.getName();
                if (prop.startsWith("bluray.") || prop.startsWith("dvb.") || prop.startsWith("mhp.")
                        || prop.startsWith("aacs.")) {
                    //logger.info(perm + " granted");
                    return;
                }
                if (prop.startsWith("user.dir")) {
                    //logger.info(perm + " granted\n" + Logger.dumpStack());
                    return;
                }
            }
            try {
                super.checkPermission(perm);
            } catch (Exception e) {
                logger.info(perm + " denied by system");
                throw new SecurityException("denied " + perm);
            }
        }

        else if (perm instanceof jail.java.io.FilePermission) {
            String file = getCanonPath(perm.getName());
            /* grant delete for writable files */
            if (perm.getActions().equals("delete")) {
                if (canWrite(file)) {
                    return;
                }
            }
            /* grant read access to BD files */
            if (perm.getActions().equals("read")) {
                if (canRead(file)) {
                    if (usingUdf) {
                        BDJLoader.accessFile(file);
                    }
                    return;
                }
            }
            if (perm.getActions().equals("write")) {
                if (canWrite(file)) {
                    return;
                }
            }
        }
        
        else if (perm instanceof java.io.FilePermission) {
            /* allow non-jailed code */
            Log.log(2, Log.LOG_TODO,"Local file access missing doPrivileged for ",perm.getName());
            return;
        }
        

        /* Networking */
        else if (perm instanceof java.net.SocketPermission) {
            if (new java.net.SocketPermission("*", "connect,resolve").implies(perm)) {
                return;
            }
        }
        else if (urlPermission != null &&
                 urlPermission.isInstance(perm)) {
            logger.info("grant " + perm);
            return;
        }

        /* Java TV */
        else if (perm instanceof jail.javax.tv.service.ReadPermission) {
            return;
        } else if (perm instanceof jail.javax.tv.service.selection.ServiceContextPermission) {
            return;
        } else if (perm instanceof jail.javax.tv.service.selection.SelectPermission) {
            return;
        } else if (perm instanceof jail.javax.tv.media.MediaSelectPermission) {
            return;
        }

        /* DVB */
        else if (perm instanceof jail.org.dvb.application.AppsControlPermission) {
            return;
        } else if (perm instanceof jail.org.dvb.media.DripFeedPermission) {
            return;
        } else if (perm instanceof jail.org.dvb.user.UserPreferencePermission) {
            return;
        }

        /* bluray */
        else if (perm instanceof jail.org.bluray.vfs.VFSPermission) {
            return;
        }

        else if (perm instanceof java.awt.AWTPermission) {
            /* silence failures from clipboard access */
            if (perm.getName().equals("accessClipboard")) {
                java.security.AccessController.checkPermission(perm);
                return;
            }
        }

        try {
            acc.checkPermission(perm);
        } catch (java.security.AccessControlException ex) {
            System.err.println(" *** caught " + ex + " at\n" + Logger.dumpStack());
            throw ex;
        }
    }

    /*
     *
     */

    public void checkExec(String cmd) {
        logger.error("Exec(" + cmd + ") denied\n" + Logger.dumpStack());
        throw new SecurityException("exec denied");
    }

    public void checkExit(int status) {
        logger.error("Exit(" + status + ") denied\n" + Logger.dumpStack());
        throw new SecurityException("exit denied");
    }

    public void checkSystemClipboardAccess() {
        throw new SecurityException("clipboard access denied");
    }

    /*
     * file read access
     */

    private boolean canRead(String file) {
        if (cacheRoot != null && file.startsWith(cacheRoot)) {
            return true;
        }
        if (discRoot != null && file.startsWith(discRoot)) {
            return true;
        }
        if (budaRoot != null && file.startsWith(budaRoot)) {
            return true;
        }
        if (persistentRoot != null && file.startsWith(persistentRoot)) {
            return true;
        }
        
        Log.log(2,Log.LOG_LIBBLURAY, "BD-J read ",file," denied at");
        return false;
    }

    /*
     * File write access
     */

    private boolean canWrite(String file) {
        // Xlet can write to persistent storage and binding unit

        if (budaRoot != null && file.startsWith(budaRoot)) {
            return true;
        }
        if (persistentRoot != null && file.startsWith(persistentRoot)) {
            return true;
        }

        BDJXletContext ctx = BDJXletContext.getCurrentContext();
        if (ctx != null) {
            Log.log(2,Log.LOG_LIBBLURAY, "Xlet write ",file," denied at");
            return false;
        }

        // BD-J core can write to cache
        if (cacheRoot != null && file.startsWith(cacheRoot)) {
            return true;
        }

        Log.log(2,Log.LOG_LIBBLURAY, "BD-J write ",file," denied at");
        return false;
    }

    private boolean isAbsolutePath(String path) {
        File f = new File(path);
        return f.isAbsolute();
    }

    private String getCanonPath(String origPath) {
        String suffix = "";

        if (!isAbsolutePath(origPath)) {
            String home = BDJXletContext.getCurrentXletHome();
            if (home == null) {
                logger.error("Relative path " + origPath + " outside Xlet context\n" + Logger.dumpStack());
                return origPath;
            }
            //origPath = home + origPath;
        }

        if (origPath.endsWith(File.separator + "*")) {
            suffix = File.separator + "*";
            origPath = origPath.substring(0, origPath.length() - 2);
        }

        final String path = origPath;
        String cpath = (String) AccessController.doPrivileged(new PrivilegedAction() {
            public Object run() {
                try {
                    return new File(path).getCanonicalPath();
                } catch (Exception ioe) {
                    logger.error("error canonicalizing " + path + ": " + ioe);
                    return null;
                }
            }
        });
        if (cpath == null) {
            throw new SecurityException("cant canonicalize " + path);
        }
        return cpath + suffix;
    }

    /*
     *
     */

    private static final Logger logger = Logger.getLogger(BDJSecurityManager.class.getName());
}
