/*
   This file is part of MDic Dictionary

   Copyright (C) 2007-2010 Mehrdad Momeny <mehrdad.momeny@gmail.com
   Copyright (C) 2007-2010 Majid Ramezanpour <majid.ramezanpour@gmail.com>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License,
   (at your option) any later version

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details

   You should have received a copy of the GNU General Public License
   along with this program; if not, see http://www.gnu.org/licenses/

*/


#define REC_PATH ":/"

#include "mainwindow.h"

#include <QtGui/QTextBrowser>
#include <QtGui/QPushButton>
#include <QtGui/QLabel>
#include <QtGui/QIcon>

#include <QtGui/QVBoxLayout>
#include <QtGui/QAction>
#include <QtGui/QToolBar>
#include <QtGui/QMenu>

#include <QtGui/QClipboard>
#include <QtGui/QMouseEvent>

#include <QtGui/QScrollBar>
#include <QtGui/QScrollArea>

#include <QtGui/QMessageBox>
#include <QtGui/QDesktopWidget>

#include <QtCore/QUrl>
#include <QtCore/QTimer>
#include <QtCore/QString>
#include <QtCore/QStringList>
#include <QtCore/QDebug>

#include "stdio.h"

#include "ui_helpform.h"
#include "config/configdialog.h"
#include "spell/spellchecker.h"
#include "dictcore.h"
#include <QProcess>
#include <QToolButton>
#include <QDesktopServices>
#include "dicts/gtranslate.h"

MainWindow::MainWindow()
{
    config = Settings::self();
    spellcheck = SpellChecker::self();
    readConfig();
}

void MainWindow::init()
{
    flgNotSearched = false;
//     histPointer = -1;
    modifierKey = -1;

    lastClipboard = QApplication::clipboard()->text(QClipboard::Selection);

    createUI();
    createSysTrayContextMenu();

    isEntered = false;

    txtGet->setFocus();         //Get Ready For Input

    if (config->showOnStart) { //if showOnStart is True then Show MainWindow On StartUp!
        flgOnScreen = true;
        this->show(false);
    }

    MessageTimer = new QTimer(this);         //time to show mean message on screen
    connect(MessageTimer,   SIGNAL(timeout()), this, SLOT(sltHideMessage()));            //just connect them

    clipboardTimer = new QTimer(this);       //ClipBoard Watching Timer
    connect(clipboardTimer, SIGNAL(timeout()), this, SLOT(sltClipBoardTime()));

    clipboardTimer->setInterval(1000);         //check clipboard every one second
    if (config->clipboardWatcher)
        clipboardTimer->start();

    setSelectionScannerEnabled(config->clipboardWatcher);
}

void MainWindow::initSysTrayContextMenu()
{
    actnSelectionScan->setChecked(config->clipboardWatcher);
}

void MainWindow::createUI()
{
    this->setWindowTitle(tr("MDic Dictionary"));
    this->setWindowIcon(QIcon(QString(REC_PATH) + QString("mdic.png")));

    if (config->onTop) {
        this->setWindowFlags(Qt::WindowStaysOnTopHint);
    }

    QPixmap pronouncePixmap(QString(REC_PATH) + QString("speak.png"));
    QPixmap settingPixmap(QString(REC_PATH) + QString("configs.png"));
//     QPixmap searchPixmap(QString(REC_PATH) + QString("translate.png"));

    btnSetting = new QToolButton(this);
    btnSetting->setSizePolicy(QSizePolicy::Maximum, QSizePolicy::Maximum);
    btnSetting->setIcon(QIcon(settingPixmap));
    btnSetting->setPopupMode(QToolButton::InstantPopup);
    btnSetting->setToolTip(tr("Configure MDic... (Ctrl+S)"));

    btnBackward = new QToolButton(this);
    btnBackward->setSizePolicy(QSizePolicy::Maximum, QSizePolicy::Minimum);
    btnBackward->setIcon(QIcon(QPixmap(QString(REC_PATH) + QString("backward.png"))));
    btnBackward->setToolTip(tr("Backward (Alt+Left)"));

    btnForward = new QToolButton(this);
    btnForward->setSizePolicy(QSizePolicy::Maximum, QSizePolicy::Minimum);
    btnForward->setIcon(QIcon(QPixmap(QString(REC_PATH) + QString("forward.png"))));
    btnForward->setToolTip(tr("Forward (Alt+Right)"));
  
    txtGet = new MDicComboBox(this);                         //Place To Get Word
    txtGet->setSizePolicy(QSizePolicy::Expanding,QSizePolicy::Fixed);

    btnPronounce = new QToolButton(this);
    btnPronounce->setIcon(QIcon(pronouncePixmap));
    btnPronounce->setToolTip(tr("Pronounce (Ctrl+P)"));

    txtShowMean = new QTextBrowser(this);
    txtShowMean->setVisible(false);

    searchToolBar = new QHBoxLayout;
    searchToolBar->addWidget(btnSetting);
    searchToolBar->addWidget(btnBackward);
    searchToolBar->addWidget(btnForward);
    searchToolBar->addWidget(txtGet);
    searchToolBar->addWidget(btnPronounce);
    reloadRemoteDictsMenu();

    QWidget *mainWid = new QWidget(this);
    QVBoxLayout *mainLayout = new QVBoxLayout(mainWid);
    mainLayout->setSpacing(0);
    mainLayout->setMargin(0);
    showMeanWidget = new DicWidget(this);
    connect( showMeanWidget, SIGNAL(anchorClicked(QUrl)), SLOT(sltLink(QUrl)) );

    mainLayout->addLayout(searchToolBar);
    mainLayout->addWidget(showMeanWidget);

    setCentralWidget(mainWid);

    trayMenu = new QMenu(tr("MDic Dictionary"), this);
    sysTrayIcon = new QSystemTrayIcon(this);
    sysTrayIcon->setIcon(QIcon(QString(REC_PATH) + QString("mdic.png")));
    sysTrayIcon->setContextMenu(trayMenu);
    sysTrayIcon->setToolTip(tr("MDic Dictionary"));
    sysTrayIcon->show();
    connect( sysTrayIcon, SIGNAL(messageClicked()), SLOT(show()) );

    txtGetOnTray = new MDicComboBox();
    connect(txtGetOnTray, SIGNAL(returnPressed()), this, SLOT(sltGetMeanByClick()));
    txtGetOnTray->setWindowFlags(Qt::WindowStaysOnTopHint | Qt::SubWindow | Qt::FramelessWindowHint);
    txtGetOnTray->setVisible(false);

    this->setMinimumWidth(300);
    this->setMinimumHeight(165);
    this->setMaximumHeight(900);

    connect(btnSetting,   SIGNAL(clicked(bool))  , this, SLOT(sltShowConfigs()));

    connect(btnBackward,  SIGNAL(clicked(bool)), this, SLOT(sltBackward()));
    connect(btnForward,   SIGNAL(clicked(bool)), this, SLOT(sltForward()));

    connect(btnPronounce, SIGNAL(clicked(bool)), this, SLOT(sltSay()));

//     connect(QApplication::clipboard(), SIGNAL(selectionChanged()), this, SLOT(sltGetMeanBySelection()));
//     connect(QApplication::clipboard(), SIGNAL(dataChanged())     , this, SLOT(sltGetMeanBySelection()));


    connect(txtGet ,      SIGNAL(returnPressed()), this, SLOT(sltGetMeanByClick()));

    connect(sysTrayIcon,  SIGNAL(activated(QSystemTrayIcon::ActivationReason)), this, SLOT(sltActivated(QSystemTrayIcon::ActivationReason)));

}

void MainWindow::createSysTrayContextMenu()
{
    QAction *actnQuit;
    QAction *actnAbout;
    QAction *actnConfig;
    QAction *actnHelp;
    actnConfig = new QAction(QIcon(QString(REC_PATH) + QString("configs.png")), tr("&Configure MDic..."), this);
    actnConfig->setShortcut(tr("Ctrl+S"));
    connect(actnConfig,   SIGNAL(triggered(bool)), this, SLOT(sltShowConfigs()));

    actnQuit   = new QAction(QIcon(QString(REC_PATH) + QString("quit.png")), tr("&Quit")   , this);
    actnQuit->setShortcut(tr("Ctrl+Q"));
    connect(actnQuit,     SIGNAL(triggered(bool)), this, SLOT(sltQuit()));

    actnAbout  = new QAction(QIcon(QString(REC_PATH) + QString("mdic.png")), tr("&About MDic"), this);
    connect(actnAbout,    SIGNAL(triggered(bool)), this, SLOT(sltAboutUs()));

    actnSelectionScan = new QAction(tr("Selection Scanner"), this);
    actnSelectionScan->setCheckable(true);
    actnSelectionScan->setChecked(config->clipboardWatcher);
    connect(actnSelectionScan, SIGNAL(triggered(bool)), this, SLOT(setSelectionScannerEnabled(bool)));

    actnHelp = new QAction(QIcon(QString(REC_PATH) + QString("help.png")), tr("Help"), this);
    connect(actnHelp, SIGNAL(triggered(bool)), this, SLOT(sltShowHelp()));

    trayMenu->addAction(actnConfig);
    trayMenu->addSeparator();//--------------
    trayMenu->addAction(actnSelectionScan);
    trayMenu->addSeparator();//--------------
    trayMenu->addAction(actnHelp);
    trayMenu->addAction(actnAbout);
    trayMenu->addSeparator();//--------------
    trayMenu->addAction(actnQuit);
    btnSetting->setMenu(trayMenu);
}

void MainWindow::sltClipBoardTime()
{
    if (!config->clipboardWatcher) {
        clipboardTimer->stop();
        return;
    } else if (!clipboardTimer->isActive())
        clipboardTimer->start();

    setModifier(config->mod);

    QString tmpStr = QApplication::clipboard()->text(QClipboard::Selection);

    if ( (flgNotSearched && config->modStatus && modifierKey &&
         Keyboard::activeModifiers().testFlag(static_cast<Qt::KeyboardModifier>(modifierKey)) )
        || ( tmpStr.compare(lastClipboard, Qt::CaseInsensitive) != 0 ) ) {

//         qDebug("MainWindow::sltClipBoardTime");
        lastClipboard = tmpStr;
        if ( this->getMean(Selection) )
            this->show(true);
    }
}

void MainWindow::sltShowConfigs()
{
    MessageTimer->stop();
    isEntered = false;
    if(configDialog){
        configDialog->raise();
    } else {
        configDialog = new ConfigDialog(this);
        configDialog->exec();
    }
}

void MainWindow::sltGetMeanByClick()
{
//     qDebug("MainWindow::sltGetMeanByClick");
    this->MessageTimer->stop();

    if (txtGetOnTray->isVisible()){
        lastClipboard = txtGetOnTray->currentText();
        this->getMean(Other,txtGetOnTray->currentText());
    } else {
        lastClipboard = txtGet->currentText();
        this->getMean(TxtGet);
    }

    this->txtGetOnTray->setVisible(false);
}

/*
void MainWindow::sltGetMeanBySelection()
{
    qDebug("MainWindow::sltGetMeanByClick");
    if (config->clipboardWatcher) {
        QString newStr = QApplication::clipboard()->text(QClipboard::Selection);
        if ( newStr.compare(lastClipboard, Qt::CaseInsensitive) != 0 ) {
            lastClipboard = QApplication::clipboard()->text(QClipboard::Selection);
        }
        if( config->triggerMaxChar && lastClipboard.count() > config->triggerMaxChar ){
            return;
        } else {
            if (this->getMean(Selection))
                this->show(true);
        }
    }
}
*/

bool MainWindow::getMean(MainWindow::SearchPlace where, const QString& phrase)
{
//     qDebug()<<"MainWindow::GetMean: "<<where<<phrase;

    if ((where == Selection || where == MainWin) && config->modStatus && modifierKey &&
        !Keyboard::activeModifiers().testFlag(static_cast<Qt::KeyboardModifier>(modifierKey))) {
        flgNotSearched = true;
//         qDebug()<<"MainWindow::getMean: No Modifier! return false";
        return false;
    }

    flgNotSearched = false;
    bool result = false;
    QString wordToFind;
    QString wordToSpell;

    if( where == Other ){
        wordToFind = phrase;
    } else if ( where == TxtGet ) {
        wordToSpell = wordToFind = txtGet->currentText();
//         lastClipboard = wordToFind;
        QApplication::clipboard()->setText(lastClipboard, QClipboard::Selection);
    } else {
        wordToFind = QApplication::clipboard()->text(QClipboard::Selection);
    }

    wordToFind = wordToFind.toLower().simplified();
//     qDebug()<<"MainWindow::GetMean2: "<<where<<wordToFind;

    if(where == TxtGet){
        QString baseStr = txtGet->currentText();
        if(wordToFind != baseStr)
            txtGet->removeItem(baseStr);
    }

    txtGet->setCurrentText(wordToFind);

    if( DictCore::self()->loadedDicts().count() < 1 ){
        QMessageBox::critical(this, tr("No dictionary databases found!"), tr("You have to add at least one database dictionary to dictionaries list, to use this application."));
        lastClipboard = wordToFind;
        flgNotSearched = true;
        sltShowConfigs();
        return false;
    }

    if ( !wordToFind.isEmpty() ) {
        if( !lastNotFoundWord.isEmpty()/* && lastNotFoundWord != txtGet->currentText()*/ )
            txtGet->removeItem(lastNotFoundWord);
        QList<Translation> resultList = DictCore::self()->translate(wordToFind);
        if ( !resultList.isEmpty() ) {
            result = true;
            lastNotFoundWord.clear();
            bool mouseOverMainWin = isMouseOver();
            if( where != TxtGet && config->isMove && config->output == Settings::Main) {
                if (!mouseOverMainWin) {
                    moveToCursorPos();
                    if (!flgOnScreen)
                        MessageTimer->start(config->showInterval*1000);
                } else
                    isEntered = true;
            } else {
                if (!flgOnScreen)
                    MessageTimer->start(config->showInterval*1000);
            }

            QString nonHtmlMean;

            if(wordToFind == txtGet->currentText()){
                showMeanWidget->clear();
                foreach (const Translation &trans, resultList) {
                        nonHtmlMean.append( trans.translation()+ '\n');
                        showMeanWidget->appendMean(trans.dictName(), DictCore::translationToHtml(trans));
                }
                showMeanWidget->moveScrollUp();
            } else {
//                 qDebug()<<"MainWindow::getMean: Not equal to current text! Return false";
                return false;
            }

            if (config->showJustFirst) {
                nonHtmlMean = resultList[0].translation();
            }
            if ( !mouseOverMainWin && (where == Selection || where == Other) ) {
//                 qDebug()<<"MainWindow::getMean: Calling showSearchResult()"<<where;
                showSearchResult(nonHtmlMean, wordToFind);
            }/* else
                qDebug()<<"MainWindow::getMean: Not mouse over!";*/

            ///for Text to Speech:
            if ( config->speech && !wordToFind.isRightToLeft() && wordToFind.count() > 100 ) {
                say(wordToFind);
            }

        } else {
            qDebug()<<"Phrase Not Found";
            lastNotFoundWord = wordToFind;
            if (config->notifyNotAvailable)
                sysTrayIcon->showMessage(tr("Sorry!"), tr("Phrase not found!"), QSystemTrayIcon::Information, 2000);

            if ( config->speakAny && !wordToFind.isRightToLeft() && wordToFind.count() > 100 )
                say(wordToFind);
        }
    } else {
//         qDebug()<<"MainWindow::getMean: wordToFind is empty! Return false";
        return false;
    }

    txtGet->selectAll();
    txtGet->setFocus();

    if (config->isSpellingEnabled)
        if ( !result && where == TxtGet /*&& wordToSpell == txtGet->currentText()*/ )
            checkSpelling(wordToSpell);

//     qDebug()<<"MainWindow::getMean: return result! End of function! result: "<<result;
    return result;
}

void MainWindow::showSearchResult(const QString& nonHtmlMean, const QString& wordToFind)
{
//     qDebug()<<"MainWindow::showSearchResult";
    QString cmd;
    switch ( config->output ) {
    case Settings::SysTray:
        txtShowMean->setHtml(nonHtmlMean);
        sysTrayIcon->showMessage(wordToFind, txtShowMean->toPlainText(), QSystemTrayIcon::Information,
                                    config->showInterval*1000);
        sltHideMessage();
        break;
    case Settings::LibNotify:
        txtShowMean->setHtml(nonHtmlMean);
        cmd = QString("notify-send -t ") + QString::number(config->showInterval * 1000) + QString(" -u low -i mdic \"") + wordToFind + QString("\" \"") + txtShowMean->toPlainText() +
        '\"';
        QProcess::execute(cmd);
        sltHideMessage();
        break;
    case Settings::KPopup:
        cmd = QString("kdialog  --title \"  ") + wordToFind +
                            QString("  \"  --passivepopup \"  ") + nonHtmlMean + QString("  \" ") +
                            QString::number(config->showInterval);
        QProcess::execute(cmd);
        sltHideMessage();
        break;
    case Settings::Main:
    default:
        break;
    };
}

void MainWindow::moveToCursorPos()
{
    int posX = 0, posY = 0;
    if (QCursor::pos().y() > QApplication::desktop()->height() - this->size().height() - 30) {
        posY = QCursor::pos().y() - (this->size().height() + 50);
    } else {
        posY = QCursor::pos().y() + 50;
    }
    if (QCursor::pos().x() < this->width() / 2) {
        posX = 0;
    } else if (QCursor::pos().x() > QApplication::desktop()->width() - this->width() / 2) {
        posX = QApplication::desktop()->width() - this->width();
    } else {
        posX = QCursor::pos().x() - this->width() / 2;
    }
    this->move(posX, posY);
}

void MainWindow::sltHideMessage()
{
//     qDebug("MainWindow::sltHideMessage");
    flgOnScreen = false;
    this->MessageTimer->stop();
    this->hide();
}

void MainWindow::sltAboutUs()
{
    MessageTimer->stop();
    isEntered = false;

    About *aboutDialog = new About(this);
    aboutDialog->show();
}

void MainWindow::sltActivated(QSystemTrayIcon::ActivationReason reason)//for Handling Click on System Tray Icon
{
    switch(reason){
    case QSystemTrayIcon::Context:
        txtGetOnTray->hide();
        initSysTrayContextMenu();
        break;
    case QSystemTrayIcon::MiddleClick:
        txtGetOnTray->setGeometry(sysTrayIcon->geometry());
        txtGetOnTray->resize(150,30);
        txtGetOnTray->setVisible( !txtGetOnTray->isVisible() );
        break;
    case QSystemTrayIcon::Trigger:
    case QSystemTrayIcon::DoubleClick:
    default:
        txtGetOnTray->hide();
        if (this->isVisible()) {
            sltHideMessage();
        } else {
            flgOnScreen = true;
            this->MessageTimer->stop();
            this->show(false);
            txtGet->selectAll();
            txtGet->setFocus();
        }
        break;
    };
}

void MainWindow::keyPressEvent(QKeyEvent * event)
{
    txtGet->setFocus();
    if (event->key() == Qt::Key_Up)
        sltBackward();
    else if (event->key() == Qt::Key_Down)
        sltForward();


    if (event->key() == Qt::Key_Escape)
        sltHideMessage();
    else
        if (event->modifiers() == Qt::ControlModifier)
            switch (event->key()) {
            case Qt::Key_Q:
                sltQuit();
                break;
            case Qt::Key_S:
                sltShowConfigs();
                break;
            case Qt::Key_P:
                sltSay();
                break;
            case Qt::Key_L:
                checkSpelling(txtGet->currentText());
                break;
            }
        else if (event->modifiers() == Qt::AltModifier)
            switch (event->key()) {
            case Qt::Key_Left:
                sltBackward();
                break;
            case Qt::Key_Right:
                sltForward();
                break;
            }
    QMainWindow::keyPressEvent(event);
}

void MainWindow::readConfig()
{
    QSettings config;
    QPoint pos = config.value("pos", QPoint(300, 200)).toPoint();
    QSize size = config.value("size", QSize(368, 265)).toSize();

    resize(size);
    move(pos);
}

void MainWindow::writeConfig()
{
    QSettings config;

    config.setValue("pos" , pos());
    config.setValue("size", size());
    config.sync();
}

void MainWindow::sltQuit()
{
    config->writeConfig();//writes all configs to file except pos and size
    writeConfig(); //writing pos and size configs in this function for using it every time that the program get closed.
    DictCore::self()->deleteLater();
    deleteLater();
    qApp->quit();
}

MainWindow::~MainWindow()
{
    txtGetOnTray->deleteLater();
}

void MainWindow::contextMenuEvent(QContextMenuEvent * event)
{
    if (event->y() < 25) {
        initSysTrayContextMenu();
        this->trayMenu->exec(event->globalPos());
    }
}

void MainWindow::enterEvent(QEvent *)
{
    if (config->stay)
        if (isEntered)
            MessageTimer->stop();
}

void MainWindow::leaveEvent(QEvent *)
{
    if (config->stay)
        if (isEntered)
            MessageTimer->start(config->stayTime * 1000);
}

void MainWindow::setSelectionScannerEnabled(bool isChecked)
{
    if (isChecked) {
        clipboardTimer->start();
        config->clipboardWatcher = true;

        sysTrayIcon->setIcon(this->windowIcon());
    } else {
        clipboardTimer->stop();
        config->clipboardWatcher = false;

        //Make Icon grayScale:
        QImage result = QImage(sysTrayIcon->icon().pixmap(48).toImage());
        for ( int y = 0; y < result.height(); ++y ) {
            for ( int x = 0; x < result.width(); ++x ) {
                int pixel = result.pixel( x, y );
                int gray = qGray( pixel );
                int alpha = qAlpha( pixel );
                result.setPixel( x, y, qRgba( gray, gray, gray, alpha ) );
            }
        }
        sysTrayIcon->setIcon( QIcon(QPixmap::fromImage( result )) );
    }
}

void MainWindow::show(bool isBySelection)
{
    if (config->output == Settings::Main || !isBySelection) {
//         qDebug("MainWindow::show");
        isEntered = isBySelection;
        QMainWindow::show();
    }
}

void MainWindow::sltSay()
{
    say(txtGet->currentText());
}

bool MainWindow::isMouseOver()
{
    if( isVisible() )
        if (QCursor::pos().x() > this->pos().x() && QCursor::pos().x() < this->pos().x() + this->width())
            if (QCursor::pos().y() > this->pos().y() && QCursor::pos().y() < this->pos().y() + this->height())
                return true;
    return false;
}

void MainWindow::say(const QString& txt)
{
    if(!speechProcess)
        speechProcess = new QProcess(this);
    if(speechProcess->state() != QProcess::NotRunning)
        speechProcess->kill();
    if (config->spchSysIdx == 1) {
        speechProcess->start( config->speechSysText, config->speechArgs, QIODevice::WriteOnly);
        speechProcess->write(txt.toUtf8());
        speechProcess->closeWriteChannel();
    } else {
        if ( !Settings::self()->noEspeak ){
            QStringList tmp(config->speechArgs);
            tmp.append(QString("\"" + txt + "\""));
            QString test = config->speechSysText;
            for (int i = 0; i < tmp.count(); ++i) {
                test.append(' ' + tmp[i]);
            }
            speechProcess->startDetached(config->speechSysText, tmp);
        }
    }
}

void MainWindow::sltLink(const QUrl & link)
{
    QString w;
//    qDebug("MainWindow::sltLink: %s", link.toString().toLatin1().data());   /**DEBUG**/
    if (link.scheme() == "bword") {
        w = link.host();
        if (link.host() == txtGet->currentText())
            w.prepend('@');
        txtGet->setCurrentText(w);
//        qDebug("MainWindow::sltLink: %s", w.toLatin1().data());   /**DEBUG**/
        getMean(TxtGet);
    } else
        QDesktopServices::openUrl(link);
}

void MainWindow::sltBackward()
{
    int cur = txtGet->currentIndex();
    if( cur > 0)
        txtGet->setCurrentIndex( --cur );
    getMean(TxtGet);
    txtGet->setFocus();
}

void MainWindow::sltForward()
{
    int cur = txtGet->currentIndex();
    if( cur < txtGet->count() - 1)
        txtGet->setCurrentIndex( ++cur );
    getMean(TxtGet);
    txtGet->setFocus();
}

void MainWindow::setModifier(int Mod)
{
    //0 meta(win key) % 1 Alt % 2 Control % 3 Shift
    switch (Mod) {
    case 0:
        modifierKey = Qt::MetaModifier;
        break;
    case 1:
        modifierKey = Qt::AltModifier;
        break;
    case 2:
        modifierKey = Qt::ControlModifier;
        break;
    case 3:
        modifierKey = Qt::ShiftModifier;
        break;
    }
}

bool MainWindow::checkSpelling(const QString & phrase)
{
    QStringList wordsList;
    QMenu spellingMenu(this);

    wordsList = spellcheck->spell(phrase);

    QStringListIterator listIterator(wordsList);
    while ( listIterator.hasNext() ) {
            spellingMenu.addAction( listIterator.next() );
    }
    if(spellingMenu.isEmpty())
        return false;
    QAction * act = spellingMenu.exec(mapToGlobal(QPoint(txtGet->pos().x(),
                                                          txtGet->pos().y() + txtGet->height())));
    if(act){
        txtGet->setCurrentText(act->text());
//         qDebug("MainWindow::checkSpelling:: Calling sltGetMeanByClick");
        sltGetMeanByClick();
    }
    return true;
}

void MainWindow::sltShowHelp()
{
    QDialog *wdg = new QDialog(this);
    wdg->setAttribute(Qt::WA_DeleteOnClose);
    Ui_HelpDialog ui;
    ui.setupUi(wdg);
    wdg->show();
}

void MainWindow::reloadRemoteDictsMenu()
{
    if( !btnSearchRemote ){
        btnSearchRemote = new QToolButton(this);
        btnSearchRemote->setIcon(QIcon(":/translate.png"));//TODO We need a better Icon for this :D
        btnSearchRemote->setToolTip(tr("Translate Text"));
        btnSearchRemote->setPopupMode(QToolButton::InstantPopup);
        searchToolBar->insertWidget(searchToolBar->count()-1, btnSearchRemote);
    } else {
        btnSearchRemote->menu()->deleteLater();
    }
    QList<DictCore::Dictionary> dicts = DictCore::self()->loadedRemoteDicts();
    QMenu *menu = new QMenu(btnSearchRemote);
    foreach(const DictCore::Dictionary &dic, dicts){
        QAction *act = new QAction(dic.name(), menu);
        act->setData(dic.code());
        connect( act, SIGNAL(triggered(bool)), SLOT(slotRemoteSearchTriggered()) );
        menu->addAction(act);
    }
    btnSearchRemote->setMenu(menu);
}

void MainWindow::slotRemoteSearchTriggered()
{
    QAction *act = qobject_cast<QAction*>(sender());
    if(!act)
        return;
    QString wordToFind = txtGet->currentText();
    QList<Translation> resultList = GTranslate::translate( DictCore::self()->dictionary(act->data().toString()),
                                                           wordToFind);

    if(wordToFind == txtGet->currentText()){
        showMeanWidget->clear();
        foreach (const Translation &trans, resultList) {
                showMeanWidget->appendMean(trans.dictName(), DictCore::translationToHtml(trans));
        }
        showMeanWidget->moveScrollUp();
    }
}

void MainWindow::slotSettingsChanged()
{
    reloadRemoteDictsMenu();
}

#include "mainwindow.moc"
