/*
   This file is part of MDic Dictionary

   Copyright (C) 2007-2010 Mehrdad Momeny <mehrdad.momeny@gmail.com
   Copyright (C) 2007-2010 Majid Ramezanpour <majid.ramezanpour@gmail.com>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License,
   (at your option) any later version

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details

   You should have received a copy of the GNU General Public License
   along with this program; if not, see http://www.gnu.org/licenses/

*/

#ifndef DICT_H
#define DICT_H

#include <QStringList>
#include <QDir>
#include <QCoreApplication>
#include <QVariant>

/**
* This class represents information about dictionary.
*/
class DictInfo
{
public:
    /**
        * Construct empty DictInfo object.
        */
    DictInfo()
        : m_wordsCount(-1L)
    { }
    /**
        * Construct DictInfo object from data.
        * @param name A dictionary name
        * @param author A dictionary author
        * @param description A dictionary description
        * @param wordsCount A count of words that available in dictionary
        */
    explicit DictInfo( const QString &name,
                const QString &author = QString(),
                const QString &description = QString(),
                long wordsCount = -1L)
        :   m_name(name),
            m_author(author),
            m_description(description),
            m_wordsCount(wordsCount)
    { }

    const QString &name() const
    { return m_name; }
    const QString &author() const
    { return m_author; }
    const QString &description() const
    { return m_description; }
    uint wordsCount() const
    { return m_wordsCount; }
    QString sourceLanguage() const
    { return m_srcLang; }
    QString destinationLanguage() const
    { return m_destLang; }
    QString license() const
    { return m_license; }

    void setName(const QString &name)
    { m_name = name; }
    void setAuthor(const QString &author)
    { m_author = author; }
    void setDescription(const QString &description)
    { m_description = description; }
    void setWordsCount(uint wordsCount)
    { m_wordsCount = wordsCount; }
    void setSourceLanguage(const QString &source)
    { m_srcLang = source; }
    void setDestinationLanguage(const QString &dest)
    { m_destLang = dest; }
    void setLicense(const QString &license)
    { m_license = license; }

private:
    QString m_srcLang;
    QString m_destLang;
    QString m_license;
    QString m_name;
    QString m_author;
    QString m_description;
    uint m_wordsCount;
};

/**
* This class represent a translation.
*/
class Translation
{
public:
    /**
    * Construct an empty translation.
    */
    Translation()
    { }

    /**
    * Construct a translation from data.
    * @param phrase phrase to translate
    * @param dictName A full dictionary name
    * @param translation A translation
    */
    Translation(const QString &phrase,
            const QString &dictName,
            const QString &translation)
        : m_phrase(phrase),
            m_dictName(dictName),
            m_translation(translation)
    { }

    /**
    Return true, if translation is empty and invalid!
    */
    bool isEmpty()
    {
        return m_translation.isEmpty();
    }

    /**
    * Return the translation title.
    */
    const QString &phrase() const
    { return m_phrase; }

    /**
    * Return the dictionary name.
    */
    const QString &dictName() const
    { return m_dictName; }

    /**
    * Return the translation.
    */
    const QString &translation() const
    { return m_translation; }

    /**
    * Set a translation title.
    */
    void setPhrase(const QString &phrase)
    { m_phrase = phrase; }

    /**
    * Set a dictionary name.
    */
    void setDictName(const QString &dictName)
    { m_dictName = dictName; }

    /**
    * Set a translation.
    */
    void setTranslation(const QString &translation)
    { m_translation = translation; }

private:
    QString m_phrase;
    QString m_dictName;
    QString m_translation;
};


#endif // DICT_H

// vim: tabstop=4 softtabstop=4 shiftwidth=4 expandtab cindent

