/*
 * Decompiled with CFR 0.152.
 */
package org.openlcb;

import java.util.Timer;
import java.util.TimerTask;
import java.util.logging.Logger;
import org.openlcb.AddressedMessage;
import org.openlcb.Connection;
import org.openlcb.InitializationCompleteMessage;
import org.openlcb.Message;
import org.openlcb.MessageDecoder;
import org.openlcb.NodeID;
import org.openlcb.ProtocolIdentification;
import org.openlcb.ProtocolIdentificationReplyMessage;
import org.openlcb.ProtocolIdentificationRequestMessage;
import org.openlcb.StreamDataCompleteMessage;
import org.openlcb.StreamDataProceedMessage;
import org.openlcb.StreamDataSendMessage;
import org.openlcb.StreamInitiateReplyMessage;
import org.openlcb.StreamInitiateRequestMessage;
import org.openlcb.implementations.DatagramService;
import org.openlcb.implementations.MemoryConfigurationService;

public class LoaderClient
extends MessageDecoder {
    private static final Logger logger = Logger.getLogger(LoaderClient.class.getName());
    private static final byte SRC_STREAM_ID = 4;
    private static final int PIP_TIMEOUT_MSEC = 3000;
    private static final int FREEZE_REBOOT_TIMEOUT_MSEC = 60000;
    private static final int STREAM_INIT_TIMEOUT_MSEC = 10000;
    private static final int STREAM_DATA_PROCEED_TIMEOUT_MSEC = 120000;
    Connection connection;
    MemoryConfigurationService mcs;
    DatagramService dcs;
    State state;
    NodeID src;
    NodeID dest;
    int space;
    long address;
    byte[] content;
    LoaderStatusReporter feedback;
    private static final int ERR_CHECKSUM_FAILED = 8328;
    private static final int ERR_FILE_CORRUPTED = 4233;
    private static final int ERR_FILE_INAPPROPRIATE = 4232;
    private static Timer timer;
    private TimerTask task = null;
    private int bufferSize;
    private int nextIndex;
    private int errorCounter;
    private byte destStreamID;

    public LoaderClient(Connection _connection, MemoryConfigurationService _mcs, DatagramService _dcs) {
        this.connection = _connection;
        this.dcs = _dcs;
        this.mcs = _mcs;
        if (timer == null) {
            timer = new Timer("OpenLCB LoaderClient Timeout Timer");
        }
    }

    public void doLoad(NodeID _src, NodeID _dest, int _space, long _address, byte[] _content, LoaderStatusReporter _feedback) {
        this.src = _src;
        this.dest = _dest;
        this.space = _space;
        this.address = _address;
        this.content = _content;
        this.state = State.IDLE;
        this.feedback = _feedback;
        this.sendFreeze();
    }

    private void sendFreeze() {
        this.state = State.FREEZE;
        this.dcs.sendData(new DatagramService.DatagramServiceTransmitMemo(this.dest, new int[]{32, 161, this.space}){

            @Override
            public void handleSuccess(int flags) {
            }

            @Override
            public void handleFailure(int errorCode) {
            }
        });
        this.state = State.INITCOMPL;
        this.startTimeout(60000);
    }

    private void startTimeout(int period_msec) {
        this.task = new TimerTask(){

            @Override
            public void run() {
                LoaderClient.this.timerExpired();
            }
        };
        timer.schedule(this.task, period_msec);
    }

    private void endTimeout() {
        if (this.task != null) {
            this.task.cancel();
        } else {
            this.state = State.FAIL;
        }
        this.task = null;
    }

    private void timerExpired() {
        this.failWith(1, "Timed out in state " + this.state.name());
    }

    private boolean isReply(Message msg) {
        if (!msg.getSourceNodeID().equals(this.dest)) {
            return false;
        }
        return !(msg instanceof AddressedMessage) || ((AddressedMessage)msg).getDestNodeID().equals(this.src);
    }

    @Override
    public void handleInitializationComplete(InitializationCompleteMessage msg, Connection sender) {
        if (this.state == State.INITCOMPL && this.isReply(msg)) {
            this.endTimeout();
            this.state = State.PIP;
            this.sendPipRequest();
        }
        if (this.state == State.SUCCESS && this.isReply(msg)) {
            this.state = State.IDLE;
        }
    }

    private void sendPipRequest() {
        this.state = State.PIPREPLY;
        ProtocolIdentificationRequestMessage msg = new ProtocolIdentificationRequestMessage(this.src, this.dest);
        this.connection.put(msg, this);
        this.startTimeout(3000);
    }

    @Override
    public void handleProtocolIdentificationReply(ProtocolIdentificationReplyMessage msg, Connection sender) {
        if (this.state == State.PIPREPLY && this.isReply(msg)) {
            this.endTimeout();
            ProtocolIdentification pi = new ProtocolIdentification(msg.getSourceNodeID(), msg);
            if (!pi.hasProtocol(ProtocolIdentification.Protocol.FirmwareUpgradeActive)) {
                this.failWith(1, "Target not in Upgrade state.");
            } else if (pi.hasProtocol(ProtocolIdentification.Protocol.Stream)) {
                this.state = State.SETUPSTREAM;
                this.setupStream();
            } else if (pi.hasProtocol(ProtocolIdentification.Protocol.Datagram)) {
                this.state = State.DG;
                this.sendDGs();
            } else {
                this.failWith(1, "Target has no Streams nor Datagrams!");
            }
        }
    }

    private void setupStream() {
        this.bufferSize = 16384;
        this.state = State.STREAM;
        this.mcs.request(new MemoryConfigurationService.McsWriteStreamMemo(this.dest, this.space, this.address, 4){

            @Override
            public void handleSuccess() {
                LoaderClient.this.sendStream();
            }

            @Override
            public void handleFailure(String where, int errorCode) {
                String f = "Failed to setup stream at " + where + ": error 0x" + Integer.toHexString(errorCode);
                logger.warning(f);
                LoaderClient.this.failWith(errorCode, f);
            }
        });
    }

    private void sendStream() {
        StreamInitiateRequestMessage m = new StreamInitiateRequestMessage(this.src, this.dest, this.bufferSize, 4, this.destStreamID);
        this.connection.put(m, this);
        this.startTimeout(10000);
    }

    @Override
    public void handleStreamInitiateReply(StreamInitiateReplyMessage msg, Connection sender) {
        if (this.state == State.STREAM && this.isReply(msg) && 4 == msg.getSourceStreamID()) {
            this.endTimeout();
            this.bufferSize = msg.getBufferSize();
            this.destStreamID = msg.getDestinationStreamID();
            this.nextIndex = 0;
            this.state = State.STREAMDATA;
            this.sendStreamNext();
        }
    }

    private void sendStreamNext() {
        int size = Math.min(this.bufferSize, this.content.length - this.nextIndex);
        int[] data = new int[size];
        for (int i = 0; i < size; ++i) {
            data[i] = this.content[this.nextIndex + i];
        }
        AddressedMessage m = new StreamDataSendMessage(this.src, this.dest, this.destStreamID, data);
        this.connection.put(m, this);
        this.nextIndex += size;
        this.feedback.onProgress(100.0f * (float)this.nextIndex / (float)this.content.length);
        if (this.nextIndex < this.content.length) {
            this.startTimeout(120000);
            return;
        }
        m = new StreamDataCompleteMessage(this.src, this.dest, 4, this.destStreamID);
        this.connection.put(m, this);
        this.sendUnfreeze();
        this.state = State.SUCCESS;
    }

    @Override
    public void handleStreamDataProceed(StreamDataProceedMessage msg, Connection sender) {
        if (this.state == State.STREAMDATA && this.isReply(msg)) {
            this.endTimeout();
            this.sendStreamNext();
        }
    }

    private void sendDGs() {
        this.nextIndex = 0;
        this.bufferSize = 64;
        this.errorCounter = 0;
        this.sendDGNext();
    }

    private void sendDGNext() {
        final int size = Math.min(this.bufferSize, this.content.length - this.nextIndex);
        byte[] data = new byte[size];
        System.arraycopy(this.content, this.nextIndex, data, 0, size);
        this.mcs.requestWrite(this.dest, this.space, this.nextIndex, data, new MemoryConfigurationService.McsWriteHandler(){

            @Override
            public void handleFailure(int errorCode) {
                if (++LoaderClient.this.errorCounter > 3) {
                    LoaderClient.this.failWith(errorCode, "Repeated errors writing to firmware space.");
                } else {
                    LoaderClient.this.sendDGNext();
                }
            }

            @Override
            public void handleSuccess() {
                LoaderClient.this.nextIndex = LoaderClient.this.nextIndex + size;
                LoaderClient.this.errorCounter = 0;
                float p = 100.0f * (float)LoaderClient.this.nextIndex / (float)LoaderClient.this.content.length;
                LoaderClient.this.feedback.onProgress(p);
                if (LoaderClient.this.nextIndex < LoaderClient.this.content.length) {
                    LoaderClient.this.sendDGNext();
                } else {
                    LoaderClient.this.state = State.SUCCESS;
                    LoaderClient.this.sendUnfreeze();
                }
            }
        });
    }

    private void sendUnfreeze() {
        this.dcs.sendData(new DatagramService.DatagramServiceTransmitMemo(this.dest, new int[]{32, 160, this.space}){

            @Override
            public void handleSuccess(int flags) {
                if (LoaderClient.this.state == State.SUCCESS) {
                    LoaderClient.this.feedback.onProgress(100.0f);
                    LoaderClient.this.feedback.onDone(0, "");
                }
            }

            @Override
            public void handleFailure(int errorCode) {
                if (errorCode == 65792) {
                    this.handleSuccess(0);
                } else if (LoaderClient.this.state == State.SUCCESS) {
                    LoaderClient.this.failWith(errorCode, "Download Failed in UnFreeze");
                }
            }
        });
    }

    private void failWith(int errorCode, String errorString) {
        boolean b = this.state != State.FAIL && this.state != State.UNFREEEZE;
        this.state = State.FAIL;
        String tmpString = null;
        if (errorCode == 8328) {
            tmpString = "Failed download checksum; try again";
        } else if (errorCode == 4233) {
            tmpString = "File corrupted";
        } else if (errorCode == 4232) {
            tmpString = "The firmware data is incompatible with this hardware node";
        }
        if (tmpString != null) {
            errorString = tmpString + " - " + errorString;
        }
        this.feedback.onDone(errorCode, errorString);
        if (b) {
            this.sendUnfreeze();
        }
    }

    public void dispose() {
        if (timer != null) {
            timer.cancel();
            timer = null;
        }
    }

    public static abstract class LoaderStatusReporter {
        public abstract void onProgress(float var1);

        public abstract void onDone(int var1, String var2);
    }

    static enum State {
        IDLE,
        ABORT,
        FREEZE,
        INITCOMPL,
        PIP,
        PIPREPLY,
        SETUPSTREAM,
        STREAM,
        STREAMDATA,
        DG,
        UNFREEEZE,
        SUCCESS,
        FAIL;

    }
}

