/*
 *   options.c
 *
 *   (C) Richard Drummond 2001-2007
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 */

#include "defs.h"
#include "options.h"
#include <string.h>

/*
 * Parse options according to supplied template and stuff parsed values
 * into structure opts using types/offsets specified in template.
 *
 * Returns RETURN_FAILURE if an error occured, RETURN_SUCCESS otherwise
 *
 * TODO: fix error handling/reporting
 */
int Options_parse( const OptionTemplate *tmplt, void *opts, int argc, const char *argv[] )
{
  while( --argc > 0 )
  {
    const char           *option = *(++argv);
    int                   found  = 0;
    const OptionTemplate *ot     = (OptionTemplate*) tmplt;

    while( ot->name != 0 )
    {
      if( (found = !strcmp( option, ot->name )) )
      {
        if( ot->type == OPT_TYPE_BOOLEAN )
        {
          int *ptr = (int *) (((ptrdiff_t) opts) + ot->offset);
          *ptr = 1;
        }
        else
        {
          const char *value = argc > 1 ? argv[1] : 0;

          if( value && value[0] != '-' )
          {
            if( ot->type == OPT_TYPE_STRING )
            {
              const char **ptr = (const char **) (((ptrdiff_t) opts) + ot->offset);

              *ptr = value;
            }
            else if( ot->type == OPT_TYPE_INT )
            {
              int  *ptr = (int *) (((ptrdiff_t) opts) + ot->offset );
              char *endptr;
              *ptr = strtod( value, &endptr );

              if( *endptr )
              {
                fprintf( stderr, PACKAGE ": Option %s requires an integer value\n", option );
                return RETURN_FAILURE;
              }
            }
            argc--; argv++;
          }
          else
          {
            fprintf( stderr, PACKAGE ": Option %s requires a value\n", option );
            return RETURN_FAILURE;
          }
        }
        break;
      }
      // Try next option template
      ot++;
    } // while

    if( !found )
    {
      fprintf( stderr, PACKAGE ": Unrecognized option: %s\n", option );
      return RETURN_FAILURE;
    }
  }
  return RETURN_SUCCESS;
}


/*
 * Print command-line usage as specified by options template stream
 */
void Options_printUsage( const OptionTemplate *opt, FILE *stream )
{
  fprintf( stream, "Usage:\n" );

  while( opt->name )
  {
    int col; // keep track of no of char output and hence current column no.

    col = fprintf( stream, "  %s", opt->name );

    if( opt->type == OPT_TYPE_INT )
      col += fprintf( stream, " <n>" );
    else if( opt->type == OPT_TYPE_STRING )
      col += fprintf( stream, " <string>" );

    fprintf( stream, "%*s %s\n", 16-col, "", opt->help );
    opt++;
  }
}
