/* #includes */ /*{{{C}}}*//*{{{*/
#undef  _POSIX_SOURCE
#define _POSIX_SOURCE   1
#undef  _POSIX_C_SOURCE
#define _POSIX_C_SOURCE 2

#include "config.h"

#include <sys/types.h>
#include <ctype.h>
#include <errno.h>
#ifdef HAVE_GETTEXT
#include <libintl.h>
#define _(String) gettext(String)
#else
#define _(String) String
#endif
#include <locale.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

#include "getopt.h"
#include "misc.h"
/*}}}*/

/* variables */ /*{{{*/
static int specletter;
static int pic=0,tbl=0,eqn=0,refer=0,refer2=0,macro=0,inlist=0,inheader=0;
static int words=0,skipheaders=0,skiplists=0,pretty=0,ignore_sonx=0;
static enum { OPTIONS, FORMAT, DATA } tblstate;
static char tblTab;
static char linebuf[512],*s;
static struct tm now;
static char **source;
static const char *file;
static int lineShould,lineIs;
static int sourceSize;
static int sourceCapacity;
static struct StringRegister
{
  char *name;
  char *value;
  struct StringRegister *next;
} *ds;
static char tr[256];
/*}}}*/
/* prototypes */ /*{{{*/
static int escChar(void);
static int textArg(void);
static int word(void);
static int number(void);
static int text(void);
static void deroff(FILE *fp, const char *fileName);
/*}}}*/

static int condputchar(int c) /*{{{*/
{
  static int nls=2;
  static int lastc='\n';

  if (!pic && !eqn && !refer && !refer2 && !macro && (!skiplists || !inlist) && (!skipheaders || !inheader))
  {
    if (pretty)
    {
      if (c=='\n') 
      {
        if (++nls>2) return c;
      }
      else
      {
        nls=0;
      }
      if (c=='\n') ++lineIs;
      lastc=c;
      return putchar(tr[(unsigned int)((unsigned char)c)]);
    }
    else return putchar(tr[(unsigned int)((unsigned char)c)]);
  }
  else if (!pretty && c=='\n') return putchar(tr[(unsigned int)((unsigned char)c)]);
  else return c;
}
/*}}}*/
static int condputs(const char *str) /*{{{*/
{
  while (*str) if (condputchar(*str++)==-1) return -1;
  return 0;
}
/*}}}*/

static int prch(char ch) /*{{{*/
{
  return (ch && ch!=' ' && ch!='\t' && ch!='\n');
}
/*}}}*/
static int white(char ch) /*{{{*/
{
  return (ch==' ' || ch=='\t' || ch=='\n');
}
/*}}}*/
static int digit(int ch) /*{{{*/
{
  return (ch>='0' && ch<='9');
}
/*}}}*/
static int letter(char ch) /*{{{*/
{
  return (isalpha((int)ch) || (ch=='_') /* underscore is used in C identifiers */
#if 0
  /* german umlauts in latin1 */
  || ch==(char)0xe4 || ch==(char)0xc4 || ch==(char)0xf6 || ch==(char)0xd6 || ch==(char)0xfc || ch==(char)0xdc || ch==(char)0xdf
#endif
  );
}
/*}}}*/

static int comment(void) /*{{{*/
{
  /*
  comment ::= '\\' '"' character ...
  */
  if (*s=='\\' && *(s+1)=='"')
  {
    while (*s && *s!='\n') ++s;
    return 1;
  }
  return 0;
}
/*}}}*/
static int font(void) /*{{{*/
{
  /*
  font ::= '\\' 'f' '(' prch prch
  font ::= '\\' 'f' prch
  font ::= '\\' 'f' '[' prch ... ']'
  */

  if (*s=='\\' && *(s+1)=='f')
  {
    if (*(s+2)=='(' && prch(*(s+3)) && prch(*(s+4))) { s+=5; return 1; }
    else if (*(s+2)=='[')
    {
      s+=2; while (prch(*s) && *s!=']') ++s;
      if (*s==']') ++s;
    }
    else if (prch(*(s+2))) { s+=3; return 1; }
  }
  return 0;
}
/*}}}*/
static int numreq(void) /*{{{*/
{
  /*
  numreq ::= '\\' [ 'h' | 'v' | 'w' | 'u' | 'd' ]  '\'' escChar { escChar } '\''
  */

  if (*s=='\\' && (*(s+1)=='h' || *(s+1)=='v' || *(s+1)=='w' || *(s+1)=='u' || *(s+1)=='d') && *(s+2)=='\'')
  {
    ++macro;
    s+=3;
    while (*s!='\'' && escChar());
    if (*s=='\'') ++s;
    --macro;
    return 1;
  }
  return 0;
}
/*}}}*/
static int size(void) /*{{{*/
{
  /*
  size ::= '\\' 's' digit { digit }
  size ::= '\\' 's' [ '+' | '-' ] digit { digit }
  */

  if (*s=='\\' && *(s+1)=='s' && (digit(*(s+2)) || ((*(s+2)=='-' || *(s+2)=='+') && digit(*(s+3)))))
  {
    s+=3; while (digit(*s)) ++s;
    return 1;
  }
  return 0;
}
/*}}}*/
static int var(void) /*{{{*/
{
  /*
  var ::= '\\' ( '*' | 'n' ) '[' prch+ ']'
  var ::= '\\' ( '*' | 'n' ) '(' prch prch
  var ::= '\\' ( '*' | 'n' ) prch
  */

  if (*s=='\\' && *(s+1)=='n')
  {
    if (*(s+3)=='d' && *(s+4)=='y')
    {
      if (now.tm_mday/10) condputchar(now.tm_mday/10+'0');
      condputchar(now.tm_mday%10+'0');
      s+=5;
      return 1;
    }
    else if (*(s+2)=='(' && prch(*(s+3)) && prch(*(s+4)))
    {
      s+=5; return 1;
    }
    else if (*(s+2)=='[' && prch(*(s+3)))
    {
      s+=3;
      while (*s && *s!=']') ++s;
      return 1;
    }
    else if (prch(*(s+2))) { s+=3; return 1; }
  }
  else if (*s=='\\' && *(s+1)=='*')
  {
    char reg[64];
    struct StringRegister *r;

    if (*(s+2)=='(' && prch(*(s+3)) && prch(*(s+4)))
    {
      reg[0]=*(s+3);
      reg[1]=*(s+4);
      reg[2]='\0';
      s+=5;
    }
    else if (*(s+2)=='[' && prch(*(s+3)))
    {
      int i;

      for (s+=3,i=0; *s && *s!=']'; ++s) if (i<sizeof(reg)-1) reg[i++]=*s;
      if (*s==']')
      {
        reg[i]='\0';
        ++s;
      }
      else return 0;
    }
    else if (prch(*(s+2)))
    {
      reg[0]=*(s+2);
      reg[1]='\0';
      s+=3;
    }
    else return 0;
    for (r=ds; r; r=r->next) if (strcmp(reg,r->name)==0)
    {
      char *old_s;

      old_s=s;
      s=r->value;
      textArg();
      s=old_s;
      break;
    }
    return 1;
  }
  return 0;
}
/*}}}*/
static int spec(void) /*{{{*/
{
  specletter=0;
  if (*s=='\\' && *(s+1)=='(' && prch(*(s+2)) && prch(*(s+3)))
  {
    switch (*(s+2)|((*(s+3))<<8))
    {
      case 'm'|('i'<<8):
      case 'e'|('n'<<8):
      case 'h'|('y'<<8): if (!words) condputchar('-'); break;
      case 'e'|('m'<<8): if (!words) condputs("--"); break;
      case 'l'|('q'<<8): if (!words) condputs("``"); break;
      case 'r'|('q'<<8): if (!words) condputs("\'\'"); break;
      case 'B'|('q'<<8): if (!words) condputs(",,"); break;
      case 'o'|('q'<<8): if (!words) condputchar('`'); break;
      case 'c'|('q'<<8): if (!words) condputchar('\''); break;
      case 'a'|('q'<<8): if (!words) condputchar('\''); break;
      case 'd'|('q'<<8): if (!words) condputchar('"'); break;
      case 'o'|('r'<<8): if (!words) condputchar('|'); break;
      case 'a'|('t'<<8): if (!words) condputchar('@'); break;
      case 's'|('h'<<8): if (!words) condputchar('#'); break;
      case 'E'|('u'<<8):
      case 'e'|('u'<<8): if (!words) condputchar('\244'); break;
      case 'D'|('o'<<8): if (!words) condputchar('$'); break;
      case 'c'|('t'<<8): if (!words) condputchar('\242'); break;
      case 'F'|('o'<<8): if (!words) condputchar('\253'); break;
      case 'F'|('c'<<8): if (!words) condputchar('\273'); break;
      case 'f'|('o'<<8): if (!words) condputchar('<'); break;
      case 'f'|('c'<<8): if (!words) condputchar('>'); break;
      case 'r'|('!'<<8): if (!words) condputchar('\241'); break;
      case 'r'|('?'<<8): if (!words) condputchar('\277'); break;
      case 'O'|('f'<<8): if (!words) condputchar('\252'); break;
      case 'O'|('m'<<8): if (!words) condputchar('\272'); break;
      case 'p'|('c'<<8): if (!words) condputchar('\267'); break;
      case 'S'|('1'<<8): if (!words) condputchar('\271'); break;
      case 'S'|('2'<<8): if (!words) condputchar('\262'); break;
      case 'S'|('3'<<8): if (!words) condputchar('\263'); break;
      case '<'|('-'<<8): if (!words) condputs("<-"); break;
      case '-'|('>'<<8): if (!words) condputs("->"); break;
      case '<'|('>'<<8): if (!words) condputs("<->"); break;
      case 'u'|('a'<<8): if (!words) condputchar('^'); break;
      case 'd'|('a'<<8): if (!words) condputchar('v'); break;
      case 'l'|('A'<<8): if (!words) condputs("<="); break;
      case 'r'|('A'<<8): if (!words) condputs("=>"); break;
      case 'h'|('A'<<8): if (!words) condputs("<=>"); break;
      case 'u'|('A'<<8): if (!words) condputs("^^"); break;
      case 'd'|('A'<<8): if (!words) condputs("vv"); break;
      case 'b'|('a'<<8):
      case 'b'|('b'<<8):
      case 'b'|('r'<<8):
      case 'b'|('v'<<8): if (!words) condputchar('|'); break;
      case 'r'|('u'<<8):
      case 'u'|('l'<<8): if (!words) condputchar('_'); break;
      case 'c'|('i'<<8): if (!words) condputchar('O'); break;
      case 'b'|('u'<<8): if (!words) condputchar('o'); break;
      case 'c'|('o'<<8): if (!words) condputchar('\251'); break;
      case 'r'|('g'<<8): if (!words) condputchar('\256'); break;
      case 't'|('m'<<8): if (!words) condputs("(TM)"); break;
      case 'd'|('d'<<8): if (!words) condputs("||"); break;
      case 'd'|('g'<<8): if (!words) condputchar('|'); break;
      case 'p'|('s'<<8): if (!words) condputchar('\266'); break;
      case 's'|('c'<<8): if (!words) condputchar('\247'); break;
      case 'd'|('e'<<8): if (!words) condputchar('\260'); break;
      case '%'|('0'<<8): if (!words) condputs("0/00"); break;
      case '1'|('4'<<8): if (!words) condputchar('\274'); break;
      case '1'|('2'<<8): if (!words) condputchar('\275'); break;
      case '3'|('4'<<8): if (!words) condputchar('\276'); break;
      case 'f'|('/'<<8):
      case 's'|('l'<<8): if (!words) condputchar('/'); break;
      case 'r'|('s'<<8): if (!words) condputchar('\\'); break;
      case 's'|('q'<<8): if (!words) condputs("[]"); break;
      case 'f'|('m'<<8): if (!words) condputchar('\''); break;
      case 'h'|('a'<<8): if (!words) condputchar('^'); break;
      case 't'|('i'<<8): if (!words) condputchar('~'); break;
      case 'l'|('B'<<8): if (!words) condputchar('['); break;
      case 'r'|('B'<<8): if (!words) condputchar(']'); break;
      case 'l'|('C'<<8): if (!words) condputchar('{'); break;
      case 'r'|('C'<<8): if (!words) condputchar('}'); break;
      case 'l'|('a'<<8): if (!words) condputchar('<'); break;
      case 'r'|('a'<<8): if (!words) condputchar('>'); break;
      case 'l'|('h'<<8): if (!words) condputs("<="); break;
      case 'r'|('h'<<8): if (!words) condputs("=>"); break;
      case 't'|('f'<<8): if (!words) condputs("therefore"); break;
      case '~'|('~'<<8): if (!words) condputs("~~"); break;
      case '~'|('='<<8): if (!words) condputs("~="); break;
      case '!'|('='<<8): if (!words) condputs("!="); break;
      case '*'|('*'<<8): if (!words) condputchar('*'); break;
      case '+'|('-'<<8): if (!words) condputchar('\261'); break;
      case '<'|('='<<8): if (!words) condputs("<="); break;
      case '='|('='<<8): if (!words) condputs("=="); break;
      case '='|('~'<<8): if (!words) condputs("=~"); break;
      case '>'|('='<<8): if (!words) condputs(">="); break;
      case 'A'|('N'<<8): if (!words) condputs("\\/"); break;
      case 'O'|('R'<<8): if (!words) condputs("/\\"); break;
      case 'n'|('o'<<8): if (!words) condputchar('\254'); break;
      case 't'|('e'<<8): if (!words) condputs("there exists"); break;
      case 'f'|('a'<<8): if (!words) condputs("for all"); break;
      case 'A'|('h'<<8): if (!words) condputs("aleph"); break;
      case 'I'|('m'<<8): if (!words) condputs("imaginary"); break;
      case 'R'|('e'<<8): if (!words) condputs("real"); break;
      case 'i'|('f'<<8): if (!words) condputs("infinity"); break;
      case 'm'|('d'<<8): if (!words) condputs("\267"); break;
      case 'm'|('o'<<8): if (!words) condputs("member of"); break;
      case 'm'|('u'<<8): if (!words) condputchar('\327'); break;
      case 'n'|('m'<<8): if (!words) condputs("not member of"); break;
      case 'p'|('l'<<8): if (!words) condputchar('+'); break;
      case 'e'|('q'<<8): if (!words) condputchar('='); break;
      case 'p'|('t'<<8): if (!words) condputs("oc"); break;
      case 'p'|('p'<<8): if (!words) condputs("perpendicular"); break;
      case 's'|('b'<<8): if (!words) condputs("(="); break;
      case 's'|('p'<<8): if (!words) condputs("=)"); break;
      case 'i'|('b'<<8): if (!words) condputs("(-"); break;
      case 'i'|('p'<<8): if (!words) condputs("-)"); break;
      case 'a'|('p'<<8): if (!words) condputchar('~'); break;
      case 'i'|('s'<<8): if (!words) condputchar('I'); break;
      case 's'|('r'<<8): if (!words) condputs("root"); break;
      case 'p'|('d'<<8): if (!words) condputchar('d'); break;
      case 'c'|('*'<<8): if (!words) condputs("(x)"); break;
      case 'c'|('+'<<8): if (!words) condputs("(+)"); break;
      case 'c'|('a'<<8): if (!words) condputs("cap"); break;
      case 'c'|('u'<<8): if (!words) condputchar('U'); break;
      case 'd'|('i'<<8): if (!words) condputchar('\367'); break;
      case 'g'|('r'<<8): if (!words) condputchar('V'); break;
      case 'e'|('s'<<8): if (!words) condputs("{}"); break;
      case 'C'|('R'<<8): if (!words) condputs("_|"); break;
      case 's'|('t'<<8): if (!words) condputs("such that"); break;
      case '/'|('_'<<8): if (!words) condputs("/_"); break;
      case 'l'|('z'<<8): if (!words) condputs("<>"); break;
      case 'a'|('n'<<8): if (!words) condputchar('-'); break;
      /* output composed latin1 letters */
      case '-'|('D'<<8): condputchar('\320'); specletter=1; break;
      case 'S'|('d'<<8): condputchar('\360'); specletter=1; break;
      case 'T'|('p'<<8): condputchar('\376'); specletter=1; break;
      case 'T'|('P'<<8): condputchar('\336'); specletter=1; break;
      case 'A'|('E'<<8): condputchar('\306'); specletter=1; break;
      case 'a'|('e'<<8): condputchar('\346'); specletter=1; break;
      case 'O'|('E'<<8): condputs("OE"); specletter=1; break;
      case 'o'|('e'<<8): condputs("oe"); specletter=1; break;
      case ':'|('a'<<8): condputchar('\344'); specletter=1; break;
      case ':'|('A'<<8): condputchar('\304'); specletter=1; break;
      case ':'|('e'<<8): condputchar('\353'); specletter=1; break;
      case ':'|('E'<<8): condputchar('\313'); specletter=1; break;
      case ':'|('i'<<8): condputchar('\357'); specletter=1; break;
      case ':'|('I'<<8): condputchar('\317'); specletter=1; break;
      case ':'|('o'<<8): condputchar('\366'); specletter=1; break;
      case ':'|('O'<<8): condputchar('\326'); specletter=1; break;
      case ':'|('u'<<8): condputchar('\374'); specletter=1; break;
      case ':'|('U'<<8): condputchar('\334'); specletter=1; break;
      case ':'|('y'<<8): condputchar('\377'); specletter=1; break;
      case 's'|('s'<<8): condputchar('\337'); specletter=1; break;
      case '\''|('A'<<8): condputchar('\301'); specletter=1; break;
      case '\''|('E'<<8): condputchar('\311'); specletter=1; break;
      case '\''|('I'<<8): condputchar('\315'); specletter=1; break;
      case '\''|('O'<<8): condputchar('\323'); specletter=1; break;
      case '\''|('U'<<8): condputchar('\332'); specletter=1; break;
      case '\''|('Y'<<8): condputchar('\335'); specletter=1; break;
      case '\''|('a'<<8): condputchar('\341'); specletter=1; break;
      case '\''|('e'<<8): condputchar('\351'); specletter=1; break;
      case '\''|('i'<<8): condputchar('\355'); specletter=1; break;
      case '\''|('o'<<8): condputchar('\363'); specletter=1; break;
      case '\''|('u'<<8): condputchar('\372'); specletter=1; break;
      case '\''|('y'<<8): condputchar('\375'); specletter=1; break;
      case '^'|('A'<<8): condputchar('\302'); specletter=1; break;
      case '^'|('E'<<8): condputchar('\312'); specletter=1; break;
      case '^'|('I'<<8): condputchar('\316'); specletter=1; break;
      case '^'|('O'<<8): condputchar('\324'); specletter=1; break;
      case '^'|('U'<<8): condputchar('\333'); specletter=1; break;
      case '^'|('a'<<8): condputchar('\342'); specletter=1; break;
      case '^'|('e'<<8): condputchar('\352'); specletter=1; break;
      case '^'|('i'<<8): condputchar('\356'); specletter=1; break;
      case '^'|('o'<<8): condputchar('\364'); specletter=1; break;
      case '^'|('u'<<8): condputchar('\373'); specletter=1; break;
      case '`'|('A'<<8): condputchar('\300'); specletter=1; break;
      case '`'|('E'<<8): condputchar('\310'); specletter=1; break;
      case '`'|('I'<<8): condputchar('\314'); specletter=1; break;
      case '`'|('O'<<8): condputchar('\322'); specletter=1; break;
      case '`'|('U'<<8): condputchar('\331'); specletter=1; break;
      case '`'|('a'<<8): condputchar('\340'); specletter=1; break;
      case '`'|('e'<<8): condputchar('\350'); specletter=1; break;
      case '`'|('i'<<8): condputchar('\354'); specletter=1; break;
      case '`'|('o'<<8): condputchar('\362'); specletter=1; break;
      case '`'|('u'<<8): condputchar('\371'); specletter=1; break;
      case '~'|('A'<<8): condputchar('\303'); specletter=1; break;
      case '~'|('N'<<8): condputchar('\321'); specletter=1; break;
      case '~'|('O'<<8): condputchar('\325'); specletter=1; break;
      case '~'|('a'<<8): condputchar('\343'); specletter=1; break;
      case '~'|('n'<<8): condputchar('\361'); specletter=1; break;
      case '~'|('o'<<8): condputchar('\365'); specletter=1; break;
      case ','|('C'<<8): condputchar('\307'); specletter=1; break;
      case ','|('c'<<8): condputchar('\347'); specletter=1; break;
      case '/'|('l'<<8): condputs("/l"); specletter=1; break;
      case '/'|('L'<<8): condputs("/L"); specletter=1; break;
      case '/'|('o'<<8): condputchar('\370'); specletter=1; break;
      case '/'|('O'<<8): condputchar('\330'); specletter=1; break;
      case 'o'|('A'<<8): condputchar('\305'); specletter=1; break;
      case 'o'|('a'<<8): condputchar('\345'); specletter=1; break;
      /* ouput ligatures */
      case 'f'|('i'<<8):
      case 'f'|('f'<<8):
      case 'f'|('l'<<8):
      {
        condputchar('f'); condputchar(*(s+3)); specletter=1;
        break;
      }
      case 'F'|('i'<<8):
      case 'F'|('f'<<8):
      case 'F'|('l'<<8):
      {
        condputs("ff"); condputchar(*(s+3)); specletter=1;
        break;
      }
      /* output greek */
      case '*'|('A'<<8): if (!words) condputs("Alpha"); break;
      case '*'|('B'<<8): if (!words) condputs("Beta"); break;
      case '*'|('C'<<8): if (!words) condputs("Xi"); break;
      case '*'|('D'<<8): if (!words) condputs("Delta"); break;
      case '*'|('E'<<8): if (!words) condputs("Epsilon"); break;
      case '*'|('F'<<8): if (!words) condputs("Phi"); break;
      case '*'|('G'<<8): if (!words) condputs("Gamma"); break;
      case '*'|('H'<<8): if (!words) condputs("Theta"); break;
      case '*'|('I'<<8): if (!words) condputs("Iota"); break;
      case '*'|('K'<<8): if (!words) condputs("Kappa"); break;
      case '*'|('L'<<8): if (!words) condputs("Lambda"); break;
      case '*'|('M'<<8): if (!words) condputs("Mu"); break;
      case '*'|('N'<<8): if (!words) condputs("Nu"); break;
      case '*'|('O'<<8): if (!words) condputs("Omicron"); break;
      case '*'|('P'<<8): if (!words) condputs("Pi"); break;
      case '*'|('Q'<<8): if (!words) condputs("Psi"); break;
      case '*'|('R'<<8): if (!words) condputs("Rho"); break;
      case '*'|('S'<<8): if (!words) condputs("Sigma"); break;
      case '*'|('T'<<8): if (!words) condputs("Tau"); break;
      case '*'|('U'<<8): if (!words) condputs("Upsilon"); break;
      case '*'|('W'<<8): if (!words) condputs("Omega"); break;
      case '*'|('X'<<8): if (!words) condputs("Chi"); break;
      case '*'|('Y'<<8): if (!words) condputs("Eta"); break;
      case '*'|('Z'<<8): if (!words) condputs("Zeta"); break;
      case '*'|('a'<<8): if (!words) condputs("alpha"); break;
      case '*'|('b'<<8): if (!words) condputs("beta"); break;
      case '*'|('c'<<8): if (!words) condputs("xi"); break;
      case '*'|('d'<<8): if (!words) condputs("delta"); break;
      case '*'|('e'<<8): if (!words) condputs("epsilon"); break;
      case '*'|('f'<<8): if (!words) condputs("phi"); break;
      case '+'|('f'<<8): if (!words) condputs("phi"); break;
      case '*'|('g'<<8): if (!words) condputs("gamma"); break;
      case '*'|('h'<<8): if (!words) condputs("theta"); break;
      case '+'|('h'<<8): if (!words) condputs("theta"); break;
      case '*'|('i'<<8): if (!words) condputs("iota"); break;
      case '*'|('k'<<8): if (!words) condputs("kappa"); break;
      case '*'|('l'<<8): if (!words) condputs("lambda"); break;
      case '*'|('m'<<8): if (!words) condputs("\265"); break;
      case '*'|('n'<<8): if (!words) condputs("nu"); break;
      case '*'|('o'<<8): if (!words) condputs("omicron"); break;
      case '*'|('p'<<8): if (!words) condputs("pi"); break;
      case '+'|('p'<<8): if (!words) condputs("omega"); break;
      case '*'|('q'<<8): if (!words) condputs("psi"); break;
      case '*'|('r'<<8): if (!words) condputs("rho"); break;
      case '*'|('s'<<8): if (!words) condputs("sigma"); break;
      case '*'|('t'<<8): if (!words) condputs("tau"); break;
      case '*'|('u'<<8): if (!words) condputs("upsilon"); break;
      case '*'|('w'<<8): if (!words) condputs("omega"); break;
      case '*'|('x'<<8): if (!words) condputs("chi"); break;
      case '*'|('y'<<8): if (!words) condputs("eta"); break;
      case '*'|('z'<<8): if (!words) condputs("zeta"); break;
      case 't'|('s'<<8): if (!words) condputs("sigma"); break;
    }
    s+=4; return 1;
  }
  else if (*s=='\\' && *(s+1)=='%')
  {
    specletter=1;
    s+=2;
    return 1;
  }
  else return 0;
}
/*}}}*/
static int esc(void) /*{{{*/
{
  if (*s=='\\' && *(s+1))
  {
    switch (*(s+1))
    {
      case 'e':
      case 'E': if (!words) condputchar('\\'); break;
      case 't': if (!words) condputchar('\t'); break;
      case '0':
      case '~': if (!words) condputchar(' '); break;
      case '|': 
      case '^':
      case '&':
      case ':': break;
      default: if (!words) condputchar(*(s+1)); break;
    }
    s+=2; return 1;
  }
  else return 0;
}
/*}}}*/

/*
ereq ::= '\\' 'z' '\'' escChar { escChar } '\''
ereq ::= '\\' 'z' prch
*/

static int escChar(void) /*{{{*/
{
  return (comment() || font() || size() || numreq() || var() || spec() || esc() || word() || number());
}
/*}}}*/
static int textArg(void) /*{{{*/
{
  if (!escChar())
  {
    if (*s && !white(*s))
    {
      if (!words) condputchar(*s); ++s;
    }
    else return 0;
  }
  for (;;)
  {
    if (!escChar())
    {
      if (*s && !white(*s)) 
      {
        if (!words) condputchar(*s);
        ++s;
      }
      else return 1;
    }
  }
}
/*}}}*/
static int quotedArg(void) /*{{{*/
{
  if (*s=='"')
  {
    ++s;
    while (*s && *s!='"')
    {
      if (!escChar()) if (*s)
      {
        if (!words) condputchar(*s);
        ++s;
      }
    }
    return 1;
  }
  else return 0;
}
/*}}}*/
static int requestOrMacro(void) /*{{{*/
{
  int nobody;

  ++s;
  switch (*s)
  {
    case '\\': if (*(s+1)=='"') { if (!pretty) condputchar('\n'); return 1; } else break;
    case '[': refer=1; if (!pretty) condputchar('\n'); return 1;
    case ']':
    {
      refer=0;
      ++s;
      return text();
    }
    case '.': macro=0; if (!pretty) condputchar('\n'); return 1;
  }
  nobody=0;
  switch (*s|(*(s+1)<<8))
  {
    case 'S'|('H'<<8):
    {
      if (strncmp(s+2," SYNOPSIS",8)==0) { if (!words && pretty) condputchar('\n'); inheader=1; }
      else if (strncmp(s+2," \"SYNOPSIS",9)==0) { if (!words && pretty) condputchar('\n'); inheader=1; }
      else if (strncmp(s+2," BERSICHT",10)==0) { if (!words && pretty) condputchar('\n'); inheader=1; }
      else if (strncmp(s+2," \"BERSICHT",11)==0) { if (!words && pretty) condputchar('\n'); inheader=1; }
      else { inheader=0; if (!words && pretty) condputchar('\n'); }
      nobody=1;
      break;
    }
    case 'S'|('S'<<8):
    case 'I'|('P'<<8):
    case 'H'|(' '<<8): if (!words && pretty) condputchar('\n'); nobody=1; break;
    case 'I'|(' '<<8):
    case 'I'|('R'<<8):
    case 'I'|('B'<<8):
    case 'B'|(' '<<8):
    case 'B'|('R'<<8):
    case 'B'|('I'<<8):
    case 'R'|(' '<<8):
    case 'R'|('B'<<8):
    case 'R'|('I'<<8):
    case 'A'|('B'<<8): break;
    case ']'|(' '<<8): refer=0; break;
    case 'P'|('S'<<8): if (white(*(s+2))) pic=1; if (!pretty) condputchar('\n'); return 1;
    case 'P'|('E'<<8): if (white(*(s+2))) pic=0; if (!pretty) condputchar('\n'); return 1;
    case 'T'|('S'<<8): if (white(*(s+2))) { tbl=1; tblstate=OPTIONS; } if (!pretty) condputchar('\n'); return 1;
    case 'T'|('&'<<8): if (white(*(s+2))) { tbl=1; tblstate=FORMAT; } if (!pretty) condputchar('\n'); return 1;
    case 'T'|('E'<<8): if (white(*(s+2))) tbl=0; if (!pretty) condputchar('\n'); return 1;
    case 'E'|('Q'<<8): if (white(*(s+2))) eqn=1; if (!pretty) condputchar('\n'); return 1;
    case 'E'|('N'<<8): if (white(*(s+2))) eqn=0; if (!pretty) condputchar('\n'); return 1;
    case 'R'|('1'<<8): if (white(*(s+2))) refer2=1; if (!pretty) condputchar('\n'); return 1;
    case 'R'|('2'<<8): if (white(*(s+2))) refer2=0; if (!pretty) condputchar('\n'); return 1;
    case 'd'|('e'<<8): macro=1; if (!pretty) condputchar('\n'); return 1;
    case 'B'|('L'<<8):
    case 'V'|('L'<<8):
    case 'A'|('L'<<8):
    case 'L'|('B'<<8):
    case 'R'|('L'<<8):
    case 'M'|('L'<<8):
    case 'D'|('L'<<8): if (white(*(s+2))) inlist=1; if (!pretty) condputchar('\n'); return 1;
    case 'B'|('V'<<8): if (*(s+2)=='L' && white(*(s+3))) inlist=1; if (!pretty) condputchar('\n'); return 1;
    case 'L'|('E'<<8): if (white(*(s+2))) inlist=0; if (!pretty) condputchar('\n'); return 1;
    case 'L'|('P'<<8):
    case 'P'|('P'<<8):
    case 'P'|('\n'<<8):
    {
      if (!words) { if (pretty) condputchar('\n'); condputchar('\n'); }
      return 1;
    }
    case 'd'|('s'<<8):
    {
      char *reg,*end;

      s+=2;
      while (white(*s)) ++s;
      if (*s)
      {
        reg=s;
        while (!white(*s)) ++s;
        if (*s)
        {
          struct StringRegister *r;

          *s='\0';
          ++s;
          while (white(*s)) ++s;
          end=s+strlen(s);
          if (end>s && *(end-1)=='\n') *--end='\0';
          for (r=ds; r; r=r->next) if (strcmp(reg,r->name)==0) break;
          if (r==(struct StringRegister*)0)
          {
            if
            (
              (r=(struct StringRegister*)malloc(sizeof(struct StringRegister)))==(struct StringRegister*)0
              || (r->name=(char*)malloc(strlen(reg)+1))==(char*)0
              || (r->value=(char*)malloc(strlen(s)+1))==(char*)0
            )
            {
              fprintf(stderr,_("deroff: Memory allocation failed (%s).\n"),strerror(errno));
              exit(2);
            }
            strcpy(r->name,reg);
            r->next=ds;
            ds=r;
          }
          else
          {
            free(r->value);
            if ((r->value=(char*)malloc(strlen(s)+1))==(char*)0)
            {
              fprintf(stderr,_("deroff: Memory allocation failed (%s).\n"),strerror(errno));
              exit(2);
            }
          }
          strcpy(r->value,s);
        }
      }
      if (!pretty) condputchar('\n'); return 1;
    }
    case 's'|('o'<<8):
    case 'n'|('x'<<8):
    {
      int so=(*s=='s');
      char *t;
      FILE *fp;
      int oldLineShould;

      if (!pretty) condputchar('\n');
      if (!ignore_sonx)
      {
        s+=2;
        while (white(*s)) ++s;
        t=s;
        while (*t && *t!='\n') ++t;
        *t='\0';
        if ((fp=fopen(s,"r"))==(FILE*)0)
        {
          fprintf(stderr,_("deroff: Opening `%s' failed (%s).\n"),s,strerror(errno));
          exit(1);
        }
        oldLineShould=lineShould;
        deroff(fp,s);
        fclose(fp);
        if (so)
        {
          lineIs=lineShould=oldLineShould;
          return 1;
        }
        else return 0;
      }
    }
    case 't'|('r'<<8):
    {
      s+=2;
      while (white(*s)) ++s;
      while (*s && *s!='\n')
      {
        int c;

        c=*s;
        ++s;
        tr[c]=(*s=='\n' || *s=='\0' ? ' ' : *s);
        if (*s) ++s;
      }
      return 1;
    }
    case 's'|('p'<<8): if (pretty) condputchar('\n'); condputchar('\n'); return 1;
    default: if (!pretty) condputchar('\n'); return 1;
  }
  if (skipheaders && nobody) return 1;
  while (white(*s)) ++s;
  while (*s && !white(*s)) ++s;
  while (white(*s)) ++s;
  for (;;)
  {
    if (!quotedArg() && !textArg())
    {
      if (*s)
      {
        if (!words) condputchar(*s);
        ++s;
      }
      else return 1;
    }
  }
}
/*}}}*/
static int number(void) /*{{{*/
{
  if (((*s=='-' || *s=='+') && digit(*(s+1))) || digit(*s))
  {
    if (!words) condputchar(*s);
    ++s;
    while (digit(*s)) { if (!words) condputchar (*s); ++s; }
    return 1;
  }
  return 0;
}
/*}}}*/
static int word(void) /*{{{*/
{
  if (letter(*s))
  {
    condputchar(*s);
    ++s;
    for (;;)
    {
      if (spec() && !specletter) break;
      else
      {
        if (letter(*s)) { condputchar(*s); ++s; }
        else break;
      }
    }
    if (words) condputchar('\n');
    return 1;
  }
  return 0;
}
/*}}}*/
static int text(void) /*{{{*/
{
  for (;;)
  {
    if (!escChar())
    {
      if (*s) { if (!words) condputchar(*s); ++s; }
      else break;
    }
  }
  return 1;
}
/*}}}*/
static int doTbl(void) /*{{{*/
{
  switch (tblstate)
  {
    case OPTIONS: /*{{{*/
    {
      const char *option,*arg;

      while (*s && *s!=';' && *s!='\n')
      {
        while (*s==' ' || *s=='\t') ++s;
        if (isalpha(*s)) /* parse option */ /*{{{*/
        {
          option=s;
          arg=(const char*)0;
          while (isalpha(*s)) ++s;
          if (*s=='(')
          {
            *s='\0';
            arg=++s;
          }
          else
          {
            *s='\0';
            while (*s==' ' || *s=='\t') ++s;
            if (*s=='(') arg=++s;
          }
          if (arg)
          {
            while (*s && *s!=')') ++s;
            if (*s) *s++='\0';
          }
          if ((strcmp(option,"tab")==0 || strcmp(option,"TAB")==0) && arg) tblTab=*arg;
        }
        /*}}}*/
      }
      tblstate=FORMAT;
      if (!pretty) condputchar('\n');
      break;
    }
    /*}}}*/
    case FORMAT: /*{{{*/
    {
      while (*s && *s!='.' && *s!='\n')
      {
        while (*s==' ' || *s=='\t') ++s;
        if (*s) ++s;
      }
      if (*s=='.') tblstate=DATA;
      if (!pretty) condputchar('\n');
      break;
    }
    /*}}}*/
    case DATA: /*{{{*/
    {
      char *old;

      old=s;
      while (*s)
      {
        if (tblTab!='\t' && *s==tblTab) *s='\t';
        ++s;
      }
      s=old;
      text();
      break;
    }
    /*}}}*/
  }
  return 1;
}
/*}}}*/
static int doLine(void) /*{{{*/
{
  if (*s=='.' || *s=='\'')
  {
    if (requestOrMacro()==0) return 0;
  }
  else if (tbl)
  {
    doTbl();
  }
  else
  {
    text();
  }
  return 1;
}
/*}}}*/
static void deroff(FILE *fp, const char *fileName) /*{{{*/
{
  lineShould=1;
  lineIs=1;
  tblTab='\t';
  if (fileName)
  {
    int i;

    for (i=0; i<sourceSize; ++i) if (strcmp(source[i],fileName)==0)
    {
      fprintf(stderr,_("deroff: Ignoring repeated request to process `%s'.\n"),fileName);
      return;
    }
    if (sourceSize==sourceCapacity && (source=realloc(source,sizeof(char*)*(sourceCapacity*=2)))==(char**)0)
    {
      fprintf(stderr,_("deroff: Memory allocation failed (%s).\n"),strerror(errno));
      exit(2);
    }
    if ((source[sourceSize]=(char*)malloc(strlen(fileName)+1))==(char*)0)
    {
      fprintf(stderr,("deroff: Memory allocation failed (%s).\n"),strerror(errno));
      exit(2);
    }
    strcpy(source[sourceSize],fileName);
    file=source[sourceSize];
    ++sourceSize;
  }
  while (fgets(linebuf,sizeof(linebuf),fp))
  {
    s=linebuf;
    if (doLine()==0) break;
    ++lineShould;
  }
}
/*}}}*/

int main(int argc, char *argv[]) /*{{{*/
{
  /* variables */ /*{{{*/
  FILE *in;
  int usage=0;
  int c;
  time_t nowt;
  static struct option lopts[]=
  {
    { "word-list", no_argument, 0, 'w' },
    { "skip-headers", no_argument, 0, 's' },
    { "skip-lists", no_argument, 0, 'L' },
    { "ignore", no_argument, 0, 'i' },
    { "pretty-print", no_argument, 0, 'p' },
    { "help", no_argument, 0, 'h' },
    { "version", no_argument, 0, 'v' },
    { (const char*)0, 0, 0, '\0' }
  };
  /*}}}*/

  setlocale(LC_MESSAGES,"");
  setlocale(LC_CTYPE,"");
#ifdef HAVE_GETTEXT
  bindtextdomain("deroff",LOCALEDIR);
  textdomain("deroff");
#endif
  nowt=time((time_t*)0);
  now=*localtime(&nowt);
  for (c=0; c<256; ++c) tr[c]=c;
  if ((source=(char**)malloc(sizeof(char*)*(sourceCapacity=64)))==(char**)0)
  {
    fprintf(stderr,_("deroff: Memory allocation failed (%s).\n"),strerror(errno));
    exit(2);
  }
  /* parse arguments */ /*{{{*/
  while ((c=getopt_long(argc,argv,"wsim:C?hp",lopts,(int*)0))!=EOF) switch(c)
  {
    case 'w': words=1; break;
    case 's': skipheaders=1; break;
    case 'L': skiplists=1; break;
    case 'i': ignore_sonx=1; break;
    case 'm':
    {
      if (strcmp(optarg,"l")==0) skiplists=1;
      else if (strcmp(optarg,"s") && strcmp(optarg,"m")) usage=1;
      break;
    }
    case 'h': usage=2; break;
    case 'p': pretty=1; break;
    case 'v': printf("deroff " VERSION "\n"); exit(0);
    default: usage=1;
  }
  if (usage==1)
  {
    fprintf(stderr,_("Usage: deroff [-w] [-s] [-ml] [-ms] [-mm] [-i] [-p] [file ...]\n"));
    fprintf(stderr,_("       deroff [--word-list] [--skip-headers] [--skip-lists] [--ignore]\n"
                     "              [--pretty-print] [file ...]\n"));
    fprintf(stderr,_("       deroff --version\n"));
    fprintf(stderr,"\n");
    fprintf(stderr,_("Try deroff -h or deroff --help for more information.\n"));
    exit(1);
  }
  if (usage==2)
  {
    printf(_("Usage: deroff [-w] [-s] [-ml] [-ms] [-mm] [-i] [-p] [file ...]\n"));
    printf(_("       deroff [--word-list] [--skip-headers] [--skip-lists] [--ignore]\n"
                     "              [--pretty-print] [file ...]\n"));
    printf(_("       deroff --version\n"));
    printf("\n");
    printf(_("Remove roff, tbl, eqn, refer and pic constructs from documents.\n"));
    printf("\n");
    printf(_("-w, --word-list        output a word list\n"));
    printf(_("-s, --skip-headers     skip headers\n"));
    printf(_("-ml, --skip-lists      suppress lists\n"));
    printf(_("-ms, -mm               ignored for compatibility\n"));
    printf(_("-i, --ignore           ignore .so and .nx requests\n"));
    printf(_("-p, --pretty-print     pretty printed output\n"));
    fputs("\n",stdout);
    fputs(_("Report bugs to <michael@moria.de>.\n"),stdout);
    exit(0);
  }
  /*}}}*/
  /* deroff stdin or files, if any */ /*{{{*/
  if (optind<argc) while (optind<argc)
  {
    int i;

    if ((in=fopen(argv[optind],"r"))==(FILE*)0)
    {
      fprintf(stderr,_("deroff: Opening `%s' failed (%s).\n"),argv[optind],strerror(errno));
      exit(1);
    }
    for (i=0; i<sourceSize; ++i) free(source[i]);
    sourceSize=0;
    deroff(in,argv[optind]);
    fclose(in);
    ++optind;
  }
  else
  {
    deroff(stdin,(const char*)0);
  }
  if (fclose(stdout)==-1)
  {
    fprintf(stderr,_("deroff: Closing standard output failed (%s).\n"),strerror(errno));
    return 1;
  }
  /*}}}*/
  return 0;
}
/*}}}*/
