/*
 * This application is free software and is released under the terms of
 * the BSD license. See LICENSE file for details.
 *
 * Copyright (c) 2020 Volker Poplawski (volker@openbios.org)
 */
#ifndef FILEFORMAT_H
#define FILEFORMAT_H

#include <QtCore>

enum class FileFormat
{
  RAW,
  RFF
};


struct RffFileInfo
{
  QString startDate;
  QString endDate;
  QString name;
  QString version;
  int     errorLogging;
  int     errorCount;
  int     origin;
  int     latestOfs;
  int     oldestOfs;
  int     fileSize;
  QString rffId;
  int     recFormat;
  int     dataFormat;
  QString sysId;
  int     unitId;
  int     ifType;
  QString ifName;
  QString  ifId;
  int     protocol;
  int     channel;
  int     speed;

  static RffFileInfo read(QIODevice* file)
  {
    RffFileInfo info;
    info.startDate = QString::fromUtf8(file->read(20));
    info.endDate = QString::fromUtf8(file->read(20));
    info.name = QString::fromUtf8(file->read(13));
    info.version = QString::fromUtf8(file->read(5));
    info.errorLogging = qFromLittleEndian<quint8>(file->read(1));
    info.errorCount = qFromLittleEndian<quint16>(file->read(2));
    info.origin = qFromLittleEndian<quint8>(file->read(1));
    info.latestOfs = qFromLittleEndian<quint32>(file->read(4));
    info.oldestOfs = qFromLittleEndian<quint32>(file->read(4));
    info.fileSize = qFromLittleEndian<quint32>(file->read(4));
    file->read(6);  // padding
    info.rffId = QString::fromUtf8(file->read(8));
    info.recFormat = qFromLittleEndian<quint8>(file->read(1));
    info.dataFormat = qFromLittleEndian<quint8>(file->read(1));
    info.sysId = QString::fromUtf8(file->read(6));
    info.unitId = qFromLittleEndian<quint8>(file->read(1));
    info.ifType = qFromLittleEndian<quint8>(file->read(1));
    info.ifName = QString::fromUtf8(file->read(17));
    info.ifId = QString::fromUtf8(file->read(6));
    info.protocol = qFromLittleEndian<quint8>(file->read(1));
    info.channel = qFromLittleEndian<quint16>(file->read(2));
    info.speed = qFromLittleEndian<quint32>(file->read(4));
    return info;
  }

  static QString formatRffFileInfo(const RffFileInfo& info)
  {
    QString s;
    s += " Start Date: " + info.startDate + "\n";
    s += "   End Date: " + info.endDate + "\n";
    s += "       Name: " + info.name + "\n";
    s += "    Version: " + info.version + "\n";
    s += " ErrLogging: " + QString::number(info.errorLogging) + "\n";
    s += "   ErrCount: " + QString::number(info.errorCount) + "\n";
    s += "  Orig Type: " + QString::number(info.origin) + "\n";
    s += " Latest Ofs: " + QString::number(info.latestOfs) + "\n";
    s += " Oldest Ofs: " + QString::number(info.oldestOfs) + "\n";
    s += " File Size:  " + QString::number(info.fileSize) + "\n";
    s += "     Rff Id: " + info.rffId + "\n";
    s += " Rec Format: " + QString::number(info.recFormat) + "\n";
    s += "Data Format: " + QString::number(info.dataFormat) + "\n";
    s += "     Sys Id: " + info.sysId + "\n";
    s += "    Unit Id: " + QString::number(info.unitId) + "\n";
    s += "  Intf Type: " + QString::number(info.ifType) + "\n";
    s += "  Intf Name: " + info.ifName + "\n";
    s += "    Intf Id: " + info.ifId + "\n";
    s += "   Protocol: " + QString::number(info.protocol) + "\n";
    s += "    Channel: " + QString::number(info.channel) + "\n";
    s += "      Speed: " + QString::number(info.speed) + "\n";
    return s;
  }

};

#endif // FILEFORMAT_H
