/*
 * ===========================
 * VDK Visual Develeopment Kit
 * Version 2.0.0
 * november 2000
 * ===========================
 *
 * Copyright (C) 1998, Mario Motta
 * Developed by Mario Motta <mmotta@guest.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#ifndef _drawingcompo_h
#define _drawingcompo_h
#include <vdk/vdk.h>
#define MAX_SHAPES 10
typedef VDKValueList<VDKPoint> PointList;
typedef VDKValueListIterator<VDKPoint> PointListIterator;
/*
 */
class DrawingShape;
class DrawingComponent : public VDKBox
{
  VDKDrawingArea *drawing;
  VDKLabel  *label;
  VDKCustomButton *mlButton;
  VDKCustomButton *clearButton;
  DrawingShape* shape_array[MAX_SHAPES];
  PointList plist;
 public:
  DrawingComponent(VDKForm* owner): VDKBox(owner){}
  ~DrawingComponent();
  void Setup();
  bool OnMotion(VDKObject* sender, GdkEvent* event);
  bool OnExpose(VDKObject*, GdkEvent* event);
  bool Clear(VDKObject*);
  bool OnButtonPress(VDKObject*, GdkEvent* ev);
  bool OnButtonRelease(VDKObject*, GdkEvent* ev);
  DECLARE_SIGNAL_MAP(DrawingComponent);
  DECLARE_EVENT_LIST(DrawingComponent);
};

/*
some shapes
*/
class DrawingShape
{
 protected:
  VDKDrawingArea* drawing;
  VDKRgb color;
  bool filled;
 public:
  VDKRect bound;
  DrawingShape(VDKDrawingArea* owner,
	      int x, int y, int w, int h, 
	      VDKRgb color, bool filled = false): 
    drawing(owner),color(color),filled(filled),bound(x,y,w,h) {}
  virtual ~DrawingShape() {}
  virtual void Draw() = 0;
  void DrawBound()
    {
      GdkFunction function = drawing->Pen->Function;
      GdkLineStyle style = drawing->Pen->Style;
      drawing->Pen->Function = GDK_INVERT;
      drawing->Pen->Style = GDK_LINE_ON_OFF_DASH;
      drawing->DrawRect(false,bound.left,bound.top,bound.W(),bound.H());
      drawing->Pen->Function = function;
      drawing->Pen->Style = style;
    }
};

class 
DrawingRectangle: public DrawingShape
{
 private:
  bool tiled;
 public:
  DrawingRectangle(VDKDrawingArea* owner,
		   int x, int y, int w, int h,
		   VDKRgb color,
		   bool filled = false,
		   bool tiled = false): 
    DrawingShape(owner,x,y,w,h,color,filled),tiled(tiled) { }
  virtual ~DrawingRectangle() {}
  void Draw() 
  {
    drawing->Pen->Color = color;
    drawing->Pen->Style = GDK_LINE_SOLID;
    if(filled)
      {
	drawing->Pen->Fill = tiled ? GDK_TILED : GDK_SOLID;
	if(tiled) 
	  { 
	    drawing->Pen->SetTile("fuzzy.png");
	    drawing->DrawRect(filled,bound.Origin().x+1,
		     bound.Origin().y+1,
		     bound.w-1,bound.h-1);
	  }
      }
    drawing->Pen->Fill = GDK_SOLID;
    drawing->DrawRect(tiled ? false : filled, bound.Origin().x,
		     bound.Origin().y,
		     bound.w,bound.h);
  }
};

/*
  Arc constructor is probably one that needs more
  explanation becaues it has quite a few parameters:
  x,y origin of arc bounding rect
  w,h width and height of arc bounding rect
  start_angle,end_angle are a bit more complicated to explain
  since unlike most arc function aren't expressed in radians
  or degrees. The angles should be converted into degrees and
  multiplied by 64 (which is done by the Draw() method itself).
  So pass start_angle and end_angle in degrees with angle 0 at
  12 o'clock and counting clockwise.
 */
class
DrawingArc: public DrawingShape
{
 private:
  int start_angle, end_angle;
 public:
    DrawingArc(VDKDrawingArea* owner,
	      int x, int y, int w, int h,
	      int start_angle, int end_angle,
	      VDKRgb color,
	      bool filled = false): 
    DrawingShape(owner,x,y,w,h,color,filled),
      start_angle(start_angle),end_angle(end_angle) {}
    virtual ~DrawingArc() {}
    virtual void Draw()
      {
	drawing->Pen->Color = color;
	drawing->Pen->Style = GDK_LINE_SOLID;
	drawing->Pen->Fill = GDK_SOLID;
	drawing->DrawArc(filled,
			   bound.Origin().x,
			   bound.Origin().y,
			   bound.w,bound.h,
			   start_angle*64,end_angle*64);
      }
};

class 
DrawingCircle: public DrawingArc
{
 public:
  DrawingCircle(VDKDrawingArea* owner, 
	       int x,int y, 
	       int radius, 
	       VDKRgb color,
	       bool filled = false):
    DrawingArc(owner,x-radius,y-radius,radius*2,radius*2,
		0,360*64,color,filled) {}
  ~DrawingCircle() {}
    
};

class 
DrawingPixmap: public DrawingShape
{
 private:
  char pixfile[256];
  VDKPixbuf* pixbuf;
 public:
  DrawingPixmap(VDKDrawingArea* owner, int x, int y, char* pixfile):
    DrawingShape(owner,x,y,0,0,VDKRgb(-1,-1,-1))
    { 
      pixbuf = new VDKPixbuf(owner,pixfile); 
      VDKRect local(bound.left,bound.top, pixbuf->Width(), pixbuf->Height());
      bound = local;
    }
  virtual ~DrawingPixmap(){ pixbuf->Destroy(); }
  void Draw() 
    { 
      drawing->DrawPixbuf(bound.Origin().x,bound.Origin().y,pixbuf);
    }
};

typedef VDKArray<VDKPoint> PointArray;
class 
DrawingPolygon: public DrawingShape
{
 private:
  GdkPoint *points;
  int npoints;
 public:
  DrawingPolygon(VDKDrawingArea *owner,
		PointArray& array, 
		VDKRgb color,
		bool filled = false);
  virtual ~DrawingPolygon() { delete[] points; }
  void Draw()
    {
      drawing->Pen->Color = color;
      drawing->Pen->Style = GDK_LINE_SOLID;
      drawing->Pen->Fill = GDK_SOLID;
      drawing->DrawPolygon(filled,points,npoints);
    }

};
#endif
