/*
* ===========================
* VDK Visual Development Kit
* Version 1.2.3
* October 1998, August 2000
* ===========================
*
* Copyright (C) 1998, Mario Motta
* Developed by Mario Motta <mmotta@guest.net>
*
* This library is free software; you can redistribute it and/or
* modify it under the terms of the GNU Library General Public
* License as published by the Free Software Foundation; either
* version 2 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Library General Public License for more details.
*
* You should have received a copy of the GNU Library General Public
* License along with this library; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
* 02111-1307, USA.
*
*/
#include "vdk/vdkustring.h"
#include <glib.h>
#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <cstdarg>

#if defined _WIN32 || defined _WIN64
#include <windows.h>
#endif

extern "C"
{
#include <ctype.h>
}

VDKUString::VDKUString()
{
	p = new USTRING;
	p->s = 0;
	p->ref = 1;
}

VDKUString::VDKUString(const char *s)
{
	p = new USTRING;
	if (s) {
		p->s = new char[std::strlen(s)+1];
		std::strcpy(p->s, s);
	}
	else p->s = 0;
	p->ref = 1; 
}

VDKUString::VDKUString (const char& c)
{
	p = new USTRING;
	if (c != '\0') {
		p->s = new char[2];
		(p->s)[0] = c;
		(p->s)[1] = '\0';
	}
	else p->s = 0;
	p->ref = 1;
}

VDKUString::VDKUString(const VDKUString& s)
{
	s.p->ref++;
	p = s.p;
}

VDKUString::~VDKUString()
{
	if (--p->ref == 0) {
		if (p->s != 0) delete[] p->s;
		delete p;  
	}
}

VDKUString& 
VDKUString::operator = (const VDKUString& s) 
{
	if (this == &s) return *this;
	else s.p->ref++;
	if (--p->ref == 0) { 
		if (p->s != 0) delete[] p->s;
		delete p; 
	} 
	p = s.p; 
	return *this;
}

VDKUString& 
VDKUString::operator = (const char* s)
{
	if (p->ref > 1) {
		p->ref--;
		p = new USTRING;
	}
	else if (p->ref == 1 && p->s != 0) delete[] p->s;
	if (s) {
		p->s = new char[std::strlen(s)+1];
		std::strcpy(p->s,s);
	}
	else p->s = 0;
	p->ref = 1;
	return *this;
}

VDKUString  
VDKUString::operator + (const char* s)  const
{
	VDKUString temp(*this);
	temp += s;
	return temp; 
}

VDKUString 
VDKUString::operator + (const VDKUString& s) const
{ 
	return *this + s.c_str();	
}

VDKUString
operator + (const char* s, const VDKUString& vdks)
{
	VDKUString temp(s);
	temp += vdks;
	return temp;
}

VDKUString& 
VDKUString::operator += (const char* s) 
{
	if (isNull()) *this = VDKUString(s);
	else if (s) {
			char* local = new char[std::strlen(p->s) + std::strlen(s) + 1];
			if (!local) return *this;
			std::strcat(std::strcpy(local, p->s), s);
			*this = VDKUString(local);
			delete[] local;
	}
	return *this;
}

VDKUString& 
VDKUString::operator += (const VDKUString& s)
{
	*this += s.c_str();
	return *this;		
}

int 
VDKUString::operator == (const VDKUString& s) const 
{ 
	if (p->s == s.p->s) return true; // can be 0 either
	else if ((p->s == 0) || (s.p->s == 0)) return false;
	else return !std::strcmp(p->s, s.p->s); 
}

int 
VDKUString::operator < (const VDKUString& s) const 
{
  if ((p->s == 0) || (s.p->s == 0)) return false;
  else return std::strcmp(p->s, s.p->s) < 0; 
}

int 
VDKUString::operator > (const VDKUString& s) const 
{
  if ((p->s == 0) || (s.p->s == 0)) return false;
  else return std::strcmp(p->s, s.p->s) > 0; 
}

int 
VDKUString::operator <= (const VDKUString& s) const 
{ 
  return (*this < s || *this == s); 
}

int 
VDKUString::operator >= (const VDKUString& s) const 
{ 
  return (*this > s || *this == s); 
}

int 
VDKUString::operator != (const VDKUString& s) const 
{ 
  return !(*this == s);	 
}

char
VDKUString::operator [] (unsigned int ix) const
{
  if ((p->s == 0) || (ix > std::strlen(p->s))) return '\0';
  else return p->s[ix];
}

const char*
VDKUString::c_str() const
{
	return p->s;
}

bool 
VDKUString::isNull() const
{ 
	return (p->s == 0); 
}

int 
VDKUString::size() const
{
	if (!p->s) return 0; 
	return std::strlen(p->s);
}

bool
VDKUString::isUTF8Valid() const
{
	return g_utf8_validate(p->s, -1, NULL);
}

unsigned int
VDKUString::Len() const
{
	if (isEmpty()) return 0;
	if (isUTF8Valid()) return g_utf8_strlen(p->s, -1);
	return std::strlen(p->s);
}

char*
VDKUString::GetChar(unsigned int ix) const
{
	gunichar c;
	char *buf;
	unsigned int len;
	
	if (!p->s) return NULL;
	buf = new char[10];
	if (!buf) return NULL;
	bool utf8valid = isUTF8Valid();
	if (utf8valid ? ix > (unsigned int)g_utf8_strlen(p->s, -1) : ix > std::strlen(p->s)) return NULL;
	if (utf8valid) {
		c = g_utf8_get_char(g_utf8_offset_to_pointer((const char*)p->s, ix));
		len = g_unichar_to_utf8(c, buf);
		buf[len] = 0;
		return buf;
	}
	buf[0] = p->s[ix];
	buf[1] = 0;
	return buf;
	//return p->s + ix;
}

VDKUString&
VDKUString::DelSelection(unsigned int begin, unsigned int len)
{
	unsigned int L;
	
	if (isEmpty() || (len == 0)) return *this;
	bool utf8valid = isUTF8Valid();
	if (utf8valid) L = Len();
	else L = size();
	if (begin > L) return *this;
	VDKUString temp = p->s;
	temp.Cut(begin);
	if (begin + len < L) {
		if (utf8valid) {
			temp += (g_utf8_offset_to_pointer((const char*)p->s, begin + len));
		}
		else {
			temp += (p->s + begin + len);
		}
	}
	*this = temp;
	return *this;
}

VDKUString&
VDKUString::RTrim()
{
	if (isEmpty()) return *this;
	unsigned int iPos = (unsigned int)size();
	unsigned int iCar = *(p->s + iPos - 1);
	while ((iCar == ' ') || (iCar == '\t')) {
		iPos--;
		iCar = *(p->s + iPos);
	}
	VDKUString temp = p->s;
	*this = temp.Cut(iPos + 1);
	return *this;
}

VDKUString&
VDKUString::LTrim()
{
	if (isEmpty()) return *this;
	unsigned int iPos = 0;
	unsigned int iCar = *(p->s);
	while ((iCar == ' ') || (iCar == '\t')) {
		iPos++;
		iCar = *(p->s + iPos);
	}
	*this = DelSelection(0, iPos);
	return *this;
}

VDKUString&
VDKUString::Trim()
{
	if (isEmpty()) return *this;
	RTrim();
	LTrim();
	return *this;
}

unsigned int
VDKUString::CharCount(const char car) const
{
	if (isEmpty()) return 0;
	unsigned int i = 0, nbcar = 0;
	char Car = p->s[i];
	while (Car != 0) {
		if (Car == car) nbcar++;
		i++;
		Car = p->s[i];
	}
	return nbcar;
}

unsigned int
VDKUString::CharCount(const char *str) const
{
	if (isEmpty() || !str || (str[0] == 0)) return 0;
	unsigned int nbcar = 0, l = std::strlen(str);
	char *Car = p->s;
	Car = strstr(p->s, str);
	while (Car) {
		nbcar++;
		Car = strstr(Car + l, str);
	}
	return nbcar;
}

VDKUString&
VDKUString::oldUpperCase()
{
	if (isEmpty()) return *this;
	unsigned int L = (unsigned int)size();
	char *szTemp = new char[L + 1];
	// Memory error
	if (szTemp == 0) return *this;
	for (unsigned int i = 0; i < L; i++) {
		szTemp[i] = (char)toupper(p->s[i]);
	}
	szTemp[L] = 0;
	*this = szTemp;
	delete [] szTemp;
	return *this;
}

// It is supposed here that upper case letters
// never have accents
VDKUString&
VDKUString::oldLowerCase()
{
	if (isEmpty()) return *this;
	unsigned int L = (unsigned int)size();
	char *szTemp = new char[L + 1];
	// Memory error
	if (szTemp == 0) return *this;
	for (unsigned int i = 0; i < L; i++) {
		szTemp[i] = (char)tolower(p->s[i]);
	}
	szTemp[L] = 0;
	*this = szTemp;
	delete [] szTemp;
	return *this;
}

VDKUString&
VDKUString::UpperCase()
{
	VDKUString tmp;
	char *car, *buf;
	gunichar c;
	unsigned int i = 0, len, olen = Len();
	
	if (isEmpty()) return *this;
	if (isUTF8Valid()) {
		buf = new char[10];
		if (!buf) return *this;
		car = g_utf8_offset_to_pointer(p->s, i);
		for (i = 1; i <= olen; i++) {
			c = g_utf8_get_char(car);
			c = g_unichar_toupper(c);
			len = g_unichar_to_utf8(c, buf);
			buf[len] = 0;
			tmp += buf;
			car = g_utf8_offset_to_pointer(p->s, i);
		}
		delete[] buf;
		*this = tmp;
	}
	else oldUpperCase();
	return *this;
}

VDKUString&
VDKUString::LowerCase()
{
	VDKUString tmp;
	char *car, *buf;
	gunichar c;
	int i = 0, len, olen = Len();
	
	if (isEmpty()) return *this;
	if (isUTF8Valid()) {
		buf = new char[10];
		if (!buf) return *this;
		car = g_utf8_offset_to_pointer(p->s, i);
		for (i = 1; i <= olen; i++) {
			c = g_utf8_get_char(car);
			c = g_unichar_tolower(c);
			len = g_unichar_to_utf8(c, buf);
			buf[len] = 0;
			tmp += buf;
			car = g_utf8_offset_to_pointer(p->s, i);
		}
		delete[] buf;
		*this = tmp;
	}
	else oldLowerCase();
	return *this;
}

bool
VDKUString::isEmpty() const
{
	// This method is modified to win some time
	if (!p->s) return true;
	if (p->s[0] == 0) return true;
	return false;
	// old code using strlen
	//return (size() == 0); 
}

VDKUString&
VDKUString::Concatf(const char* format, ...)
{
	va_list argptr;
	int cnt;

	// buffer is already full up, give up
	if (size() > MAXPRINTFLEN - 1) return *this;
	// alloc a large string
	char* szBuffer = new char[MAXPRINTFLEN];
	// Memory error, nothing is done yet
	if (szBuffer == 0) return *this;
	// constructing the new string
	va_start(argptr, format);
	cnt = g_vsnprintf(szBuffer, MAXPRINTFLEN, format, argptr);
	va_end(argptr);
	if (cnt == EOF) {
		// EOF encountered, returns backward
		delete[] szBuffer;
		return *this;
	}
	*this = (*this += szBuffer);	
	delete[] szBuffer;
	return *this;
}

VDKUString&
VDKUString::Sprintf(const char* format, ...)
{
	va_list argptr;
	int cnt;

	// buffer is already full up, give up
	if (size() > MAXPRINTFLEN - 1) return *this;
	// alloc a large string
	char* szBuffer = new char[MAXPRINTFLEN];
	// Memory error, nothing is done yet
	if (szBuffer == 0) return *this;
	// constructing the new string
	va_start(argptr, format);
	cnt = g_vsnprintf(szBuffer, MAXPRINTFLEN, format, argptr);
	va_end(argptr);
	if (cnt == EOF) {
		// EOF encountered, returns backward
		delete[] szBuffer;
		return *this;
	}
	*this = szBuffer;
	delete[] szBuffer;
	return *this;
}

VDKUString&
VDKUString::GetPart(unsigned int i, const char sep)
{
	char szSep[2];
	
	szSep[0] = sep;
	szSep[1] = 0;
	return GetPart(i, szSep);
}

VDKUString&
VDKUString::GetPart(unsigned int i, const char *sep)
{
	unsigned int L, lensep, ccount;
	char *szSep, *szBuff;
	VDKUString temp;

	if (isEmpty()) return *this;
	ccount = CharCount(sep);
	if ((i == 1) && (ccount == 0)) return *this;
	if ((i == 0) || (i > ccount + 1)) {
		*this = temp;
		return *this;
	}
	lensep = std::strlen(sep);
	// Make a string copy first
	szBuff = new char[size() + 1];
	if (!szBuff) return *this;
	std::strcpy(szBuff, p->s);
	szSep = new char[lensep + 1];
	if (!szSep) return *this;
	std::strcpy(szSep, sep);
	unsigned int j = 1;
	char* szS = std::strstr(szBuff, szSep);
	if (!szS) return *this;	// no sep found
	char* szP = szBuff;
	while (j < i) {
		szP = szS + lensep;
		szS = std::strstr(szP, szSep);
		j++;
	}
	if (szS != 0) L = szS - szP;
	else L = (szBuff + size()) - szP;
	szP[L] = 0;
	*this = szP;
	delete[] szBuff;
	delete[] szSep;
	return *this;
}

int
VDKUString::GetFCharPos(const char car) const
{
	char szCar[2];
	
	szCar[0] = car;
	szCar[1] = 0;
	return GetFCharPos(szCar);
}

int
VDKUString::GetFCharPos(const char *car) const
{
	gunichar c;
	char *Car;
	
	if (isEmpty()) return -1;
	c = g_utf8_get_char_validated(car, -1);
	if (c < 0) return 0;
	Car = g_utf8_strchr((const char*)p->s, size(), c);
	if (!Car) return -1;
	return g_utf8_pointer_to_offset(p->s, (const char*)Car);
}

int
VDKUString::GetLCharPos(const char car) const
{
	char szCar[2];
	
	szCar[0] = car;
	szCar[1] = 0;
	return GetLCharPos(szCar);
}

int
VDKUString::GetLCharPos(const char *car) const
{
	gunichar c;
	char *Car;
	
	if (isEmpty()) return -1;
	c = g_utf8_get_char_validated(car, -1);
	if (c < 0) return 0;
	Car = g_utf8_strrchr((const char*)p->s, size(), c);
	if (!Car) return -1;
	return g_utf8_pointer_to_offset(p->s, (const char*)Car);
}

double
VDKUString::StrtoDouble() const
{
	VDKUString tmp;
	char *decsep, *thousandssep, *grouping;

	if (isEmpty()) return 0;
	tmp = p->s;
	if (!tmp.get_i18n_sysparams(decsep, thousandssep, grouping)) return 0;
	// Delete thousand separator
	// It seems that atof or strtod support internationalized decimal separator
	// but not thousand separator - V 2.5.0, 2009/12/29
	tmp.StripChar(thousandssep);
	
#if defined _WIN32 || defined _WIN64
	if (decsep) delete[] decsep;
	if (thousandssep) delete[] thousandssep;
	if (grouping) delete[] grouping;
#endif

	return atof(tmp.c_str());
}

int
VDKUString::StrtoInt() const
{
	if (isEmpty()) return 0;
	else return std::atoi(p->s);
}

VDKUString&
VDKUString::SubStr(unsigned int start, unsigned int len)
{
	unsigned int L = Len();
	if (isEmpty() || (start > L)) return *this;
	char *szTemp = new char[size() + 1];
	if (!szTemp) return *this;
	std::strcpy(szTemp, p->s);
	char *s = g_utf8_offset_to_pointer(szTemp, start);
	if (!s) return *this;
	if (start + len <= L) {
		char *e = g_utf8_offset_to_pointer(s, len);
		if (!e) return *this;
		e[0] = 0;
	}
	*this = s;
	return *this;
}

VDKUString&
VDKUString::Cut(unsigned int len)
{
	if (isEmpty() || (len >= Len())) return *this;
	VDKUString temp = p->s;
	if (isUTF8Valid()) *(g_utf8_offset_to_pointer(temp.p->s, len)) = 0;
	else *(temp.p->s + len) = 0;
	*this = temp.p->s;
	return *this;
}

VDKUString&
VDKUString::LPad(unsigned int len, const char car)
{
	if (isEmpty()) return *this;
	unsigned int L = (unsigned int)size();
	int s = len - L; // s = number of missing chars
	if (s > 0) {
		char *local = new char[s + 1];
		if (!local) return *this;
		std::memset(local, car, s);
		local[s] = 0;
		VDKUString temp = local;
		*this = (temp += *this);
		delete[] local;
	}
	return *this;
}

VDKUString&
VDKUString::RPad(unsigned int len, const char car)
{
	if (isEmpty()) return *this;
	unsigned int L = (unsigned int)size();
	int s = len - L; // s = number of missing chars
	if (s > 0) {
		char *local = new char[s + 1];
		if (!local) return *this;
		std::memset(local, car, s);
		local[s] = 0;
		VDKUString temp = local;
		*this += temp;
		delete[] local;
	}
	return *this;
}

VDKUString&
VDKUString::DoubleChar(const char car)
{
	char szCar[2];
	
	szCar[0] = car;
	szCar[1] = 0;
	return DoubleChar(szCar);
}

VDKUString&
VDKUString::DoubleChar(const char *car)
{
	VDKUString torep, rep;
	
	if (isEmpty() || !car || (car[0] == 0)) return *this;
	torep = car;
	rep = car;
	rep += car;
	return ReplaceChar(torep.c_str(), rep.c_str());
}

VDKUString&
VDKUString::InsertChar(const char car, unsigned int pos)
{
	char szCar[2];
	
	szCar[0] = car;
	szCar[1] = 0;
	return InsertChar(szCar, pos);
}

VDKUString&
VDKUString::InsertChar(const char *car, unsigned int pos)
{
	VDKUString rslt, begpart, endpart;

	if (isEmpty() || (pos >= Len())) return *this;
	begpart = *this;
	begpart.Cut(pos);
	endpart = *this;
	endpart.SubStr(pos, Len());
	rslt = begpart + car + endpart;
	*this = rslt;
	return *this;
}

VDKUString&
VDKUString::StripChar(const char car)
{
	char szCar[2];
	
	szCar[0] = car;
	szCar[1] = 0;
	return StripChar(szCar);
}

VDKUString&
VDKUString::StripChar(const char *car)
{
	VDKUString temp, rslt;
	unsigned int i, ncar;

	if (isEmpty() || !car || (car[0] == 0)) return *this;
	ncar = CharCount(car);
	if (ncar == 0) return *this;
	for (i = 1; i <= ncar + 1; i++) {
		temp = p->s;
		temp.GetPart(i, car);
		rslt += temp;
	}
	*this = rslt;
	return *this;
}

VDKUString&
VDKUString::ReplaceChar(const char torep, const char rep)
{
	char szTorep[2], szRep[2];
	
	szTorep[0] = torep;
	szTorep[1] = 0;
	szRep[0] = rep;
	szRep[1] = 0;
	return ReplaceChar(szTorep, szRep);
}

VDKUString&
VDKUString::ReplaceChar(const char *torep, const char *rep)
{
	VDKUString temp, rslt;
	unsigned int i, ncar;

	if (isEmpty() || !torep) return *this;
	ncar = CharCount(torep);
	if (ncar == 0) return *this;
	for (i = 1; i <= ncar + 1; i++) {
		temp = p->s;
		temp.GetPart(i, torep);
		if (i < ncar + 1) temp += rep;
		rslt += temp;
	}
	*this = rslt;
	return *this;
}

VDKUString&
VDKUString::FormatDate(const char sep, int orig, int ret)
{
	char day[3], month[3], year[5], rslt[11];
	unsigned int doffset, moffset, yoffset;

	if (isEmpty() || (size() < 8)) return *this;
	bool HasSep = (!isdigit(p->s[2]) || !isdigit(p->s[4]));
	std::memset(day, 0, 3);
	std::memset(month, 0, 3);
	std::memset(year, 0, 5);
	switch(orig) {
		case ENG_DATE :
			if (HasSep) {
				doffset = 3; moffset = 0; yoffset = 6;
			}
			else {
				doffset = 2; moffset = 0; yoffset = 4;
			}
			break;
		case EUR_DATE :
			if (HasSep) {
				doffset = 0; moffset = 3; yoffset = 6;
			}
			else {
				doffset = 0; moffset = 2; yoffset = 4;
			}
			break;
		default : // defaults to INT_DATE
			if (HasSep) {
				doffset = 8; moffset = 5; yoffset = 0;
			}
			else {
				doffset = 6; moffset = 4; yoffset = 0;
			}
	}
	std::memcpy(day, p->s + doffset, 2);
	std::memcpy(month, p->s + moffset, 2);
	std::memcpy(year, p->s + yoffset, 4);
	switch(ret) {
		case ENG_DATE :
			if (sep != 0)
				sprintf(rslt, "%s%c%s%c%s", month, sep, day, sep, year);
			else
				sprintf(rslt, "%s%s%s", month, day, year);
			break;
		case EUR_DATE :
			if (sep != 0)
				sprintf(rslt, "%s%c%s%c%s", day, sep, month, sep, year);
			else
				sprintf(rslt, "%s%s%s", day, month, year);
			break;
		default : // defaults to INT_DATE
			if (sep != 0)
				sprintf(rslt, "%s%c%s%c%s", year, sep, month, sep, day);
			else
				sprintf(rslt, "%s%s%s", year, month, day);
	}
	*this = rslt;
	return *this;
}

VDKUString&
VDKUString::FloattoUserFormat()
{
	VDKUString rslt, intpart, decpart;
	char *decsep, *thousandssep, *grouping;
	unsigned int L, thousandslen = 3;
	int pos;
	
	if (isEmpty()) return *this;
	if (!get_i18n_sysparams(decsep, thousandssep, grouping)) return *this;

#if defined _WIN32 || defined _WIN64

	char thlen[2];
	
	thlen[0] = grouping[0];
	thlen[1] = 0;
	thousandslen = atoi(thlen);

#else

	thousandslen = grouping[0];

#endif
	// Detects if decimal separator is correct
	bool decsepok = (CharCount(decsep) > 0);
	// Detects if thousand separator is correct
	bool thousandsepok = (CharCount(thousandssep) > 0);
	// Nothing to do
	if (decsepok && thousandsepok) {
	
#if defined _WIN32 || defined _WIN64
		if (decsep) delete[] decsep;
		if (thousandssep) delete[] thousandssep;
		if (grouping) delete[] grouping;
#endif

		return *this;
	}
	// Thousand separator insertion
	intpart = p->s;
	intpart.GetPart(1, decsepok ? decsep : ".");
	L = intpart.Len();
	if (!thousandsepok) {
		pos = L - thousandslen;
		while (pos > 0) {
			intpart.InsertChar(thousandssep, pos);
			pos -= thousandslen;
		}
	}
	decpart = p->s;
	decpart.GetPart(2, decsepok ? decsep : ".");
	rslt = intpart + decsep + decpart;

#if defined _WIN32 || defined _WIN64
	if (decsep) delete[] decsep;
	if (thousandssep) delete[] thousandssep;
	if (grouping) delete[] grouping;
#endif

	*this = rslt;
	return *this;
}

VDKUString&
VDKUString::FloattoCFormat()
{
	char *decsep, *thousandssep, *grouping;

	if (isEmpty()) return *this;
	if (!get_i18n_sysparams(decsep, thousandssep, grouping)) return *this;
	// Delete thousand separator
	if (thousandssep) StripChar(thousandssep);
	// Replace decimal separator
	if (decsep) ReplaceChar(decsep, ".");
	
#if defined _WIN32 || defined _WIN64
	if (decsep) delete[] decsep;
	if (thousandssep) delete[] thousandssep;
	if (grouping) delete[] grouping;
#endif

	return *this;
}

bool
VDKUString::get_i18n_sysparams(char *&decsep, char *&thousandssep, char *&grouping)
{
	decsep = NULL;
	thousandssep = NULL;
	grouping = NULL;

#if defined _WIN32 || defined _WIN64

	int size = 0;
	char *tmpthsep = NULL, *tmpdecsep = NULL, *tmpgroup = NULL;
	
	// Decimal separator
	// Get buffer size first
	size = GetLocaleInfo(LOCALE_USER_DEFAULT, LOCALE_SDECIMAL, NULL, 0);
	if (size <= 0) goto failed;
	tmpdecsep = new char[size];
	if (!tmpdecsep) goto failed;
	// Get real data
	if (GetLocaleInfo(LOCALE_USER_DEFAULT, LOCALE_SDECIMAL, tmpdecsep, size) <= 0) goto failed;
	// Translate to UTF8
	decsep = (char*)g_locale_to_utf8(tmpdecsep, -1, NULL, NULL, NULL);
	delete[] tmpdecsep;
	// Thousand separator
	size = GetLocaleInfo(LOCALE_USER_DEFAULT, LOCALE_STHOUSAND, NULL, 0);
	if (size <= 0) goto failed;
	tmpthsep = new char[size];
	if (!tmpthsep) goto failed;
	if (GetLocaleInfo(LOCALE_USER_DEFAULT, LOCALE_STHOUSAND, tmpthsep, size) <= 0) goto failed;
	// Translate to UTF8
	thousandssep = (char*)g_locale_to_utf8(tmpthsep, -1, NULL, NULL, NULL);
	delete[] tmpthsep;
	// Thousand grouping
	size = GetLocaleInfo(LOCALE_USER_DEFAULT, LOCALE_SGROUPING, NULL, 0);
	if (size <= 0) goto failed;
	tmpgroup = new char[size];
	if (!tmpgroup) goto failed;
	if (GetLocaleInfo(LOCALE_USER_DEFAULT, LOCALE_SGROUPING, tmpgroup, size) <= 0) goto failed;
	// Translate to UTF8
	grouping = (char*)g_locale_to_utf8(tmpgroup, -1, NULL, NULL, NULL);
	delete[] tmpgroup;
	return true;

#else

	struct lconv *lc = NULL;
	
	lc = localeconv();
	if (!lc) return false;
	decsep = (char*)g_locale_to_utf8(lc->decimal_point, -1, NULL, NULL, NULL);
	thousandssep = (char*)g_locale_to_utf8(lc->thousands_sep, -1, NULL, NULL, NULL);
	grouping = (char*)g_locale_to_utf8(lc->grouping, -1, NULL, NULL, NULL);

#endif

#if defined _WIN32 || defined _WIN64
failed:
	if (tmpdecsep) delete[] tmpdecsep;
	if (decsep) delete[] decsep;
	if (tmpthsep) delete[] tmpthsep;
	if (thousandssep) delete[] thousandssep;
	if (tmpgroup) delete[] tmpgroup;
	if (grouping) delete[] grouping;
	return false;
#endif

	return true;
}

