/* GtkDatabox - An extension to the gtk+ library
 * Copyright (C) 1998 - 2002  Dr. Roland Bock
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
/* gtkdatabox.c */

#include <string.h>
#include "gtk/gtksignal.h"
#include "gtk/gtktable.h"
#include "gtk/gtktogglebutton.h"
#include "gtk/gtkdrawingarea.h"
#include "gtk/gtkhscrollbar.h"
#include "gtk/gtkvscrollbar.h"
#include "gtk/gtkruler.h"
#include "gtk/gtkhruler.h"
#include "gtk/gtkvruler.h"
#include <vdk/gtkdatabox.h>
#include <vdk/gtkdataboxmarshal.h>

#define CROSS_BORDER 10

enum
{
   GTK_DATABOX_SHOW_RULERS = 0,
   GTK_DATABOX_SHOW_SCROLLBARS,
   GTK_DATABOX_ENABLE_SELECTION,
   GTK_DATABOX_SHOW_SELECTION_FILLED,
   GTK_DATABOX_ENABLE_ZOOM,
   GTK_DATABOX_REDRAW_REQUEST,
   GTK_DATABOX_SELECTION_STOPPED,
};

enum
{
   GTK_DATABOX_DATA_HAS_GC = 0,
};

typedef struct _GtkDataboxData GtkDataboxData;

typedef void (*GtkDataboxDrawFunc) (GtkDatabox * box, GtkDataboxData * data);
#define GTK_DATABOX_DRAW_FUNC(f)           ((GtkDataboxDrawFunc) (f))

static GtkObjectClass *parent_class = NULL;

struct _GtkDataboxData
{
   gfloat *X;			/* X (horizontal) values */
   gfloat *Y;			/* Y (vertical) values */
   guint length;		/* Number of data points */

   GtkDataboxDataType type;	/* How this data set is to be displayed */
   GtkDataboxDrawFunc draw;	/* Function that draws the data */

   GdkColor color;		/* Dot-color */
   guint size;			/* Dot-size */
   GdkGC *gc;			/* Dot-gc */

   glong flags;			/* ..HAS_GC, etc. */

   guint hlines;		/* Only used for grid */
   guint vlines;		/* Only used for grid */
};

static void gtk_databox_class_init (GtkDataboxClass * klass);
static void gtk_databox_init (GtkDatabox * box);
static gint gtk_databox_destroy_callback (GtkWidget * widget,
					  GtkDatabox * box);
static gint gtk_databox_expose_callback (GtkWidget * widget,
					 GdkEventExpose * event,
					 GtkDatabox * box);
static gint gtk_databox_configure_callback (GtkWidget * widget,
					    GdkEventConfigure * event,
					    GtkDatabox * box);
static void gtk_databox_zoom_to_selection (GtkWidget * widget,
					   GtkDatabox * box);
static void gtk_databox_zoom_out (GtkWidget * widget, GtkDatabox * box);
static void gtk_databox_zoom_home (GtkWidget * widget, GtkDatabox * box);
static void gtk_databox_zoomed (GtkWidget * widget, GtkDatabox * box,
				gboolean redraw_flag);
static void gtk_databox_x_adjustment_callback (GtkWidget * widget,
					       GtkDatabox * box);
static void gtk_databox_y_adjustment_callback (GtkWidget * widget,
					       GtkDatabox * box);
static gint gtk_databox_button_press_callback (GtkWidget * widget,
					       GdkEventButton * event,
					       GtkDatabox * box);
static gint gtk_databox_button_release_callback (GtkWidget * widget,
						 GdkEventButton * event,
						 GtkDatabox * box);
static gint gtk_databox_motion_notify_callback (GtkWidget * widget,
						GdkEventMotion * event,
						GtkDatabox * box);
static void gtk_databox_draw_request_full (GtkWidget * widget, gboolean now,
					   GtkDatabox * box);
static gint gtk_databox_draw_selection (GtkWidget * widget, GtkDatabox * box,
					GdkRectangle * rect);
static void gtk_databox_draw (GtkWidget * widget, GtkDatabox * box,
			      GdkEventExpose * event);
static void gtk_databox_draw_points (GtkDatabox * box, GtkDataboxData * data);
static void gtk_databox_draw_lines (GtkDatabox * box, GtkDataboxData * data);
static void gtk_databox_draw_bars (GtkDatabox * box, GtkDataboxData * data);
static void gtk_databox_draw_cross_simple (GtkDatabox * box,
					   GtkDataboxData * data);
static void gtk_databox_draw_grid (GtkDatabox * box, GtkDataboxData * data);
static void gtk_databox_new_data_gc (GtkWidget * widget, GtkDatabox * box,
				     GtkDataboxData * data);
static void gtk_databox_update_x_ruler (GtkDatabox * box);
static void gtk_databox_update_y_ruler (GtkDatabox * box);
static void gtk_databox_data_calc_extrema (GtkDatabox * box,
					   GtkDataboxValue * min,
					   GtkDataboxValue * max);
static gint gtk_databox_check_x_links (GList * list, gfloat * values);
static gint gtk_databox_check_y_links (GList * list, gfloat * values);
static void gtk_databox_destroy_data (GtkDatabox * box, GtkDataboxData * data,
				      GList * list, gboolean free_flag);
static gint gtk_databox_data_destroy_with_flag (GtkDatabox * box, gint index,
						gboolean free_flag);
static gint gtk_databox_data_destroy_all_with_flag (GtkDatabox * box,
						    gboolean free_flag);
enum
{
   GTK_DATABOX_ZOOMED_SIGNAL,
   GTK_DATABOX_MARKED_SIGNAL,
   GTK_DATABOX_SELECTION_STARTED_SIGNAL,
   GTK_DATABOX_SELECTION_CHANGED_SIGNAL,
   GTK_DATABOX_SELECTION_STOPPED_SIGNAL,
   GTK_DATABOX_SELECTION_CANCELLED_SIGNAL,
   LAST_SIGNAL
};

static gint gtk_databox_signals[LAST_SIGNAL] = { 0 };

/* FIXME: We should take a closer look at some other widgets for better
   understanding... */
guint
gtk_databox_get_type ()
{
   static GType databox_type = 0;

   if (!databox_type)
   {
      static const GTypeInfo databox_info = 
      {
	 sizeof (GtkDataboxClass),
	 NULL,                                     /* base_init */
	 NULL,                                     /* base_finalize */
	 (GClassInitFunc) gtk_databox_class_init,
	 NULL,                                     /* class_finalize */
	 NULL,                                     /* class_data */
	 sizeof (GtkDatabox),
	 0,                                        /* no pre-allocation */
         (GInstanceInitFunc) gtk_databox_init,
      };

      databox_type = g_type_register_static (GTK_TYPE_VBOX,
                                             "GtkDatabox", 
					     &databox_info, 
					     0);
   }

   return databox_type;
}

/* FIXME: We should take a closer look at some other widgets for better
   understanding... */
static void
gtk_databox_class_init (GtkDataboxClass * klass)
{
   GtkObjectClass *object_class = (GtkObjectClass *) klass;
/*   GtkWidgetClass *widget_class = (GtkWidgetClass *) klass;*/

   parent_class = g_type_class_peek_parent (klass);

/* FIXME Many Gtk widgets use something like the following */
/*       Should that be used here as well? */
/*   
   object_class->destroy               = gtk_databox_destroy;
   widget_class->configure_event       = gtk_databox_configure;
   widget_class->expose_event          = gtk_databox_expose;
   widget_class->button_press_event    = gtk_databox_button_press;
   widget_class->button_release_event  = gtk_databox_button_release;
   widget_class->motion_notify_event   = gtk_databox_motion_notify;
   */

   gtk_databox_signals[GTK_DATABOX_ZOOMED_SIGNAL] =
      g_signal_new ("gtk_databox_zoomed", 
		      G_TYPE_FROM_CLASS (object_class),
                      G_SIGNAL_RUN_FIRST,
		      G_STRUCT_OFFSET (GtkDataboxClass, gtk_databox_zoomed),
		      NULL,  /* accumulator */
		      NULL,  /* accumulator_data */
		      gtk_databox_marshal_VOID__POINTER_POINTER, 
		      G_TYPE_NONE, 
		      2, G_TYPE_POINTER, G_TYPE_POINTER);
   gtk_databox_signals[GTK_DATABOX_MARKED_SIGNAL] =
      g_signal_new ("gtk_databox_marked", 
		      G_TYPE_FROM_CLASS (object_class),
                      G_SIGNAL_RUN_FIRST,
		      G_STRUCT_OFFSET (GtkDataboxClass, gtk_databox_marked),
		      NULL,  /* accumulator */
		      NULL,  /* accumulator_data */
		      gtk_databox_marshal_VOID__POINTER, 
		      G_TYPE_NONE, 
		      1, G_TYPE_POINTER);
   gtk_databox_signals[GTK_DATABOX_SELECTION_STARTED_SIGNAL] =
      g_signal_new ("gtk_databox_selection_started", 
		      G_TYPE_FROM_CLASS (object_class),
                      G_SIGNAL_RUN_FIRST,
		      G_STRUCT_OFFSET (GtkDataboxClass, gtk_databox_selection_started),
		      NULL,  /* accumulator */
		      NULL,  /* accumulator_data */
		      gtk_databox_marshal_VOID__POINTER, 
		      G_TYPE_NONE, 
		      1, G_TYPE_POINTER);
   gtk_databox_signals[GTK_DATABOX_SELECTION_CHANGED_SIGNAL] =
      g_signal_new ("gtk_databox_selection_changed", 
		      G_TYPE_FROM_CLASS (object_class),
                      G_SIGNAL_RUN_FIRST,
		      G_STRUCT_OFFSET (GtkDataboxClass, gtk_databox_selection_changed),
		      NULL,  /* accumulator */
		      NULL,  /* accumulator_data */
		      gtk_databox_marshal_VOID__POINTER_POINTER, 
		      G_TYPE_NONE, 
		      2, G_TYPE_POINTER, G_TYPE_POINTER);
   gtk_databox_signals[GTK_DATABOX_SELECTION_STOPPED_SIGNAL] =
      g_signal_new ("gtk_databox_selection_stopped", 
		      G_TYPE_FROM_CLASS (object_class),
                      G_SIGNAL_RUN_FIRST,
		      G_STRUCT_OFFSET (GtkDataboxClass, gtk_databox_selection_stopped),
		      NULL,  /* accumulator */
		      NULL,  /* accumulator_data */
		      gtk_databox_marshal_VOID__POINTER_POINTER, 
		      G_TYPE_NONE, 
		      2, G_TYPE_POINTER, G_TYPE_POINTER);
   gtk_databox_signals[GTK_DATABOX_SELECTION_CANCELLED_SIGNAL] =
      g_signal_new ("gtk_databox_selection_cancelled", 
		      G_TYPE_FROM_CLASS (object_class),
                      G_SIGNAL_RUN_FIRST,
		      G_STRUCT_OFFSET (GtkDataboxClass, gtk_databox_selection_cancelled),
		      NULL,  /* accumulator */
		      NULL,  /* accumulator_data */
		      gtk_databox_marshal_VOID__VOID, 
		      G_TYPE_NONE, 
		      0);

   klass->gtk_databox = NULL;
   klass->gtk_databox_zoomed = NULL;
   klass->gtk_databox_marked = NULL;
   klass->gtk_databox_selection_started = NULL;
   klass->gtk_databox_selection_changed = NULL;
   klass->gtk_databox_selection_stopped = NULL;
   klass->gtk_databox_selection_cancelled = NULL;
}

static void
gtk_databox_init (GtkDatabox * box)
{
   GtkWidget *widget = NULL;

   box->table = gtk_table_new (3, 3, FALSE);
   gtk_container_add (GTK_CONTAINER (box), box->table);
   gtk_widget_show (box->table);

   widget = box->draw = gtk_drawing_area_new ();
   gtk_widget_set_events (widget,
			  GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
			  GDK_POINTER_MOTION_MASK |
			  GDK_POINTER_MOTION_HINT_MASK);
   g_signal_connect (GTK_OBJECT (widget), "destroy",
		       GTK_SIGNAL_FUNC (gtk_databox_destroy_callback), box);
   g_signal_connect (GTK_OBJECT (widget), "configure_event",
		       GTK_SIGNAL_FUNC (gtk_databox_configure_callback), box);
   g_signal_connect (GTK_OBJECT (widget), "expose_event",
		       GTK_SIGNAL_FUNC (gtk_databox_expose_callback), box);
   g_signal_connect (GTK_OBJECT (widget), "button_press_event",
		       GTK_SIGNAL_FUNC (gtk_databox_button_press_callback),
		       box);
   g_signal_connect (GTK_OBJECT (widget), "button_release_event",
		       GTK_SIGNAL_FUNC (gtk_databox_button_release_callback),
		       box);
   g_signal_connect (GTK_OBJECT (widget), "motion_notify_event",
		       GTK_SIGNAL_FUNC (gtk_databox_motion_notify_callback),
		       box);
   gtk_widget_set_size_request (widget, 20, 30);

   gtk_table_attach (GTK_TABLE (box->table), widget, 1, 2, 1, 2,
		     GTK_FILL | GTK_EXPAND | GTK_SHRINK,
		     GTK_FILL | GTK_EXPAND | GTK_SHRINK, 0, 0);
   gtk_widget_show (widget);

   box->adjX =
      GTK_ADJUSTMENT (gtk_adjustment_new (0.0, 0.0, 1.0, 0.1, 0.9, 1.0));
   box->adjY =
      GTK_ADJUSTMENT (gtk_adjustment_new (0.0, 0.0, 1.0, 0.1, 0.9, 1.0));
   g_object_ref (GTK_OBJECT (box->adjX));
   g_object_ref (GTK_OBJECT (box->adjY));

   g_signal_connect (GTK_OBJECT (box->adjY), "value_changed",
		       GTK_SIGNAL_FUNC (gtk_databox_y_adjustment_callback),
		       box);
   g_signal_connect (GTK_OBJECT (box->adjX), "value_changed",
		       GTK_SIGNAL_FUNC (gtk_databox_x_adjustment_callback),
		       box);

   box->flags = 0;
   gtk_databox_show_rulers (box);
   gtk_databox_show_scrollbars (box);
   gtk_databox_enable_zoom (box);
   gtk_databox_enable_selection (box);
   gtk_databox_hide_selection_filled (box);
   gtk_databox_set_zoom_limit (box, 0.01);

   box->pixmap = NULL;
   box->data = NULL;
   box->max_points = 0;
   box->points = NULL;
   box->select_gc = NULL;

   gtk_databox_rescale (box);
}

GtkWidget *
gtk_databox_new ()
{
   return gtk_widget_new (GTK_TYPE_DATABOX, NULL);
}

void
gtk_databox_show_rulers (GtkDatabox * box)
{
   g_return_if_fail (GTK_IS_DATABOX (box));

   if (!(box->flags & (1 << GTK_DATABOX_SHOW_RULERS)))
   {
      box->hrule = gtk_hruler_new ();
      gtk_ruler_set_metric (GTK_RULER (box->hrule), GTK_PIXELS);
      gtk_ruler_set_range (GTK_RULER (box->hrule), 1.5, -0.5, 0.5, 20);
      g_signal_connect_closure (box->draw, "motion_notify_event",
      		g_cclosure_new_object_swap (GTK_SIGNAL_FUNC(GTK_WIDGET_GET_CLASS(box->hrule)->motion_notify_event), G_OBJECT (box->hrule)),
                FALSE);
      box->vrule = gtk_vruler_new ();
      gtk_ruler_set_metric (GTK_RULER (box->vrule), GTK_PIXELS);
      gtk_ruler_set_range (GTK_RULER (box->vrule), 1.5, -0.5, 0.5, 20);
      g_signal_connect_closure (box->draw, "motion_notify_event",
      		g_cclosure_new_object_swap (GTK_SIGNAL_FUNC(GTK_WIDGET_GET_CLASS(box->vrule)->motion_notify_event), G_OBJECT (box->vrule)),
                FALSE);

      gtk_table_attach (GTK_TABLE (box->table), box->hrule, 1, 2, 0, 1,
			GTK_EXPAND | GTK_SHRINK | GTK_FILL, GTK_FILL, 0, 0);
      gtk_table_attach (GTK_TABLE (box->table), box->vrule, 0, 1, 1, 2,
			GTK_FILL, GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);

      gtk_widget_show (box->hrule);
      gtk_widget_show (box->vrule);
      box->flags |= 1 << GTK_DATABOX_SHOW_RULERS;
   }
}

void
gtk_databox_hide_rulers (GtkDatabox * box)
{
   g_return_if_fail (GTK_IS_DATABOX (box));

   if (box->flags & (1 << GTK_DATABOX_SHOW_RULERS))
   {
      gtk_widget_destroy (box->hrule);
      box->hrule = NULL;
      gtk_widget_destroy (box->vrule);
      box->vrule = NULL;
   }
   box->flags &= ~(1 << GTK_DATABOX_SHOW_RULERS);
}

void
gtk_databox_show_scrollbars (GtkDatabox * box)
{
   g_return_if_fail (GTK_IS_DATABOX (box));

   if (!(box->flags & (1 << GTK_DATABOX_SHOW_SCROLLBARS)))
   {
      box->hscroll = gtk_hscrollbar_new (box->adjX);
      box->vscroll = gtk_vscrollbar_new (box->adjY);
      gtk_table_attach (GTK_TABLE (box->table), box->hscroll, 1, 2, 2, 3,
			GTK_FILL | GTK_EXPAND | GTK_SHRINK, GTK_FILL, 0, 0);
      gtk_table_attach (GTK_TABLE (box->table), box->vscroll, 2, 3, 1, 2,
			GTK_FILL, GTK_FILL | GTK_EXPAND | GTK_SHRINK, 0, 0);

      gtk_widget_show (box->hscroll);
      gtk_widget_show (box->vscroll);
   }
   box->flags |= 1 << GTK_DATABOX_SHOW_SCROLLBARS;
}

void
gtk_databox_hide_scrollbars (GtkDatabox * box)
{
   g_return_if_fail (GTK_IS_DATABOX (box));

   if ((box->flags & (1 << GTK_DATABOX_SHOW_SCROLLBARS)))
   {
      gtk_widget_destroy (box->hscroll);
      gtk_widget_destroy (box->vscroll);
   }
   box->flags &= ~(1 << GTK_DATABOX_SHOW_SCROLLBARS);
}

void
gtk_databox_enable_selection (GtkDatabox * box)
{
   g_return_if_fail (GTK_IS_DATABOX (box));

   box->flags |= 1 << GTK_DATABOX_ENABLE_SELECTION;
}

void
gtk_databox_disable_selection (GtkDatabox * box)
{
   g_return_if_fail (GTK_IS_DATABOX (box));

   box->flags &= ~(1 << GTK_DATABOX_ENABLE_SELECTION);

   box->selection_flag = 0;
   g_signal_emit (GTK_OBJECT (box),
		    gtk_databox_signals
		    [GTK_DATABOX_SELECTION_CANCELLED_SIGNAL], 0);

}

void
gtk_databox_show_selection_filled (GtkDatabox * box)
{
   g_return_if_fail (GTK_IS_DATABOX (box));

   box->flags |= 1 << GTK_DATABOX_SHOW_SELECTION_FILLED;
}

void
gtk_databox_hide_selection_filled (GtkDatabox * box)
{
   g_return_if_fail (GTK_IS_DATABOX (box));

   box->flags &= ~(1 << GTK_DATABOX_SHOW_SELECTION_FILLED);
}

void
gtk_databox_enable_zoom (GtkDatabox * box)
{
   g_return_if_fail (GTK_IS_DATABOX (box));

   box->flags |= 1 << GTK_DATABOX_ENABLE_ZOOM;
}

void
gtk_databox_disable_zoom (GtkDatabox * box)
{
   g_return_if_fail (GTK_IS_DATABOX (box));

   box->flags &= ~(1 << GTK_DATABOX_ENABLE_ZOOM);
}

void
gtk_databox_set_zoom_limit (GtkDatabox * box, gfloat zoom_limit)
{
   g_return_if_fail (GTK_IS_DATABOX (box));

   box->zoom_limit = zoom_limit;
}


gfloat
gtk_databox_get_zoom_limit (GtkDatabox * box)
{
   g_return_val_if_fail (GTK_IS_DATABOX (box), 0);

   return box->zoom_limit;
}


static gint
gtk_databox_destroy_callback (GtkWidget * widget, GtkDatabox * box)
{
   if (box->pixmap)
      g_object_unref (box->pixmap);
   if (box->select_gc)
   {
      g_object_unref (box->select_gc);
   }
   g_object_unref (GTK_OBJECT (box->adjX));
   g_object_unref (GTK_OBJECT (box->adjY));
   
   return FALSE;
}

static gint
gtk_databox_configure_callback (GtkWidget * widget, GdkEventConfigure * event,
				GtkDatabox * box)
{
   gdk_drawable_get_size (widget->window, &(box->size.x), &(box->size.y));

   if (box->pixmap)
      g_object_unref (box->pixmap);

   box->pixmap =
      gdk_pixmap_new (widget->window, box->size.x, box->size.y, -1);

   gdk_draw_rectangle (box->pixmap, widget->style->bg_gc[0], TRUE, 0, 0,
		       box->size.x, box->size.y);

   if (box->selection_flag)
   {
      box->selection_flag = 0;
      g_signal_emit (GTK_OBJECT (box),
		       gtk_databox_signals
		       [GTK_DATABOX_SELECTION_CANCELLED_SIGNAL], 0);
   }
   gtk_databox_zoomed (widget, box, FALSE);

   return FALSE;
}

static gint
gtk_databox_expose_callback (GtkWidget * widget, GdkEventExpose * event,
			     GtkDatabox * box)
{
   gtk_databox_draw (box->draw, box, event);

   gdk_draw_drawable (widget->window,
		    widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
		    box->pixmap, event->area.x, event->area.y, event->area.x,
		    event->area.y, event->area.width, event->area.height);

   return FALSE;
}

static gint
gtk_databox_button_press_callback (GtkWidget * widget, GdkEventButton * event,
				   GtkDatabox * box)
{
   gint x;
   gint y;
   guint button;
   GdkRectangle rect;


   if (event->type != GDK_BUTTON_PRESS)
      return FALSE;

   box->flags &= ~(1 << GTK_DATABOX_SELECTION_STOPPED);
   
   button = event->button;
   x = event->x;
   y = event->y;

   if (box->selection_flag)
   {
      rect.x = MIN (box->marked.x, box->select.x);
      rect.y = MIN (box->marked.y, box->select.y);
      rect.width = MAX (box->marked.x, box->select.x) - rect.x + 1;
      rect.height = MAX (box->marked.x, box->select.x) - rect.y + 1;

      gtk_databox_draw_selection (box->draw, box, &rect);
   }

   if (button == 1 || button == 2)
   {
      if (box->selection_flag)
      {
	 box->selection_flag = 0;
	 if (rect.x < x && x < MAX (box->marked.x, box->select.x)
	     && rect.y < y && y < MAX (box->marked.y, box->select.y))
	 {
	    gtk_databox_zoom_to_selection (widget, box);
	 }
	 else
	 {
	    g_signal_emit (GTK_OBJECT (box),
			     gtk_databox_signals
			     [GTK_DATABOX_SELECTION_CANCELLED_SIGNAL], 0);
	 }
      }
   }
   else if (button == 3)
   {
      if (box->selection_flag) 
      {
         box->selection_flag = 0;
         g_signal_emit (GTK_OBJECT (box),
			     gtk_databox_signals
			     [GTK_DATABOX_SELECTION_CANCELLED_SIGNAL], 0);
        
      }
      if (event->state & GDK_SHIFT_MASK)
      {
	 gtk_databox_zoom_home (widget, box);
      }
      else
      {
	 gtk_databox_zoom_out (widget, box);
      }
   }
   box->marked.x = x;
   box->marked.y = y;
   g_signal_emit (GTK_OBJECT (box),
		    gtk_databox_signals[GTK_DATABOX_MARKED_SIGNAL], 0,
		    &box->marked);

   return FALSE;
}

static gint
gtk_databox_button_release_callback (GtkWidget * widget,
				     GdkEventButton * event, GtkDatabox * box)
{
   if (event->type != GDK_BUTTON_RELEASE)
      return FALSE;

   if (box->selection_flag)
   {
      box->flags |= 1 << GTK_DATABOX_SELECTION_STOPPED;
   
      g_signal_emit (GTK_OBJECT (box),
		       gtk_databox_signals
		       [GTK_DATABOX_SELECTION_STOPPED_SIGNAL], 0, &box->marked,
		       &box->select);
   }

   return FALSE;
}

static gint
gtk_databox_motion_notify_callback (GtkWidget * widget,
				    GdkEventMotion * event, GtkDatabox * box)
{
   gint x, y;
   GdkModifierType state;

   x = event->x;
   y = event->y;
   state = event->state;

   if (event->is_hint || (event->window != widget->window))
      gdk_window_get_pointer (widget->window, &x, &y, &state);

   if (state & GDK_BUTTON1_MASK
       && (box->flags & (1 << GTK_DATABOX_ENABLE_SELECTION))
       && !(box->flags & (1 << GTK_DATABOX_SELECTION_STOPPED)))
   {
      GdkRectangle rect;
      gint width;
      gint height;

      gdk_drawable_get_size (widget->window, &width, &height);
      x = MAX (0, MIN (width - 1, x));
      y = MAX (0, MIN (height - 1, y));

      if (box->selection_flag)
      {
	 /* Clear current selection from pixmap */
	 gtk_databox_draw_selection (box->draw, box, NULL);
      }
      else
      {
	 box->selection_flag = 1;
	 box->marked.x = x;
	 box->marked.y = y;
	 box->select.x = x;
	 box->select.y = y;
	 g_signal_emit (GTK_OBJECT (box),
			  gtk_databox_signals
			  [GTK_DATABOX_SELECTION_STARTED_SIGNAL], 0,
			  &box->marked);
      }

      /* Determine the exposure rectangle (covering old selection and new) */
      rect.x = MIN (MIN (box->marked.x, box->select.x), x);
      rect.y = MIN (MIN (box->marked.y, box->select.y), y);
      rect.width = MAX (MAX (box->marked.x, box->select.x), x) - rect.x + 1;
      rect.height = MAX (MAX (box->marked.y, box->select.y), y) - rect.y + 1;

      box->select.x = x;
      box->select.y = y;

      /* Draw new selection */
      gtk_databox_draw_selection (box->draw, box, &rect);

      g_signal_emit (GTK_OBJECT (box),
		       gtk_databox_signals
		       [GTK_DATABOX_SELECTION_CHANGED_SIGNAL], 0, &box->marked,
		       &box->select);
   }

   return FALSE;
}

void
gtk_databox_data_get_value (GtkDatabox * box, GtkDataboxCoord point,
			    GtkDataboxValue * coord)
{
   g_return_if_fail (GTK_IS_DATABOX (box) && coord);

   coord->x =
      box->top_left.x + point.x / box->factor.x;
   coord->y =
      box->top_left.y + point.y / box->factor.y;
}

void
gtk_databox_data_get_marked_value (GtkDatabox * box, GtkDataboxValue * coord)
{
   gtk_databox_data_get_value (box, box->marked, coord);
}

void
gtk_databox_data_get_delta_value (GtkDatabox * box, GtkDataboxValue * coord)
{
   GtkDataboxValue drooc;

   g_return_if_fail (GTK_IS_DATABOX (box) && coord);
   
   gtk_databox_data_get_value (box, box->marked, &drooc);
   gtk_databox_data_get_value (box, box->select, coord);
   coord->x -= drooc.x;
   coord->y -= drooc.y;
}

static void
gtk_databox_data_calc_extrema (GtkDatabox * box, GtkDataboxValue * min,
			       GtkDataboxValue * max)
{
   gint i;
   GtkDataboxData *data = NULL;
   GList *list = NULL;
   GtkDataboxValue border;

   g_return_if_fail (GTK_IS_DATABOX (box) && min && max);

   if (!box->data)
   {
      min->x = -0.5;
      min->y = -0.5;
      max->x = 1.5;
      max->y = 1.5;
      return;
   }

   list = box->data;
   if (list)
      data = (GtkDataboxData *) list->data;
   else
      data = NULL;

   min->x = data->X[0];
   min->y = data->Y[0];
   max->x = data->X[0];
   max->y = data->Y[0];

   while (data)
   {
      for (i = 0; i < data->length; i++)
      {
	 min->x = MIN (data->X[i], min->x);
	 max->x = MAX (data->X[i], max->x);
      }
      for (i = 0; i < data->length; i++)
      {
	 min->y = MIN (data->Y[i], min->y);
	 max->y = MAX (data->Y[i], max->y);
      }
      list = g_list_next (list);
      if (list)
	 data = (GtkDataboxData *) list->data;
      else
	 data = NULL;
   }

   border.x = (max->x - min->x) / 10.;
   border.y = (max->y - min->y) / 10.;

   min->x = min->x - border.x;
   min->y = min->y - border.y;
   max->x = max->x + border.x;
   max->y = max->y + border.y;

   return;
}

void
gtk_databox_data_get_extrema (GtkDatabox * box, GtkDataboxValue * min,
			      GtkDataboxValue * max)
{
   g_return_if_fail (GTK_IS_DATABOX (box) && min && max);

   *min = box->min;
   *max = box->max;

   return;
}

void
gtk_databox_data_get_visible_extrema (GtkDatabox * box, GtkDataboxValue * min,
				      GtkDataboxValue * max)
{
/*   GtkDataboxValue pre_min;
   GtkDataboxValue pre_max;
   GtkDataboxCoord top_left;
   GtkDataboxCoord bottom_right;
   */

   g_return_if_fail (GTK_IS_DATABOX (box) && min && max);

/*   top_left.x = top_left.y = 0;
   bottom_right.x = box->size.x - 1;
   bottom_right.y = box->size.y - 1;


   gtk_databox_data_get_value (box, top_left, &pre_min);
   gtk_databox_data_get_value (box, bottom_right, &pre_max);

   min->x = MIN (pre_min.x, pre_max.x);
   min->y = MIN (pre_min.y, pre_max.y);
   max->x = MAX (pre_min.x, pre_max.x);
   max->y = MAX (pre_min.y, pre_max.y);
*/
   min->x = box->top_left.x;
   max->x = box->bottom_right.x;

   min->y = box->bottom_right.y;
   max->y = box->top_left.y;

   return;
}

void
gtk_databox_rescale_with_values (GtkDatabox * box, GtkDataboxValue min,
				 GtkDataboxValue max)
{
   g_return_if_fail (GTK_IS_DATABOX (box));

   box->min.x = min.x;
   box->max.x = max.x;
   box->min.y = min.y;
   box->max.y = max.y;

   if (box->max.x - box->min.x < 1e-10)
   {
      box->min.x -= 0.5e-10;
      box->max.x += 0.5e-10;
   }
   if (box->max.y - box->min.y < 1e-10)
   {
      box->min.y -= 0.5e-10;
      box->max.y += 0.5e-10;
   }

   gtk_databox_zoom_home (box->draw, box);
}

void
gtk_databox_rescale (GtkDatabox * box)
{
   GtkDataboxValue min, max;

   g_return_if_fail (GTK_IS_DATABOX (box));

   gtk_databox_data_calc_extrema (box, &min, &max);

   gtk_databox_rescale_with_values (box, min, max);
}

void
gtk_databox_redraw (GtkDatabox * box)
{
   gtk_databox_draw_request_full (box->draw, TRUE, box);
}


static void
gtk_databox_zoom_to_selection (GtkWidget * widget, GtkDatabox * box)
{
   if (!(box->flags & (1 << GTK_DATABOX_ENABLE_ZOOM)))
      return;

   /* We always scroll from 0 to 1.0 */
   box->adjX->lower = 0;
   box->adjX->upper = 1.0;

   /* The left border of the selection box is used for calculating
    * the left end of the visible "page" in scollbar-coordinates
    * (remember: we scroll from 0 to 1.0)
    */
   box->adjX->value +=
      (gfloat) (MIN (box->marked.x, box->select.x)) * box->adjX->page_size /
      box->size.x;

   /* The size of the selection box is used for calculating
    * the size of the visible "page" in scollbar-coordinates
    * (remember: we scroll from 0 to 1.0)
    */
   box->adjX->page_size *=
      (gfloat) (ABS (box->marked.x - box->select.x) + 1) / box->size.x;
      
   /* If we zoom to far into the data, we will get funny results, because
    * of overflow effects. Therefore zooming is limited to box->zoom_limit.
    */
   if (box->adjX->page_size < box->zoom_limit)
   {
      box->adjX->value = (gfloat) MAX (0, 
                 box->adjX->value 
		 - (box->zoom_limit - box->adjX->page_size) / 2.0);
      box->adjX->page_size = box->zoom_limit;
   }
   
   /* Setting the scroll increments */
   box->adjX->step_increment = box->adjX->page_size / 20;
   box->adjX->page_increment = box->adjX->page_size * 0.9;
   
   /* And now the analog steps for the vertical scrollbar */
   box->adjY->lower = 0;
   box->adjY->upper = 1.0;

   box->adjY->value +=
      (gfloat) (MIN (box->marked.y, box->select.y)) * box->adjY->page_size /
      box->size.y;
   
   box->adjY->page_size *=
      (gfloat) (ABS (box->marked.y - box->select.y) + 1) / box->size.y;
   
   if (box->adjY->page_size < box->zoom_limit)
   {
      box->adjY->value = (gfloat) MAX (0, 
                 box->adjY->value 
		 - (box->zoom_limit - box->adjY->page_size) / 2.0);
      box->adjY->page_size = box->zoom_limit;
   }
   
   box->adjY->step_increment = box->adjY->page_size / 20;
   box->adjY->page_increment = box->adjY->page_size * 0.9;

   gtk_databox_zoomed (widget, box, TRUE);
}

static void
gtk_databox_zoom_out (GtkWidget * widget, GtkDatabox * box)
{
   if (!(box->flags & (1 << GTK_DATABOX_ENABLE_ZOOM)))
      return;

   box->adjX->lower = 0;
   box->adjY->lower = 0;
   box->adjX->page_size = MIN (1.0, box->adjX->page_size * 2);
   box->adjY->page_size = MIN (1.0, box->adjY->page_size * 2);
   box->adjX->value =
      (box->adjX->page_size ==
       1.0) ? 0 : (MAX (0, (box->adjX->value - box->adjX->page_size / 4)));
   box->adjY->value =
      (box->adjY->page_size ==
       1.0) ? 0 : (MAX (0, (box->adjY->value - box->adjY->page_size / 4)));
   box->adjX->upper = 1.0;
   box->adjY->upper = 1.0;
   box->adjY->step_increment = box->adjY->page_size / 20;
   box->adjY->page_increment = box->adjY->page_size * 0.9;
   box->adjX->step_increment = box->adjX->page_size / 20;
   box->adjX->page_increment = box->adjX->page_size * 0.9;

   gtk_databox_zoomed (widget, box, TRUE);
}

static void
gtk_databox_zoom_home (GtkWidget * widget, GtkDatabox * box)
{

   if (!(box->flags & (1 << GTK_DATABOX_ENABLE_ZOOM)))
      return;

   box->selection_flag = 0;

   box->adjX->lower = 0;
   box->adjY->lower = 0;
   box->adjX->page_size = 1.0;
   box->adjY->page_size = 1.0;
   box->adjX->value = 0;
   box->adjY->value = 0;
   box->adjX->upper = 1.0;
   box->adjY->upper = 1.0;
   box->adjY->step_increment = box->adjY->page_size / 20;
   box->adjY->page_increment = box->adjY->page_size * 0.9;
   box->adjX->step_increment = box->adjX->page_size / 20;
   box->adjX->page_increment = box->adjX->page_size * 0.9;

   gtk_databox_zoomed (widget, box, TRUE);
}

static void
gtk_databox_zoomed (GtkWidget * widget, GtkDatabox * box,
		    gboolean redraw_flag)
{
   /* This function is called after configure events even if zoom 
    * is disabled, and we need it because box->factor is re-calculated here.
    * (It is also called after zoom events, of course...)
    */ 
   box->flags |= 1 << GTK_DATABOX_REDRAW_REQUEST;

   gtk_adjustment_changed (box->adjX);
   gtk_adjustment_changed (box->adjY);
   gtk_databox_x_adjustment_callback (widget, box);
   gtk_databox_y_adjustment_callback (widget, box);

   box->factor.x = box->size.x / (box->bottom_right.x - box->top_left.x);
   box->factor.y = box->size.y / (box->bottom_right.y - box->top_left.y);

   if (redraw_flag)
   {
      box->flags &= ~(1 << GTK_DATABOX_REDRAW_REQUEST);
      gtk_databox_draw_request_full (box->draw, TRUE, box);
   }

   g_signal_emit (GTK_OBJECT (box),
		    gtk_databox_signals[GTK_DATABOX_ZOOMED_SIGNAL], 0,
		    &box->top_left, &box->bottom_right);
}

static void
gtk_databox_x_adjustment_callback (GtkWidget * widget, GtkDatabox * box)
{
   if (box->adjX->page_size == 1.0)
   {
      box->top_left.x = box->min.x;
      box->bottom_right.x = box->max.x;
   }
   else
   {
      box->top_left.x =
	 box->min.x + (box->max.x - box->min.x) * box->adjX->value;
      box->bottom_right.x =
	 box->top_left.x + (box->max.x - box->min.x) * box->adjX->page_size;
   }

   gtk_databox_update_x_ruler (box);
   gtk_databox_draw_request_full (box->draw, TRUE, box);
}

static void
gtk_databox_y_adjustment_callback (GtkWidget * widget, GtkDatabox * box)
{
   if (box->adjY->page_size == 1.0)
   {
      box->top_left.y = box->max.y;
      box->bottom_right.y = box->min.y;
   }
   else
   {
      box->top_left.y =
	 box->max.y - (box->max.y - box->min.y) * box->adjY->value;
      box->bottom_right.y =
	 box->top_left.y - (box->max.y - box->min.y) * box->adjY->page_size;
   }

   gtk_databox_update_y_ruler (box);
   gtk_databox_draw_request_full (box->draw, TRUE, box);
}

static void
gtk_databox_update_x_ruler (GtkDatabox * box)
{
   if (box->flags & (1 << GTK_DATABOX_SHOW_RULERS))
   {
      gtk_ruler_set_range (GTK_RULER (box->hrule), box->top_left.x,
			   box->bottom_right.x,
			   0.5 * (box->top_left.x + box->bottom_right.x), 20);
   }
}

static void
gtk_databox_update_y_ruler (GtkDatabox * box)
{
   if (box->flags & (1 << GTK_DATABOX_SHOW_RULERS))
   {
      gtk_ruler_set_range (GTK_RULER (box->vrule), box->top_left.y,
			   box->bottom_right.y,
			   0.5 * (box->top_left.y + box->bottom_right.y), 20);
   }
}

static void
gtk_databox_draw_request_full (GtkWidget * widget, gboolean now,
			       GtkDatabox * box)
{
   GdkRectangle redraw_rect;

   redraw_rect.x = 0;
   redraw_rect.y = 0;
   redraw_rect.width = box->size.x;
   redraw_rect.height = box->size.y;

   if (box->flags & (1 << GTK_DATABOX_REDRAW_REQUEST))
   {
      return;
   }

   box->flags |= 1 << GTK_DATABOX_REDRAW_REQUEST;

   if (now)
      gtk_widget_queue_draw_area (widget, 0, 0, box->size.x, box->size.y);
/* FIXME: Maybe we want to send an expose event instead. This guarantees
          immediate redrawing */
/*      gtk_widget_draw (widget, &redraw_rect);*/
}

static gint
gtk_databox_draw_selection (GtkWidget * widget, GtkDatabox * box,
			    GdkRectangle * rect)
{
   if (!box->select_gc)
   {
      gboolean color_allocate_success;
      GdkGCValues values;
      GdkColormap *colormap;
      GdkColor color;

      color.red = 65535;
      color.green = 65535;
      color.blue = 65535;
      colormap = gtk_widget_get_colormap (widget);
      color_allocate_success = gdk_colormap_alloc_color (colormap, 
                              &color, 
			      FALSE, 
			      TRUE);
      /* FIXME We should add a message here ... */
      g_return_val_if_fail (color_allocate_success, FALSE);

      values.foreground = color;
      values.function = GDK_XOR;
      box->select_gc =
	 gdk_gc_new_with_values (widget->window, &values,
				 GDK_GC_FUNCTION | GDK_GC_FOREGROUND);
   }


   gdk_draw_rectangle (box->pixmap, box->select_gc,
		       box->flags & (1 << GTK_DATABOX_SHOW_SELECTION_FILLED),
		       MIN (box->marked.x, box->select.x), MIN (box->marked.y,
								box->select.
								y),
		       ABS (box->marked.x - box->select.x),
		       ABS (box->marked.y - box->select.y));

   if (rect)
      gdk_draw_drawable (widget->window,
		       widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
		       box->pixmap, rect->x, rect->y, rect->x, rect->y,
		       rect->width, rect->height);


   return TRUE;
}

gint
gtk_databox_data_get_color (GtkDatabox * box, gint index, GdkColor * color)
{
   GtkDataboxData *data = NULL;

   g_return_val_if_fail (GTK_IS_DATABOX (box), -1);
   g_return_val_if_fail (color, -1);

   data = (GtkDataboxData *) g_list_nth_data (box->data, index);
   g_return_val_if_fail (data, -1);

   *color = data->color;

   return 0;
}

gint
gtk_databox_data_get_type (GtkDatabox * box, gint index,
			   GtkDataboxDataType * type, guint * dot_size)
{
   GtkDataboxData *data = NULL;

   g_return_val_if_fail (GTK_IS_DATABOX (box), -1);
   g_return_val_if_fail (type, -1);
   g_return_val_if_fail (dot_size, -1);

   data = (GtkDataboxData *) g_list_nth_data (box->data, index);
   g_return_val_if_fail (data, -1);

   *type = data->type;
   *dot_size = data->size;

   return 0;
}

gint
gtk_databox_data_set_type (GtkDatabox * box, gint index,
			   GtkDataboxDataType type, guint dot_size)
{
   GtkDataboxData *data = NULL;

   g_return_val_if_fail (GTK_IS_DATABOX (box), -1);

   data = (GtkDataboxData *) g_list_nth_data (box->data, index);
   g_return_val_if_fail (data, -1);

   if (data->flags & (1 << GTK_DATABOX_DATA_HAS_GC))
   {
      g_object_unref (data->gc);
      data->flags &= ~(1 << GTK_DATABOX_DATA_HAS_GC);
   }

   switch (type)
   {
   case GTK_DATABOX_NOT_DISPLAYED:
      data->draw = GTK_DATABOX_DRAW_FUNC (NULL);
      break;
   case GTK_DATABOX_POINTS:
      data->draw = GTK_DATABOX_DRAW_FUNC (gtk_databox_draw_points);
      break;
   case GTK_DATABOX_LINES:
      data->draw = GTK_DATABOX_DRAW_FUNC (gtk_databox_draw_lines);
      break;
   case GTK_DATABOX_BARS:
      data->draw = GTK_DATABOX_DRAW_FUNC (gtk_databox_draw_bars);
      break;
   case GTK_DATABOX_CROSS_SIMPLE:
      data->draw = GTK_DATABOX_DRAW_FUNC (gtk_databox_draw_cross_simple);
      break;
   case GTK_DATABOX_GRID:
      data->draw = GTK_DATABOX_DRAW_FUNC (gtk_databox_draw_grid);
      break;

   default:
      data->draw = GTK_DATABOX_DRAW_FUNC (NULL);
   }

   data->type = type;
   data->size = dot_size;

   return 0;
}

gint
gtk_databox_data_set_color (GtkDatabox * box, gint index, GdkColor color)
{
   GtkDataboxData *data = NULL;

   g_return_val_if_fail (GTK_IS_DATABOX (box), -1);

   data = (GtkDataboxData *) g_list_nth_data (box->data, index);
   g_return_val_if_fail (data, -1);

   if (data->flags & (1 << GTK_DATABOX_DATA_HAS_GC))
   {
      g_object_unref (data->gc);
      data->flags &= ~(1 << GTK_DATABOX_DATA_HAS_GC);
   }

   data->color = color;

   return 0;
}

gint
gtk_databox_data_get_grid_config (GtkDatabox * box, gint index,
				  guint * hlines, guint * vlines)
{
   GtkDataboxData *data = NULL;

   g_return_val_if_fail (GTK_IS_DATABOX (box), -1);

   data = (GtkDataboxData *) g_list_nth_data (box->data, index);
   g_return_val_if_fail (data, -1);

   *hlines = data->hlines;
   *vlines = data->vlines;

   return 0;

}

gint
gtk_databox_data_set_grid_config (GtkDatabox * box, gint index, guint hlines,
				  guint vlines)
{
   GtkDataboxData *data = NULL;

   g_return_val_if_fail (GTK_IS_DATABOX (box), -1);

   data = (GtkDataboxData *) g_list_nth_data (box->data, index);
   g_return_val_if_fail (data, -1);

   data->hlines = hlines;
   data->vlines = vlines;

   return 0;
}

static void
gtk_databox_new_data_gc (GtkWidget * widget, GtkDatabox * box,
			 GtkDataboxData * data)
{
   GdkGCValues values;
   gboolean color_allocate_success;
   GdkColormap *colormap = NULL;

   g_return_if_fail (GTK_IS_DATABOX (box));
   g_return_if_fail (GTK_IS_WIDGET (widget));
   g_return_if_fail (data);

   colormap = gtk_widget_get_colormap (widget);
   g_return_if_fail (colormap);
   color_allocate_success = gdk_colormap_alloc_color (colormap, 
                              &data->color, 
			      FALSE, 
			      TRUE);
   /* FIXME We should add a message here ... */
   g_return_if_fail (color_allocate_success);

   values.foreground = data->color;
   values.function = GDK_COPY;
   values.line_width = data->size;
   if (data->type == GTK_DATABOX_GRID)
   {
      values.line_style = GDK_LINE_ON_OFF_DASH;
      values.cap_style = GDK_CAP_BUTT;
      values.join_style = GDK_JOIN_MITER;
   }
   else
   {
      values.line_style = GDK_LINE_SOLID;
      values.cap_style = GDK_CAP_BUTT;
      values.join_style = GDK_JOIN_MITER;
   }
   data->gc =
      gdk_gc_new_with_values (widget->window, &values,
			      GDK_GC_FUNCTION | GDK_GC_FOREGROUND |
			      GDK_GC_LINE_WIDTH | GDK_GC_LINE_STYLE |
			      GDK_GC_CAP_STYLE | GDK_GC_JOIN_STYLE);

   data->flags |= 1 << GTK_DATABOX_DATA_HAS_GC;
}

static void
gtk_databox_draw (GtkWidget * widget, GtkDatabox * box,
		  GdkEventExpose * event)
{
   GList *list = NULL;
   GtkDataboxData *data = NULL;

   box->flags &= ~(1 << GTK_DATABOX_REDRAW_REQUEST);

   g_return_if_fail (GTK_IS_DATABOX (box));
   if (!GTK_WIDGET_VISIBLE (widget))
      return;

   gdk_draw_rectangle (box->pixmap, widget->style->bg_gc[0], TRUE, 0, 0,
		       box->size.x, box->size.y);

   if (!box->data || !box->max_points)
      return;

   /*  Draw last data set first so first is on top */
   list = g_list_last (box->data);
   if (list)
      data = (GtkDataboxData *) list->data;
   else
      data = NULL;

   while (data)
   {
      if (!data->gc || !(data->flags & (1 << GTK_DATABOX_DATA_HAS_GC)))
      {
	 gtk_databox_new_data_gc (widget, box, data);
      }
      if (data->length && data->draw)
      {
	 data->draw (box, data);
      }

      list = g_list_previous (list);
      if (list)
	 data = (GtkDataboxData *) list->data;
      else
	 data = NULL;
   }

   if (box->selection_flag)
   {
      gtk_databox_draw_selection (widget, box, NULL);
   }

   return;
}

static void
gtk_databox_draw_points (GtkDatabox * box, GtkDataboxData * data)
{
   gint i = 0;
   gint count = 0;
   

      for (i = 0; i < data->length; i++)
      {
	 box->points[i].x = (gint16) ((data->X[i] - box->top_left.x) * box->factor.x);
	 box->points[i].y = (gint16) ((data->Y[i] - box->top_left.y) * box->factor.y);
      }
      count = data->length;
   if (data->size < 2)
   {
      /* More than 2^16 points will cause X IO error on most XServers
         (Hint from Paul Barton-Davis <pbd@Op.Net>) */
      for (i = 0; i < count; i += 65536)
      {
	 gdk_draw_points (box->pixmap, data->gc, box->points + i,
			  MIN (65536, count - i));
      }
   }
   else
   {
      for (i = 0; i < count; i++)
      {
	 /* Why on earth is there no gdk_draw_rectangles?? */
	 gdk_draw_rectangle (box->pixmap, data->gc, TRUE,
			     box->points[i].x - data->size / 2,
			     box->points[i].y - data->size / 2, data->size,
			     data->size);
      }
   }
}

static void
gtk_databox_draw_lines (GtkDatabox * box, GtkDataboxData * data)
{
   gint i;

   for (i = 0; i < data->length; i++)
   {
      box->points[i].x = (gint16) ((data->X[i] - box->top_left.x) * box->factor.x);
      box->points[i].y = (gint16) ((data->Y[i] - box->top_left.y) * box->factor.y);
   }

   /* More than 2^16 points will cause X IO error on most XServers
      (Hint from Paul Barton-Davis <pbd@Op.Net>) */
   for (i = 0; i < data->length; i += 65535)
   {
      gdk_draw_lines (box->pixmap, data->gc, box->points + i,
		      MIN (65536, data->length - i));
   }
}

static void
gtk_databox_draw_bars (GtkDatabox * box, GtkDataboxData * data)
{
   gint i = 0;
   gint count = 0;
   GdkSegment *segments = (GdkSegment *) box->points;
   gfloat axis = 0;

   axis = ((0 - box->top_left.y) * box->factor.y);
      for (i = 0; i < data->length; i++)
      {
	 segments[i].x1 = segments[i].x2 =
	    (gint16) ((data->X[i] - box->top_left.x) * box->factor.x);
	 segments[i].y1 = axis;
	 segments[i].y2 = (gint16) ((data->Y[i] - box->top_left.y) * box->factor.y);
      }
      count = data->length;
   for (i = 0; i < count; i += 65536)
   {
      gdk_draw_segments (box->pixmap, data->gc, segments,
			 MIN (65536, count - i));
   }
}

static void
gtk_databox_draw_cross_simple (GtkDatabox * box, GtkDataboxData * data)
{
   gint x = 0;
   gint y = 0;
   gboolean xflag = FALSE;
   gboolean yflag = FALSE;

   if (0 >= box->top_left.x && 0 < box->bottom_right.x)
   {
      x = (0 - box->top_left.x) * box->factor.x;
      if (x >= CROSS_BORDER && x < box->size.x - CROSS_BORDER)
      {
	 gdk_draw_line (box->pixmap, data->gc, x, CROSS_BORDER, x,
			box->size.y - CROSS_BORDER);
	 xflag = TRUE;
      }
   }
   if (0 <= box->top_left.y && 0 > box->bottom_right.y)
   {
      y = (0 - box->top_left.y) * box->factor.y;
      if (y >= CROSS_BORDER && y < box->size.y - CROSS_BORDER)
      {
	 gdk_draw_line (box->pixmap, data->gc, CROSS_BORDER, y,
			box->size.x - CROSS_BORDER, y);
	 yflag = TRUE;
      }
   }
}

static void
gtk_databox_draw_grid (GtkDatabox * box, GtkDataboxData * data)
{
   gint x = 0;
   gint y = 0;
   gint i = 0;
   guint v_lines = data->vlines;
   guint h_lines = data->hlines;

   for (i = 0; i < v_lines; i++)
   {
      x = box->min.x + (i + 1) * (box->max.x - box->min.x) / (v_lines + 1);
      x = (gint16) ((x - box->top_left.x) * box->factor.x);
      gdk_draw_line (box->pixmap, data->gc, x, 0, x, box->size.y);
   }
   for (i = 0; i < h_lines; i++)
   {
      y = box->min.y + (i + 1) * (box->max.y - box->min.y) / (v_lines + 1);
      y = (gint16) ((y - box->top_left.y) * box->factor.y);
      gdk_draw_line (box->pixmap, data->gc, 0, y, box->size.x, y);
   }
}


gint
gtk_databox_data_add_x_y (GtkDatabox * box, guint length, gfloat * X,
			  gfloat * Y, GdkColor color, GtkDataboxDataType type,
			  guint dot_size)
{
   GtkDataboxData *data;
   guint index;

   g_return_val_if_fail (GTK_IS_DATABOX (box), -1);
   g_return_val_if_fail (X, -1);
   g_return_val_if_fail (Y, -1);
   g_return_val_if_fail (length, -1);

   box->max_points = MAX (length, box->max_points);
   if (box->max_points) 
   {
      /* We need twice as many GdkPoints, because draw bars uses segments 
       * and one segment equals two points.
       */
      box->points = (GdkPoint *) g_realloc (box->points, 
                             sizeof (GdkPoint) * box->max_points * 2);
   }
   data = g_new0 (GtkDataboxData, 1);

   data->X = X;
   data->Y = Y;
   data->length = length;
   data->flags = 0;
   data->gc = NULL;

   box->data = g_list_append (box->data, data);
   index = g_list_length (box->data) - 1;

   gtk_databox_data_set_type (box, index, type, dot_size);
   gtk_databox_data_set_color (box, index, color);

   return index;
}

gint
gtk_databox_data_add_x (GtkDatabox * box, guint length, gfloat * X,
			gint shared_Y_index, GdkColor color,
			GtkDataboxDataType type, guint dot_size)
{
   GtkDataboxData *data;

   g_return_val_if_fail (GTK_IS_DATABOX (box), -1);
   g_return_val_if_fail (X, -1);

   data = (GtkDataboxData *) g_list_nth_data (box->data, shared_Y_index);
   g_return_val_if_fail (data, -1);
   g_return_val_if_fail (data->length == length, -1);

   return gtk_databox_data_add_x_y (box, length, X, data->Y, color, type,
				    dot_size);
}

gint
gtk_databox_data_add_y (GtkDatabox * box, guint length, gfloat * Y,
			gint shared_X_index, GdkColor color,
			GtkDataboxDataType type, guint dot_size)
{
   GtkDataboxData *data;

   g_return_val_if_fail (GTK_IS_DATABOX (box), -1);
   g_return_val_if_fail (Y, -1);

   data = (GtkDataboxData *) g_list_nth_data (box->data, shared_X_index);
   g_return_val_if_fail (data, -1);
   g_return_val_if_fail (data->length == length, -1);

   return gtk_databox_data_add_x_y (box, length, data->X, Y, color, type,
				    dot_size);
}

static gint
gtk_databox_check_x_links (GList * list, gfloat * values)
{
   GtkDataboxData *data;
   gint counter = 0;

   if (list)
      data = (GtkDataboxData *) list->data;
   else
      return 0;

   while (data)
   {
      if (data->X == values)
	 counter++;
      list = g_list_next (list);
      if (list)
	 data = (GtkDataboxData *) list->data;
      else
	 data = NULL;
   }

   return counter;
}

static gint
gtk_databox_check_y_links (GList * list, gfloat * values)
{
   GtkDataboxData *data;
   gint counter = 0;

   if (list)
      data = (GtkDataboxData *) list->data;
   else
      return 0;

   while (data)
   {
      if (data->Y == values)
	 counter++;
      list = g_list_next (list);
      if (list)
	 data = (GtkDataboxData *) list->data;
      else
	 data = NULL;
   }

   return counter;
}

static void
gtk_databox_destroy_data (GtkDatabox * box, GtkDataboxData * data,
			  GList * list, gboolean free_flag)
{
   GdkColormap *colormap;

   if (free_flag && gtk_databox_check_x_links (box->data, data->X) == 1)
   {
      g_free (data->X);
   }
   if (free_flag && gtk_databox_check_y_links (box->data, data->Y) == 1)
   {
      g_free (data->Y);
   }
   if (data->flags & (1 << GTK_DATABOX_DATA_HAS_GC))
   {
      colormap = gtk_widget_get_colormap (box->draw);
      gdk_colormap_free_colors (colormap, &data->color, 1);
   }
   if (data->gc)
      g_object_unref (data->gc);

   g_free (data);

}

gint
gtk_databox_data_destroy_all_with_flag (GtkDatabox * box, gboolean free_flag)
{
   GList *list = NULL;
   GtkDataboxData *data = NULL;

   g_return_val_if_fail (GTK_IS_DATABOX (box), 0);

   if (!box->data)
      return 0;

   list = box->data;
   if (list)
      data = (GtkDataboxData *) list->data;
   else
      data = NULL;

   while (data)
   {
      gtk_databox_destroy_data (box, data, list, free_flag);

      list = g_list_next (list);
      if (list)
	 data = (GtkDataboxData *) list->data;
      else
	 data = NULL;
   }

   g_list_free (box->data);

   box->data = NULL;
   box->max_points = 0;

   g_free (box->points);
   box->points = NULL;

   return 0;
}

gint
gtk_databox_data_destroy_with_flag (GtkDatabox * box, gint index,
				    gboolean free_flag)
{
   GList *list = NULL;
   GtkDataboxData *data = NULL;

   g_return_val_if_fail (GTK_IS_DATABOX (box), 0);

   if (!box->data)
      return -1;

   list = g_list_nth (box->data, index);
   if (list)
      data = (GtkDataboxData *) list->data;
   else
      return -1;

   gtk_databox_destroy_data (box, data, list, free_flag);

   box->data = g_list_remove_link (box->data, list);
   g_list_free_1 (list);

   return 0;
}

gint
gtk_databox_data_remove_all (GtkDatabox * box)
{
   return gtk_databox_data_destroy_all_with_flag (box, FALSE);
}

gint
gtk_databox_data_remove (GtkDatabox * box, gint index)
{
   return gtk_databox_data_destroy_with_flag (box, index, FALSE);
}

gint
gtk_databox_data_destroy_all (GtkDatabox * box)
{
   return gtk_databox_data_destroy_all_with_flag (box, TRUE);
}

gint
gtk_databox_data_destroy (GtkDatabox * box, gint index)
{
   return gtk_databox_data_destroy_with_flag (box, index, TRUE);
}
