/*
* ===========================
* VDK Visual Development Kit
* Version 1.2.3
* October 1998, August 2000
* ===========================
*
* Copyright (C) 1998, Mario Motta
* Developed by Mario Motta <mmotta@guest.net>
*
* This library is free software; you can redistribute it and/or
* modify it under the terms of the GNU Library General Public
* License as published by the Free Software Foundation; either
* version 2 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Library General Public License for more details.
*
* You should have received a copy of the GNU Library General Public
* License along with this library; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
* 02111-130
*/ 

#ifndef VDKUSTRING_H
#define VDKUSTRING_H

#include <locale.h>

#define MAXPRINTFLEN	65535	// max size for Sprintf and Concatf buffer
#define INT_DATE		0		// for FormatDate
#define ENG_DATE		1
#define EUR_DATE		2

struct USTRING 
{          
	char* s;              
	unsigned int ref ;    
};

/*!
\class VDKUString
\brief Implements famous cont referenced string objects for UTF-8 string encoding.
This string is (and have to be) compatible with byte encoded strings
*/
class VDKUString
{
	private:
		// Just copied from VDKString to keep compatibility
		VDKUString& oldUpperCase();
		VDKUString& oldLowerCase();
		// Always return UTF-8 strings
		bool get_i18n_sysparams(char *&decsep, char *&thousandssep, char *&grouping);

	protected:
	
		USTRING* p;
		
	public:
		/*!
		Constructor, makes an empty string
		\code
		VDKUString s;
		\endcode
		*/
		VDKUString();
		/*!
		Constructor
		\param s a null terminated C string
		\code
		VDKUString s = "uncle bill";
		\endcode
		*/
		VDKUString(const char*s);
		/*!
		Constructor
		\param c a single character
		\code
		VDKUString s(c);
		\endcode
		*/
		VDKUString(const char& c);
		/*!
		Copy-initializer
		\param s a VDKUString reference
		\code
		VDKUString s = "uncle bill";
		VDKUString s1 = s;
		\endcode
		*/
		VDKUString(const VDKUString& s);
		/*!
		Destructor
		*/
		~VDKUString();
		/*!
		VDKUString to char* casting
		** warning ** can violate data hiding OO concept
		*/
		operator char*() { return p->s; }
		/*!
		Assignement operator
		\param s a VDKUString reference
		\code
		VDKUString s = "uncle bill";
		VDKUString s1 = s;
		VDKUString s2 = "uncle sam";
		s = s2;
		\endcode
		*/
		VDKUString& operator = (const VDKUString& s);
		/*!
		Assignement operator
		\param s a null terminated C string
		\code
		VDKUString s = "uncle bill";
		s = "uncle sam";
		\endcode
		*/
		VDKUString& operator = (const char* s);
		/*!
		Equality operator
		*/
		int operator == (const VDKUString& s) const ;
		/*!
		less than operator
		*/
		int operator < (const VDKUString& s) const ;
		/*!
		greater than operator
		*/
		int operator > (const VDKUString& s)  const ;
		/*!
		less-equal operator
		*/
		int operator <= (const VDKUString& s) const ;
		/*!
		greater-equal operator
		*/
		int operator >= (const VDKUString& s) const ;
		/*!
		disequality operator
		*/
		int operator != (const VDKUString& s) const ;
		/*!
		cat to this
		\param s a null terminated string
		\code
		VDKUString s = "uncle bill";
		s += " is a smart boy";
		\endcode
		*/
		VDKUString& operator += (const char* s);
		/*!
		cat to this
		\param s a VDKUString
		\code
		VDKUString s = "uncle bill";
		VDKUString s1 = " is a smart boy";
		s += s1;
		\endcode
		*/
		VDKUString& operator += (const VDKUString& s);
		/*!
		Returns a VDKUString concatenated\param s a null terminated string
		\code
		VDKUString s = "uncle bill";
		VDKUString s1 = s + " is a smart boy";
		\endcode
		*/
		VDKUString operator + (const char* s) const;
		friend VDKUString operator + (const char* s, const VDKUString& vdks);
		/*!
		Returns a VDKUString concatenated
		\param s a VDKUString
		*/
		VDKUString operator + (const VDKUString& s) const;
		/*!
		index operator for const instances returns NULL if ix >= size
		*/
		char operator [] (unsigned int ix) const;
		/*! 
		string pointer access for const instances
		*/
		const char* c_str() const;
		/*!
		Returns true if this is an empty string
		*/
		bool isNull() const;
		/*!
		as strlen()
		*/
		int size() const;
		/*!
		Added for UTF-8 support, returns the length of the string in characters.
		The result is a byte count for not UTF-8 strings
		*/
		unsigned int Len() const;
		/*!
		index operator for const instances returns empty string if ix >= len, ix is character based
		\warning Returned buffer has to be deleted, may be NULL
		*/
		char* GetChar(unsigned int ix) const;
		/*!
		Removes a part of the string, beginning at 'begin' on 'len' length.
		Modifies and returns the resulting VDKUString.
		\param begin char number where begins the selection (0 based)
		\param len   selection length
		*/
		VDKUString& DelSelection(unsigned int begin, unsigned int len);
		/*!
		Removes all trailing spaces.
		Modifies and returns the resulting VDKUString.
		*/
		VDKUString& RTrim();
		/*!
		Removes all leading spaces.
		Modifies and returns the resulting VDKUString.
		*/
		VDKUString& LTrim();
		/*!
		Removes all leading and trailing spaces.
		Modifies and returns the resulting VDKUString.
		*/
		VDKUString& Trim();
		/*!
		Returns the number of the specified char 'car' contained in the string.
		\param car char to be counted
		\code
		VDKUString s = "uncle bill";
		int NumCar = s.CharCount('l');	// NumCar value is 3
		\endcode
		*/
		unsigned int CharCount(const char car) const;
		/*!
		Returns the number of the specified string 'str' contained in the string.
		\param str string to be counted
		\code
		VDKUString s = "uncle bill brother bill";
		int NumCar = s.CharCount("bill");	// NumCar value is 2
		\endcode
		*/
		unsigned int CharCount(const char *str) const;
		/*!
		Returns the upper case VDKUString after having modify it.
		\warning Does not modify unknown characters.
		\warning Upper case characters are assumed without accents.
		*/
		VDKUString& UpperCase();
		/*!
		Returns the lower case VDKUString after having modify it.
		\warning Upper case characters are assumed without accents.
		*/
		VDKUString& LowerCase();
		/*!
		Returns true if this is an empty string meaning
		NULL buffer or strlen() == 0.
		*/
		bool isEmpty() const;
		/*!
		Strcat() to the existing string (printf style).
		Modifies and returns the resulting VDKUString.
		\warning Final string is 65534 chars max.
		\warning Returns the previous string in case of memory overflow
		or buffer overflow.
		\param format	a NULL terminated string
		\param ...		a list of parameters
		\code
		VDKUString s = "uncle bill";
		s.Concatf("%s", " is a smart boy");	// s value is "uncle bill is a smart boy"
		\endcode
		*/
		VDKUString& Concatf(const char* format, ...);
		/*!
		Assignment to string (printf style).
		Modifies and returns the resulting VDKUString.
		\warning Final string is 65534 chars max.
		\warning Returns the previous string in case of memory overflow
		or buffer overflow.
		\param format	a NULL terminated string
		\param ...		a list of parameters
		\code
		VDKUString s;
		s.Sprintf("%s is %d years old", "uncle bill", 40);	// s value is "uncle bill is 40 years old"
		\endcode
		*/
		VDKUString& Sprintf(const char* format, ...);
		/*!
		Extract the specified part of a formatted string.
		Modifies and returns the resulting VDKUString.
		\warning Returns an isNull() string if the specified part not found.
		\param i	the desired part position (starting at 1)
		\param sep	the parts separator, '|' by default
		\code
		VDKUString s = "one|two|three|four";
		VDKUString p = s;
		p.GetPart(2);		// p value is "two"
		\endcode
		*/
		VDKUString& GetPart(unsigned int i, const char sep);
		/*!
		Extract the specified part of a formatted string.
		Modifies and returns the resulting VDKUString.
		\warning Returns an isNull() string if the specified part not found.
		\param i	the desired part position (starting at 1)
		\param sep	the parts separator, "|" by default
		\code
		VDKUString s = "one#!#two#!#three#!#four";
		VDKUString p = s;
		p.GetPart(2, "#!#");		// p value is "two"
		\endcode
		*/
		VDKUString& GetPart(unsigned int i, const char *sep = "|");
		/*!
		Returns the first occurrence position of the specified char 'car' (0 based)
		or -1 if 'car ' not found.
		\param car char to be searched for
		*/
		int GetFCharPos(const char car) const;
		/*!
		Returns the last occurrence position of the specified char 'car' (0 based)
		or -1 if 'car ' not found.
		\param car char to be searched for
		*/
		int GetLCharPos(const char car) const;
		/*!
		Returns the first occurrence position of the specified UTF-8 character 'car' (0 based)
		or -1 if 'car' not found.
		\param car char* to be searched for
		*/
		int GetFCharPos(const char *car) const;
		/*!
		Returns the last occurrence position of the specified UTF-8 character 'car' (0 based)
		or -1 if 'car' not found.
		\param car char* to be searched for
		*/
		int GetLCharPos(const char *car) const;
		/*!
		Returns the converted string to double.
		See atof() for details.
		*/
		double StrtoDouble() const;
		/*!
		Returns the converted string to int.
		See atoi() for details.
		*/
		int StrtoInt() const;
		/*!
		Extract a part of the string beginning at 'start' upon 'len' length.
		start and len are expressed in UTF-8 characters and not bytes.
		Modifies and returns the resulting VDKUString.
		\param start	first char position  (0 based)
		\param len		maximum length of the resulting string
		*/
		VDKUString& SubStr(unsigned int start, unsigned int len);
		/*!
		Cut the string at 'len' length.
		Modifies and returns the resulting VDKUString.
		\param len		length of the resulting string
		*/
		VDKUString& Cut(unsigned int len);
		/*!
		Returns true if string is a valid UTF-8 one
		If not, string can be an ASCII or malformed UTF-8 string
		Data read from a file or the network should be checked with isUTF8Valid()
		before doing anything else with it.
		*/
		bool isUTF8Valid() const;
		/*!
		Pad left of string with a specified char 'car' upon 'len' length.
		Modifies and returns the resulting VDKUString.
		\param len		length of the resulting string
		\param car		char to be padded
		*/
		VDKUString& LPad(unsigned int len, const char car);
		/*!
		Pad right of string with a specified char 'car' upon 'len' length.
		Modifies and returns the resulting VDKUString.
		\param len		length of the resulting string
		\param car		char to be padded
		*/
		VDKUString& RPad(unsigned int len, const char car);
		/*!
		Double all 'car' chars in the string (for SQL purpose).
		Modifies and returns the resulting VDKUString.
		\param car		char to be doubled, '\'' (cote) by default
		\code
		VDKUString s = "Don't do that";
		VDKUString p = s;
		p.DoubleChar(); // p value is "Don''t do that"
		\endcode
		*/
		VDKUString& DoubleChar(const char car);
		/*!
		Same as DoubleChar(char) but for UTF-8 strings
		*/
		VDKUString& DoubleChar(const char *car = "'");
		/*!
		Insert car at pos position (0 based)
		*/
		VDKUString& InsertChar(const char car, unsigned int pos);
		/*!
		Same as InsertChar(char) but for UTF-8 strings
		*/
		VDKUString& InsertChar(const char *car, unsigned int pos);
		/*!
		Remove all 'car' chars in the string.
		Modifies and returns the resulting VDKUString.
		\param car		char to be removed
		\code
		VDKUString s = "Don't do that";
		VDKUString p = s;
		p.StripChar("'"); // p value is "Dont do that"
		\endcode
		*/
		VDKUString& StripChar(const char car);
		/*!
		Same as StripChar(char) but for UTF-8 strings
		*/
		VDKUString& StripChar(const char *car);
		/*!
		Replace all 'torep' chars in the string (for Windows/Unix compatibility purpose).
		Modifies and returns the resulting VDKUString.
		\param torep		char to be replaced, '\\' (antislash) by default
		\param rep			char to replace torep, '\\' (slash) by default
		\code
		VDKUString s = "c:\\windows\\foo.txt";
		VDKUString p = s;
		p.ReplaceChar(); // p value is "c:/windows/foo.txt"
		\endcode
		*/
		VDKUString& ReplaceChar(const char torep, const char rep);
		/*!
		Same as ReplaceChar(char, char) but for UTF-8 strings
		*/
		VDKUString& ReplaceChar(const char *torep = "\\", const char *rep = "/");
		/*!
		Returns a VDKUString containing a formatted date according to
		parameters settings.
		Modifies and returns the resulting VDKUString.
		\warning Only complete dates are supported.
		That's to say days and months on two digits
		and years on 4 digits. For ex. : 02/03/2000.
		\param sep	desired separator. If 0, no separator left
		\param orig	date style staying in VDKUString buffer
		\param ret	date style to return
		\code
		VDKUString s = "12/25/2000";
		VDKUString p = s;
		p.FormatDate(0, ENG_DATE, INT_DATE); // p value is "20001225"

		VDKUString s = "12/25/2000";
		VDKUString p = s;
		p.FormatDate('-', ENG_DATE, EUR_DATE); // p value is "25-12-2000"
		\endcode
		*/
		VDKUString& FormatDate(const char sep, int orig, int ret);
		/*!
		Formats a float string to user float format (internationalization)
		Modifies and returns the resulting VDKUString.
		*/
		VDKUString& FloattoUserFormat();
		/*!
		Formats a user float string to C float format (reverse internationalization)
		Modifies and returns the resulting VDKUString.
		*/
		VDKUString& FloattoCFormat();
};

#endif




