/*
 * ===========================
 * VDK Visual Develeopment Kit
 * Version 0.4
 * October 1998
 * ===========================
 *
 * Copyright (C) 1998, Mario Motta
 * Developed by Mario Motta <mmotta@guest.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#ifndef DLIST_H
#define DLIST_H

#include <cstdlib>

// just a cosmetic capitalizing (suggestion by Ionutz Borcoman)
#define VDKListIterator VDKListiterator

/*
  ============
  VDKItem class
  ============
  */
template <class T> class VDKItem 
{

 public:
  T* x;
  VDKItem* next,*prev;
  VDKItem(T* x);
	~VDKItem();
};
/*!
  \class VDKList
  \brief Provides a reference semantic double linked list

  VDKList has a reference semantic, all managed objects are pointers to
  their original values. This calls is widely used for vdk internals.
  \par Implementation notes
  I suggest to use typedef like this:
  \code
  typedef VDKList<someClass> SomeClassList;
  typedef VDKListIterator<someClass> SomeClassListIterator;
  \endcode
*/

template <class T> class VDKList 
{
  //private:
  VDKItem<T>* head,* tail;
  int count;
  
  // append to list
  void addToTail(VDKItem<T>* i);
  // over head loading
  void addToHead(VDKItem<T>* i);
  // insert at pos
  void insertVDKItem(VDKItem<T>* i, int pos);
  
  // fetch an VDKItem at <n> position
  VDKItem<T>* fetch(int n);  
  
  // assign VDKItems to container
  void assign(VDKList& l);
  // on reference semantic copy-assign is prohibited 
  VDKList(VDKList& c);
  
  VDKList& operator=(VDKList<T>& l);

  /* class interface methods */
 public:
  /*!
    Constructor, makes an empty lis
  */
  VDKList();
  /*!
    Destructor. VDKList does not owns object pointed thus not provide 
    to their destruction. Destruction of pointed objects must be explicit.
    \code
    // delete pointed objects
    SomeClassListIterator li(list);
    for(li;li++)
    delete li.current();
    \endcode
   */
  ~VDKList();
  /*!
    Appends a pointer to type T to the list. To mantain
    reference integrity no same pointer will be added twice.
    \param t type T pointer
  */
  void add(T* t);
  /*!
    Prepends a pointer to type T to the list. To mantain
    reference integrity no same pointer will be added twice.
    \param t type T pointer
  */
  void addH(T* t);
  /*!
    Insert a pointer to type T to the list. To mantain
    reference integrity no same pointer will be added twice.
    \param t type T pointer
    \param pos ordinal position
  */
  void insertAt(T* t, int pos);
  /*!
    Membership operator, return NULL if not found
    \param x address to be searched for
   */
  T* find(T* x);
  /*!
    \internal
   */
  T* listhead();
  /*! 
    find position of type<T> object, returns ordinal position, -1 on failure
    \param x address to be searched for
  */
  int at(T* x);
  /*!
    Ordinal access operator
   */
  T* operator[](int n);
  /*!
    Remove a pointer from list
    \param x address to be removed
   */
  int  remove(T* x);
  /*!
    Returns list size
   */
  int size();
  /*!
    Flushes list
   */
  void flush();
  /*!
    \internal
   */
  VDKItem<T>* Head();
  /*!
    \internal
   */
  VDKItem<T>* Tail();
};

/*!
  \class VDKListiterator
  \brief Provides a VDKList iterator
*/
template <class T> class VDKListiterator 
{

  VDKItem<T> *head,*tail,*p;
 public:
  /*!
    Constructor
    \param a VDKList reference
   */
  VDKListiterator(VDKList<T>& c);
  /*!
    Destructor
   */
  virtual ~VDKListiterator();
  /*!
    Incremental operator (postfix)
  */
  void operator ++ ();
  /*!
    Incremental operator (infix)
  */
  void operator++(int);
  /*!
    Decremental operator (postfix)
  */
  void operator -- ();
  /*!
    Decremental operator (infix)
  */
  void operator -- (int);
  /*!
    Return 0 at the list end
   */
  operator int();
    /*!
    Goes to list head
   */
  void first();
  /*!
    Goes to list tail
   */
  void last();
  /*!
    Returns currently pointed object
   */
  T* current();
  /*!
    \internal
   */
  VDKItem<T>* Next(VDKItem<T> *t);
  /*!
    \internal
   */
  VDKItem<T>* Head();
  /*!
    \internal
   */
  T* Now(VDKItem<T> * t);
  /*!
    Rewind iterator
  */
  void restart();
};

// VDKItem class
template <class T>
VDKItem<T>::VDKItem(T* x) : x(x), next((VDKItem<T>*)0), prev((VDKItem<T>*)0) 
{
}

template <class T>
VDKItem<T>::~VDKItem()
{
}

// VDKList class
template <class T>
VDKList<T>::VDKList() : head(0), tail(0), count(0)
{
}  

template <class T>
VDKList<T>::VDKList(VDKList& c)
{
	count = 0;
	head = tail = (VDKItem<T>* )0;
	assign(c);
}

template <class T>
VDKList<T>::~VDKList()
{
	flush();
}

template <class T>
void VDKList<T>::addToTail(VDKItem<T>* i) 
{
	if(! head) head = tail = i;
	else {
		tail->next = i;
		i->prev=tail; tail = i;
	}
	count++;
}

template <class T>
void VDKList<T>::addToHead(VDKItem<T>* i) 
{
	if(! head) head = tail = i;
	else {
		head->prev = i;
		i->next = head;
		head = i;
	}
	count++;
}

template <class T>
void VDKList<T>::assign(VDKList<T>& l) 
{
	VDKListiterator<T> ci(l);
	while(ci) { add(ci.current()); ci++; }
}

template <class T>
VDKItem<T>* VDKList<T>::fetch(int n) {
  int t = 0;
  if(n >= count || n < 0) return (VDKItem<T>*) 0;
  VDKItem<T>* p = head;
  for( ;p && (t < n) ; t++, p = p->next);
  return p;
}

template <class T>
VDKList<T>& VDKList<T>::operator = (VDKList<T>& l) 
{
	// avoid l = l;
	if(this != &l) { flush(); assign(l); }
	return *this;
}

template <class T>
T* VDKList<T>::operator[](int n)
{
	return fetch(n)->x;
}

template <class T>
void VDKList<T>::add(T* t) 
{ 
	if(!find(t))
	addToTail(new VDKItem<T>(t)); 
}

template <class T>
void VDKList<T>::addH(T* t) 
{
	if(! find(t))
	addToHead(new VDKItem<T>(t)); 
}

template <class T>
void VDKList<T>::insertAt(T* t, int pos)
{
	if(!find(t))
	insertVDKItem(new VDKItem<T>(t),pos); 
}

template <class T>
T* VDKList<T>::listhead()
{
	return fetch(0)->x;
}

template <class T>
int VDKList<T>::size()
{
	return count;
}

template <class T>
VDKItem<T>* VDKList<T>::Head()
{
	return head;
}

template <class T>
VDKItem<T>* VDKList<T>::Tail()
{
	return tail;
}

template <class T>
int VDKList<T>::at(T* x) {
	register int t = 0;
	VDKItem<T>* p = head;
	for(; p && (p->x != x);p = p->next,t++) ;
	return p ? t : -1;
}

template <class T>
void VDKList<T>::flush() 
{
	VDKItem<T>*p = head;
	VDKItem<T>*p1;
	while(p) {
		p1 = p->next;
		delete p;
		p = p1;
	}
	head = tail = 0;
	count = 0;
}

template <class T>
T* VDKList<T>::find(T* x)
{
	VDKItem<T>* p;
	for(p = head; p && (p->x != x); p = p->next) ;
	return p ? p->x : (T*) 0;
}

template <class T>
int VDKList<T>::remove(T* x)
{
	VDKItem<T>* cur;
	int n = at(x);
	// object not found
	if(n < 0) return 0;
	else cur = fetch(n);
	// removing head
	if(cur == head) {
		head = cur->next;
		// one element list
		if(head != (VDKItem<T>*) 0) head->prev = (VDKItem<T>*) 0;
		else tail = (VDKItem<T>*) 0;
	}
	else { // remove tail or intermediate element
		cur->prev->next = cur->next;
		if(cur != tail) cur->next->prev = cur->prev;
		// remove tail
		else tail = cur->prev;
	}
	delete cur;
	count--;
	return 1;
}
/*
insert an item a <t> position
*/
template <class T>
void VDKList<T>::insertVDKItem(VDKItem<T>* i, int pos)
{
	int t = 0;
	VDKItem<T>* p = NULL;
	for(p = head; p && t < pos; p=p->next,t++);
	if(!p) addToTail(i);
	else if(p->prev) {
		p->prev->next = i;
		i->prev = p->prev;
		p->prev = i;
		i->next = p;
		count++;
	}
	else addToHead(i);
}

// VDKListiterator class
template <class T> VDKListiterator<T>::VDKListiterator(VDKList<T>& c)
{
	p = head = c.Head(); tail = c.Tail();
}

template <class T> VDKListiterator<T>::~VDKListiterator()
{
}

template <class T>
void VDKListiterator<T>::operator ++ ()
{
	p = p->next;
}

template <class T>
void VDKListiterator<T>::operator ++ (int)
{
	p = p->next;
}

template <class T> 
void VDKListiterator<T>::operator -- ()
{
	p = p->prev;
}

template <class T>
void VDKListiterator<T>::operator -- (int)
{
	p = p->prev;
}

template <class T>
VDKListiterator<T>::operator int()
{
	return p != (VDKItem<T>*) 0;
}

template <class T> 
void VDKListiterator<T>::first()
{
	p = head;
}

template <class T>
void VDKListiterator<T>::last()
{
	p = tail;
}

template <class T>
T* VDKListiterator<T>::current()
{
	return p->x;
}

template <class T>
VDKItem<T>* VDKListiterator<T>::Next(VDKItem<T> *t)
{
	return t->next;
}

template <class T>
VDKItem<T>* VDKListiterator<T>::Head()
{
	return head;
}

template <class T>
T* VDKListiterator<T>::Now(VDKItem<T> * t)
{
	return t->x;
}

template <class T>
void VDKListiterator<T>::restart()
{
	p = head;
}

#endif





