
/*
 * Copyright (C) 2002-2012 Edscott Wilson Garcia
 * EMail: edscott@users.sf.net
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; .
 */
#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include "rodent.h"
#include "rfm_modules.h"
#include "rodent_tip.i"
/*************************************************************************/
/*************************************************************************/

/**************************************************************/
/////////////////////   tip_event_t    /////////////////////////
/**************************************************************/
static GMutex *
get_tip_mutex(void){
    static GMutex *mutex = NULL;
    static gsize initialized = 0;
    if (g_once_init_enter (&initialized)){
	rfm_mutex_init(mutex);
      g_once_init_leave (&initialized, 1);
    }
    return mutex;
}

static GThreadPool *tip_queue;

typedef struct tip_event_t {
    gboolean 		tooltip_active;		// tooltip status   
    GtkWindow		*tooltip_window;
    record_entry_t	*tooltip_entry;  	// tooltip record_entry
    GdkPixbuf		*tooltip_pixbuf;	// tooltip image preview
    gchar 		*tooltip_text;		// tooltip status   
}tip_event_t;
//static gint current_tip_serial=0;
static tip_event_t tip_event ={FALSE, NULL, NULL, NULL, NULL};

#define MAX_TIP_THREADS MAX_PREVIEW_THREADS

static void *
trigger_tooltip_f(gpointer data){
    NOOP(stderr, "context tooltip trigget\n");
    rfm_global_t *rfm_global_p = rfm_global();
    gtk_widget_trigger_tooltip_query(rfm_global_p->window);
    return NULL;
}

static void
trigger_tooltip(void){
    TRACE("trigger_tooltip...\n");
    rfm_context_function (trigger_tooltip_f, NULL);
}


gboolean
rodent_tip_get_active (void) {
    GMutex *tip_mutex = get_tip_mutex();
    g_mutex_lock(tip_mutex);
    gboolean retval = tip_event.tooltip_active;
    g_mutex_unlock(tip_mutex);
    return retval;
}

void
rodent_hide_tip (void) {
    rfm_global_t *rfm_global_p = rfm_global();
    if (!rfm_rw_lock_reader_trylock (&(rfm_global_p->setup_lock))) return;
    rfm_rw_lock_reader_unlock (&(rfm_global_p->setup_lock));
    GMutex *tip_mutex = get_tip_mutex();
    g_mutex_lock(tip_mutex);
    tip_event.tooltip_active = FALSE;
    g_mutex_unlock(tip_mutex);
    NOOP("rodent_hide_tip...\n");
}

gboolean
rodent_tip_function (
        GtkWidget * window, 
        gint x, 
        gint y, 
        gboolean keyboard_mode, 
        GtkTooltip * tooltip,
        gpointer user_data
) {
#ifdef DEBUG_TRACE
    static gint c=0; TRACE( "rodent_tip_function %d ... \n", c++);
#endif
    rfm_global_t *rfm_global_p = rfm_global();
    if (!rfm_rw_lock_reader_trylock (&(rfm_global_p->setup_lock))){
	TRACE("rodent_tip_function(): unable to lockrfm_global_p->setup_lock!\n");
	return FALSE;
    }
    rfm_rw_lock_reader_unlock (&(rfm_global_p->setup_lock));
    g_mutex_lock(rfm_global_p->status_mutex);
    gint status = rfm_global_p->status;
    g_mutex_unlock(rfm_global_p->status_mutex);
    if (status == STATUS_EXIT) {
	TRACE("rodent_tip_function(): rfm_global_p->status == STATUS_EXIT!\n");
	return FALSE;
    }

    widgets_t *widgets_p = rfm_get_widget ("widgets_p");
    if (!widgets_p || !widgets_p->view_p) return FALSE;

    view_t *view_p = widgets_p->view_p;
    if (!rfm_view_list_lock(view_p, "rodent_tip_function")) return FALSE;
    // We now have a read lock on the view list
    gboolean retval = FALSE;
    status = view_p->flags.status;
    if (view_p->flags.status == STATUS_EXIT) {
	TRACE("rodent_tip_function(): view_p->flags.status == STATUS_EXIT!\n");
        goto done;
    }

    GMutex *tip_mutex = get_tip_mutex();
    if (!g_mutex_trylock(tip_mutex))  {
	TRACE("rodent_tip_function(): failed to lock tip_mutex!\n");
        goto done;
    }
	


    record_entry_t *en = tip_event.tooltip_entry;
    gboolean tip_not_ok = FALSE;
    if (!tip_event.tooltip_active) {
	tip_not_ok = TRUE;
	TRACE("rodent_tip_function(): xx !tip_event.tooltip_active\n");
    }
    if (!en) {
	tip_not_ok = TRUE;
	TRACE("rodent_tip_function(): xx !en\n");
    }
    else if (IS_UP_TYPE (en->type) && !g_path_is_absolute(en->path)) {
	tip_not_ok = TRUE;
	TRACE("rodent_tip_function(): xx IS_UP_TYPE (en->type) && !g_path_is_absolute(en->path)\n");
    }
    if (g_object_get_data(G_OBJECT(rfm_global_p->window), "tip_widgets_p") 
	    != (void *)widgets_p) {
	tip_not_ok = TRUE;
	TRACE("rodent_tip_function(): xx tip_widgets_p != (void *)widgets_p\n");
    }
#if 10
    //hacks
    if (!view_p->mouse_event.saturated_p && !view_p->mouse_event.label_p) {
	tip_not_ok = TRUE;
	TRACE("rodent_tip_function(): xx !view_p->mouse_event.saturated_p && !view_p->mouse_event.label_p 0x%x 0x%x\n",
		GPOINTER_TO_INT(view_p->mouse_event.saturated_p),
		GPOINTER_TO_INT(view_p->mouse_event.label_p));
    }
    if (view_p->mouse_event.label_p && !tip_event.tooltip_text) {
	tip_not_ok = TRUE;
	TRACE("rodent_tip_function(): xx view_p->mouse_event.label_p && !tip_event.tooltip_text\n");
    }
#endif
	    
    if (tip_not_ok){
	g_mutex_unlock(tip_mutex);
	goto done;
    }

    NOOP("TIP function proceeding: %s\n", en->path);
    
     if(tip_event.tooltip_pixbuf) {
	 if (tip_event.tooltip_text) {
	     gboolean create = (tip_event.tooltip_window == NULL
		     ||
		     g_object_get_data(G_OBJECT(tip_event.tooltip_window), "normal_entry") != en);
	     if (create) {
		gchar *basename = NULL;
		if (g_path_is_absolute(en->path)) {
                    gchar *g = g_path_get_basename(en->path);
                    gchar *v = rfm_utf_string(g);
                    basename = g_markup_escape_text(v, -1);
                    g_free(g);
                    g_free(v);
                } else {
                    gchar *v = rfm_utf_string(en->path);
                    basename = g_markup_escape_text(en->path, -1);
                    g_free(v);
                }

		
		tip_event.tooltip_window = 
		    GTK_WINDOW(rfm_create_tooltip_window(rfm_global_p->window, 
                            GTK_WIDGET(tip_event.tooltip_window),
			    tip_event.tooltip_pixbuf, 
			    tip_event.tooltip_text,
			    basename));
		g_free(basename);
		g_object_set_data(G_OBJECT(tip_event.tooltip_window), 
			"normal_entry", en);
	     } 
	 } else {
	    gint width = 0; 
	    gint height = 0; 
	    GdkPixbuf *pixbuf = NULL;
	    gint p_width = -1;
	    gint p_height = -1;
	    if (tip_event.tooltip_window) {
		width = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(tip_event.tooltip_window), "width")); 
		height = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(tip_event.tooltip_window), "height")); 
		pixbuf = g_object_get_data(G_OBJECT(tip_event.tooltip_window), "pixbuf");
		if (tip_event.tooltip_pixbuf) {
		    p_width = gdk_pixbuf_get_width(tip_event.tooltip_pixbuf);
		    p_height = gdk_pixbuf_get_height(tip_event.tooltip_pixbuf);
		}
	    }
	    if (tip_event.tooltip_window && 
		    pixbuf == tip_event.tooltip_pixbuf && 
		    p_width == width && p_height == height){
		// Option 1: window size has not changed. Use the same window.
		GtkWidget *image = g_object_get_data(G_OBJECT(tip_event.tooltip_window), "image"); 
		gtk_image_set_from_pixbuf (GTK_IMAGE (image), tip_event.tooltip_pixbuf);
	    } else {
		TRACE("creating new tip window...\n");
		tip_event.tooltip_window = 
		    GTK_WINDOW(rfm_create_tooltip_window(rfm_global_p->window, 
                            GTK_WIDGET(tip_event.tooltip_window),
			    tip_event.tooltip_pixbuf, 
			    NULL,
			    NULL));
	    }
        }
	NOOP(stderr, "tip_function TRUE ... \n");
	retval = TRUE;
    } else {
	NOOP(stderr, "tip_function FALSE ... en=0x%x, pixbuf=0x%x, tooltip_active=%d\n",
		GPOINTER_TO_INT(en), 
		GPOINTER_TO_INT(tip_event.tooltip_pixbuf),
		tip_event.tooltip_active);
    }
    gchar *path = g_strdup(tip_event.tooltip_entry->path);
    g_mutex_unlock(tip_mutex);
    if (rfm_population_try_read_lock(view_p, "rodent_tip_function")){
	population_t **population_pp = view_p->population_pp;
	for (; population_pp && *population_pp; population_pp++){
	    if (!(*population_pp)->en || !(*population_pp)->en->path) continue;
	    if (strcmp((*population_pp)->en->path, path)==0){
		(*population_pp)->flags &= (POPULATION_TIP_BUSY ^ 0xffffffff);
		break;
	    }
	}
	rfm_population_read_unlock(view_p, "rodent_tip_function");
    }
    g_free(path);		
done:
    rfm_view_list_unlock("rodent_tip_function");
    return retval;
}

static void
signal_pool_f(void *data, void *pool_data){
    // This is a stop light for initial setup:
    rfm_global_t *rfm_global_p = rfm_global();
    rfm_rw_lock_reader_lock (&(rfm_global_p->setup_lock));
    rfm_rw_lock_reader_unlock (&(rfm_global_p->setup_lock));

    void **arg = data;
    view_t *view_p = arg[1];
    void *function_data = arg[2];
    g_free(arg);

    g_mutex_lock(rfm_global_p->status_mutex);
    gint status = rfm_global_p->status;
    g_mutex_unlock(rfm_global_p->status_mutex);
    if (status == STATUS_EXIT) return;

    const gchar *dbg_text = "signal_pool_f (tip)";

	
    if (view_p->flags.status == STATUS_EXIT) {
        return;
    }
    
    rfm_thread_reference(view_p, g_thread_self(), dbg_text);
    rodent_activate_tip_f(view_p, function_data);
    rfm_thread_unreference(view_p, g_thread_self());
    return;
    
}


typedef struct activate_tip_t {
    view_t *view_p;
    population_t *thread_population_p;
    gint saturation_serial;
    gboolean normal_tip;
} activate_tip_t;

static void free_activate_tip(activate_tip_t *activate_tip_p){
    rfm_destroy_entry(activate_tip_p->thread_population_p->en);
    if (GDK_IS_PIXBUF(activate_tip_p->thread_population_p->preview_pixbuf)){
	g_object_unref(activate_tip_p->thread_population_p->preview_pixbuf);
    }

    g_free(activate_tip_p->thread_population_p->icon_id);
    g_free(activate_tip_p->thread_population_p);
    g_free(activate_tip_p);
}

    
/*************************************************************************/
/******************* tip functions ***************************************/
/*************************************************************************/


//static
//void show_tip (population_t * population_p);

static gchar *
path_info (const population_t *population_p, record_entry_t *en, const gchar *warning) {
    gchar *s1 = NULL, *s2 = NULL;
    gchar *info = NULL;
    if(!en || !en->path)
        return NULL;
    if(IS_ROOT_TYPE (en->type) && 
	    !rfm_g_file_test (en->path, G_FILE_TEST_EXISTS))
        return NULL;


    //gchar *b = g_path_get_basename (en->path);
    //rfm_chop_excess (b);
    //g_free (b);
    if(IS_LOCAL_TYPE(en->type) && IS_SLNK (en->type)) {
	NOOP(stderr, "local lnk  type...\n");
        gchar lpath[_POSIX_PATH_MAX + 1];
        memset (lpath, 0, _POSIX_PATH_MAX + 1);
        if(readlink (en->path, lpath, _POSIX_PATH_MAX) > 0) {
            gchar *v = rfm_utf_string(lpath);
            gchar *escaped_markup = g_markup_escape_text(v, -1);
            g_free(v);
            gchar *q = rfm_utf_string (escaped_markup);
            g_free(escaped_markup);
            gchar *linkto=g_strdup_printf (_("Link to %s"), q);
            s1 = g_strdup_printf ("%s\n<i>%s</i>\n\n", linkto, warning);
            g_free(linkto);
            g_free (q);
        }
    } else if (population_p) {
        gchar *p = g_strdup_printf ("<i>%s</i>\n\n", warning);
        if(POPULATION_MODULE(population_p)) {
	    gchar *plugin_info=rfm_void(PLUGIN_DIR, POPULATION_MODULE(population_p), "plugin_info");
	    if (plugin_info) {
		s1 = g_strconcat (p, plugin_info, NULL);
		g_free (p);
		g_free(plugin_info);
	    } else {
		gchar *q = g_strdup_printf ("%s: %s", _("Installed Plugins"), 
			POPULATION_MODULE(population_p));
		s1 = g_strconcat (p, q, NULL);
		g_free (p);
		g_free (q);
	    }
        } else {
            s1 = p;
        }
    }

    gchar *s12 = NULL;
	
	
    
    rfm_set_mime_dtype(en);
    if (!en->mimetype) {
	NOOP(stderr, "getting mimetype: %s\n", en->path);
	en->mimetype = MIME_type(en->path, en->st); 
    }
    
    if (IS_LOCAL_TYPE(en->type)){
	if (!en->mimemagic || strcmp(en->mimemagic, _("unknown"))==0) {
	    gchar *old = en->mimemagic;
	    NOOP(stderr, "getting magic type: %s\n", en->path);
	    en->mimemagic = rfm_rational(RFM_MODULE_DIR, "mime", en, "mime_magic", "mime_function"); 
	    g_free(old);
	    
	    if (!en->mimemagic) en->mimemagic = g_strdup(_("unknown"));
	}
	if (!en->filetype || strcmp(en->filetype, _("unknown"))==0) {
	    NOOP(stderr, "getting file type: %s\n", en->path);
	    gchar *old = en->filetype;
	    en->filetype = rfm_rational(RFM_MODULE_DIR, "mime", en, "mime_file", "mime_function"); 
	    g_free(old);
	    if (!en->filetype) en->filetype = g_strdup(_("unknown"));
	}
	if (!en->encoding || strcmp(en->encoding, _("unknown"))==0) {
	    gchar *old = en->encoding;
	    NOOP(stderr, "getting file encoding: %s\n", en->path);
	    en->encoding = rfm_rational(RFM_MODULE_DIR, "mime", en, "mime_encoding", "mime_function"); 
	    g_free(old);
	    if (!en->encoding) en->encoding = g_strdup(_("unknown"));
	}

    } 
    else {
	NOOP(stderr, "Not a local type: %s\n", en->path);
    }
    if ((en->mimetype && strstr(en->mimetype, "x-trash")) || 
	en->path[strlen(en->path)-1] =='~' ||
	en->path[strlen(en->path)-1] =='%' ) {
	g_free(en->filetype);
	en->filetype = g_strdup(_("Backup file"));
    }
    s12 = g_strdup_printf("<b>%s</b>: %s\n<b>%s</b> (freedesktop): %s\n<b>%s</b> (libmagic): %s\n<b>%s</b>: %s\n\n",
	    _("File Type"), en->filetype,
	    _("MIME Type"), (en->mimetype)?en->mimetype:_("unknown"),
	    _("MIME Type"), en->mimemagic,
	    _("Encoding"), en->encoding);


    if(en->st) {
        gchar *grupo=rfm_group_string(en->st);
        gchar *owner=rfm_user_string(en->st);
        gchar *tag = rfm_sizetag ((off_t) en->st->st_size, -1);

        //    gchar *ss= rfm_time_to_string(en->st->st_mtime);   

        gchar *t = g_path_get_dirname (en->path);
        gchar *v = rfm_utf_string(t);
        gchar *escaped_markup = g_markup_escape_text(v, -1);
        g_free(v);
        gchar *dirname = rfm_utf_string (escaped_markup);
        g_free(t);
        g_free(escaped_markup);
	gchar *mode_string=rfm_mode_string (en->st->st_mode);
        s2 = g_strdup_printf (
                "<b>%s/%s</b>: %s/%s\n<b>%s</b>: %s\n<b>%s</b>: %s\n\n<b>%s</b>: %s",
                 _("Owner"),_("Group"), owner, grupo,
                _("Permissions"), mode_string,
                _("Folder"), dirname, 
                _("Size"),  tag);

        //    g_free(q);
        g_free (owner);
        g_free (grupo);
        g_free (tag);
        g_free (dirname);
        g_free (mode_string);

	gchar buf[1024];

	gchar *date_string=rfm_date_string(en->st->st_ctime);

        sprintf (buf, "<b>%s :</b> %s", _("Status Change"), date_string);
	g_free(date_string);

        gchar *s3 = g_strconcat (s2, "\n", buf, NULL);
        g_free (s2);
        s2 = s3;

	date_string=rfm_date_string(en->st->st_mtime);
        sprintf (buf, "<b>%s</b> %s", _("Modification Time :"), date_string);
	g_free(date_string);


        s3 = g_strconcat (s2, "\n", buf, NULL);
        g_free (s2);
        s2 = s3;

	date_string=rfm_date_string(en->st->st_atime);
        sprintf (buf, "<b>%s</b> %s", _("Access Time :"), date_string);
	g_free(date_string);

        s3 = g_strconcat (s2, "\n", buf, NULL);
        g_free (s2);
        s2 = s3;

	gchar *hard_links = g_strconcat(_("Links")," (", _("hard"), ")", NULL);
        s3 = g_strdup_printf ("%s\n\n<b>%s</b>: %ld\n<b>%s</b>: %ld",
		s2, hard_links,
		(long)en->st->st_nlink, _("Inode"), (long)en->st->st_ino);
	g_free(hard_links);
		
        g_free (s2);
        s2 = s3;
    }

    if(!s1)
        s1 = g_strdup ("");
    if(!s2)
        s2 = g_strdup ("");
#ifdef CORE
    info = g_strdup_printf("%s%s%s\n<b>Icon id:</b> %s",
	    s1, s12, s2, 
	    (population_p)?population_p->icon_id:"See label tooltip");
#else
    info = g_strconcat (s1, s12, s2, NULL);
#endif
    g_free (s1);
    g_free (s2);
    g_free (s12);
    return info;
}


gchar *
rodent_get_tip_text (view_t * view_p, const population_t * population_p, record_entry_t *en, const gchar * prepend_txt) {
    const gchar *warning = "";
    if (view_p->en && view_p->en->module == NULL){
	if (IS_LOCAL_TYPE(en->type)){
	    warning = _("Local File");
	} else {
	    warning = _("Remote File");
	}
    } else {
	warning = _("Plugin services");
    }
    gchar *g=NULL;
    if (view_p->en && view_p->en->module == NULL) {
	if (!IS_LOCAL_TYPE(en->type)){
	    gchar *info = path_info (population_p, en, warning);
	    g = g_strdup ((info)?info:"");
	    g_free (info);
	} else if(en && IS_SDIR (en->type)) {
	    gint files = xfdir_count_files (en->path);
	    gint hidden = xfdir_count_hidden_files (en->path);

	    if(files) {
		warning = _("Local Directory");
		gchar *files_string = g_strdup_printf (ngettext (" (containing %'d item)", " (containing %'d items)", files),files);
	
		gchar *plural_string = 
		    g_strdup_printf(ngettext ("%'u item","%'u items",hidden), hidden);
		gchar *hidden_string = 
		    g_strdup_printf ("%s: %s.",_("Hidden"), plural_string);
		g_free(plural_string);

		g = g_strdup_printf ("%s\n%s\n%s", warning, files_string, hidden_string);
		g_free(hidden_string);
		g_free (files_string);
		gchar *info = path_info (population_p, en, g);
		g_free(g);
		g = info;
	    } else if (rfm_g_file_test(en->path, G_FILE_TEST_EXISTS)){
		warning = _("Local Directory");
		g = g_strdup_printf ("%s\n(%s)", warning, _("The location is empty."));
		gchar *info = path_info (population_p, en, g);
		g_free(g);
		g = info;
	    } else {
		warning = _("Local Directory");
		gchar *info = path_info (population_p, en, warning);
		g = info;
	    }
	} else {
	    gchar *info = path_info (population_p, en, warning);
	    g = g_strdup ((info)?info:"");
	    g_free (info);
	}
    } else if (view_p->en == NULL  
	    && en && !en->module) {
        gchar *v = NULL;
	if (en->tag) {
            v = rfm_utf_string(en->tag);
        } else {
            v = rfm_utf_string(en->path);
        }
        g = g_markup_escape_text(v, -1);
        g_free(v);
    }


    // modules can prepend text to the tooltip
    if(prepend_txt) {
        gchar *v = rfm_utf_string(prepend_txt);
        gchar *pt = g_markup_escape_text(v, -1);
        g_free(v);
        gchar *gg = g_strconcat (prepend_txt, "\n", (g)?g:"", NULL);
        g_free (pt);
        g_free (g);
        g = gg;
    }
    return g;
}

static gboolean 
is_normal_condition(population_t *population_p){
    if (!population_p) return FALSE;
    view_t *view_p=population_p->view_p;

    // Preview already done.
    if (population_p->preview_pixbuf) {
	return FALSE;
    }

    // User does not want previews in the directory.
    if (!SHOWS_IMAGES(view_p->flags.preferences)) {
	 NOOP("normal tip; !SHOWS_IMAGES(view_p->flags.preferences)\n");
	return TRUE;
    }
    // Short circuit.
    if (population_p->flags & POPULATION_NORMAL_TIP) {
	 NOOP(stderr,  "* normal tip; short circuit:%s \n", population_p->en->path);
	return TRUE;
    }
    
    NOOP(stderr,  "* FAILED short circuit:%s \n", population_p->en->path);
    
    if (population_p->en==NULL || population_p->en->st==NULL) {
	 NOOP("normal tip; population_p->en==NULL || population_p->en->st==NULL\n");
	return TRUE;
    }
 
    // Modules will have normal tip.
    if (POPULATION_MODULE(population_p)) {
	 NOOP("normal tip; POPULATION_MODULE\n");
	return TRUE;
    }

    
    // For some reason Rodent has previously determined that preview
    // is not possible for the population element.
    if(g_object_get_data (G_OBJECT (view_p->widgets.paper), "normal_tip")) {
	 NOOP("normal tip; g_object_get_data (G_OBJECT (view_p->widgets.paper), normal_tip) is set for population item\n");
	return TRUE;
    }

    gboolean retval = TRUE;
    // Directories are now set to preview.
    if (IS_SDIR(population_p->en->type) ) retval = FALSE;

    // Empty files are easy to preview with a blank page.
    // (This, of course, if the stat we're looking at is not an empty stat)
    else if (population_p->en->st->st_size == 0){
	if (population_p->en->st->st_ino != 0) retval = FALSE; 
	else{
	     NOOP("normal tip; (population_p->en->st->st_size == 0\n");
	    retval = TRUE;
	}
    }
     
    // Application/xxx mimetypes are now set to preview.
    else if (population_p->en->mimetype || population_p->en->mimemagic){
	const gchar *type = population_p->en->mimetype;
	if (!type || strcmp(type, _("unknown"))==0) type = population_p->en->mimemagic;
	if (!type || strcmp(type, _("unknown"))==0) {
	     NOOP("normal tip; strcmp(type, _(\"unknown\"))==0\n");
	    return TRUE;
	}
	// XXX I'm not sure at this time whether desktop files
	// should or should not have previews...
	//if(strcmp(population_p->en->mimetype, "application/x-desktop")==0){
	//    return TRUE;
	//}
	if(strcmp(type, "application/x-executable")==0){
	     NOOP("normal tip; application/x-executable\n");
	    return TRUE;
	}
	if (strncmp(type, "application/",strlen("application/"))==0){
	    retval = FALSE;
	}
	else if (strncmp(type,"image/",strlen("image/"))==0){
	    retval = FALSE;
	}
	else if (strncmp(type, "text/",strlen("text/"))==0){
	    retval = FALSE;
	}
    }

    // Images should preview, even if mimetype not correctly set.
    else if (rfm_entry_is_image (population_p->en)) retval = FALSE;

    // Open documents should preview, even if mimetype not correctly set.
    else if (population_p->en->mimemagic && strstr(population_p->en->mimemagic, "opendocument")) {
	retval = FALSE;
    }

    // Are we capable of doing the preview?
    if(rfm_void(RFM_MODULE_DIR, "mime",  "module_active") == NULL) {
	 NOOP( "* capable of doing the preview is false: %s\n", 
		population_p->en->path);
	return TRUE;
    }

    // User wants previews in the directory.
    if (SHOWS_IMAGES(view_p->flags.preferences)) {
	NOOP("normal tip; ! (view_p->flags.preferences & __SHOW_IMAGES)\n");
	return FALSE;
    }
    // Default is the normal tip (for the time being).
     NOOP( "* Default is the normal tip: %s\n", 
		population_p->en->path);
    return retval;
}

static gboolean
set_tooltip_info(view_t *view_p, const gchar *text, const record_entry_t *en, GdkPixbuf *tip_pixbuf){
    if (!rfm_population_try_read_lock(view_p, "set_tooltip_info")) {
	NOOP(stderr, "set_tooltip_info !rfm_population_try_read_lock(view_p)) \n");
	return FALSE;
    }

    rfm_global_t *rfm_global_p = rfm_global();
    if(tip_pixbuf) g_object_ref(tip_pixbuf);
    record_entry_t *tip_entry = (en)?rfm_copy_entry (en):NULL;
    gchar *tip_text = (text)?g_strdup(text):NULL;
    
    GMutex *tip_mutex = get_tip_mutex();
    g_mutex_lock(tip_mutex);
    // assign new tip values.
    gchar *oldtext=tip_event.tooltip_text;
    tip_event.tooltip_text = tip_text;

    record_entry_t *olden=tip_event.tooltip_entry;
    tip_event.tooltip_entry = tip_entry;

    GdkPixbuf *oldpixbuf = tip_event.tooltip_pixbuf;
    tip_event.tooltip_pixbuf = tip_pixbuf; 

    tip_event.tooltip_active = TRUE;
    const gchar *path = (tip_event.tooltip_entry)?(tip_event.tooltip_entry->path):NULL;
    g_object_set_data(G_OBJECT(rfm_global_p->window), 
	    "tip_widgets_p", rfm_get_widget("widgets_p"));
    g_mutex_unlock(tip_mutex);

    g_free (oldtext);
    population_t **population_pp = view_p->population_pp;
    for (; population_pp && *population_pp; population_pp++){
        if (!(*population_pp)->en || !(*population_pp)->en->path) continue;
        if (path && strcmp((*population_pp)->en->path, path)==0){
            (*population_pp)->flags &= (POPULATION_TIP_BUSY ^ 0xffffffff);
            break;
        }
    }
    rfm_population_read_unlock(view_p, "set_tooltip_info");
    
    rfm_destroy_entry(olden);
    if (oldpixbuf && G_IS_OBJECT(oldpixbuf)) g_object_unref(oldpixbuf);

    return TRUE;
}

static gboolean
normal_tip(const population_t * population_p){
    view_t *view_p = population_p->view_p;
    gchar *module_txt = NULL;
    if (!view_p->en && POPULATION_MODULE(population_p)) {
	module_txt =
	    rfm_void (PLUGIN_DIR, POPULATION_MODULE(population_p), "module_entry_tip");	
        if (!module_txt) {
            module_txt = g_strdup_printf("FIXME: module_entry_tip() in plugin %s returns null.",
                    POPULATION_MODULE(population_p));
        }
    } else if(POPULATION_MODULE(population_p)) {
        module_txt =	
	    rfm_natural (PLUGIN_DIR, POPULATION_MODULE(population_p), 
			population_p->en, "item_entry_tip");
    }
    NOOP(stderr, "Not a module determined tip...\n");

    GdkPixbuf *tip_pixbuf=NULL;
    gchar *tip_text=NULL;
    //record_entry_t *tip_entry;
    if (module_txt){
	tip_text = g_strdup(module_txt);
    } else {
	tip_text = rodent_get_tip_text (view_p, population_p, population_p->en, module_txt);
    }
    g_free (module_txt);

    // This would put the preview instead of the icon in the text 
    // tip window. This seemed like a good idea at first, but later
    // on I became disenchanted.
    //gboolean icon_tips = (getenv ("RFM_ENABLE_TIPS")!=NULL && strlen (getenv ("RFM_ENABLE_TIPS")));
    //if (!icon_tips) 
    //   tip_pixbuf = rfm_natural(RFM_MODULE_DIR, "mime", 
    //		(void *)population_p, "mime_preview"); // refs
    

    if (!tip_pixbuf) {
	const gchar *icon_id = population_p->icon_id;
	if (!icon_id && population_p->en) icon_id = population_p->en->mimetype;
	if (!icon_id){
	    tip_pixbuf = rfm_get_pixbuf ("rodent", BIG_ICON_SIZE); //refs
	} else {
	    tip_pixbuf = rfm_get_pixbuf (icon_id, BIG_ICON_SIZE); //refs
	}
    }
    
    NOOP(stderr, "iconid=%s\n", population_p->icon_id);
    gboolean retval = set_tooltip_info(view_p, tip_text, population_p->en, tip_pixbuf);
    if(tip_pixbuf && G_IS_OBJECT(tip_pixbuf)) g_object_unref(tip_pixbuf);
    
    g_free(tip_text);
    return retval;
}


static gboolean
set_tooltip_pixbuf(view_t *view_p, population_t *population_p){
    return set_tooltip_info(view_p, NULL, population_p->en, population_p->preview_pixbuf);
}

static population_t*
get_population_p(activate_tip_t *activate_tip_p){
    // This needs a readlock...
    if (!rfm_population_read_lock(activate_tip_p->view_p, "get_population_p")){
	DBG("get_population_p() does not get population readlock\n");
	return NULL;
    }
    // find population item
    population_t **pp=activate_tip_p->view_p->population_pp;
    gboolean found =FALSE;
    population_t *population_p=*pp;
    for (; pp && *pp; pp++){
	 population_p=*pp;
	if (!population_p->en) continue;
	if (strcmp(population_p->en->path, activate_tip_p->thread_population_p->en->path)==0)
	{
	    found=TRUE;
	    break;
	}
    }

    rfm_population_read_unlock(activate_tip_p->view_p, "get_population_p");
    if (found) return population_p;
    return NULL;
}

static gboolean
tip_preview (gpointer data) {
    gboolean retval = FALSE;
    gboolean status = FALSE;
     NOOP("TIP: tip_preview NOW...\n");
    activate_tip_t *activate_tip_p=data;
    view_t *view_p = activate_tip_p->view_p;
    
    // Do the real preview.
    GdkPixbuf *result=rfm_natural(RFM_MODULE_DIR, "mime", 
	    activate_tip_p->thread_population_p, "mime_preview"); //refs

    if (!rfm_view_list_lock(view_p, "tip_preview")) return FALSE;

    if (result && GDK_IS_PIXBUF(result)) {
 	if (view_p->mouse_event.saturated_p && view_p->flags.saturation_serial == activate_tip_p->saturation_serial) 
           NOOP("rodent_mouse: SHOW_TIP: population_p->preview_pixbuf OK!\n");
	{ 
	    retval = 
		set_tooltip_info(view_p, NULL, activate_tip_p->thread_population_p->en, result);
	} 
	if (!retval) goto done;
    } 

    if (!rfm_population_read_lock(view_p, "tip_preview")) goto done;
    population_t *population_p=get_population_p(activate_tip_p);

    if (population_p) {
	if (!result) {
	    // normal tip is set to short circuit further attempts.
	    population_p->flags |= POPULATION_NORMAL_TIP;
	} else {
	    if (population_p->preview_pixbuf
		    && G_IS_OBJECT(population_p->preview_pixbuf)) {
		g_object_unref(population_p->preview_pixbuf);
	    }
	    population_p->preview_pixbuf=result;
	    g_object_ref(population_p->preview_pixbuf);
	    population_p->flags |= POPULATION_PREVIEW_DONE;
	}
    }

    
    NOOP(stderr, "rodent_mouse: SHOW_TIP: tip_preview DONE...\n");
    status = TRUE;
    rfm_population_read_unlock(view_p, "tip_preview");
done:
    rfm_view_list_unlock("tip_preview");
    // eliminate the reference which comes from mime-mouse-magic.i
    if(result && G_IS_OBJECT(result))  g_object_unref(result); 
    return status;
}


void *
rodent_activate_tip_f(view_t *view_p, void *data){
    // stop other unused threads. This will effectively
    // leave a single threadpool thread running, at least.
    if (g_thread_pool_get_num_unused_threads() > 2) {
	g_thread_pool_stop_unused_threads();
    }
    if (!rfm_view_list_lock(view_p, "rodent_activate_tip_f")) return NULL;
 
    TRACE("rodent_activate_tip_f\n");
    activate_tip_t *activate_tip_p=data;
    if (activate_tip_p->thread_population_p == NULL 
	    ||activate_tip_p->thread_population_p->en == NULL){
	g_error("rodent_activate_tip_f(): !population_p || !population_p->en\n");
    }



    NOOP(stderr, "TIP: activate_tip_thread: 0x%x: %s\n", GPOINTER_TO_INT(g_thread_self()),
	    activate_tip_p->thread_population_p->en->path);

    if (activate_tip_p->normal_tip || is_normal_condition(activate_tip_p->thread_population_p)) {
	NOOP(stderr, "normal tip\n");
        if (normal_tip(activate_tip_p->thread_population_p)){
	    NOOP(stderr, "normal tip triggered\n");
	    trigger_tooltip();
	}
    } 
    else {
	NOOP(stderr, "preview tip\n");
	 // is the work already done?
	gboolean preview_ok = (activate_tip_p->thread_population_p->preview_pixbuf != NULL);

	if(preview_ok) {
	    if (set_tooltip_pixbuf(view_p, activate_tip_p->thread_population_p)){
		trigger_tooltip();
	    }
	    TRACE( "DONE: image tip from cache 0x%x: %s\n", GPOINTER_TO_INT(g_thread_self()), activate_tip_p->thread_population_p->en->path);
	} else {
	    NOOP(stderr, "entering tip_preview NOW\n");
	    if (tip_preview((gpointer) activate_tip_p)){
		TRACE( "DONE: image tip 0x%x\n", GPOINTER_TO_INT(g_thread_self()));
		trigger_tooltip();
	    }
	}
    }

    // This needs a readlock...
    while (!rfm_population_read_lock(view_p, "rodent_activate_tip_f")) rfm_threadwait();
    population_t *population_p=get_population_p(activate_tip_p);
    if (population_p){
	population_p->flags &= (POPULATION_PREVIEW_BUSY ^ 0xffffffff);	
    }
    rfm_population_read_unlock(view_p, "rodent_activate_tip_f");
    free_activate_tip(activate_tip_p);
    rfm_view_list_unlock("rodent_activate_tip_f");
    return NULL;
}



void 
rodent_activate_tip(view_t *view_p, population_t *population_p, gboolean image_tip){
    if(population_p == NULL || !population_p->en) return;
        
    rfm_global_t *rfm_global_p = rfm_global();
    if (!rfm_rw_lock_reader_trylock (&(rfm_global_p->setup_lock)))
	return;
    rfm_rw_lock_reader_unlock (&(rfm_global_p->setup_lock));

    // if item has yet to be stat'ed, don't do a preview
    // since that would give you an empty file preview
    // which would be incorrect.
    // Probability that the inode number is actually zero
    // is practically nil.

    if (population_p->en && population_p->en->st && g_file_test( population_p->en->path, G_FILE_TEST_EXISTS) && !population_p->en->st->st_ino){
	TRACE("skipping preview for %s until item is stat'ed\n", 
		population_p->en->path);
	return;
    }

    // These two tests are done later down the line, since a new
    // preview will be returned if conditions determine that
    // the actual preview is out of date:
    /*if (population_p->preview_pixbuf) {
	NOOP(stderr, "return on population_p->preview_pixbuf\n");
	return;
    }*/
    /*if ((population_p->flags  & POPULATION_PREVIEW_DONE)) {
	NOOP(stderr, "return on POPULATION_PREVIEW_DONE\n");
	return;
    }*/

    if ((population_p->flags  & POPULATION_TIP_BUSY)) {
	NOOP(stderr, "return on POPULATION_TIP_BUSY\n");
	return;
    }
    if ((population_p->flags  & POPULATION_PREVIEW_BUSY)) {
	NOOP(stderr, "return on POPULATION_PREVIEW_BUSY\n");
	return;
    }
    widgets_t *widgets_p = &(view_p->widgets);
   
    population_p->flags |= POPULATION_TIP_BUSY;
    population_p->flags |= POPULATION_PREVIEW_BUSY;


    if (!tip_queue) {
	tip_queue = rfm_thread_queue_new(signal_pool_f, NULL, MAX_TIP_THREADS);
    }


    activate_tip_t *activate_tip_p=
	(activate_tip_t *)malloc(sizeof(activate_tip_t));
    if (!activate_tip_p) g_error("malloc: %s", strerror(errno));
    activate_tip_p->view_p=view_p;
    activate_tip_p->saturation_serial=view_p->flags.saturation_serial;
    activate_tip_p->thread_population_p = 
	(population_t *)malloc(sizeof(population_t));
    if (!activate_tip_p->thread_population_p) 
	g_error("malloc: %s", strerror(errno));
    memcpy(activate_tip_p->thread_population_p, population_p, sizeof(population_t));
    activate_tip_p->thread_population_p->icon_id = g_strdup(population_p->icon_id);

    if (!image_tip) activate_tip_p->normal_tip = TRUE; 		
    else activate_tip_p->normal_tip = FALSE;   
    
    /*if (!image_tip){
	activate_tip_p->thread_population_p->flags |= POPULATION_NORMAL_TIP;
    } else if (!(population_p->flags & POPULATION_NORMAL_TIP)){
	activate_tip_p->thread_population_p->flags &=
	    (POPULATION_NORMAL_TIP ^ 0xffffffff);
    }*/
    

    rfm_set_mime_dtype(population_p->en);
    if (population_p->en && !population_p->en->mimetype) {
	population_p->en->mimetype = rfm_rational(RFM_MODULE_DIR, "mime", population_p->en, "mime_type", "mime_function");
	if (!population_p->en->mimetype) {
	    population_p->en->mimetype = g_strdup(_("unknown"));
	}
    }
    if (population_p->en && IS_LOCAL_TYPE(population_p->en->type)
		    && !population_p->en->mimemagic){
		population_p->en->mimemagic =
		    rfm_rational(RFM_MODULE_DIR, "mime", population_p->en,
			    "mime_magic", "mime_function");
    }
    if (population_p->en && !population_p->en->mimemagic){
	population_p->en->mimemagic = g_strdup(_("unknown"));
    }	
    activate_tip_p->thread_population_p->en = 
	rfm_copy_entry(population_p->en);

    if (GDK_IS_PIXBUF(population_p->preview_pixbuf)){
	g_object_ref(population_p->preview_pixbuf);
    }
    g_object_set_data(G_OBJECT(rfm_global_p->window), "tip_widgets_p", widgets_p);

    rfm_threadqueue_push(tip_queue, 0, view_p, activate_tip_p);

}

void 
rodent_clear_tooltip(void){
    GMutex *tip_mutex = get_tip_mutex();
    g_mutex_lock(tip_mutex);
    // assign new tip values.
    g_free(tip_event.tooltip_text);
    rfm_destroy_entry(tip_event.tooltip_entry);
    if (tip_event.tooltip_pixbuf && G_IS_OBJECT(tip_event.tooltip_pixbuf)) g_object_unref(tip_event.tooltip_pixbuf);
    g_mutex_unlock(tip_mutex);
}
