
/*
 * Copyright (C) 2002-2012 Edscott Wilson Garcia
 * EMail: edscott@users.sf.net
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; 
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif


#include "rodent.h"
#include "rfm_modules.h"
#include "rodent_gridview.i"

static void
pathbar_go(GtkButton * button, gpointer pathbar){
    GList *children_list = gtk_container_get_children(GTK_CONTAINER(pathbar));
    GList *children = children_list;
    for (;children && children->data; children=children->next){
        if (button == children->data){
            gchar *path = g_object_get_data(G_OBJECT(button), "path");
            widgets_t *widgets_p = rfm_get_widget("widgets_p");        
            if (g_file_test(path, G_FILE_TEST_IS_DIR)){
                NOOP("PATHBAR_GO: %s\n", path);
                record_entry_t *target_en=NULL;
                if (strcmp("RFM_ROOT", path)) target_en = rfm_stat_entry(path, 0);
                if (!rodent_refresh (widgets_p, target_en)) rfm_destroy_entry(target_en);
            } else {
                if (!rodent_refresh (widgets_p, NULL)) continue;
            }
        } 
    }
}


GtkWidget *rodent_new_pathbar(void){
    GtkWidget *pathbar = rfm_hbox_new(FALSE,0);
    g_object_set_data(G_OBJECT(pathbar), "callback", pathbar_go);
    //GtkWidget *pb_button = rfm_pathbar_button("xffm/emblem_computer", NULL);
    GtkWidget *pb_button = rfm_pathbar_button( NULL, ".");

    //g_object_set (pb_button, "can-focus", FALSE, "relief", GTK_RELIEF_NONE, NULL);
    gtk_box_pack_start (GTK_BOX (pathbar), pb_button, FALSE, FALSE, 0);
    g_object_set_data(G_OBJECT(pb_button), "name", g_strdup("RFM_ROOT"));
    g_signal_connect (G_OBJECT(pb_button) , "clicked", G_CALLBACK (pathbar_go), pathbar);
    gtk_widget_show(pb_button);
  /*  pb_button = rfm_pathbar_button(NULL, "<");
    g_object_set_data(G_OBJECT(pb_button), "name", g_strdup("<"));
    //gtk_widget_show(pb_button);
    gtk_box_pack_start (GTK_BOX (pathbar), pb_button, FALSE, FALSE, 0);*/
    return pathbar;
}

void 
rodent_new_gridview(widgets_t *widgets_p, const gchar *in_path){
    //rfm_global_t *rfm_global_p = rfm_global();
    view_t *view_p = widgets_p->view_p;
    gchar *exec = (view_p->flags.type != DESKVIEW_TYPE)?"rodent-fm":"rodent";
    NOOP(stderr, "in_path: %s\n", in_path);

    gchar *path=NULL;
    if (in_path!=NULL) {
	if (strncmp(in_path,"rodent-plug",strlen("rodent-plug"))==0){
	    exec = "rodent-plug";
            path = g_strdup(in_path + strlen("rodent-plug"));
            g_strstrip(path);
	} else {
            path = g_strdup(in_path);
        }
    }
    gchar *argv[]={exec, path, NULL};
    NOOP(stderr, "child_constructor: %s %s\n", argv[0], (argv[1])?argv[1]:"NULL");
    NOOP("child_constructor: %s \"%s\"\n", argv[0], (argv[1])?argv[1]:"NULL");
    rfm_thread_run_argv (widgets_p, argv, FALSE);
}

gint 
rodent_text_color(const population_t *population_p){
    gint text_color = TEXT_COLOR;
    if (!population_p) return text_color;
    //view_t *view_p=population_p->view_p;
    if(population_p->en && population_p->en->path && !g_utf8_validate (population_p->en->path, -1, NULL)){
        text_color = INVALID_UTF8_TEXT_COLOR;
    }
    if(population_p->flags & LABEL_SATURATED){
//        text_color = SATURATE_LABEL_COLOR;
    }
    if(population_p->flags & POPULATION_SELECTED){
        text_color = SELECT_TEXT_COLOR;
    }
    else if(population_p->flags & POPULATION_SATURATED){
//        text_color = SATURATE_TEXT_COLOR;
        text_color = SELECT_TEXT_COLOR;
    }
    return text_color;
}


gboolean
rodent_set_upper_adjustment (view_t * view_p, int width, int height) {

    GtkAdjustment *adjustment;
    NOOP( "rodent_set_upper_adjustment(): %d x %d = %d\n", width, height, width * height);
    if(!view_p || !view_p->widgets.paper) {
        DBG ("!view_p || !view_p->widgets.paper\n");
        return TRUE;
    }

    if(!CELLWIDTH(view_p)) {
        NOOP ("!CELLWIDTH(view_p)\n");
        return TRUE;
    } else {
        NOOP ("CELLWIDTH(view_p)=%d\n", CELLWIDTH(view_p));
    }

    NOOP ("rodent_set_upper_adjustment:rows and columns: %d,%d\n", GRID_ROWS(view_p), GRID_COLUMNS(view_p));
    GtkScrolledWindow *scrolled_window = g_object_get_data(G_OBJECT(view_p->widgets.paper), "scrolled_window");
    if(GTK_IS_SCROLLED_WINDOW (scrolled_window)) {
        adjustment = gtk_scrolled_window_get_vadjustment (scrolled_window);
        NOOP ("adjustment upper=%lf, adjustment pagesize=%lf\nf",  gtk_adjustment_get_upper(adjustment), gtk_adjustment_get_page_size(adjustment));
        gtk_adjustment_set_upper(adjustment, GRID_ROWS(view_p) * CELLHEIGHT(view_p));
    }
    return TRUE;
}

// this arbitrarily set scroll position
void
rodent_set_scroll (view_t * view_p, double scroll_position) {
    NOOP (stderr, "SCROLL: scroll_position = %lf\n", scroll_position);
    GtkScrolledWindow *scrolled_window = g_object_get_data(G_OBJECT(view_p->widgets.paper), "scrolled_window");
    if(!GTK_IS_SCROLLED_WINDOW (scrolled_window))
        return;

    double upper = gtk_adjustment_get_upper (
	    gtk_scrolled_window_get_vadjustment (
		scrolled_window));
    double page = gtk_adjustment_get_page_size (
	    gtk_scrolled_window_get_vadjustment (
		scrolled_window));
    if(scroll_position < upper - page) {
        gtk_adjustment_set_value (
		gtk_scrolled_window_get_vadjustment (
		    scrolled_window), scroll_position);

    } else {
        NOOP ("SCROLL: geometryZ3: setting scroll position to 0 \n");
        gtk_adjustment_set_value (
		gtk_scrolled_window_get_vadjustment (
		    scrolled_window), upper - page);
    }
    gtk_adjustment_changed (
	    gtk_scrolled_window_get_vadjustment (
		scrolled_window));
}

double 
rodent_scroll_position(view_t * view_p){
    GtkScrolledWindow *scrolled_window = g_object_get_data(G_OBJECT(view_p->widgets.paper), "scrolled_window");


    view_geometry_t *view_geometry_p = rodent_get_view_geometry_p (view_p);


    gdouble upper=gtk_adjustment_get_upper (
	    gtk_scrolled_window_get_vadjustment (
		scrolled_window));
    gdouble page=gtk_adjustment_get_page_size (
	    gtk_scrolled_window_get_vadjustment (
		scrolled_window));
    
    double scroll_position = floor((upper+page) * view_geometry_p->scrollX +0.5);
    NOOP (stderr, "SCROLL: geometryZ2: setting scroll position to %lf geometry height=%d, upper=%lf, page=%lf\n", scroll_position, view_geometry_p->h, upper, page);
    g_free(view_geometry_p);

    return scroll_position;
}

// this sets scroll position from saved geometry
void *
rodent_set_scroll_position (gpointer data) {
    view_t * view_p = data;
    GtkScrolledWindow *scrolled_window = g_object_get_data(G_OBJECT(view_p->widgets.paper), "scrolled_window");
    if(!GTK_IS_SCROLLED_WINDOW (scrolled_window))
        return NULL;
    double scroll_position = rodent_scroll_position(view_p);
    rodent_set_scroll (view_p, scroll_position);
    return NULL;

}

static GMutex *
get_geometry_mutex(void){
    static GMutex *mutex = NULL;
    static gsize initialized = 0;
    if (g_once_init_enter (&initialized)){
	rfm_mutex_init(mutex);
      g_once_init_leave (&initialized, 1);
    }
    return mutex;
}

view_geometry_t *
rodent_get_view_geometry_p (view_t * view_p) {
    DBHashTable *geometry;
    const gchar *key;
    GString *gs;
    view_geometry_t *view_geometry_p;
    view_geometry_t view_geometry;
    view_geometry_t path_view_geometry;
    gchar *f;

    if(!view_p)
        return NULL;
    /* defaults */
    view_geometry.w = DEFAULT_WIDTH;
    view_geometry.h = DEFAULT_HEIGHT;
    view_geometry.x = view_geometry.y = -1;
    view_geometry.scrollX = 0;

    if(view_p->en && view_p->en->path && strlen (view_p->en->path)){
        key = view_p->en->path;
    } else {
        key = "RODENT_ROOT";
    }
    NOOP ("rodent_get_view_geometry_p(%s)\n", key);

    f = g_build_filename (GRID_GEOMETRY_FILE, NULL);
    NOOP ("looking for geometry file %s with key=%s\n", f, key);
    if(!rfm_g_file_test (f, G_FILE_TEST_EXISTS)) {
        NOOP ("Geometry file does not exist (yet): %s\n", f);
        g_free (f);
	view_geometry_p = (view_geometry_t *) malloc(sizeof(view_geometry_t));
	if(!view_geometry_p) g_error("malloc: %s\n", strerror(errno));
	memcpy(view_geometry_p, &view_geometry, sizeof(view_geometry_t));
        return view_geometry_p;
    }
    GMutex *geometry_mutex = get_geometry_mutex();
    g_mutex_lock(geometry_mutex);

    TRACE("opening %s...\n",f); 
    geometry = dbh_new (f, NULL, DBH_READ_ONLY|DBH_PARALLEL_SAFE);
    TRACE("open %s.\n",f); 
    g_free (f);
    if (!geometry){
	g_mutex_unlock(geometry_mutex);
	return NULL;
    }
    dbh_set_parallel_lock_timeout(geometry, 3);

    /* load root record geometry */
    gs = g_string_new ("RODENT_ROOT");
    sprintf ((gchar *)DBH_KEY (geometry), "%10u", g_string_hash (gs));
    g_string_free (gs, TRUE);
    if(!dbh_load (geometry)) {
        NOOP ("no \"root\" geometry record for %s (%s). Using defaults...",
		"RODENT_ROOT", (gchar *)DBH_KEY (geometry));
        dbh_close (geometry);
	g_mutex_unlock(geometry_mutex);
	view_geometry_p = (view_geometry_t *) malloc(sizeof(view_geometry_t));
	if(!view_geometry_p) g_error("malloc: %s\n", strerror(errno));
	memcpy(view_geometry_p, &view_geometry, sizeof(view_geometry_t));
        return view_geometry_p;
    }
    memcpy (&view_geometry, DBH_DATA (geometry), sizeof (view_geometry_t));

    gs = g_string_new (key);
    sprintf ((gchar *)DBH_KEY (geometry), "%10u", g_string_hash (gs));
    g_string_free (gs, TRUE);

    if(!dbh_load (geometry)) {
        NOOP("no \"path\" geometry record for %s (%s)\n", 
		key, (gchar *)DBH_KEY (geometry));
    } else {
        memcpy (&path_view_geometry, DBH_DATA (geometry), sizeof (view_geometry_t));
        view_geometry.scrollX = path_view_geometry.scrollX;
#ifdef PER_PATH_GEOMETRY
        /* this is disabled by default */
        view_geometry.x = path_view_geometry.x;
        view_geometry.y = path_view_geometry.y;
        view_geometry.w = path_view_geometry.w;
        view_geometry.h = path_view_geometry.h;
#endif
    }
    //view_geometry.scrollX *= GRID_ROWS(view_p);
    NOOP (stderr, "got geometry with key=%s  (scroll %lf,x=%d y=%d, w=%d h=%d)\n", key, view_geometry.scrollX,
         view_geometry.x, view_geometry.y, view_geometry.w, view_geometry.h);
    dbh_close (geometry);
    g_mutex_unlock(geometry_mutex);
    if(view_geometry.w <= 0 || view_geometry.h <= 0) {
        view_geometry.w = DEFAULT_WIDTH;
        view_geometry.h = DEFAULT_HEIGHT;
    }
    view_geometry_p = (view_geometry_t *) malloc(sizeof(view_geometry_t));
    if(!view_geometry_p) g_error("malloc: %s\n", strerror(errno));
	memcpy(view_geometry_p, &view_geometry, sizeof(view_geometry_t));
    return view_geometry_p;
}

static
void
rodent_save_view_geometry_p (view_t * view_p, const char *inkey) {
    NOOP ("rodent_save_view_geometry_p()\n");
    view_geometry_t view_geometry;
    DBHashTable *geometry;
    GString *gs;
    gchar *f;
    const gchar *key;
    if(!view_p)
        return;

    if(!view_p->en || !view_p->en->path)
        key = "RODENT_ROOT";
    else if(inkey)
        key = inkey;
    else
        key = view_p->en->path;

    f = g_build_filename (GRID_GEOMETRY_FILE, NULL);
    
    GMutex *geometry_mutex = get_geometry_mutex();
    g_mutex_lock(geometry_mutex);
    TRACE("opening %s...\n",f); 
    geometry = dbh_new (f, NULL, DBH_PARALLEL_SAFE);
    TRACE("opened %s.\n",f); 
    if(!geometry) {
        NOOP ("Creating geometry file: %s", f);
	unsigned char keylength=11;
        gchar *directory = g_path_get_dirname(f);
        if (!g_file_test(directory, G_FILE_TEST_IS_DIR)){
            g_mkdir_with_parents(directory, 0700);
        }
        g_free(directory);
        geometry = dbh_new (f, &keylength, DBH_CREATE|DBH_PARALLEL_SAFE);
    }

    if(!geometry) {
        DBG ("Cannot open geometry file %s for write.\n", f);
        g_free (f);
        g_mutex_unlock(geometry_mutex);
        return;
    }
    g_free (f);

    gs = g_string_new (key);
    sprintf ((char *)DBH_KEY (geometry), "%10u", g_string_hash (gs));
    g_string_free (gs, TRUE);

    GdkRectangle allocation;
    rfm_global_t *rfm_global_p = rfm_global();
    gtk_widget_get_allocation (rfm_global_p->window, &allocation);
    view_geometry.w = allocation.width;
    view_geometry.h = allocation.height;

    if(strcmp (key, "RODENT_ROOT") == 0){
        view_geometry.scrollX = 0;
   } else {
	GtkScrolledWindow *scrolled_window = g_object_get_data(G_OBJECT(view_p->widgets.paper), "scrolled_window");
	gdouble value=gtk_adjustment_get_value (
		gtk_scrolled_window_get_vadjustment (
		    scrolled_window));
	gdouble upper=gtk_adjustment_get_upper (
		gtk_scrolled_window_get_vadjustment (
		    scrolled_window));
	gdouble page=gtk_adjustment_get_page_size (
		gtk_scrolled_window_get_vadjustment (
		    scrolled_window));
	// size=upper+page;
	// place=value/size;
        view_geometry.scrollX = value/(upper+page);
	// inverse:
	// floor((upper+page) * scrollX +0.5)
 
	NOOP (stderr, "geometry graphics: saving ajustment at %lf (%s), page=%lf upper=%lf\n",
	    view_geometry.scrollX, key, page, upper);
   }

    gint x,y;
    gtk_window_get_position ((GtkWindow *) rfm_global_p->window, &x, &y);

    view_geometry.x = x;
    view_geometry.y = y; 

    NOOP (stderr, "rodent_save_view_geometry_p():  saving geometry %d,%d size %d,%d\n",
         view_geometry.x, view_geometry.y, view_geometry.w, view_geometry.h);

    /* here we save the per path geometry */
    memcpy (DBH_DATA (geometry), &view_geometry, sizeof (view_geometry_t));
    dbh_set_recordsize (geometry, sizeof (view_geometry_t));
    if(!dbh_update (geometry))
        DBG ("!dbh_update(geometry)\n");
    NOOP ("1.saved geometry with key=%s (%lf,%d,%d)\n", key, view_geometry.scrollX, view_geometry.w, view_geometry.h);

    dbh_close (geometry);
    g_mutex_unlock(geometry_mutex);
    return;
}

void
rodent_save_local_view_geometry_p (view_t * view_p) {
    rodent_save_view_geometry_p (view_p, NULL);
}

void
rodent_save_root_view_geometry_p (view_t * view_p) {
    rodent_save_view_geometry_p (view_p, "RODENT_ROOT");
}
