/*
 * Copyright (C) 2002-2013 Edscott Wilson Garcia
 * EMail: edscott@users.sf.net
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; 
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include "rfm.h"
#include "rfm_modules.h"

// Do checks, remove obsolete functions.
/////////////////////////////////////////////////////////////////////////////////////////////////


typedef struct mime_winner_thread_t {
    gchar *path;
    struct stat st;
    gchar *mimetype;
} mime_winner_thread_t;

// This cleans up the thumbnail cache when reload is pressed in order
// that thumbnails be unconditionally regenerated.

gint 
rfm_get_preview_image_size(void){
    gchar *s_size = getenv("RFM_PREVIEW_IMAGE_SIZE");
    if (!s_size || !strlen(s_size)) return PREVIEW_IMAGE_SIZE;
    errno = 0;
    long size = strtol(s_size, NULL, 10);
    if (errno){
        fprintf(stderr, "strtol(%s): %s\n", s_size, strerror(errno));
        return PREVIEW_IMAGE_SIZE;
    }
    if (size < 24 || size > 1000) return PREVIEW_IMAGE_SIZE;
    return (gint)size;
}

void *
rfm_cleanup_thumbnails(void *data){
    gchar *path = data;
    TRACE("rfm_cleanup_thumbnails()...\n");
    // This should always resolve to a local absolute path...
    if (!g_file_test(path, G_FILE_TEST_IS_DIR)){
        DBG("rfm_cleanup_thumbnails(): %s not a dir\n", path);
        return NULL;
    }
    // Comments are for reading and understanding code:
    // clear any thumbnail previews:
    gchar *test_image = g_build_filename (path, "test.jpg", NULL);
    gchar *filename = rfm_get_thumbnail_path (test_image, 48);

    gchar *dirname = g_path_get_dirname (filename);
    g_free (filename);
    if (!g_file_test(dirname, G_FILE_TEST_EXISTS)){
        TRACE("rfm_cleanup_thumbnails(): %s does not exist\n", dirname);
	g_free (test_image);
	g_free (dirname);
	g_free (path);
	return NULL;
    }

    DIR *directory;
    struct dirent *d;
    directory = opendir (dirname);
    if(directory) {
        while((d = readdir (directory)) != NULL) {
            if(strcmp (d->d_name, ".") == 0) continue;
            if(strcmp (d->d_name, "..") == 0) continue;
            gchar *fullpath = g_build_filename (dirname, d->d_name, NULL);
            //if (rename(fullpath,test_image));
            if (g_file_test(fullpath, G_FILE_TEST_EXISTS) && unlink(fullpath) < 0)
		DBG("unlink(%s): %s\n", fullpath, strerror(errno));
            g_free (fullpath);
        }
        closedir(directory);
    }
    // Now clean up the pixbuf hash...
    directory = opendir (path);
    if(directory) {
        while((d = readdir (directory)) != NULL) {
            if(strcmp (d->d_name, ".") == 0) continue;
            if(strcmp (d->d_name, "..") == 0) continue;
            gchar *fullpath = g_build_filename (path, d->d_name, NULL);
            rfm_rm_from_pixbuf_hash(fullpath, rfm_get_preview_image_size());
            rfm_rm_from_pixbuf_hash(fullpath, TINY_ICON_SIZE);
            rfm_rm_from_pixbuf_hash(fullpath, SMALL_ICON_SIZE);
            rfm_rm_from_pixbuf_hash(fullpath, MEDIUM_ICON_SIZE);
            rfm_rm_from_pixbuf_hash(fullpath, BIG_ICON_SIZE);
            g_free (fullpath);
        }
        closedir(directory);
    }

    g_free (test_image);
    g_free (dirname);
    g_free (path);
    return NULL;
}

static 
const gchar *
resolve_folder_icon_by_path (const gchar *path){
    record_entry_t *en = rfm_stat_entry (path, 0);
    const gchar *t;
    if(IS_NOACCESS_TYPE (en->type)) {
	    t = "xffm/stock_directory/compositeC/emblem_unreadable";
    } else if(IS_NOWRITE_TYPE (en->type)) {
    // without write access: xffm/stock_directory
	    t = "xffm/stock_directory";
    } else {
    // with write access: xffm/stock_directory
	    t = "xffm/stock_directory/compositeSE/emblem_write-ok";
    }
    rfm_destroy_entry(en);
    return t;
}

static off_t
st_sum (struct stat *st) {
    off_t sum;
    gint build=0;
    if(!st){
        sum = 0;
    } else {
        sum = st->st_mtime + st->st_size + st->st_mode + st->st_nlink + st->st_uid + st->st_gid + build;
    }
    return sum;
}

static pthread_mutex_t *
get_winner_mutex(void){
    static pthread_mutex_t mutex = PTHREAD_MUTEX_INITIALIZER;
    return &mutex;
}

static
gchar *
get_mime_winner_from_cache(const gchar *path, struct stat *st){
    // return NULL; // stress test
    DBHashTable *winner_cache;
    gchar *mimetype=NULL;
    gchar *g = g_build_filename (MIME_COUNT_DBH_FILE, NULL);
    // Here we use an external mutex instead of DBH_THREAD_SAFE
    pthread_mutex_t *winner_mutex = get_winner_mutex();
    pthread_mutex_lock(winner_mutex);
    TRACE("opening %s...\n",g); 
    if((winner_cache = dbh_new (g, NULL, DBH_READ_ONLY|DBH_PARALLEL_SAFE)) != NULL) {
	dbh_set_parallel_lock_timeout(winner_cache, 3);
	GString *gs = g_string_new (path);
	sprintf ((char *)DBH_KEY (winner_cache), "%10u", g_string_hash (gs));
	NOOP(stderr, "%s -> %10u\n", path, g_string_hash (gs));
	g_string_free (gs, TRUE);
	if (dbh_load (winner_cache) != 0){
	    mime_winner_dbh_t *mime_winner_dbh_p =
		(mime_winner_dbh_t *)winner_cache->data;
	    if (st_sum(st) == mime_winner_dbh_p->st_sum) {
		mimetype=g_strdup(mime_winner_dbh_p->mimetype);
	    } else {
		NOOP(stderr, "st mismatch for %s %d != %d\n",
			path, (int)st_sum(st), (int)mime_winner_dbh_p->st_sum);
	    }
	} else {
	    NOOP(stderr, "Could not load record for %s\n", path);
	}
	dbh_close (winner_cache);
    } else {
	NOOP(stderr, "Cannot open %s\n", g);
    }
    TRACE("opened %s.\n",g); 
    pthread_mutex_unlock(winner_mutex);
    g_free(g);
    /*if (mimetype && strcmp(mimetype, "unknown")==0) {
	g_free(mimetype);
	mimetype = NULL;
    }*/
    return mimetype;
}
static
gpointer
save_mime_winner_to_cache(gpointer data){
    mime_winner_thread_t *mime_winner_thread_p=data;

    // save to dbh
    DBHashTable *winner_cache;

    gchar *g = g_build_filename (MIME_COUNT_DBH_FILE, NULL);
    pthread_mutex_t *winner_mutex = get_winner_mutex();
    pthread_mutex_lock(winner_mutex);
    gchar *directory = g_path_get_dirname(g);
    if (!g_file_test(directory, G_FILE_TEST_IS_DIR)){
        g_mkdir_with_parents(directory, 0700);
    }
    g_free(directory);
    TRACE("opening %s...\n",g); 
    if((winner_cache = dbh_new (g, NULL, DBH_PARALLEL_SAFE)) == NULL) {
	unsigned char keylength = 11;
	winner_cache = dbh_new (g, &keylength, DBH_PARALLEL_SAFE|DBH_CREATE);
    }
    TRACE("opened %s.\n",g); 
    dbh_set_parallel_lock_timeout(winner_cache, 3);
    NOOP(stderr, "saving cache value for %s: %s\n",
	    mime_winner_thread_p->path, mime_winner_thread_p->mimetype);
    if(winner_cache == NULL) {
	    DBG("could not create %s\n", g);
    } else {
	GString *gs = g_string_new (mime_winner_thread_p->path);
	memset(DBH_KEY(winner_cache), 0, 11);
	sprintf ((char *)DBH_KEY (winner_cache), "%10u", g_string_hash (gs));
	NOOP(stderr, "S: %s -> %10u\n", mime_winner_thread_p->path, g_string_hash (gs));
	g_string_free (gs, TRUE);
	dbh_set_recordsize(winner_cache, sizeof(mime_winner_dbh_t));
	mime_winner_dbh_t *mime_winner_dbh_p =
	    (mime_winner_dbh_t *)winner_cache->data;
	memset(mime_winner_dbh_p->mimetype, 0, 80);
	strncpy(mime_winner_dbh_p->mimetype, mime_winner_thread_p->mimetype, 79);
	mime_winner_dbh_p->st_sum = st_sum(&(mime_winner_thread_p->st));
	if (dbh_update (winner_cache) == 0){
	    DBG("could not update %s\n", g);
	}
	NOOP(stderr, "saved cache value for %s at %s\n", 
		mime_winner_thread_p->path,g );
	dbh_close (winner_cache);
    }
    pthread_mutex_unlock(winner_mutex);
    g_free(g);

// cleanup
    g_free(mime_winner_thread_p->path);
    g_free(mime_winner_thread_p->mimetype);
    g_free(mime_winner_thread_p);
    return NULL;
}

static
gchar *
get_mayor_mime_type(const gchar *path){
    NOOP(stderr, "get_mayor_mime_type(%s)\n",path);
    struct stat st;
    if (stat(path, &st) < 0){
	return NULL;
    }
#if 10
    gchar *cache_value=get_mime_winner_from_cache(path, &st);
    if (cache_value) {
	NOOP(stderr, "cache value! %s got mimetype: %s\n", path, cache_value);
	return cache_value;
    } else {
	NOOP(stderr, "no cache value for %s\n", path);
    }
#endif

    // If cache out of date or empty, get the new value

    DIR *directory;
    directory = opendir (path);
    NOOP(stderr, "path=%s directory=0x%x\n", path,GPOINTER_TO_INT(directory));
    if (!directory){
	return NULL;
    }
    // This is a stack hash table:
    // thus, no need to mutex protect.
    GHashTable *type_hash=g_hash_table_new_full(g_str_hash, g_str_equal, g_free, NULL);
    struct dirent *d;
    // count first
    gint total = 0;
    while((d = readdir (directory)) != NULL) {
	total++;
	if (total > 200) break;
    }
    closedir (directory);
    if (total > 200){
	// Ugly composite in gnome:
	//return g_strdup("xffm/emblem_documents");
	return NULL;
    }


    directory = opendir (path);

    //
    while((d = readdir (directory)) != NULL) {
        if(*(d->d_name)=='.') continue;
        gchar *fullpath = g_strdup_printf ("%s/%s", path, d->d_name);
	gchar *hash_key=MIME_type(fullpath,NULL);
	// Don't do mime magic so that hash key may be NULL
	// and do things faster (this is a ballpark figure anyways)
	
	if (!hash_key) {
	    hash_key=MIME_magic(fullpath);
	    if (!hash_key) continue;
	}
	
	void *p = g_hash_table_lookup (type_hash, hash_key);
	gint count;
	if (p==NULL) {
	    count=1;
	    g_hash_table_insert (type_hash, hash_key, GINT_TO_POINTER(count));
	} else {
	    count=GPOINTER_TO_INT(p)+1;
	    g_hash_table_replace (type_hash, hash_key, GINT_TO_POINTER(count));
	}
	NOOP(stderr, "get_mayor_mime_type: %s (%s) %s==%d\n",
		path,d->d_name,hash_key,count);

	g_free(fullpath);
    }
    closedir (directory);

    GList *tmp;
    GList *hash_keys = g_hash_table_get_keys (type_hash);
    gint max=0;
    gchar *winner=NULL;
    for (tmp=hash_keys; tmp && tmp->data; tmp=tmp->next){
	gint value=GPOINTER_TO_INT(g_hash_table_lookup (type_hash, (gchar *)tmp->data));
	NOOP(stderr, "hashkey=%s value=%d\n", (gchar *)tmp->data, value);
	if (value > max) {
	    max=value;
	    winner = tmp->data;
	}
    }
    g_list_free(hash_keys);
    if (!winner) return NULL;
    if (strstr(winner, "No Read Permission")){
	return NULL;
    } 
    winner=g_strdup(winner);

    if (type_hash) {
	g_hash_table_remove_all (type_hash);
 
	g_hash_table_destroy(type_hash);
    }
    NOOP(stderr, "%s ---> %s\n", path, winner); 

    //
    mime_winner_thread_t *mime_winner_thread_p=
	(mime_winner_thread_t *)malloc(sizeof(mime_winner_thread_t));
    if (!mime_winner_thread_p){
	g_error("cannot allocate mime_winner_thread_p\n");
    }


    mime_winner_thread_p->mimetype = g_strdup(winner);
    mime_winner_thread_p->path = g_strdup(path);
    memcpy(&(mime_winner_thread_p->st), &st, sizeof(struct stat));
    rfm_view_thread_create(NULL, save_mime_winner_to_cache, mime_winner_thread_p, "save_mime_winner_to_cache");
    if (strcmp(winner, "unknown") == 0) return NULL;
    return winner;
}

static void
save_thumbnail (const gchar * thumbnail_path, 
	const gchar * original_path, 
	GdkPixbuf * tgt)
{
    if(!tgt || !GDK_IS_PIXBUF (tgt)) return;
    struct stat st;
    if(!thumbnail_path || !original_path) return;
    NOOP ("creating thumbnail for %s (%s)\n", original_path, thumbnail_path);
    if(stat (original_path, &st) != 0) return;
    rfm_pixbuf_save(tgt, thumbnail_path);
}

static gint
mimetype_is_image(const gchar *mimetype){
    static GSList *pix_mimetypes = NULL;
    static gsize initialized = 0;
    if (g_once_init_enter(&initialized)){
	// This gdk call is thread safe. 
	GSList *pix_formats = gdk_pixbuf_get_formats ();// check OK
	GSList *list = pix_formats;
	for(; list && list->data; list = list->next) {
	    gchar **pix_mimetypes_p;
	    GdkPixbufFormat *fmt = list->data;
	    // This gdk call is thread safe.
	    pix_mimetypes_p = gdk_pixbuf_format_get_mime_types (fmt);// check OK
	    pix_mimetypes = g_slist_prepend(pix_mimetypes, pix_mimetypes_p);
	}
	g_slist_free(pix_formats);
	g_once_init_leave(&initialized, 1);
    }
    /* check for image support types */
    GSList *list = pix_mimetypes;
    for(; list && list->data; list = list->next) {
	gchar **pix_mimetypes_p = list->data;
	for(; pix_mimetypes_p && *pix_mimetypes_p; pix_mimetypes_p++) {
	    NOOP(stderr, "allowable pix_format=%s --> %s\n", *pix_mimetypes_p, mimetype);
	    if(g_ascii_strcasecmp (*pix_mimetypes_p, mimetype) == 0) {
		return 1;
	    }
	}
    }
    return 0;
}

gint
rfm_path_is_image(const gchar *path){
    gchar *mimetype = (gchar *)
	rfm_natural(RFM_MODULE_DIR,"mime", (void *)path, "mime_type_plain");
    if (!mimetype){
	NOOP("%s --> null plain mimetype\n", path);
	return FALSE;
    }
    gint retval = mimetype_is_image(mimetype);
    // if (retval) NOOP("%s --> image.\n", path);
    g_free(mimetype);
    return retval;
    
}

gint
rfm_entry_is_image (record_entry_t *en) {
    if (!en){
	TRACE("rfm_entry_is_image(): !en\n");
	return 0;
    }
    if (en->st && !S_ISREG(en->st->st_mode)) {
	TRACE("rfm_entry_is_image(): !S_ISREG(en->st->st_mode)\n");
        return 0;
    }
    if (!en->mimetype) {
	en->mimetype = MIME_type (en->path, en->st);
    }
    TRACE("rfm_entry_is_image(): mimetype = %s\n", en->mimetype);
    gboolean want_magic = (!en->mimetype || strcmp(en->mimetype, _("unknown"))==0);
    const gchar *mimetype = en->mimetype;
    if (want_magic) {
	if (!en->mimemagic) {
	    en->mimemagic = rfm_natural(RFM_MODULE_DIR, "mime", en->path, "mime_magic");
	}
	if (en->mimemagic) mimetype = en->mimemagic;
	if (!mimetype) mimetype = _("unknown");
        TRACE("rfm_entry_is_image(): mimemagic= %s\n", mimetype);
    }
    gint retval = mimetype_is_image(mimetype);

    return retval;

}
gchar * 
rfm_content_icon_id_f (view_t *view_p, const gchar *path) {

    if (!path) return NULL;
    if (strcmp(path, g_get_home_dir())==0){
	return g_strdup("xffm/stock_directory/compositeC/stock_home");
    }
    if (getenv("RFM_DESKTOP_DIR") 
	    && strcmp(path, getenv("RFM_DESKTOP_DIR"))==0){
		return g_strdup("xffm/emblem_desktop");
    }

    if (view_p->module) {
	NOOP(stderr, "view_p->module\n");
	return NULL;
    }
    
    gint have_mime_plugin=GPOINTER_TO_INT(rfm_void(RFM_MODULE_DIR, "mime", "module_active"));
    if (!have_mime_plugin) {
	NOOP(stderr, "!have_mime_plugin\n");
	return NULL;
    }
    /*gint have_icons_plugin=GPOINTER_TO_INT(rfm_void(RFM_MODULE_DIR, "icons", "module_active"));
    if (!have_icons_plugin) {
	NOOP(stderr, "!have_icons_plugin\n");
	return NULL;
    }*/

    if (FSTAB_is_in_fstab(path)) {
	NOOP(stderr, "FSTAB_is_in_fstab(path)\n");
	return NULL;
    }

    gchar *inserted_icon=get_mayor_mime_type(path);
    NOOP(stderr, "mayor mimetype: %s\n", inserted_icon);
    if (inserted_icon) {
	const gchar *icon_id=resolve_folder_icon_by_path (path);
	gchar *content_icon=g_strdup_printf("%s/compositeC/%s", icon_id, inserted_icon);
	NOOP(stderr, "compositeC %s (%s)\n", c_icon,  path);
	g_free(inserted_icon);
	return content_icon;
    }
    return NULL;
}


GdkPixbuf *
rfm_create_preview (const gchar * file, gint size) {
    if(!file){
	g_warning("create_preview(): file is NULL\n");
        return NULL;
    }
    if (!g_path_is_absolute (file)) return NULL;
    GdkPixbuf *pixbuf = NULL;

    gchar *thumbnail_path = rfm_get_thumbnail_path (file, size);
    if (!thumbnail_path){
	DBG("create_preview(): cannot get thumbnail path for %s\n", file);
	return NULL;
    }

    if(g_file_test (thumbnail_path, G_FILE_TEST_EXISTS)) {
        struct stat thumbnail_st;
        struct stat file_st;
	gboolean stat_ok = TRUE;
        if (stat (thumbnail_path, &thumbnail_st) < 0)stat_ok = FALSE;
	    
	// This already succeded in the path_is_image() call.
        if (stat (file, &file_st) != 0) goto done;
        if(stat_ok && thumbnail_st.st_mtime > file_st.st_mtime) {
            NOOP ("PREVIEW thread: retrieving thumbnail for %s (%s)\n", file, thumbnail_path);
	    if (g_file_test(thumbnail_path, G_FILE_TEST_EXISTS)){
		pixbuf = rfm_pixbuf_new_from_file(thumbnail_path, -1, -1); // check- OK contexted.
	    }
            if(pixbuf && GDK_IS_PIXBUF(pixbuf) ) {
		g_object_ref(pixbuf);
                goto done;
            }
        }
	if (g_file_test(thumbnail_path, G_FILE_TEST_EXISTS) && unlink(thumbnail_path)<0){
	    DBG("unlink(%s): %s\n", thumbnail_path, strerror(errno));
	}
    }
    NOOP ("PREVIEW thread: no thumbnail for %s (%s) at size %d\n", file, thumbnail_path, size);
    pixbuf = rfm_pixbuf_new_from_file(file, size, size); // check- OK contexted.    
    if(pixbuf) {
	g_object_ref(pixbuf);
	save_thumbnail (thumbnail_path, file, pixbuf); // check - OK contexted.
	NOOP ("save_thumbnail %s\n", thumbnail_path);
    } else {
	DBG("could not create pixbuf from: %s at size %dx%d\n", file, size, size);
    }
done:
    g_free (thumbnail_path);
	    
    return pixbuf;
}


