/*
 * Copyright (C) 2002-2012 Edscott Wilson Garcia
 * EMail: edscott@users.sf.net
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; 
 */

#define RFM_PRIMARY_COMPAT_C

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include "rfm.h"
#include "rfm_modules.h"

////////////////////////////////////////////////////////////////////////////
/// glib 2.32 compatibility
////////////////////////////////////////////////////////////////////////////
//
void rfm_replace_menu_label(GtkWidget *parent, const gchar *text){
     GtkWidget *label = g_object_get_data(G_OBJECT(parent), "label");
     gtk_label_set_markup ((GtkLabel *) label, text);
}
void rfm_replace_menu_image(GtkWidget *parent, const gchar *icon_id){
      GdkPixbuf *p = rfm_get_pixbuf (icon_id, SIZE_BUTTON);
      if(p==NULL) {
          DBG("unable to get pixbuf for menu item \"%s\"\n",icon_id);
          return;
      }

    GtkWidget *label = g_object_get_data(G_OBJECT(parent), "label");
    GtkWidget *image;
    GtkWidget *replacement;
    if (!label) {
        replacement = rfm_hbox_new(FALSE,0);
        label = gtk_bin_get_child(GTK_BIN(parent));
	if (label && GTK_IS_WIDGET(label)) {
            g_object_ref(label);
            gtk_container_remove(GTK_CONTAINER(parent), label);
        }
        gtk_container_add(GTK_CONTAINER(parent), replacement);
    } else {
        image = g_object_get_data(G_OBJECT(parent), "image");
        replacement = gtk_bin_get_child(GTK_BIN(parent));
	if (image && GTK_IS_WIDGET(image)) {
            gtk_container_remove(GTK_CONTAINER(replacement), image);
        }
	if (label && GTK_IS_WIDGET(label)) {
            g_object_ref(label);
            gtk_container_remove(GTK_CONTAINER(replacement), label);
        }
    }
    image = gtk_image_new_from_pixbuf (p);

    gtk_box_pack_start(GTK_BOX(replacement), image, FALSE,FALSE,0);
    if (label && GTK_IS_WIDGET(label)) {
        gtk_box_pack_start(GTK_BOX(replacement), label, FALSE,FALSE,0);
        g_object_unref(label);
        gtk_widget_show(label);
    }
    gtk_widget_show(image);
    gtk_widget_show(replacement);
    g_object_set_data(G_OBJECT(parent), "image", image);
    g_object_set_data(G_OBJECT(parent), "label", label);
}
// GtkAccelLabel
void rfm_set_menu_image(GtkWidget *parent, GtkWidget *image){
    if (!image || !GTK_IS_WIDGET(image)) return;
    GtkWidget *label = gtk_bin_get_child(GTK_BIN(parent));
    if (label && GTK_IS_WIDGET(label)) {
        g_object_ref(label);
        gtk_container_remove(GTK_CONTAINER(parent), label);
    }
    GtkWidget *replacement = rfm_hbox_new(FALSE,0);
    gtk_container_add(GTK_CONTAINER(parent), replacement);
    gtk_box_pack_start(GTK_BOX(replacement), image, FALSE,FALSE,0);
    if (label && GTK_IS_WIDGET(label)) {
        gtk_box_pack_start(GTK_BOX(replacement), label, FALSE,FALSE,0);
        g_object_unref(label);
        gtk_widget_show(label);
    }
    gtk_widget_show(image);
    gtk_widget_show(replacement);
    g_object_set_data(G_OBJECT(parent), "image", image);
    g_object_set_data(G_OBJECT(parent), "label", label);
    
}

GtkWidget * rfm_menu_item_new(const gchar *icon_id, const gchar *text)
{
    GdkPixbuf *pb = (icon_id)? rfm_get_pixbuf (icon_id, SIZE_BUTTON): NULL;    
                // no replacement for gtk_image_menu_item
    GtkWidget *w = gtk_menu_item_new_with_label ("");
    GtkWidget *replacement = rfm_hbox_new(FALSE,0);
    GtkWidget *label = gtk_bin_get_child(GTK_BIN(w));
    if (label && GTK_IS_WIDGET(label)) {
        g_object_ref(label);
        gtk_container_remove(GTK_CONTAINER(w), label);
    }

    if (pb){
        GtkWidget *image = gtk_image_new_from_pixbuf (pb);
        gtk_widget_show (image);
        gtk_box_pack_start(GTK_BOX(replacement), image, FALSE,FALSE,0);
        g_object_set_data(G_OBJECT(w), "image", image);
	g_object_unref(pb);
    }
    if (label && GTK_IS_WIDGET(label)) {
        gtk_label_set_markup(GTK_LABEL(label), text);
        gtk_box_pack_start(GTK_BOX(replacement), label, FALSE,FALSE,3);
        g_object_set_data(G_OBJECT(w), "label", label);
        gtk_widget_show(label);
        g_object_unref(label);
    }
    gtk_widget_show(replacement);
    gtk_container_add(GTK_CONTAINER(w), replacement);
    return w;
}

gint
rfm_gtk_version(void){
#if GTK_MAJOR_VERSION==2
    return 2;
#else
    return 3;
#endif
}

GThread * rfm_thread_create(const gchar *name, GThreadFunc func, gpointer data, gboolean joinable){
    NOOP(stderr, "rfm_thread_create: %s\n", name);
    GThread *thread;
    GError *error = NULL;
#if GLIB_MAJOR_VERSION==2 && GLIB_MINOR_VERSION<32
    thread = g_thread_create(func, data, joinable, &error);
#else
    thread = g_thread_try_new(name, func, data, &error);
    if (!error && !joinable) g_thread_unref(thread);
#endif
    if (error){
	DBG("rfm_thread_create(): %s\n", error->message);
	g_error_free(error);
        thread = NULL;
    }
    return thread;
}

gboolean rfm_cond_timed_wait(GCond *signal, GMutex *mutex, gint seconds){
    gboolean retval;
#if GLIB_MAJOR_VERSION==2 && GLIB_MINOR_VERSION<32
    GTimeVal tv;
    g_get_current_time (&tv);
    tv.tv_sec += seconds;
    retval = g_cond_timed_wait(signal, mutex, &tv);
#else
    gint64 end_time;
    end_time = g_get_monotonic_time () + seconds * G_TIME_SPAN_SECOND;
    retval = g_cond_wait_until (signal, mutex, end_time);
#endif
    return retval;
}

void rfm_rw_lock_init(RfmRWLock *rw_lock){
#if GLIB_MAJOR_VERSION==2 && GLIB_MINOR_VERSION>=32
    g_rw_lock_init(rw_lock);
#else
    g_static_rw_lock_init(rw_lock); 
#endif
}

void rfm_rw_lock_clear(RfmRWLock *rw_lock){
#if GLIB_MAJOR_VERSION==2 && GLIB_MINOR_VERSION>=32
    g_rw_lock_clear (rw_lock);
#else
    g_static_rw_lock_free(rw_lock); 
#endif
}


void rfm_rw_lock_writer_lock (RfmRWLock *rw_lock){               
#if GLIB_MAJOR_VERSION==2 && GLIB_MINOR_VERSION>=32
    g_rw_lock_writer_lock(rw_lock);
#else
    g_static_rw_lock_writer_lock (rw_lock);
#endif
}

gboolean rfm_rw_lock_writer_trylock (RfmRWLock *rw_lock){               
#if GLIB_MAJOR_VERSION==2 && GLIB_MINOR_VERSION>=32
    return g_rw_lock_writer_trylock(rw_lock);
#else
    return g_static_rw_lock_writer_trylock (rw_lock);
#endif
}

void rfm_rw_lock_writer_unlock (RfmRWLock *rw_lock){               
#if GLIB_MAJOR_VERSION==2 && GLIB_MINOR_VERSION>=32
    g_rw_lock_writer_unlock(rw_lock);
#else
    g_static_rw_lock_writer_unlock (rw_lock);
#endif
}

void rfm_rw_lock_reader_lock (RfmRWLock *rw_lock){               
#if GLIB_MAJOR_VERSION==2 && GLIB_MINOR_VERSION>=32
    g_rw_lock_reader_lock(rw_lock);
#else
    g_static_rw_lock_reader_lock (rw_lock);
#endif
}

void rfm_rw_lock_reader_unlock (RfmRWLock *rw_lock){               
#if GLIB_MAJOR_VERSION==2 && GLIB_MINOR_VERSION>=32
    g_rw_lock_reader_unlock(rw_lock);
#else
    g_static_rw_lock_reader_unlock (rw_lock);
#endif
}

gboolean rfm_rw_lock_reader_trylock (RfmRWLock *rw_lock){               
#if GLIB_MAJOR_VERSION==2 && GLIB_MINOR_VERSION>=32
    return g_rw_lock_reader_trylock(rw_lock);
#else
    return g_static_rw_lock_reader_trylock (rw_lock);
#endif
}


////////////////////////////////////////////////////////////////////////////
/// gtk+ compatibility
////////////////////////////////////////////////////////////////////////////


void rfm_text_view_set_wrap_mode(GtkTextView *text_view, gint mode){
#if GTK_MAJOR_VERSION==3
    gtk_text_view_set_wrap_mode (text_view, mode);
#endif
    return;
}


GtkWidget *rfm_combo_box_new_with_entry (void){
#if GTK_MAJOR_VERSION==2 && GTK_MINOR_VERSION<24
    // this is deprecated...
    return gtk_combo_box_entry_new ();
#else
    return gtk_combo_box_new_with_entry ();
#endif

}

void rfm_combo_box_text_remove_all(GtkWidget *comboentry){
#if GTK_MAJOR_VERSION==2
    // XXX: This is a known bug due to gtk2 limitations.
    // This will just remove first item...
    // (in Rodent only a problem if more than one bluetooth is present)
# if GTK_MAJOR_VERSION==2 && GTK_MINOR_VERSION<24
    gtk_combo_box_remove_text (GTK_COMBO_BOX(comboentry), 0);
# else
    gtk_combo_box_text_remove  (GTK_COMBO_BOX_TEXT(comboentry), 0);
# endif

#else
    // This is gtk3, and no bug is present.
    gtk_combo_box_text_remove_all  (GTK_COMBO_BOX_TEXT(comboentry));
#endif

}

void rfm_combo_box_text_prepend(GtkWidget *comboentry, const gchar *text){
#if GTK_MAJOR_VERSION==2 && GTK_MINOR_VERSION<24
    gtk_combo_box_prepend_text(GTK_COMBO_BOX(comboentry),text);
#else
    gtk_combo_box_text_prepend_text(GTK_COMBO_BOX_TEXT(comboentry),text);
#endif
}

#if GTK_MAJOR_VERSION==3 && GTK_MINOR_VERSION>=2
GtkWidget * // XXX deprecated
rfm_hscale_new_with_range(gdouble min, gdouble max, gdouble step){
    return gtk_scale_new_with_range(GTK_ORIENTATION_HORIZONTAL, min, max, step);
}

GtkWidget *
rfm_hbox_new(gboolean homogeneous, gint spacing){
    GtkWidget *box = gtk_box_new(GTK_ORIENTATION_HORIZONTAL, spacing);
    gtk_box_set_homogeneous (GTK_BOX(box),homogeneous);
    return box;
}

GtkWidget *
rfm_vbox_new(gboolean homogeneous, gint spacing){
    GtkWidget *box = gtk_box_new(GTK_ORIENTATION_VERTICAL, spacing);
    gtk_box_set_homogeneous (GTK_BOX(box),homogeneous);
    return box;
}
GtkWidget *
rfm_vpaned_new(void){
    return gtk_paned_new(GTK_ORIENTATION_VERTICAL);
}
GtkWidget *
rfm_hpaned_new(void){
    return gtk_paned_new(GTK_ORIENTATION_HORIZONTAL);
}
GtkWidget *
rfm_vbutton_box_new(void){
    return gtk_button_box_new(GTK_ORIENTATION_VERTICAL);
}
GtkWidget *
rfm_hbutton_box_new(void){
    return gtk_button_box_new(GTK_ORIENTATION_HORIZONTAL);
}
#else
GtkWidget * // XXX deprecated
rfm_hscale_new_with_range(gdouble min, gdouble max, gdouble step){
    return gtk_hscale_new_with_range(min, max, step);
}

GtkWidget *
rfm_hbox_new(gboolean homogeneous, gint spacing){
    return gtk_hbox_new(homogeneous, spacing);
}

GtkWidget *
rfm_vbox_new(gboolean homogeneous, gint spacing){
    return gtk_vbox_new(homogeneous, spacing);
}
GtkWidget *
rfm_vpaned_new(void){
    return gtk_vpaned_new();
}
GtkWidget *
rfm_hpaned_new(void){
    return gtk_hpaned_new();
}
GtkWidget *
rfm_vbutton_box_new(void){
    return gtk_vbutton_box_new();
}
GtkWidget *
rfm_hbutton_box_new(void){
    return gtk_hbutton_box_new();
}
#endif


