/*  utils.c
 *
 *  Utility functions of xwxapt application
 */

/*
 *  xwxapt: An application to decode APT signals from
 *  weather satellites and produce an image of the weather.
 *
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "utils.h"
#include "shared.h"

/*------------------------------------------------------------------*/

/*  Load_Config()
 *
 *  Loads the xwxaptrc configuration file
 */

  gboolean
Load_Config( gpointer data )
{
  char
	rc_fpath[64], /* File path to xwxaptrc */
	line[81];     /* Buffer for Load_Line  */

  /* Config file pointer */
  FILE *xwxaptrc;

  /* ALSA channel names and values */
  char *chan_name[] =
  {
	"FRONT_LEFT",
	"FRONT_RIGHT",
	"REAR_LEFT",
	"REAR_RIGHT",
	"SIDE_LEFT",
	"SIDE_RIGHT",
	"MONO",
	"NONE"
  };

  int chan_number[] =
  {
	SND_MIXER_SCHN_FRONT_LEFT,
	SND_MIXER_SCHN_FRONT_RIGHT,
	SND_MIXER_SCHN_REAR_LEFT,
	SND_MIXER_SCHN_REAR_RIGHT,
	SND_MIXER_SCHN_SIDE_LEFT,
	SND_MIXER_SCHN_SIDE_RIGHT,
	SND_MIXER_SCHN_MONO,
	-1
  };

  int idx, num_chan = 8;


  /* Setup file path to xwxaptrc and working dir */
  snprintf( rc_fpath, sizeof(rc_fpath),
	  "%s/xwxapt/xwxaptrc", getenv("HOME") );
  snprintf( rc_data.xwxapt_dir, sizeof(rc_data.xwxapt_dir),
	  "%s/xwxapt/", getenv("HOME") );

  /* Open xwxaptrc file */
  xwxaptrc = fopen( rc_fpath, "r" );
  if( xwxaptrc == NULL )
  {
	perror( rc_fpath );
	Show_Message( _("Failed to open xwxaptrc file"), "red" );
	Error_Dialog();
	return( FALSE );
  }

  /*** Read runtime configuration data ***/

  /* Read Receiver Type to use (Analogue via soundcard
   * and ALSA or a librtlsdr compatible SDR device) */
  if( Load_Line(line, xwxaptrc, "Receiver Type") != SUCCESS )
	return( FALSE );
  if( strcmp(line, "SDR") == 0 )
	SetFlag( USE_RTLSDR_RX );
  else if( strcmp(line, "SND") == 0 )
	ClearFlag( USE_RTLSDR_RX );
  else
  {
	Show_Message(
		_("Invalid Radio Receiver type\n"\
		  "Quit and correct xwxaptrc"), "red" );
	Error_Dialog();
	return( FALSE );
  }

  /* Read librtlsdr Device Index, abort if EOF */
  if( Load_Line(line, xwxaptrc, "Device Index") != SUCCESS )
	return( FALSE );
  idx = atoi( line );
  if( (idx < 0) || (idx > 8) )
  {
	Show_Message(
		_("Invalid librtlsdr Device Index\n"\
		  "Quit and correct xwxaptrc"), "red" );
	Error_Dialog();
	return( FALSE );
  }
  rc_data.rtlsdr_dev_index = idx;

  /* Read Frequency Correction Factor, abort if EOF */
  if( Load_Line(line, xwxaptrc, "Frequency Correction Factor") != SUCCESS )
	return( FALSE );
  rc_data.rtlsdr_freq_corr = atoi( line );
  if( abs(rc_data.rtlsdr_freq_corr) > 100 )
  {
	Show_Message(
		_("Invalid Frequency Correction Factor\n"\
		  "Quit and correct xwxaptrc"), "red" );
	Error_Dialog();
	return( FALSE );
  }

  /* Read Low Pass Filter Bandwidth, abort if EOF */
  if( Load_Line(line, xwxaptrc, "Low Pass Filter Bandwidth") != SUCCESS )
	return( FALSE );
  if( !rc_data.rtlsdr_lpf_bw )
	rc_data.rtlsdr_lpf_bw = atoi( line );
  if( rc_data.rtlsdr_lpf_bw < 100 )
  {
	Show_Message(
		_("Invalid Low Pass Filter Bandwidth\n"\
		  "Quit and correct xwxaptrc"), "red" );
	Error_Dialog();
	return( FALSE );
  }

  /* Read sound card name, abort if EOF */
  if( Load_Line(line, xwxaptrc, "Sound Card Name") != SUCCESS )
	return( FALSE );
  Strlcpy( rc_data.pcm_dev, line, sizeof(rc_data.pcm_dev) );

  /* Read ALSA "channel", abort if EOF */
  if( Load_Line(line, xwxaptrc, _("ALSA Channel") ) != SUCCESS )
	return( FALSE );
  for( idx = 0; idx < num_chan; idx++ )
	if( strcmp(chan_name[idx], line) == 0 )
	  break;
  if( idx == num_chan )
  {
	rc_data.channel = -1;
	fclose( xwxaptrc );
	Show_Message(
		_("Invalid ALSA channel name\n"\
		  "Quit and correct xwxaptrc"), "red" );
	Error_Dialog();
	return( FALSE );
  }
  rc_data.channel = chan_number[idx];

  /* Set right or left channel buffer index */
  if( strstr(line, "LEFT") || strstr(line, "MONO") )
	rc_data.use_chn = 0;
  else
	rc_data.use_chn = 1;

  /* Set number of channels */
  if( strstr(line, "MONO") )
	rc_data.num_chn = 1;
  else
	rc_data.num_chn = 2;

  /* Read capture source, abort if EOF */
  if( Load_Line(line, xwxaptrc, _("Capture Source")) != SUCCESS )
	return( FALSE );
  Strlcpy( rc_data.cap_src, line, sizeof(rc_data.cap_src) );

  /* Read Capture volume control, abort if EOF */
  if( Load_Line(line, xwxaptrc, _("Capture Volume Control")) != SUCCESS )
	return( FALSE );
  Strlcpy( rc_data.cap_vol, line, sizeof(rc_data.cap_vol) );

  /* Read Capture volume, abort if EOF */
  if( Load_Line(line, xwxaptrc, _("Capture Volume")) != SUCCESS )
	return( FALSE );
  rc_data.cap_lev = atoi( line );

  /* Read default decode duration, abort if EOF */
  if( Load_Line(line, xwxaptrc, _("Decoding Duration")) != SUCCESS )
	return( FALSE );
  rc_data.default_dur = atoi( line );
  gbl_duration = rc_data.default_dur;

  /* Warn if decoding duration is too long */
  if( gbl_duration > 600 )
  {
	char mesg[134];
	snprintf( mesg, sizeof(mesg),
		_("Default decoding duration specified\n"\
		  "in xwxaptrc (%d sec) seems excessive\n"\
		  "Default decoding duration limited to 600 sec"),
		gbl_duration );
	Show_Message( mesg, "red" );
	gbl_duration = 600;
	Error_Dialog();
  }

  /* Read default main window height, abort if EOF */
  if( Load_Line(line, xwxaptrc, _("Decoding Duration")) != SUCCESS )
	return( FALSE );
  rc_data.window_height = atoi( line );

  /*** Do the global config variables ***/
  /* Load pseudo-colorization maps */
  if( !Load_Colormap(xwxaptrc, &rc_data.noaa_A_map) )
  {
	  Show_Message( _("Error loading colorization maps from xwxaptrc"), "red" );
	  Error_Dialog();
	  return( FALSE );
  }

  if( !Load_Colormap(xwxaptrc, &rc_data.meteor_map) )
  {
	  Show_Message( _("Error loading colorization maps from xwxaptrc"), "red" );
	  Error_Dialog();
	  return( FALSE );
  }

  /* Initialize top window etc */
  Initialize_Top_Window();

  fclose( xwxaptrc );

  return( FALSE );
} /* End of Load_Config() */

/*------------------------------------------------------------------*/

/* Load_Colormap()
 *
 * Loads the pseudo-colorization maps from rc file
 */

  gboolean
Load_Colormap( FILE *xwxaptrc, color_map_t *map )
{
  /* Buffer for Load_Line  */
  char line[81];
  int idx, match;

  /* Read number of ranges */
  if( Load_Line(line, xwxaptrc, _("Colormap Number of Ranges")) != SUCCESS )
	return( FALSE );
  map->num_ranges = atoi(line);
  if( (map->num_ranges < 0) || (map->num_ranges > 6) )
  {
	Show_Message( _("Number of colorization ranges incorrect in xwxaptrc"), "red" );
	Error_Dialog();
	return( FALSE );
  }

  /* Read color map */
  for( idx = 0; idx < map->num_ranges; idx++ )
  {
	if( Load_Line(line, xwxaptrc, _("Color Map")) != SUCCESS )
	  return( FALSE );

	match = sscanf( line,
		"%3hhu %3hhu %3hhu %3hhu %3hhu %3hhu %3hhu %3hhu",
		&map->gray_from[idx],  &map->gray_to[idx],
		&map->red_from[idx],   &map->red_to[idx],
		&map->green_from[idx], &map->green_to[idx],
		&map->blue_from[idx],  &map->blue_to[idx] );

	if( match != 8 )
	{
	  perror( "xwxapt: sscanf" );
	  fclose( xwxaptrc );
	  Show_Message( _("Error scanning colorization ranges in xwxaptrc"), "red" );
	  Error_Dialog();
	  return( FALSE );
	}
  } /* for( idx = 0; idx < map->num_ranges; idx++ ) */

  /* Read white (cloud) threshold */
  if( Load_Line(line, xwxaptrc, _("White (cloud) Threshold")) != SUCCESS )
	return( FALSE );
  map->white_thld = atoi(line);

  return( TRUE );
} /* Load_Colormap() */

/*------------------------------------------------------------------*/

/*  Load_Line()
 *
 *  Loads a line from a file, aborts on failure. Lines beginning
 *  with a '#' are ignored as comments. At the end of file EOF is
 *  returned. Lines assumed maximum 80 characters long.
 */

  int
Load_Line( char *buff, FILE *pfile, char *messg )
{
  int
	num_chr, /* Number of characters read, excluding lf/cr */
	chr;     /* Character read by getc() */
  char error_mesg[MESG_SIZE];

  /* Prepare error message */
  snprintf( error_mesg, MESG_SIZE,
	  _("Error reading %s\n"\
		"Premature EOF (End Of File)"), messg );

  /* Clear buffer at start */
  buff[0] = '\0';
  num_chr = 0;

  /* Get next character, return error if chr = EOF */
  if( (chr = fgetc(pfile)) == EOF )
  {
	fprintf( stderr, "xwxapt: %s\n", error_mesg );
	fclose( pfile );
	Show_Message( error_mesg, "red" );
	Error_Dialog();
	return( EOF );
  }

  /* Ignore commented lines and eol/cr and tab */
  while(
	  (chr == '#') ||
	  (chr == HT ) ||
	  (chr == CR ) ||
	  (chr == LF ) )
  {
	/* Go to the end of line (look for LF or CR) */
	while( (chr != CR) && (chr != LF) )
	  /* Get next character, return error if chr = EOF */
	  if( (chr = fgetc(pfile)) == EOF )
	  {
		fprintf( stderr, "xwxapt: %s\n", error_mesg );
		fclose( pfile );
		Show_Message( error_mesg, "red" );
		Error_Dialog();
		return( EOF );
	  }

	/* Dump any CR/LF remaining */
	while( (chr == CR) || (chr == LF) )
	  /* Get next character, return error if chr = EOF */
	  if( (chr = fgetc(pfile)) == EOF )
	  {
		fprintf( stderr, "xwxapt: %s\n", error_mesg );
		fclose( pfile );
		Show_Message( error_mesg, "red" );
		Error_Dialog();
		return( EOF );
	  }

  } /* End of while( (chr == '#') || ... */

  /* Continue reading characters from file till
   * number of characters = 80 or EOF or CR/LF */
  while( num_chr < 80 )
  {
	/* If LF/CR reached before filling buffer, return line */
	if( (chr == LF) || (chr == CR) ) break;

	/* Enter new character to line buffer */
	buff[num_chr++] = (char)chr;

	/* Get next character */
	if( (chr = fgetc(pfile)) == EOF )
	{
	  /* Terminate buffer as a string if chr = EOF */
	  buff[num_chr] = '\0';
	  return( SUCCESS );
	}

	/* Abort if end of line not reached at 80 char. */
	if( (num_chr == 80) && (chr != LF) && (chr != CR) )
	{
	  /* Terminate buffer as a string */
	  buff[num_chr] = '\0';
	  snprintf( error_mesg, MESG_SIZE,
		  _("Error reading %s\n"\
			"Line longer than 80 characters"), messg );
	  fprintf( stderr, "xwxapt: %s\n%s\n", error_mesg, buff );
	  fclose( pfile );
	  Show_Message( error_mesg, "red" );
	  Error_Dialog();
	  return( ERROR );
	}

  } /* End of while( num_chr < max_chr ) */

  /* Terminate buffer as a string */
  buff[num_chr] = '\0';

  return( SUCCESS );

} /* End of Load_Line() */

/*------------------------------------------------------------------------*/

/*  File_Name()
 *
 *  Prepare a file name, use date and time if null argument
 */

  void
File_Name( char *file_name )
{
  int len; /* String length of file_name */

  /* If file_name is null, use date and time as file name */
  if( strlen(file_name) == 0 )
  {
	/* Variables for reading time (UTC) */
	time_t tp;
	struct tm utc;

	/* Prepare a file name as UTC date-time. */
	/* Default paths are images/ and record/ */
	time( &tp );
	utc = *gmtime( &tp );

	Strlcpy( file_name, rc_data.xwxapt_dir, MAX_FILE_NAME );
	len = (int)strlen( file_name );

	if( isFlagSet(ACTION_PROCESS_DSP) )
	  strftime( &file_name[len], 24, "images/%d%b%Y-%H%M", &utc );
	else if( isFlagSet(ACTION_RECORD_APT) )
	  strftime( &file_name[len], 24, "record/%d%b%Y-%H%M", &utc );
  }
  else /* Remove leading spaces from file_name */
  {
	int idx = 0;
	len = (int)strlen(file_name);
	do idx++;
	while( (file_name[idx] == ' ') && (idx != len) );
	file_name += idx - 1;
  }

} /* End of File_Name() */

/*------------------------------------------------------------------------*/

/* Fname()
 *
 * Finds file name in a file path
 */

  char *
Fname( char *fpath )
{
  int idx;

  idx = (int)strlen( fpath );

  while( (--idx >= 0) && (fpath[idx] != '/') );

  return( &fpath[++idx] );

} /* Fname() */

/*------------------------------------------------------------------------*/

/*  Usage()
 *
 *  Prints usage information
 */

  void
Usage( void )
{
  fprintf( stderr, "%s\n",
	  _("Usage: xwxapt [-b:f:hv]") );

  fprintf( stderr, "%s\n",
	  _("       -b: Specify Low Pass Filter Cutoff in Hz"));

  fprintf( stderr, "%s\n",
	  _("       -f: Specify Tuner Center Frequency in Hz"));

  fprintf( stderr, "%s\n",
	  _("       -h: Print this usage information and exit"));

  fprintf( stderr, "%s\n",
	  _("       -v: Print version number and exit"));

} /* End of Usage() */

/*------------------------------------------------------------------------*/

/*  Show_Message()
 *
 *  Prints a message string in the Text View scroller
 */

  void
Show_Message( char *mesg, char *attr )
{
  GtkAdjustment *adjustment;

  static GtkTextIter iter;
  static gboolean first_call = TRUE;

  /* Initialize */
  if( first_call )
  {
	first_call = FALSE;
	gtk_text_buffer_get_iter_at_offset( gbl_text_buffer, &iter, 0 );
  }

  /* Print message */
  gtk_text_buffer_insert_with_tags_by_name(
	  gbl_text_buffer, &iter, mesg, -1, attr, NULL );
  gtk_text_buffer_insert( gbl_text_buffer, &iter, "\n", -1 );

  /* Scroll Text View to bottom */
  adjustment = gtk_scrolled_window_get_vadjustment
	( GTK_SCROLLED_WINDOW(gbl_text_scroller) );
  gtk_adjustment_set_value(
	  adjustment, adjustment->upper - adjustment->page_size );

  /* Wait for GTK to complete its tasks */
  while( g_main_context_iteration(NULL, FALSE) );

} /* End of Show_Message() */

/*------------------------------------------------------------------------*/

/***  Memory allocation/freeing utils ***/
gboolean mem_alloc( void **ptr, size_t req )
{
  if( req == 0 ) return(FALSE );
  free_ptr( ptr );
  *ptr = malloc( req );
  if( *ptr == NULL )
  {
	perror( "xwxapt: A memory allocation request failed" );
	Show_Message( _("A memory allocation request failed - Quit"), "red" );
	Error_Dialog();
	return( FALSE );
  }
  return( TRUE );
} /* End of void mem_alloc() */

/*------------------------------------------------------------------------*/

gboolean mem_realloc( void **ptr, size_t req )
{
  if( req == 0 ) return( FALSE );
  *ptr = realloc( *ptr, req );
  if( *ptr == NULL )
  {
	perror( "xwxapt: A memory allocation request failed" );
	Show_Message( _("A memory allocation request failed - Quit"), "red" );
	Error_Dialog();
	return( FALSE );
  }
  return( TRUE );
} /* End of void mem_realloc() */

/*------------------------------------------------------------------------*/

void free_ptr( void **ptr )
{
  if( *ptr != NULL )
	free( *ptr );
  *ptr = NULL;

} /* End of void free_ptr() */

/*------------------------------------------------------------------------*/

/* Open_File()
 *
 * Opens a file, aborts on error
 */

  gboolean
Open_File( FILE **fp, char *fname, const char *mode )
{
  /* Message buffer */
  char mesg[64];

  /* Open Channel A image file */
  *fp = fopen( fname, mode );
  if( *fp == NULL )
  {
	perror( fname );
	snprintf( mesg, sizeof(mesg),
		_("Failed to open file\n%s"), Fname(fname) );
	Show_Message( mesg, "red" );
	Error_Dialog();
	return( FALSE );
  }

  return( TRUE );
} /* End of Open_File() */

/*------------------------------------------------------------------------*/

/* File_Image()
 *
 * Write an image buffer to file
 */

gboolean
File_Image(
	FILE *fp, const char *type,
	int width, int height,
	int max_val, unsigned char *buffer)
{
  size_t size;

  /* Write header in Ch-A output PPM files */
  if( fprintf(fp, "%s\n%s\n%d %d\n%d\n",
		type, _("# Created by xwxapt"), width, height, max_val) < 0 )
  {
	perror( "xwxapt: Error writing image to file" );
	Show_Message( _("Error writing image to file"), "red" );
	Error_Dialog();
	return( FALSE );
  }

  /* P6 type (PPM) files are 3* size in pixels */
  if( strcmp(type, "P6") == 0 )
	size = (size_t)(3 * width * height);
  else
	size = (size_t)(width * height);

  /* Write image buffer to file, abort on error */
  if( fwrite(buffer, 1, size, fp) != size )
  {
	perror( "xwxapt: Error writing image to file" );
	Show_Message( _("Error writing image to file"), "red" );
	Error_Dialog();
	return( FALSE );
  }

  return( TRUE );

} /* File_Image() */

/*------------------------------------------------------------------------*/

/*  Cleanup()
 *
 *  Cleanup before quitting or not using sound card
 */

  void
Cleanup( void )
{
  if( gbl_snd_buffer != NULL )
  {
	free( gbl_snd_buffer );
	gbl_snd_buffer = NULL;
  }
  Close_Capture();
  Close_Mixer();
  Close_RTL_Device();

  gbl_image_file[0]   = '\0';
  gbl_samples_file[0] = '\0';

  /* Clear status indications */
  Set_Sync_Icon( "gtk-stop" );

  ClearFlag( ACTION_FLAGS_ALL );
  ClearFlag( ICON_FLAGS_ALL );

} /*  Cleanup() */

/*------------------------------------------------------------------------*/

/* Functions for testing and setting/clearing flags */

/* An int variable holding the single-bit flags */
static int Flags = 0;

  int
isFlagSet(int flag)
{
  return (Flags & flag);
}

  int
isFlagClear(int flag)
{
  return (~Flags & flag);
}

  void
SetFlag(int flag)
{
  Flags |= flag;
}

  void
ClearFlag(int flag)
{
  Flags &= ~flag;
}

  void
ToggleFlag(int flag)
{
  Flags ^= flag;
}

/*------------------------------------------------------------------*/

/* Strlcpy()
 *
 * Copies n-1 chars from src string into dest string. Unlike other
 * such library fuctions, this makes sure that the dest string is
 * null terminated by copying only n-1 chars to leave room for the
 * terminating char. n would normally be the sizeof(dest) string but
 * copying will not go beyond the terminating null of src string
 */
  void
Strlcpy( char *dest, const char *src, size_t n )
{
  char ch = src[0];
  int idx = 0;

  /* Leave room for terminating null in dest */
  n--;

  /* Copy till terminating null of src or to n-1 */
  while( (ch != '\0') && (n > 0) )
  {
	dest[idx] = src[idx];
	idx++;
	ch = src[idx];
	n--;
  }

  /* Terminate dest string */
  dest[idx] = '\0';

} /* Strlcpy() */

/*------------------------------------------------------------------*/

/* Strlcat()
 *
 * Concatenates at most n-1 chars from src string into dest string.
 * Unlike other such library fuctions, this makes sure that the dest
 * string is null terminated by copying only n-1 chars to leave room
 * for the terminating char. n would normally be the sizeof(dest)
 * string but copying will not go beyond the terminating null of src

 */
  void
Strlcat( char *dest, const char *src, size_t n )
{
  char ch = dest[0];
  int idd = 0; /* dest index */
  int ids = 0; /* src  index */

  /* Find terminating null of dest */
  while( (n > 0) && (ch != '\0') )
  {
	idd++;
	ch = dest[idd];
	n--; /* Count remaining char's in dest */
  }

  /* Copy n-1 chars to leave room for terminating null */
  n--;
  ch = src[ids];
  while( (n > 0) && (ch != '\0') )
  {
	dest[idd] = src[ids];
	ids++;
	ch = src[ids];
	idd++;
	n--;
  }

  /* Terminate dest string */
  dest[idd] = '\0';

} /* Strlcat() */

/*------------------------------------------------------------------*/

