/*  sound.c
 *
 *  Soundcard handling functions of xwxapt application
 */

/* xwxapt: An application to decode APT signals from
 * weather satellites and produce image(s) of the weather.
 *
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "sound.h"
#include "shared.h"

/* Receive samples buffer */
static short *recv_buffer  = NULL;

static int
  recv_buf_size,	/* Receive DSP signal samples buffer */
  recv_buf_idx;		/* Index to Receive signal samples buffer */

/* ALSA pcm capture and mixer handles */
static snd_pcm_t *capture_handle  = NULL;
static snd_mixer_t *mixer_handle  = NULL;
static snd_pcm_hw_params_t *hw_params = NULL;

/* Simple mixer elements for setting up
 * capture amd playback sources and volume */
static snd_mixer_elem_t *cap_elem = NULL;

/*------------------------------------------------------------------------*/

/* Open_PCM()
 *
 * Opens a pcm device for a given handle
 */
  gboolean
Open_PCM(
	snd_pcm_t **handle,
	snd_pcm_stream_t stream,
	char *mesg, int *error )
{
  /* Open pcm */
  *error = snd_pcm_open(
	  handle, rc_data.pcm_dev, stream, SND_PCM_ASYNC );
  if( *error < 0 )
  {
	Strlcat( mesg, _("Cannot open sound device "), MESG_SIZE );
	Strlcat( mesg, rc_data.pcm_dev, MESG_SIZE );
	return( FALSE );
  }

  /* Allocate memory to hardware parameters structure */
  *error = snd_pcm_hw_params_malloc( &hw_params );
  if( *error < 0 )
  {
	Strlcat( mesg, _("Cannot allocate hw_params struct"), MESG_SIZE );
	return( FALSE );
  }

  /* Initialize hardware parameter structure */
  *error = snd_pcm_hw_params_any( *handle, hw_params );
  if( *error < 0 )
  {
	Strlcat( mesg, _("Cannot initialize hw_params struct"), MESG_SIZE );
	return( FALSE );
  }

  /* Set access type */
  *error = snd_pcm_hw_params_set_access(
	  *handle, hw_params, SND_PCM_ACCESS );
  if( *error < 0 )
  {
	Strlcat( mesg, _("Cannot set PCM access type"), MESG_SIZE );
	return( FALSE );
  }

  /* Set sample format */
  *error = snd_pcm_hw_params_set_format(
	  *handle, hw_params, SND_PCM_FORMAT );
  if( *error < 0 )
  {
	Strlcat( mesg, _("Cannot set sample format"), MESG_SIZE );
	return( FALSE );
  }

  /* Set sample rate */
  *error = snd_pcm_hw_params_set_rate(
	  *handle, hw_params, (unsigned int)SND_DSP_RATE, EXACT_VAL );
  if( *error < 0 )
  {
	snprintf( mesg, MESG_SIZE,
		_("Cannot set sample rate to %d"), SND_DSP_RATE );
	return( FALSE );
  }

  /* Set channel count */
  *error = snd_pcm_hw_params_set_channels(
	  *handle, hw_params, (unsigned int)rc_data.num_chn );
  if( *error < 0 )
  {
	snprintf( mesg, MESG_SIZE,
		_("Cannot set channel count to %d"), rc_data.num_chn );
	return( FALSE );
  }

  /* Set number of periods */
  *error = snd_pcm_hw_params_set_periods(
	  *handle, hw_params, NUM_PERIODS, EXACT_VAL );
  if( *error < 0)
  {
	snprintf( mesg, MESG_SIZE,
		_("Cannot set number periods to %d"), NUM_PERIODS );
	return( FALSE );
  }

  /* Set period size */
  *error = snd_pcm_hw_params_set_period_size(
	  *handle, hw_params, PERIOD_SIZE, EXACT_VAL );
  if( *error < 0)
  {
	snprintf( mesg, MESG_SIZE,
		_("Cannot set period size to %d"), PERIOD_SIZE );
	return( FALSE );
  }

  /* Set parameters */
  *error = snd_pcm_hw_params( *handle, hw_params );
  if( *error < 0 )
  {
	Strlcat( mesg, _("Cannot set capture parameters"), MESG_SIZE );
	return( FALSE );
  }
  snd_pcm_hw_params_free( hw_params );
  hw_params = NULL;

  /* Prepare sound interface for use */
  *error = snd_pcm_prepare( *handle );
  if( *error < 0 )
  {
	Strlcat( mesg, _("Cannot prepare sound interface"), MESG_SIZE );
	return( FALSE );
  }

  return( TRUE );
} /* Open_PCM() */

/*------------------------------------------------------------------------*/

/* Open_Capture()
 *
 * Opens sound card for Capture
 */
  gboolean
Open_Capture( char *mesg, int *error )
{
  /* To test Return value of functions */
  gboolean ret;

  /* Return if Capture is setup */
  if( isFlagSet(CAPTURE_SETUP) ) return( TRUE );

  /*** Open/setup dsp/pcm and mixer ***/
  Show_Message( _("Setting up Sound Capture"), "black" );

  /* Open & setup pcm for Capture */
  if( !Open_PCM(
		&capture_handle,
		SND_PCM_STREAM_CAPTURE,
		mesg, error) )
	return( FALSE );

  /* Allocate memory to DSP data buffer */
  rc_data.snd_buf_size = rc_data.num_chn * PERIOD_SIZE;
  if( gbl_snd_buffer == NULL )
  {
	ret = mem_alloc( (void *)&gbl_snd_buffer,
		(size_t)rc_data.snd_buf_size * sizeof(short) );
	if( !ret )
	{
	  Show_Message( _("Failed to allocate SND buffer"), "red" );
	  *error = 0;
	  Error_Dialog();
	  return( FALSE );
	}
	memset( gbl_snd_buffer, 0,
		(size_t)rc_data.snd_buf_size * sizeof(short) );
  }

  /* Size of receive samples buffer in 'shorts' */
  recv_buf_size = PERIOD_SIZE * rc_data.num_chn;

  /* Index to recv samples buffer (set to end) */
  recv_buf_idx = recv_buf_size;

  /* Allocate memory to receive samples buffer */
  if( recv_buffer == NULL )
  {
	ret = mem_alloc( (void **)&recv_buffer,
		(size_t)recv_buf_size * sizeof(short) );
	if( !ret )
	{
	  Show_Message( _("Failed to allocate SND Recv buffer"), "red" );
	  *error = 0;
	  Error_Dialog();
	  return( FALSE );
	}
	memset( recv_buffer, 0,
		(size_t)recv_buf_size * sizeof(short) );
  }

  /* Open mixer & set playback voulume, abort on failure.
   * Failure to set volume level is not considered fatal */
  if( !Open_Mixer(mesg, error) ) return( FALSE );
  Set_Capture_Level( rc_data.cap_lev, mesg, error );

  Show_Message( _("Sound Capture Set Up OK"), "green" );
  SetFlag( CAPTURE_SETUP );
  return( TRUE );
} /* Open_Capture() */

/*------------------------------------------------------------------------*/

/* Open_Mixer()
 *
 * Opens mixer interface
 */
  gboolean
Open_Mixer( char *mesg, int *error )
{
  snd_mixer_elem_t *elem;
  snd_mixer_selem_id_t *sid;

  /* Abort if mixer already setup */
  if( isFlagSet(MIXER_SETUP) ) return( TRUE );

  /* Open mixer handle */
  *error = snd_mixer_open( &mixer_handle, 0 );
  if( *error < 0 )
  {
	Strlcat( mesg, _("Cannot open mixer handle"), MESG_SIZE );
	return( FALSE );
  }

  /* Attach mixer */
  *error = snd_mixer_attach( mixer_handle, rc_data.pcm_dev );
  if( *error < 0 )
  {
	Strlcat( mesg, _("Cannot attach mixer to "), MESG_SIZE );
	Strlcat( mesg, rc_data.pcm_dev, MESG_SIZE );
	return( FALSE );
  }

  /* Register mixer */
  *error = snd_mixer_selem_register( mixer_handle, NULL, NULL );
  if( *error < 0 )
  {
	Strlcat( mesg, _("Cannot register mixer"), MESG_SIZE );
	return *error;
  }

  /* Load mixer */
  *error = snd_mixer_load( mixer_handle );
  if( *error < 0 )
  {
	Strlcat( mesg, _("Cannot load mixer"), MESG_SIZE );
	return( FALSE );
  }

  /* Allocate selem_id structure */
  *error = snd_mixer_selem_id_malloc( &sid );
  if( *error < 0 )
  {
	Strlcat( mesg, _("Cannot allocate selem_id struct"), MESG_SIZE );
	return( FALSE );
  }

  /* Find capture source selem */
  snd_mixer_selem_id_set_index( sid, 0 );
  snd_mixer_selem_id_set_name( sid, rc_data.cap_src );
  elem = snd_mixer_find_selem( mixer_handle, sid );
  if( elem == NULL )
  {
	Strlcat( mesg, _("Cannot find capture source element "), MESG_SIZE );
	Strlcat( mesg, rc_data.cap_src, MESG_SIZE );
	snd_mixer_selem_id_free(sid);
	*error = 0;
	return( FALSE );
  }

  /* Set capture switch for capture source */
  if( snd_mixer_selem_has_capture_switch(elem) )
  {
	*error = snd_mixer_selem_set_capture_switch(
		elem, rc_data.channel, 1 );
	if( *error < 0 )
	{
	  Strlcat( mesg, _("Cannot set capture device "), MESG_SIZE );
	  Strlcat( mesg, rc_data.cap_src, MESG_SIZE );
	  snd_mixer_selem_id_free(sid);
	  return( FALSE );
	}
  }
  else
  {
	snprintf( mesg, MESG_SIZE,
		_("Device %s does not have Capture capability"),
		rc_data.cap_src );
	snd_mixer_selem_id_free(sid);
	*error = 0;
	return( FALSE );
  }

  /* Find capture volume selem if not -- */
  if( strcmp(rc_data.cap_vol, "--") != 0 )
  {
	snd_mixer_selem_id_set_index( sid, 0 );
	snd_mixer_selem_id_set_name( sid, rc_data.cap_vol );
	cap_elem = snd_mixer_find_selem( mixer_handle, sid );
	if( !cap_elem )
	{
	  snprintf( mesg, MESG_SIZE,
		  _("Cannot find Capture volume %s\n"),
		  rc_data.cap_vol );
	  Show_Message( mesg, "red" );
	}
  }
  snd_mixer_selem_id_free(sid);

  SetFlag( MIXER_SETUP );
  return( TRUE );
} /* Open_Mixer() */

/*------------------------------------------------------------------------*/

/* Set_Capture_Level()
 *
 * Sets Capture Control level
 */
  gboolean
Set_Capture_Level( int level, char *mesg, int *error )
{
  long cmin, cmax;

  /* Abort with no error if Mixer not setup */
  if( mixer_handle == NULL ) return( TRUE );

  /* Set capture volume */
  if( cap_elem != NULL )
  {
	if( snd_mixer_selem_has_capture_volume(cap_elem) )
	{
	  /* Change from % volume to sound card value */
	  long lev;
	  snd_mixer_selem_get_capture_volume_range(
		  cap_elem, &cmin, &cmax );
	  lev = cmin + ((cmax - cmin) * level) / 100;

	  /* Set capture volume */
	  *error = snd_mixer_selem_set_capture_volume(
		  cap_elem, rc_data.channel, lev );
	  if( *error < 0 )
	  {
		snprintf( mesg, MESG_SIZE,
			_("Cannot set capture volume to %d\n"\
			  "Error: %s"),
			level, snd_strerror(*error) );
		Show_Message( mesg, "red" );
		return( FALSE );
	  }
	}
  } /* if( cap_elem != NULL ) */

  return( TRUE );
} /* Set_Capture_Level() */

/*------------------------------------------------------------------------*/

/*
 * Following 2 functions close sound card interfaces
 */

  void
Close_Capture( void )
{
  if( capture_handle != NULL )
	snd_pcm_close( capture_handle );
  capture_handle = NULL;

  if( hw_params != NULL )
	snd_pcm_hw_params_free( hw_params );
  hw_params = NULL;

  ClearFlag(CAPTURE_SETUP);
}

/*------------------------------------------------------------------------*/

  void
Close_Mixer( void )
{
  if( mixer_handle != NULL )
	snd_mixer_close( mixer_handle );
  mixer_handle = NULL;
  ClearFlag(MIXER_SETUP);
}

/*------------------------------------------------------------------------*/

/* Read_SND_Buffer()
 *
 * Reads the DSP's buffer and separates Left/Right
 * channel data depending on stereo or mono mode
 */

  gboolean
Read_SND_Buffer( short *buffer, int buff_size )
{
  int idx;
  snd_pcm_sframes_t error;
  static int det_dft_idx = 0;


  /* Transfer data from dsp buffer to buffer to be processed, */
  /* separating left or right channel data if in stereo mode  */
  if( buff_size >= BLOCK_BUFFER_SIZE ) buff_size -= BLOCK_BUFFER_SIZE;
  for( idx = 0; idx < buff_size; idx++ )
  {
	if( snd_buf_idx >= rc_data.snd_buf_size )
	{
	  /* Start buffer index according to channel */
	  snd_buf_idx = rc_data.use_chn;

	  /* Fill in the samples buffer from DSP, return -1 on error */
	  error = snd_pcm_readi(
		  capture_handle, gbl_snd_buffer, PERIOD_SIZE );
	  if( error != PERIOD_SIZE )
	  {
		char mesg[MESG_SIZE];
		snprintf( mesg, MESG_SIZE,
			_("Read from sound interface failed\n"\
			  "Error: %s"), snd_strerror((int)error) );
		Show_Message( mesg, "red" );

		/* Try to recover from error */
		if( !Xrun_Recovery(capture_handle, (int)error) )
		{
		  snprintf( mesg, MESG_SIZE,
			  _("Failed to recover from %s"), snd_strerror((int)error) );
		  Show_Message( mesg, "red" );
		  Cleanup();
		  Error_Dialog();
		  return( FALSE );
		}
	  } /* if( error != NUM_FRAMES ) */
	} /* if( snd_buf_idx >= rc_data.snd_buf_size ) */

	/* Fill buffer from left or right channel. snd_buf_idx
	 * is arranged to point to the correct location */
	buffer[idx] = gbl_snd_buffer[snd_buf_idx];

	/* Simulates a pulsed 2.4kHz signal
	{
	  static double w  = 0.0;
	  static double dw = TWOPI * 2400.0 / 48000.0;
	  static int cnt = 0;

	  if( cnt < 46 )
		buffer[idx] = 0;
	  else if( cnt < 69 )
		buffer[idx] = (short)( 32000.0 * sin(w) );
	  else if( cnt < 92 )
		buffer[idx] = 0;
	  else if( cnt < 115 )
		buffer[idx] = (short)( 32000.0 * sin(w) );
	  else if( cnt < 138 )
		buffer[idx] = 0;
	  else if( cnt < 161 )
		buffer[idx] = (short)( 32000.0 * sin(w) );
	  else if( cnt < 184 )
		buffer[idx] = 0;
	  else if( cnt < 207 )
		buffer[idx] = (short)( 32000.0 * sin(w) );
	  else if( cnt < 230 )
		buffer[idx] = 0;
	  else if( cnt < 253 )
		buffer[idx] = (short)( 32000.0 * sin(w) );
	  else if( cnt < 276 )
		buffer[idx] = 0;
	  else if( cnt < 299 )
		buffer[idx] = (short)( 32000.0 * sin(w) );
	  else if( cnt < 322 )
		buffer[idx] = 0;
	  else if( cnt < 345 )
		buffer[idx] = (short)( 32000.0 * sin(w) );
	  else if( cnt < 448 )
		buffer[idx] = 0;
	  else if( cnt < 1004 )
		buffer[idx] = (short)( 10000.0 * sin(w) );
	  else
	  {
		if( (cnt / 400) & 1 )
		  buffer[idx] = (short)( 32000.0 * sin(w) );
		else buffer[idx] = 0;
	  }

	  w += dw;
	  if( w >= TWOPI ) w -= TWOPI;
	  if( cnt++ == 24000 ) cnt = 0;
	} */

	/* Increment according to mono/stereo mode */
	snd_buf_idx += rc_data.num_chn;

	/* Display waterfall when input buffer full */
	if( isFlagClear(CARRIER_SPECTRUM) &&
	  (det_dft_idx < DET_DFT_INPUT_SIZE) )
	{
	  /* Take a sample for the DFT */
	  dft_in_i[det_dft_idx] = buffer[idx];
	  det_dft_idx++;
	}

  } /* for( idx = 0; idx < buff_size; idx++ ) */

  /* Display selected waterfall */
  if( isFlagClear(CARRIER_SPECTRUM) )
  {
	det_dft_idx = 0;
	Display_Waterfall( DET_DFT_INPUT_SIZE );
  }

  return( TRUE );
} /* End of Read_SND_Buffer() */

/*------------------------------------------------------------------------*/

/* Xrun_Recovery()
 *
 * Recover from underrrun (broken pipe) and suspend
 */
  gboolean
Xrun_Recovery( snd_pcm_t *handle, int error )
{
  char mesg[MESG_SIZE];
  if( error == -EPIPE )
  {
	error = snd_pcm_prepare( handle );
	if( error < 0 )
	{
	  snprintf( mesg, sizeof(mesg),
		  _("Cannot recover from underrun, prepare failed\n"\
			"Error: %s\n"), snd_strerror(error) );
	  Show_Message( mesg, "red" );
	  return( FALSE );
	}
  }
  else if( error == -ESTRPIPE )
  {
	while( (error = snd_pcm_resume(handle)) == -EAGAIN )
	  sleep(1);
	if( error < 0 )
	{
	  error = snd_pcm_prepare( handle );
	  if( error < 0 )
	  {
		snprintf( mesg, sizeof(mesg),
			_("Cannot recover from suspend, prepare failed\n"\
			  "Error: %s\n"), snd_strerror(error) );
		Show_Message( mesg, "red" );
		return( FALSE );
	  }
	}
  }

  return( TRUE );
} /* Xrun_Recovery() */

/*------------------------------------------------------------------------*/

