/*  monitors.c
 *
 *  Visual Monitors, DFT Spectrum and FM Demodulator Scope
 */

/*
 *  xwxapt: An application to decode APT signals from
 *  weather satellites and produce image(s) of the weather.
 *
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "monitors.h"
#include "shared.h"

/* Scaled-up, integer, sin/cos tables */
static int
  *isin = NULL,
  *icos = NULL;

/* DFT In-phase and Quadature output arrays */
static int
  *dft_out_i = NULL,
  *dft_out_q = NULL;

/*------------------------------------------------------------------------*/

/* Idft_Init()
 *
 * Initializes Idft()
 */
  void
Idft_Init( int dft_input_size, int dft_bin_size )
{
  int i;
  size_t mreq;
  double w, dw;

  /* Allocate dft buffers */
  mreq = (size_t)dft_input_size * sizeof(int);
  mem_realloc( (void **)&dft_in_i, mreq );
  mem_realloc( (void **)&dft_in_q, mreq );
  mem_realloc( (void **)&isin, mreq );
  mem_realloc( (void **)&icos, mreq );

  mreq = (size_t)dft_bin_size * sizeof(int);
  mem_realloc( (void **)&dft_out_i, mreq );
  mem_realloc( (void **)&dft_out_q, mreq );

  /* Make sin/cos tables */
  dw = 2.0 * M_PI / (double)dft_input_size;
  for( i = 0; i < dft_input_size; i++ )
  {
	w = dw * (double)i;
	isin[i] = (int)( 127.0 * sin(w) + 0.5 );
	icos[i] = (int)( 127.0 * cos(w) + 0.5 );
  }

} /* Idft_Init() */

/*------------------------------------------------------------------------*/

/* Idft_Demod()
 *
 * Simple, integer-only, DFT function
 */
  void
Idft_Demod( int dft_input_size, int dft_bin_size )
{
  int i, j, w;

  /* In-phase and quadrature summation */
  int sum_i, sum_q, scale;


  /* Scale factor to keep o/p in range */
  scale = dft_input_size * DET_DFT_SCALE;

  /* Calculate output bins */
  for( i = 0; i < dft_bin_size; i++ )
  {
	sum_i = sum_q = w = 0;

	/* Summate input values */
	for( j = 0; j < dft_input_size; j++ )
	{
	  sum_i += dft_in_i[j] * isin[w];
	  sum_q += dft_in_i[j] * icos[w];
	  w += i;
	  if( w >= dft_input_size ) w -= dft_input_size;
	}

	/* Normalized summations to bins */
	dft_out_i[i] = sum_i / scale;
	dft_out_q[i] = sum_q / scale;

  } /* for( i = 0; i < dft_bin_size; i++ ) */

} /* Idft_Demod() */

/*------------------------------------------------------------------------*/

/* Idft_SDR()
 *
 * Simple, integer-only, DFT function
 */
  void
Idft_SDR( int dft_input_size, int dft_bin_size )
{
  int i, j, k, bin_mid;

  /* Index for the rotation of the reference sinewave */
  int wr;

  /* In-phase and quadrature summation */
  int sum_i, sum_q, scale;


  /* Middle bin */
  bin_mid = dft_bin_size / 2;

  /* Scale factor to keep o/p in range */
  scale = dft_input_size * SDR_DFT_SCALE;

  /* Calculate output bins for "negative" frequencies */
  for( i = 0; i < bin_mid; i++ )
  {
	sum_i = sum_q = 0;
	wr = 0;

	/* Summate input values */
	for( j = 0; j < dft_input_size; j++ )
	{
	  sum_i += dft_in_i[j] * icos[wr] + dft_in_q[j] * isin[wr];
	  sum_q += dft_in_q[j] * icos[wr] - dft_in_i[j] * isin[wr];
	  k = bin_mid - i;
	  wr -= k;
	  if( wr < 0 ) wr += dft_input_size;
	}

	/* Normalized summations to bins */
	dft_out_i[i] = sum_i / scale;
	dft_out_q[i] = sum_q / scale;

  } /* for( i = 0; i < bin_mid; i++ ) */

  /* Calculate output bins for "positive" frequencies */
  for( i = bin_mid; i < dft_bin_size; i++ )
  {
	sum_i = sum_q = 0;
	wr = 0;

	/* Summate input values */
	for( j = 0; j < dft_input_size; j++ )
	{
	  sum_i += dft_in_i[j] * icos[wr] + dft_in_q[j] * isin[wr];
	  sum_q += dft_in_q[j] * icos[wr] - dft_in_i[j] * isin[wr];
	  k = i - bin_mid;
	  wr += k;
	  if( wr >= dft_input_size ) wr -= dft_input_size;
	}

	/* Normalized summations to bins */
	dft_out_i[i] = sum_i / scale;
	dft_out_q[i] = sum_q / scale;

  } /* for( i = bin_mid; i < dft_bin_size; i++ ) */

} /* Idft_SDR() */

/*------------------------------------------------------------------------*/

/* DFT_Bin_Value()
 *
 * Calculates DFT bin values with auto level control
 */
  int
DFT_Bin_Value( int sum_i, int sum_q, gboolean reset )
{
 /* Value of dft output "bin" */
  static int bin_val = 0;

  /* Maximum value of dft bins */
  static int bin_max = 1000, max = 0;


  /* Calculate sliding window average of max bin value */
  if( reset )
  {
	bin_max = max;
	if( !bin_max ) bin_max = 1;
	max = 0;
  }
  else
  {
	/* Calculate average signal power at each frequency (bin) */
	bin_val  = bin_val * AMPL_AVE_MUL;
	bin_val += sum_i * sum_i + sum_q * sum_q;
	bin_val /= AMPL_AVE_WIN;

	/* Record max bin value */
	if( max < bin_val )
	  max = bin_val;

	/* Scale bin values to 255 depending on max value */
	int ret = (255 * bin_val) / bin_max;
	if( ret > 255 ) ret = 255;
	return( ret );
  }

  return( 0 );
} /* DFT_Bin_Value() */

/*------------------------------------------------------------------------*/

/* Display_Waterfall()
 *
 * Displays DFT Spectrum as "waterfall"
 */
  void
Display_Waterfall( int dft_input_size )
{
  int
	vert_lim,  /* Limit of vertical index for copying lines */
	idh, idv,  /* Index to hor. and vert. position in warterfall */
	pixel_val, /* Greyscale value of pixel derived from dft o/p  */
	dft_idx;   /* Index to dft output array */

  /* Pointer to current pixel */
  static guchar *pix;


  /* Copy each line of waterfall to next one */
  vert_lim = gbl_wfall_height - 2;
  for( idv = vert_lim; idv > 0; idv-- )
  {
	pix = gbl_wfall_pixels + gbl_wfall_rowstride * idv + gbl_wfall_n_channels;

	for( idh = 0; idh < gbl_wfall_width; idh++ )
	{
	  pix[0] = pix[ -gbl_wfall_rowstride];
	  pix[1] = pix[1-gbl_wfall_rowstride];
	  pix[2] = pix[2-gbl_wfall_rowstride];
	  pix += gbl_wfall_n_channels;
	}
  }

  /* Go to top left +1 hor. +1 vert. of pixbuf */
  pix = gbl_wfall_pixels + gbl_wfall_rowstride + gbl_wfall_n_channels;

  /* Do the DFT on input array */
  if( isFlagSet(CARRIER_SPECTRUM) )
	Idft_SDR( dft_input_size, gbl_wfall_width );
  else
	Idft_Demod( dft_input_size, gbl_wfall_width );

  /* Calculate bin values after DFT */
  //bin_max = 0;
  for( dft_idx = 0; dft_idx < gbl_wfall_width; dft_idx++ )
  {
	/* Calculate signal power at each frequency (bin) */
	pixel_val = DFT_Bin_Value(
		  dft_out_i[dft_idx], dft_out_q[dft_idx], FALSE );

	/* Record max bin value
	if( bin_max < bin_val ) bin_max = bin_val; */

	/* Scale pixel values to 255 depending on ave max value
	pixel_val = (255 * bin_val) / ave_max;
	if( pixel_val > 255 ) pixel_val = 255; */

	/* Color code signal strength */
	int n;
	if( pixel_val < 85 )
	{
	  pix[0] = 0;
	  pix[1] = 0;
	  pix[2] = 3 + (guchar)pixel_val * 3;
	}
	else if( pixel_val < 170 )
	{
	  n = pixel_val - 85;
	  pix[0] = 0;
	  pix[1] = 3 + (guchar)n * 3;
	  pix[2] = 255 - (guchar)n * 3;
	}
	else
	{
	  n = pixel_val - 170;
	  pix[0] = (guchar)pixel_val;
	  pix[1] = 255;
	  pix[2] = 0;
	}

	pix += gbl_wfall_n_channels;

  } /* for( dft_idx = 0; dft_idx < gbl_wfall_width; dft_idx++ ) */

  /* Reset function */
  pixel_val = DFT_Bin_Value(
	  dft_out_i[0], dft_out_q[0], TRUE );

  /* Calculate sliding window average of max bin value
  ave_max  = ave_max * BINMAX_AVE_MUL;
  ave_max += bin_max;
  ave_max /= BINMAX_AVE_WIN;
  if( !ave_max ) ave_max = 1; */

  /* At last draw waterfall */
  gtk_widget_queue_draw( gbl_waterfall );

} /* Display_Waterfall() */

/*------------------------------------------------------------------------*/

/*  Display_Signal()
 *
 *  Displays the sub-carrier signal amplitude
 */

  void
Display_Signal( int plot )
{
  /* Points to plot */
  static GdkPoint *points = NULL;
  static int points_idx = 0;
  static cairo_t *cr = NULL;

  /* Initialize on first call */
  if( points == NULL )
  {
	if( !mem_alloc( (void *)&points,
		  (size_t)gbl_wfall_width * sizeof(GdkPoint)) )
	  return;
  }

  /* Initialize cairo */
  if( cr == NULL )
	cr = gdk_cairo_create( gbl_scope->window );

  /* Save values to be plotted (scaled to fit display) */
  points[points_idx].y = gbl_scope_height - plot - 1;
  if( points[points_idx].y <= 0 )
	points[points_idx].y = 1;
  if( points[points_idx].y >= gbl_scope_height )
	points[points_idx].y = gbl_scope_height - 1;

  points[points_idx].x = points_idx;

  /* Recycle buffer idx when full and plot */
  points_idx++;
  if( points_idx >= gbl_scope_width )
  {
	int idx;

	/* Draw scope background */
	cairo_set_source_rgb( cr, 0.0, 0.3, 0.0 );
	cairo_rectangle(
		cr, 0.0, 0.0,
		(double)gbl_scope_width,
		(double)gbl_scope_height );
	cairo_fill( cr );

	/* Plot signal graph */
	cairo_set_source_rgb( cr, 0.0, 1.0, 0.0 );
	cairo_move_to( cr, (double)points[0].x, (double)points[0].y );
	for( idx = 1; idx < points_idx; idx++ )
	  cairo_line_to( cr, (double)points[idx].x, (double)points[idx].y );

	/* Stroke paths */
	cairo_stroke( cr );

	points_idx = 0;
  } /* if( points_idx >= gbl_scope_width ) */

} /* Display_Signal */

/*------------------------------------------------------------------------*/

