/*  main.c
 *
 *  Main body of xwxapt application
 */

/*
 *  xwxapt: An application to decode APT signals from
 *  weather satellites and produce image(s) of the weather.
 *
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include "main.h"
#include "shared.h"


/* Signal handler */
static void sig_handler( int signal );

/*------------------------------------------------------------------------*/

  int
main (int argc, char *argv[])
{
  /* Command line option returned by getopt() */
  int option;

  /* New and old actions for sigaction() */
  struct sigaction sa_new, sa_old;


  /* Initialize new actions */
  sa_new.sa_handler = sig_handler;
  sigemptyset( &sa_new.sa_mask );
  sa_new.sa_flags = 0;

  /* Register function to handle signals */
  sigaction( SIGINT,  &sa_new, &sa_old );
  sigaction( SIGSEGV, &sa_new, 0 );
  sigaction( SIGFPE,  &sa_new, 0 );
  sigaction( SIGTERM, &sa_new, 0 );
  sigaction( SIGABRT, &sa_new, 0 );
  sigaction( SIGCONT, &sa_new, 0 );

  /* Process command line options. Defaults below. */
  rc_data.rtlsdr_center_freq = 137100000;
  gbl_sat_type = NOAA_19;
  while( (option = getopt(argc, argv, "b:hf:v") ) != -1 )
	switch( option )
	{
	  case 'b' : /* LPF bandwidth */
		rc_data.rtlsdr_lpf_bw = atoi( optarg );
		break;

	  case 'f' : /* Get center frequency */
		rc_data.rtlsdr_center_freq = atoi( optarg );
	  break;

	  case 'h' : /* Print usage and exit */
		Usage();
		return(0);

	  case 'v' : /* Print version */
		puts( PACKAGE_STRING );
		return(0);

	  default: /* Print usage and exit */
		Usage();
		exit(-1);

	} /* End of switch( option ) */

  /* Start GTK+ */
  gtk_set_locale();
  gtk_init( &argc, &argv );

  /* Defaults/initialization */
  rc_data.default_dur = gbl_duration = 600;
  gbl_image_file[0]   = '\0';
  gbl_samples_file[0] = '\0';
  SetFlag( IMAGE_NORMALIZE | IMAGE_COLORIZE );
  SetFlag( CARRIER_SPECTRUM );

  /* Create xwxapt window */
  xwxapt_main_window = create_main_window();
  gtk_window_set_title( GTK_WINDOW(xwxapt_main_window), PACKAGE_STRING );

  /* Create global sync status widgets' table */
  gbl_status_table = lookup_widget( xwxapt_main_window, "status_frame" );

  /* Create global sync status indicator icon */
  gbl_sync_icon = lookup_widget( xwxapt_main_window, "sync_ind" );

  /* Create the text view scroller */
  gbl_text_scroller = lookup_widget(
	  xwxapt_main_window, "text_scrolledwindow" );

  /* Get text buffer */
  gbl_text_buffer = gtk_text_view_get_buffer
	( GTK_TEXT_VIEW(lookup_widget(xwxapt_main_window, "textview")) );

  /* Get waterfall and scope widgets */
  gbl_scope     = lookup_widget( xwxapt_main_window, "demod_drawingarea" );
  gbl_waterfall = lookup_widget( xwxapt_main_window, "dft_drawingarea" );

  /* Create some rendering tags */
  gtk_text_buffer_create_tag( gbl_text_buffer, "black",
	  "foreground", "black", NULL);
  gtk_text_buffer_create_tag( gbl_text_buffer, "red",
	  "foreground", "red", NULL);
  gtk_text_buffer_create_tag( gbl_text_buffer, "orange",
	  "foreground", "orange", NULL);
  gtk_text_buffer_create_tag( gbl_text_buffer, "green",
	  "foreground", "darkgreen", NULL);
  gtk_text_buffer_create_tag( gbl_text_buffer, "bold",
	  "weight", PANGO_WEIGHT_BOLD, NULL);

  /* Create the progress bars */
  gbl_level_pbar = lookup_widget( xwxapt_main_window, "signal_lev" );
  gbl_sync_pbar  = lookup_widget( xwxapt_main_window, "sync_lev" );

  /* Open xwxapt window */
  gtk_widget_show( xwxapt_main_window );

  char ver[28];
  snprintf( ver, sizeof(ver), _("Welcome to %s"), PACKAGE_STRING );
  Show_Message( ver, "bold" );

  g_idle_add( Load_Config, NULL );

  gtk_main ();

  return 0;
}

/*------------------------------------------------------------------------*/

/*  Initialize_Top_Window()
 *
 *  Initializes xwxapt's top window
 */

  void
Initialize_Top_Window( void )
{
  /* The scrolled window image container */
  GtkWidget *image_scroller;
  char text[12];

  /* Show current satellite */
  GtkLabel *label = GTK_LABEL(
	  lookup_widget(xwxapt_main_window, "sat_label") );
  snprintf( text, sizeof(text), "%s-APT", gbl_sat_names[gbl_sat_type] );
  gtk_label_set_text( label, text );

  /* Show Center_Freq to frequency entry */
  Enter_Center_Freq( rc_data.rtlsdr_center_freq );

  /* Show Bandwidth to B/W entry */
  Enter_Filter_BW();

  /* Kill existing pixbuf */
  if( gbl_pixbuf != NULL )
  {
	g_object_unref( gbl_pixbuf );
	gbl_pixbuf = NULL;
  }

  /* Create new pixbuff for sat images */
  switch( gbl_sat_type )
  {
	case NOAA_15: case NOAA_18: case NOAA_19: /* Create pixbuff for NOAA images */
	  gbl_pixbuf = gdk_pixbuf_new( GDK_COLORSPACE_RGB, FALSE, 8,
		  NOAA_IMAGE_WIDTH, gbl_duration );
	  break;

	case METEOR: /* Create pixbuff for METEOR images */
	  gbl_pixbuf = gdk_pixbuf_new( GDK_COLORSPACE_RGB, FALSE, 8,
		  METEOR_IMAGE_WIDTH/2, gbl_duration );
  }

  /* Error, not enough memory */
  if( gbl_pixbuf == NULL)
  {
	Show_Message( _("Failed to allocated memory to pixbuf - Quit"), "red" );
	Error_Dialog();
	return;
  }

  /* Fill pixbuf with background color */
  gdk_pixbuf_fill( gbl_pixbuf, 0xaaaaaaff );

  /* Get details of pixbuf */
  gbl_pixel_buf  = gdk_pixbuf_get_pixels(     gbl_pixbuf );
  gbl_rowstride  = gdk_pixbuf_get_rowstride(  gbl_pixbuf );
  gbl_n_channels = gdk_pixbuf_get_n_channels( gbl_pixbuf );

  /* Globalize image to be displayed */
  gbl_apt_image = lookup_widget( xwxapt_main_window, "image" );

  /* Set apt image from pixbuff */
  gtk_image_set_from_pixbuf( GTK_IMAGE(gbl_apt_image), gbl_pixbuf );
  gtk_widget_show( gbl_apt_image );

  /* Set window size as required */
  gint wdt, hgt;
  image_scroller = lookup_widget(
	  xwxapt_main_window, "image_scrolledwindow" );

  if( gbl_sat_type == METEOR )
	wdt = METEOR_IMAGE_WIDTH / 2;
  else
	wdt = NOAA_IMAGE_WIDTH;

  if( gbl_duration <= rc_data.default_dur )
	wdt += 9;
  else
	wdt += 27;

  if( gbl_duration <= rc_data.window_height )
	hgt = gbl_duration + 8;
  else
	hgt = rc_data.window_height + 8;

  gtk_widget_set_size_request( image_scroller, wdt, hgt );
  gtk_window_resize( GTK_WINDOW(xwxapt_main_window), 10, 10 );

} /* Initialize_Top_Window() */

/*------------------------------------------------------------------------*/

/*  sig_handler()
 *
 *  Signal Action Handler function
 */

static void sig_handler( int signal )
{
  /* Internal wakeup call */
  if( signal == SIGCONT ) return;

  Cleanup();

  fprintf( stderr, "\n" );
  switch( signal )
  {
	case SIGINT :
	  fprintf( stderr, "%s\n", _("xwxapt: Exiting via User Interrupt") );
	  exit(-1);

	case SIGSEGV :
	  fprintf( stderr, "%s\n", _("xwxapt: Segmentation Fault") );
	  exit(-1);

	case SIGFPE :
	  fprintf( stderr, "%s\n", _("xwxapt: Floating Point Exception") );
	  exit(-1);

	case SIGABRT :
	  fprintf( stderr, "%s\n", _("xwxapt: Abort Signal received") );
	  exit(-1);

	case SIGTERM :
	  fprintf( stderr, "%s\n", _("xwxapt: Termination Request received") );
	  exit(-1);
  }

} /* End of sig_handler() */

/*------------------------------------------------------------------------*/

