/*  aptsignal.c
 *
 *  APT signal processing functions of xwxapt application
 */

/*
 *  xwxapt: An application to decode APT signals from
 *  weather satellites and produce an image of the weather.
 *
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "aptsignal.h"
#include "shared.h"

/*------------------------------------------------------------------------*/

/*  Detect_NOAA_Sync()
 *
 *  Detects the Channel A sync of a NOAA APT signal.
 *  This is 7 cycles at 1040 Hz on the 2.4 KHz sub-carrier.
 */

  int
Detect_NOAA_Sync( int search_limit )
{
  /* FIFO Buffer for ch A sync detection */
  static int *sync_fifo = NULL, sync_fifo_len;

  /* An amplitude sample of 2.4kHz sub-carrier */
  short	sample_a;

  int
	sync_idx,    /* Counter used in detecting sync train position  */
	carrier_val, /* Instantaneous value of 2.4 Khz sub-carrier     */
	sync_val,    /* Value produced by sync detection algorithm     */
	sync_max,    /* Max value produced by sync detection algorithm */
	sync_ave,    /* Average value of carrier samples in sync fifo  */
	sync_pos,    /* Position in sample buffer where sync o/p peaks */
	peak_idx,    /* Index used in searching for peak in sync value */
	fifo_idx,    /* Index for sync detection fifo */
	sync_cnt,	 /* Counter of sync pulses processed */
	idx;         /* Index for various loops etc.  */


  /* Allocate sync fifo on first call */
  if( sync_fifo == NULL )
  {
	sync_fifo_len =
	  NOAA_SYNC_LEAD + NOAA_SYNC_TAIL +
	  2 * NOAA_SYNC_PULSES * NOAA_SYNC_SAMPLES;

	if( !mem_alloc((void *)&sync_fifo, sizeof(int) * sync_fifo_len) )
	{
	  ClearFlag( ACTION_FLAGS_ALL );
	  Show_Message( "Failed to allocate Sync Detect FIFO", "red" );
	  return( 0 );
	}
	bzero( sync_fifo, sizeof(int) * sync_fifo_len );
  }

  /* Move through buffer up to search_limit, looking   */
  /* for sync detector peak. On exit, the buffer index */
  /* should be at the end of 'A' sync pulse sequence.  */
  sync_max = fifo_idx = 0;
  sync_pos = gbl_line_idx;
  for( peak_idx = 0; peak_idx < search_limit; peak_idx++ )
  {
	/* Read a carrier sample */
	sample_a = gbl_line_buffer[gbl_line_idx++];

	/* This is the instantaneous magnitude of the 2.4 Khz  */
	/* subcarrier, calculated as the scalar magnitude of   */
	/* two consecutive signal samples at fi deg phase angle */
	Carrier_Amplitude( sample_a, &carrier_val );

	/* Store carrier amplitude samples in fifo in cyclic manner */
	if( ++fifo_idx >= sync_fifo_len ) fifo_idx = 0;
	sync_fifo[fifo_idx] = carrier_val;

	/* Evaluate the average of the samples in fifo, to
	 * be used as a zero reference in summation below. */
	sync_val = 0;
	for( idx = 0; idx < sync_fifo_len; idx++ )
	  sync_val += sync_fifo[idx];
	sync_ave = sync_val / sync_fifo_len;

	/* Subtract carrier samples from accumulator
	 * for the leading zero amplitude period */
	sync_val = 0;
	sync_idx = fifo_idx;
	for( idx = 0; idx < NOAA_SYNC_LEAD; idx++ )
	{
	  sync_val -= sync_fifo[sync_idx] - sync_ave;
	  if( ++sync_idx >= sync_fifo_len )
		sync_idx = 0;
	}

	/* Add and subtract NOAA_SYNC_SAMPLES samples to the  */
	/* accumulator, for each half cycle of the sync train */
	for( sync_cnt = 0; sync_cnt < NOAA_SYNC_PULSES; sync_cnt++ )
	{
	  /* Add carrier samples into accumulator */
	  for( idx = 0; idx < NOAA_SYNC_SAMPLES; idx++ )
	  {
		sync_val += sync_fifo[sync_idx] - sync_ave;
		if( ++sync_idx >= sync_fifo_len )
		  sync_idx = 0;
	  }

	  /* Subtract carrier samples from accumulator */
	  for( idx = 0; idx < NOAA_SYNC_SAMPLES; idx++ )
	  {
		sync_val -= sync_fifo[sync_idx] - sync_ave;
		if( ++sync_idx >= sync_fifo_len )
		  sync_idx = 0;
	  }

	} /* for( sync_cnt = 0; sync_cnt < NOAA_SYNC_PULSES; sync_cnt++ ) */

	/* Summate the tail end of the sync train */
	for( sync_cnt = 0; sync_cnt < NOAA_SYNC_TAIL; sync_cnt++ )
	{
	  sync_val -= sync_fifo[sync_idx] - sync_ave;
	  if( ++sync_idx >= sync_fifo_len )
		sync_idx = 0;
	}

	/* Record position in buffer where sync detector peaks */
	if( sync_val > sync_max )
	{
	  sync_max = sync_val;
	  sync_pos = gbl_line_idx;
	}

  } /* End of for( peak_idx = 0; peak_idx < search_limit; peak_idx++ ) */

  /* Decoding from DSP */
  if( isFlagClear(ACTION_PROCESS_FILE) )
  {
	/* Adjust sync detector progress bar */
	gdouble pbar_frac = (gdouble)sync_max/(gdouble)NOAA_SYNC_UPPER;
	if( pbar_frac > 1.0 ) pbar_frac = 1.0;
	gtk_progress_bar_set_fraction(
		GTK_PROGRESS_BAR(gbl_sync_pbar), pbar_frac );

	/* Initialize skew label */
	gtk_label_set_text( GTK_LABEL(lookup_widget(
			xwxapt_main_window, "err_label")), "?" );

	/* Setup buffer index to max sync position */
	/* if sync maximum is within the threshold */
	if( (sync_max > NOAA_SYNC_LOWER) &&
		(sync_max < NOAA_SYNC_UPPER) )
	{
	  char buff[12];

	  /* Display sync error */
	  gbl_line_idx = sync_pos;
	  gbl_sync_err = sync_pos - gbl_sync_ref;
	  snprintf( buff, sizeof(buff), "%d", gbl_sync_err );
	  gtk_label_set_text( GTK_LABEL(lookup_widget(
			  xwxapt_main_window, "err_label")), buff );

	  /* If sync position is in range, show 'Lock' */
	  if( (abs( gbl_sync_err ) < SYNC_DETECT) &&
		  isFlagClear(ACTION_SYNC_BUFFER) )
	  {
		if( isFlagClear(ICON_SYNC_LOCK) )
		{
		  ClearFlag( ICON_FLAGS_ALL );
		  SetFlag( ICON_SYNC_LOCK );
		  Set_Sync_Icon( "gtk-yes" );
		}
	  }
	  else /* Show 'Search' */
	  {
		if( isFlagClear(ICON_SYNC_SEARCH) )
		{
		  ClearFlag( ICON_FLAGS_ALL );
		  SetFlag( ICON_SYNC_SEARCH );
		  Set_Sync_Icon( "gtk-dialog-question" );
		}

	  } /* if( abs(gbl_line_idx - gbl_sync_ref) < 3 ) */
	}
	else /* If sync level is out of range, show 'Unlock' */
	{
	  gbl_line_idx = 0;

	  if( isFlagClear(ICON_SYNC_UNLOCK) )
	  {
		ClearFlag( ICON_FLAGS_ALL );
		SetFlag( ICON_SYNC_UNLOCK );
		Set_Sync_Icon( "gtk-no" );
	  }

	} /* if( (sync_max > NOAA_SYNC_LOWER) && */

  } /* if( isFlagClear(ACTION_PROCESS_FILE) ) */
  else
  {
	/* Setup buffer index to max sync position */
	/* if sync maximum is within the threshold */
	if( (sync_max > NOAA_SYNC_LOWER) &&
		(sync_max < NOAA_SYNC_UPPER) )
	  gbl_line_idx = sync_pos;
	else
	  gbl_line_idx = 0;
  }

  return(0);

} /* End of Detect_NOAA_Sync() */

/*------------------------------------------------------------------------*/

/*  Detect_Meteor_Sync()
 *
 *  Detects the sync train of a Meteor type APT signal.
 *  This is 5 cycles at 1200 Hz on the 2.4 KHz sub-carrier.
 */

  int
Detect_Meteor_Sync( int search_limit )
{
  /* FIFO Buffer for sync detection */
  static int sync_fifo[METEOR_SYNC_FIFO_SIZE];

      /* An amplitude sample of the 2.4kHz sub-carrier */
  short sample_a;

  int
	sync_idx,      /* Counter used in detecting sync train position  */
	carrier_val,   /* Instantaneous value of 2.4 Khz sub-carrier     */
	sync_val,      /* Value produced by sync detection algorithm     */
	sync_max,      /* Max value produced by sync detection algorithm */
	sync_ave,      /* Average value of carrier samples in sync fifo  */
	sync_pos,      /* Position in sample buffer where sync o/p peaks */
	sync_cnt,      /* Counter of sync pulses processed */
	peak_idx,      /* Index used in searching for peak in sync value */
	fifo_idx,      /* Index for sync detection fifo */
	idx;           /* Index for various loops etc.  */


  /* Move through buffer until sync detector  */
  /* peaks. On exit, the buffer index should  */
  /* be at the end of the sync pulse sequence */

  sync_max = fifo_idx = 0;
  sync_pos = gbl_line_idx;
  for( peak_idx = 0; peak_idx < search_limit; peak_idx++ )
  {
	/* Read a carrier sample */
	sample_a = gbl_line_buffer[gbl_line_idx++];

	/* This is the instantaneous magnitude of the 2.4 Khz  */
	/* subcarrier, calculated as the scalar magnitude of   */
	/* two consecutive signal samples at 90 deg phase angle */
	Carrier_Amplitude( sample_a, &carrier_val );

	/* Store samples in fifo in cyclic manner */
	if( ++fifo_idx == METEOR_SYNC_FIFO_SIZE ) fifo_idx = 0;
	sync_fifo[fifo_idx] = carrier_val;

	/* Evaluate the average of the samples in fifo, to */
	/* be used as a zero reference in summation below. */
	sync_val = 0;
	for( idx = 0; idx < METEOR_SYNC_FIFO_SIZE; idx++ )
	  sync_val += sync_fifo[idx];
	sync_ave = sync_val/METEOR_SYNC_FIFO_SIZE;

	/* Subtract 5x2-sample groups and add 5x4-sample groups */
	/* to the sync accumulator alternately, simulating      */
	/* synchromous detection. The sync train appears to be  */
	/* 5 cycles at 1200 Hz with a 2/1 mark/space ratio.     */
	/* Their average value is used as a zero reference.     */
	sync_val = 0;
	sync_idx = fifo_idx;
	for( sync_cnt = 0; sync_cnt < METEOR_SYNC_PULSES; sync_cnt++ )
	{
	  /* Subtract carrier samples from accumulator */
	  for( idx = 0; idx < METEOR_SYNC_MARK; idx++ )
	  {
		sync_val -= sync_fifo[fifo_idx] - sync_ave;
		if( ++sync_idx >= METEOR_SYNC_FIFO_SIZE )
		  sync_idx = 0;
	  }

	  /* Add carrier samples into accumulator */
	  for( idx = 0; idx < METEOR_SYNC_SPACE; idx++ )
	  {
		sync_val += sync_fifo[fifo_idx] - sync_ave;
		if( ++sync_idx >= METEOR_SYNC_FIFO_SIZE )
		  sync_idx = 0;
	  }

	} /* for( sync_cnt = 0; sync_cnt < METEOR_SYNC_PULSES; sync_cnt++ ) */

	/* Record position in buffer where sync detector peaks */
	if( sync_val > sync_max)
	{
	  sync_max = sync_val;
	  sync_pos = gbl_line_idx;
	}

  } /* End of for( peak_idx = 0; peak_idx < search_limit-1; peak_idx++ ) */

  /* Decoding from DSP */
  if( isFlagClear(ACTION_PROCESS_FILE) )
  {
	/* Adjust sync detector progress bar */
	gdouble pbar_fr = (gdouble)sync_max/(gdouble)METEOR_SYNC_UPPER;
	if( pbar_fr > 1.0 ) pbar_fr = 1.0;
	gtk_progress_bar_set_fraction(
		GTK_PROGRESS_BAR(gbl_sync_pbar), pbar_fr );

	/* Initialize skew label */
	gtk_label_set_text( GTK_LABEL(lookup_widget(
			xwxapt_main_window, "err_label")), "?" );

	/* Setup buffer index to max sync position */
	/* if sync maximum is within the threshold */
	if( (sync_max > METEOR_SYNC_LOWER) &&
		(sync_max < METEOR_SYNC_UPPER) )
	{
	  char buff[12];

	  /* Display sync error */
	  gbl_line_idx = sync_pos;
	  gbl_sync_err = sync_pos - gbl_sync_ref;
	  snprintf( buff, sizeof(buff), "%d", gbl_sync_err );
	  gtk_label_set_text( GTK_LABEL(lookup_widget(
			  xwxapt_main_window, "err_label")), buff );

	  /* If sync position is in range, show 'Lock' */
	  if( (abs( gbl_sync_err ) < SYNC_DETECT) &&
		  isFlagClear(ACTION_SYNC_BUFFER) )
	  {
		if( isFlagClear(ICON_SYNC_LOCK) )
		{
		  ClearFlag( ICON_FLAGS_ALL );
		  SetFlag( ICON_SYNC_LOCK );
		  Set_Sync_Icon( "gtk-yes" );
		}
	  }
	  else /* Show 'Search' */
	  {
		if( isFlagClear(ICON_SYNC_SEARCH) )
		{
		  ClearFlag( ICON_FLAGS_ALL );
		  SetFlag( ICON_SYNC_SEARCH );
		  Set_Sync_Icon( "gtk-dialog-question" );
		}

	  } /* if( abs(gbl_line_idx - gbl_sync_ref) < 3 ) */
	}
	else /* If sync level is out of range, show 'Unlock' */
	{
	  gbl_line_idx = 0;

	  if( isFlagClear(ICON_SYNC_UNLOCK) )
	  {
		ClearFlag( ICON_FLAGS_ALL );
		SetFlag( ICON_SYNC_UNLOCK );
		Set_Sync_Icon( "gtk-no" );
	  }

	} /* if( (sync_max > METEOR_SYNC_LOWER) && */

  } /* if( isFlagClear(ACTION_PROCESS_FILE) ) */
  else
  {
	/* Setup buffer index to max sync position */
	/* if sync maximum is within the threshold */
	if( (sync_max > METEOR_SYNC_LOWER) &&
		(sync_max < METEOR_SYNC_UPPER) )
	  gbl_line_idx = sync_pos;
	else
	  gbl_line_idx = 0;
  }

  return(0);

} /* End of Detect_Meteor_Sync() */

/*------------------------------------------------------------------------*/

/*  Detect_Sync_Train()
 *
 *  Detects sync train position in line buffer
 *  depending on the type of satellite processed
 */

  int
Detect_Sync_Train( int search_limit )
{
  gbl_line_idx = 0;
  switch( gbl_sat_type )
  {
	case NOAA_15: case NOAA_18: case NOAA_19:
	  Detect_NOAA_Sync( search_limit );
	  break;
	case METEOR:
	  Detect_Meteor_Sync( search_limit );
  }

  return( gbl_line_idx );
} /* End of Detect_Sync_Train() */

/*------------------------------------------------------------------------*/

/* Init_Reception()
 *
 * Initialize Reception of signal from Satellite
 */
  gboolean
Init_Reception( void )
{
  char mesg[MESG_SIZE];
  mesg[0] = '\0';
  int error;

  /* Setup sound card or rtlsdr device */
  if( isFlagSet(USE_RTLSDR_RX) )
  {
	/* Initialize RTLSDR device */
	if( !Rtlsdr_Init() )
	{
	  Cleanup();
	  Show_Message( _("Failed to Initialize RTLSDR device"), "red" );
	  Error_Dialog();
	  return( FALSE );
	}
  }
  else
  {
	/* Set up Sound Card */
	if( !Open_Capture(mesg, &error) )
	{
	  if( error )
	  {
		Strlcat( mesg, _("\nError: "), MESG_SIZE );
		Strlcat( mesg, snd_strerror(error), MESG_SIZE );
	  }

	  Cleanup();
	  Show_Message( mesg, "red" );
	  Error_Dialog();
	  return( FALSE );
	}

  } /* if( isFlagSet(USE_RTLSDR_RX) ) */

  return( TRUE );
} /* Init_Reception() */

/*------------------------------------------------------------------------*/

/*  Setup_Signal_Level()
 *
 *  Displays a simple graphical aid for setting
 *  up the signal level from the APT receiver
 */

  gboolean
Setup_Signal_Level( gpointer data )
{
  int
	carrier_max, /* Max value of carrier in  buffer */
	carrier_val, /* Sampled value of 2.4kHz carrier */
	idx;         /* Index used in loops etc */

  /* Progressbar fraction */
  double pbar_frac;

  /* Required max value of carrier level */
  static int level_max;

  /* First idle call */
  static gboolean first_call = TRUE;


  if( first_call )
  {
	/* Initialize Reception, abort on error */
	if( !Init_Reception() )
	{
	  first_call = TRUE;
	  return( FALSE );
	}

	/* Select required max signal level */
	switch( gbl_sat_type )
	{
	  case NOAA_15: case NOAA_18: case NOAA_19:
		level_max = MAX_SUBCARRIER_AMPL;
		break;

	  case METEOR:
		level_max = MAX_SUBCARRIER_AMPL;
	}

	Show_Message( _("Starting Signal Level Setup "), "black" );
	first_call = FALSE;
  } /* if( first_call ) */

  /* Fill in the samples buffer from DSP, abort on error */
  if( !Read_Device_Buffer(gbl_line_buffer, BLOCK_SIZE) )
  {
	first_call = TRUE;
	return( FALSE );
  }

  /*** Find max value of 2.4kHz subcarrier in buffer ***/
  carrier_max = 0;
  for( idx = 0; idx < BLOCK_SIZE; idx++ )
  {
	/* Calculate carrier amplitude from two consecutive samples */
	Carrier_Amplitude( gbl_line_buffer[idx], &carrier_val );

	/* Find maximum carrier level */
	if( carrier_val > carrier_max )
	  carrier_max = carrier_val;

  } /* for( idx = 0; idx < BLOCK_SIZE; idx++ ) */

  /* Show signal level on progress bar */
  pbar_frac = (gdouble)carrier_max / (gdouble)level_max;
  if( pbar_frac > 1.0 )	pbar_frac = 1.0;
  gtk_progress_bar_set_fraction(
	  GTK_PROGRESS_BAR(gbl_level_pbar), pbar_frac );

  if( isFlagSet(ACTION_SETUP_AUDIO) )
	return( TRUE );

  Show_Message( _("Stopping Signal Level Setup"), "black" );
  Cleanup();
  first_call = TRUE;

  return( FALSE );
} /* Setup_Signal_Level() */

/*------------------------------------------------------------------------*/

/*  Record_APT_Signal()
 *
 *  Records samples of the 2.4 Khz subcarrier in a file
 */

  gboolean
Record_APT_Signal( gpointer data )
{
  /* Number of APT image lines processed */
  static int line_cnt;

  /* First call of function flag */
  static gboolean first_call = TRUE;

  /* Messages buffer */
  char mesg[256];
  size_t sm = sizeof( mesg );

  /* Temp buffer to write line buffer to file */
  unsigned char buff[2];
  int idx;

  /* Samples written to file */
  size_t sample_cnt;


  if( first_call )
  {
	/* Initialize Reception, abort on error */
	if( !Init_Reception() )
	{
	  first_call = TRUE;
	  return( FALSE );
	}

	Show_Message( _("Recording apt signal to file "), "black" );

	/* Add extension to file name */
	size_t s = sizeof( gbl_samples_file );
	Strlcat( gbl_samples_file, "-", s );
	Strlcat( gbl_samples_file, gbl_sat_names[gbl_sat_type], s );
	Strlcat( gbl_samples_file, ".bin", s );

	/* Create a file for writing samples */
	gbl_samples_fp = fopen( gbl_samples_file, "w+" );
	if( gbl_samples_fp == NULL )
	{
	  perror( gbl_samples_file );
	  snprintf( mesg, sm,
		  _("Failed to open recorded samples file:\n%s"),
		  gbl_samples_file );
	  Show_Message( mesg, "red" );
	  Error_Dialog();
	  return( FALSE );
	}

	/* Read samples from DSP, stop on user interrupt */
	/* or at time limit in number of APT lines read. */
	/* gbl_duration is in seconds of recording time. */
	snprintf( mesg, sm,
		_("Setting recording duration to %d sec"), gbl_duration );
	Show_Message( mesg, "black" );

	/* Print some information */
	snprintf( mesg, sm,
		_("Recording APT signal in:\n%s "), gbl_samples_file );
	Show_Message( mesg, "black" );

	line_cnt = 0;

	/* Write satellite type to samples file, abort on error */
	sample_cnt = fwrite(&gbl_sat_type, 4, 1, gbl_samples_fp);
	if( sample_cnt < 1 )
	{
	  perror( "xwxapt: fwrite()" );
	  snprintf( mesg, sm,
		  _("Error writing to samples file\n%s"), gbl_samples_file );
	  Show_Message( mesg, "red" );
	  Error_Dialog();
	  return( FALSE );
	}
	if( sample_cnt != 1 )
	{
	  snprintf( mesg, sm,
		  _("Error writing to samples file\n%s"), gbl_samples_file );
	  Show_Message( mesg, "red" );
	  Error_Dialog();
	  return( FALSE );
	}

	first_call = FALSE;

  } /* if( first_call ) */

  /* Fill samples buffer from DSP */
  if( !Fill_Samples_Buffer() )
  {
	first_call = TRUE;
	return( FALSE );
  }

  /* Write carrier sample values to samples file, abort on error */
  for( idx = 0; idx < BLOCK_SIZE; idx++ )
  {
	buff[0] = gbl_line_buffer[idx] & 0xff;
	buff[1] = (gbl_line_buffer[idx] >> 8) & 0xff;

	sample_cnt = fwrite(buff, 1, 2, gbl_samples_fp);
	if( sample_cnt < 1 )
	{
	  perror( "xwxapt: fwrite()" );
	  snprintf(mesg, sm,
		  _("Error writing to samples file\n%s"), gbl_samples_file);
	  Show_Message( mesg, "red" );
	  Error_Dialog();
	  first_call = TRUE;
	  return( FALSE );
	}

	if( (sample_cnt != 2) )
	{
	  snprintf(mesg, sm,
		  _("Error writing to samples file\n%s"), gbl_samples_file);
	  Show_Message( mesg, "red" );
	  Error_Dialog();
	  first_call = TRUE;
	  return( FALSE );
	}
  } /* for( idx = 0; idx < BLOCK_SIZE; idx++ ) */

  /* Stop if line count reaches time */
  /* limit or on user stop request   */
  if( (++line_cnt/2 != gbl_duration) &&
	  isFlagSet(ACTION_RECORD_APT) )
	return( TRUE );

  /* Print some information */
  snprintf(mesg, sm,
	  _("Closing samples file:\n%s"), gbl_samples_file);
  Show_Message( mesg, "black" );

  Cleanup();
  first_call = TRUE;

  return( FALSE );
} /* End of Record_APT_Signal() */

/*------------------------------------------------------------------------*/

/*  File_Fill_Buffer()
 *
 *  Fills the signal sample buffer from source file
 */

  gboolean
File_Fill_Buffer( void )
{
  /* Messages buffer */
  char mesg[256];

  /* For reading processed dsp data from file */
  size_t sample_cnt;

  /* Temp buffer to write line buffer to file */
  unsigned char buff[2];
  int idx;


  /* Fill in the samples buffer from file, return on error */
  gbl_line_idx = 0;
  for( idx = 0; idx < BLOCK_SIZE; idx++ )
  {
	sample_cnt = fread( buff, 1, 2, gbl_samples_fp );
	if( feof(gbl_samples_fp) ) return( FALSE );
	if( sample_cnt < 2 )
	{
	  perror( "xwxapt: fread()" );
	  snprintf(mesg, sizeof(mesg),
		  _("Error reading from samples file\n%s"), gbl_samples_file);
	  Show_Message( mesg, "red" );
	  Error_Dialog();
	  return( FALSE );
	}

	else if( sample_cnt != 2 )
	{
	  snprintf(mesg, sizeof(mesg),
		  _("Error reading from samples file\n%s"), gbl_samples_file);
	  Show_Message( mesg, "red" );
	  Error_Dialog();
	  return( FALSE );
	}

	gbl_line_buffer[idx]  = buff[0];
	gbl_line_buffer[idx] |= buff[1] << 8;
	if( gbl_line_buffer[idx] & 0x8000 )
	  gbl_line_buffer[idx] -= 0x10000;

  } /* for( idx = 0; idx < BLOCK_SIZE; idx++ ) */

  /* Set buffer index just after sync train */
  /* Search for sync up to 1.5* sync refer. */
  if( !Detect_Sync_Train(gbl_sync_ref + SYNC_SEARCH_EXTEND) )
	return( TRUE );

  /* If sync detection is far out, keep */
  /* buffer index in default position */
  if( abs(gbl_line_idx - gbl_sync_ref) > SYNC_ACCEPT )
	gbl_line_idx = gbl_sync_ref;

  return( TRUE );
} /* End of File_Fill_Buffer() */

/*------------------------------------------------------------------------*/

/* Read_Device_Buffer()
 *
 * Reads the samples buffer of the RTLSDR device
 * or the samples buffer of the Sound Card DSP
 */
  gboolean
Read_Device_Buffer( short *buffer, int buff_size )
{
  if( isFlagSet(USE_RTLSDR_RX) )
  {
	/* Fill in the line buffer from RTL device */
	if( !Read_RTL_Buffer(buffer, buff_size) )
	  return( FALSE );
  }
  else
  {
	/* Fill in the line buffer from Sound DSP */
	if( !Read_SND_Buffer(buffer, buff_size) )
	  return( FALSE );
  } /* if( isFlagSet(USE_RTLSDR_RX) ) */

  return( TRUE );
} /* Read_Device_Buffer() */

/*------------------------------------------------------------------------*/

/*  Fill_Samples_Buffer()
 *
 *  Fills samples buffer from Sound DSP or RTLSDR device.
 *  Buffer filling is synchronized with APT sync train.
 */

  gboolean
Fill_Samples_Buffer( void )
{
  static int block_size = BLOCK_SIZE;

  if( isFlagSet(USE_RTLSDR_RX) )
  {
	/* Fill in the samples buffer from RTL device */
	if( !Read_RTL_Buffer(gbl_line_buffer, block_size + dsp_rate_err) )
	{
	  Show_Message( _("Error filling samples buffer from RTL"), "red" );
	  return( FALSE );
	}
  }
  else
  {
	/* Fill in the samples buffer from Sound Card DSP */
	if( !Read_SND_Buffer(gbl_line_buffer, block_size + dsp_rate_err) )
	{
	  Show_Message( _("Error filling samples buffer from SND"), "red" );
	  return( FALSE );
	}
  } /* if( isFlagSet(USE_RTLSDR_RX) ) */

  /* Search for sync up to 1.25 * sync ref, return if sync fails */
  if( !Detect_Sync_Train(gbl_sync_ref + SYNC_SEARCH_EXTEND) )
  {
	gbl_line_idx = gbl_sync_ref;
	return( TRUE );
  }

  /* Try to keep APT sync train at beginning of buffer. This  */
  /* done done by increasing or decreasing the data  block    */
  /* size argument passed to read(), by adding or subtracting */
  /* half the difference between detected sync position and a */
  /* reference value. Error is limited to +-2 to avoid coarse */
  /* corrections to sync position due to noise interference.  */
  if( gbl_sync_err > SYNC_ACCEPT )
	gbl_sync_err = SYNC_ACCEPT;
  else
	if( gbl_sync_err < -SYNC_ACCEPT )
	  gbl_sync_err = -SYNC_ACCEPT;
  gbl_line_idx += gbl_sync_err;

  /* Vary the data block size argument to read() to
   * take up any drift in sync position due to errors
   * in the DSP sampling rate and satellite movement.
   * Only half the sync error is used to slow and
   * stabilize sync correction. */
  block_size = BLOCK_SIZE + gbl_sync_err / 2;

  return( TRUE );
} /* End of Fill_Samples_Buffer() */

/*------------------------------------------------------------------------*/

/*  Synchronize_Buffer()
 *
 *  Synchronizes samples buffer with the APT sync train
 */

  gboolean
Synchronize_Buffer( gpointer data )
{
  /* First call of function flag */
  static gboolean first_call = TRUE;

  /* Messages buffer */
  char mesg[MESG_SIZE];

  /* Data block size used as read() argument  */
  int block_size;

  static int
	last_line_idx,	/* Saved value of samples buffer index */
	rate_ok_cnt;	/* Count number of similar rate measurements */

  /* Cancel on user request */
  if( isFlagClear(ACTION_SYNC_BUFFER) )
  {
	Show_Message( _("Search for APT sync train stopped"), "black" );
	Cleanup();
	first_call = TRUE;
	return( FALSE );
  }

  /* Initialize on first call */
  if( first_call )
  {
	mesg[0] = '\0';

	/* Initialize Reception, abort on error */
	if( !Init_Reception() )
	{
	  first_call = TRUE;
	  return( FALSE );
	}

	/* Print some information */
	Show_Message( _("Synchronizing with APT sync train "), "black" );
	snprintf( mesg, MESG_SIZE,
		_("Looking for a %s type sync train "), gbl_sat_names[gbl_sat_type] );
	Show_Message( mesg, "black" );

	/* Load appropriate sync reference */
	gbl_sync_ref = gbl_sync_refs[ gbl_sat_type ];

	dsp_rate_err  = 0;
	last_line_idx = 0;
	rate_ok_cnt   = 0;

	first_call = FALSE;
	SetFlag(ACTION_RATE_ERROR);

  } /* if( first_call ) */

  /*** Find DSP rate error ***/
  if( isFlagSet(ACTION_RATE_ERROR) )
  {
	/* Fill in the line buffer from Device DSP */
	if( !Read_Device_Buffer(gbl_line_buffer, BLOCK_SIZE) )
	{
	  first_call = TRUE;
	  return( FALSE );
	}

	/* Locate sync train (sets gbl_line_idx just after it) */
	if( !Detect_Sync_Train(BLOCK_SIZE) ) return( TRUE );

	/* DSP rate error is change in sync train position */
	dsp_rate_err  = gbl_line_idx - last_line_idx;
	last_line_idx = gbl_line_idx;

	/* If measured dsp sampling rate error is
	 * within limits, signal end of measurement */
	if( abs(dsp_rate_err) > SYNC_ACCEPT )
	  rate_ok_cnt = 0;
	else
	  rate_ok_cnt++;

	if( rate_ok_cnt >= RATE_OK_COUNT )
	  ClearFlag(ACTION_RATE_ERROR);
	else return( TRUE );

	/* Dummy-read samples to align buffer with sync ref. */
	block_size = gbl_line_idx - gbl_sync_ref;
	if( block_size < 0 )
	  block_size += BLOCK_SIZE;
	block_size += (dsp_rate_err * block_size)/BLOCK_SIZE;
	if( !Read_Device_Buffer(gbl_line_buffer, block_size) )
	{
	  first_call = TRUE;
	  return( FALSE );
	}

	/* Show dsp rate error */
	snprintf( mesg, MESG_SIZE,
		_("DSP sampling rate error: %d ppm"),
		(1000000 * dsp_rate_err) / BLOCK_SIZE );
	Show_Message( mesg, "orange" );
	gbl_sync_err = 0;

  } /* if( isFlagSet(ACTION_RATE_ERROR) ) */

  /*** Refine buffer alignment with sync reference ***/
  /* Dummy-read samples to align buffer with sync reference */
  block_size = BLOCK_SIZE + dsp_rate_err + gbl_sync_err;
  if( !Read_Device_Buffer(gbl_line_buffer, block_size) )
  {
	first_call = TRUE;
	return( FALSE );
  }

  /* Locate sync train. (sets gbl_line_idx just after it) */
  if( !Detect_Sync_Train(BLOCK_SIZE) ||
	  abs(gbl_sync_err) > SYNC_ACCEPT )
	return( TRUE );

  /* Print some information */
  Show_Message( _("Synchronized with APT sync train"), "green" );
  first_call = TRUE;
  ClearFlag( ACTION_SYNC_BUFFER );

  /* Start the selected process */
  if( isFlagSet(ACTION_PROCESS_DSP) )
	g_idle_add( Dsp_Process_Image, NULL );
  else if( isFlagSet(ACTION_RECORD_APT) )
	g_idle_add( Record_APT_Signal, NULL );

  return( FALSE );
} /* End of Synchronize_Buffer() */

/*------------------------------------------------------------------------*/

/* Carrier_Amplitude()
 *
 * The instantaneous amplitude a of the 2.4 KHz sub-carrier
 * is calculated from an incoming sample (the I sample) and
 * one that was delayed by 90 deg in a circular buffer
 * the Q sample). The amplitude is A = SQRT( I^2 + Q^2 ).
 */
  void
Carrier_Amplitude( short sample, int *amplitude )
{
  /* Intitalize on first call */
  static gboolean first_call = TRUE;

  /* Circular samples buffer, length is 1/4 of the
   * samples/cycle of the 2.4 kHz sub-carrier */
  static short *buffer = NULL;

  static int
	samples_cycle = 0,	/* Amplitude samples per sub-carrier cycle */
	samples_sum   = 0,  /* Summation of sub-carrier samples over 1 cycle */
	samples_cnt   = 0,	/* Count of Amplitude samples summated */
	dc_offset = 0,	/* DC offset of the demodulated signal */
	buf_size  = 0,	/* Circular buffer size */
	buf_idx   = 0;	/* Buffer index */

   static gboolean ret = TRUE;

  /* The value as float of the i and q
   * samples and the sum of their squares */
  double
	id = 0.0,
	qd = 0.0;

  /* Amplitude summation for scope */
  static int scope_sum = 0;

  /* Samples counter and limits must be in doubles to
   * avoid the discrepancies caused by integer division */
  static double
	scope_cnt = 0.0,
	scope_lim = 0.0;


  /* Abort if buffer allocation failed */
  if( !ret )
  {
	*amplitude = 0;
	return;
  }

  /* Initialize on first call */
  if( first_call )
  {
	/* Samples per cycle of sub-carrier */
	samples_cycle = SND_DSP_RATE / CARRIER_FREQ;

	/* Allocate circular buffer. Size is 1/4 of sub-carrier cycle */
	buf_size  = SND_DSP_RATE / CARRIER_FREQ / 4;
	ret = mem_alloc( (void *)&buffer, buf_size * sizeof(short) );
	if( !ret )
	{
	  Show_Message(
		  _("Failed to allocate Carrier Amplitude buffer"), "red" );
	  first_call = TRUE;
	  return;
	}
	bzero( buffer, buf_size * sizeof(short) );

	/* Covers one complete APT line */
	scope_lim = (double)BLOCK_SIZE / (double)gbl_scope_width;

	first_call = FALSE;
  } /* if( first_call ) */

  /* Summate sub-carrier samples */
  samples_sum += sample;
  samples_cnt++;

  /* Calculate DC offset per cycle */
  if( samples_cnt >= samples_cycle )
  {
	samples_cnt = 0;
	dc_offset = samples_sum / samples_cycle;
	samples_sum = 0;
  }

  /* Sum of squares of I and Q samples.
   * The calculated DC offset is removed */
  sample -= dc_offset;
  id = (double)sample;
  qd = (double)buffer[buf_idx];

  /* The instantaneous amplitude of the sub-carrier */
  *amplitude = (int)hypot( id, qd );
  if( *amplitude > MAX_SUBCARRIER_AMPL )
	*amplitude = MAX_SUBCARRIER_AMPL;

  /* Display sub-carrier amplitude */
  scope_sum += *amplitude;
  scope_cnt += 1.0;
  if( scope_cnt >= scope_lim )
  {
	scope_sum /= scope_lim;
	Display_Signal( scope_sum / SCOPE_SCALE );
	scope_sum = 0;
	scope_cnt -= scope_lim;
  }

  /* Store I sample and advance
   * index of I and Q samples */
  buffer[buf_idx] = sample;
  buf_idx++;
  if( buf_idx >= buf_size ) buf_idx = 0;

  return;
} /* End of Carrier_Amplitude() */

/*------------------------------------------------------------------------*/

/*  Set_Sync_Icon()
 *
 * Changes the sync status icon
 */

  void
Set_Sync_Icon( gchar *icon_name )
{
  /* Remove previous icon */
  gtk_container_remove( GTK_CONTAINER(gbl_status_table), gbl_sync_icon );

  /* Attach a new icon */
  gbl_sync_icon = gtk_image_new_from_stock( icon_name, GTK_ICON_SIZE_BUTTON );
  gtk_widget_show( gbl_sync_icon );

  gtk_container_add( GTK_CONTAINER(gbl_status_table), gbl_sync_icon );
  gtk_misc_set_padding (GTK_MISC (gbl_sync_icon), 0, 4);

} /* Set_Sync_Icon() */

/*------------------------------------------------------------------------*/
