/*  utils.c
 *
 *  Utility functions for gridloc
 *
 *  xgridloc: A GTK+ application to calculate the Maidenhead Grid
 *  Locator of a given position or vice-versa. Also to calculate
 *  bearing and distance to a remote position and display a Great
 *  Circle arc between Home and Remote positions using 'xplanet'.
 *
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "utils.h"

/*------------------------------------------------------------------*/

/*  Load_Line()
 *
 *  Loads a line from a file, aborts on failure. Lines beginning
 *  with a '#' are ignored as comments. At the end of file EOF is
 *  returned. Lines assumed maximum 195 characters long.
 */

  int
Load_Line( char *buff, FILE *pfile, char *mesg )
{
  int
	num_chr, /* Number of characters read, excluding lf/cr */
	chr;     /* Character read by getc() */
  char error_mesg[MESG_SIZE];

  /* Prepare error message */
  snprintf( error_mesg, sizeof(error_mesg),
	  _("Error reading %s\n"\
		"Premature EOF (End Of File)"), mesg );

  /* Clear buffer at start */
  buff[0] = '\0';
  num_chr = 0;

  /* Get next character, return error if chr = EOF */
  if( (chr = fgetc(pfile)) == EOF )
  {
	fprintf( stderr, "xgridloc: %s\n", error_mesg );
	fclose( pfile );
	Error_Dialog( error_mesg );
	return( EOF );
  }

  /* Ignore commented lines and eol/cr */
  while(
	  (chr == '#') ||
	  (chr == ' ') ||
	  (chr == HT ) ||
	  (chr == CR ) ||
	  (chr == LF ) )
  {
	/* Go to the end of line (look for LF or CR) */
	while( (chr != CR) && (chr != LF) )
	  /* Get next character, return error if chr = EOF */
	  if( (chr = fgetc(pfile)) == EOF )
	  {
		fprintf( stderr, "xgridloc: %s\n", error_mesg );
		fclose( pfile );
		Error_Dialog( error_mesg );
		return( EOF );
	  }

	/* Dump any CR/LF remaining */
	while( (chr == CR) || (chr == LF) )
	  /* Get next character, return error if chr = EOF */
	  if( (chr = fgetc(pfile)) == EOF )
	  {
		fprintf( stderr, "xgridloc: %s\n", error_mesg );
		fclose( pfile );
		Error_Dialog( error_mesg );
		return( EOF );
	  }

  } /* End of while( (chr == '#') || ... */

  /* Continue reading characters from file till
   * number of characters = 195 or EOF or CR/LF */
  while( num_chr < 195 )
  {
	/* If LF/CR reached before filling buffer, return line */
	if( (chr == LF) || (chr == CR) ) break;

	/* Enter new character to line buffer */
	buff[num_chr++] = (char)chr;

	/* Get next character */
	if( (chr = fgetc(pfile)) == EOF )
	{
	  /* Terminate buffer as a string if chr = EOF */
	  buff[num_chr] = '\0';
	  return( SUCCESS );
	}

	/* Abort if end of line not reached at 195 char. */
	if( (num_chr == 195) && (chr != LF) && (chr != CR) )
	{
	  /* Terminate buffer as a string */
	  buff[num_chr] = '\0';
	  snprintf( error_mesg, sizeof(error_mesg),
		  _("Error reading %s\n"\
			"Line longer than 195 characters"), mesg );
	  fprintf( stderr, "xgridloc: %s\n%s\n", error_mesg, buff );
	  fclose( pfile );
	  Error_Dialog( error_mesg );
	  return( ERROR );
	}

  } /* End of while( num_chr < max_chr ) */

  /* Terminate buffer as a string */
  buff[num_chr] = '\0';

  return( SUCCESS );

} /* End of Load_Line() */

/*------------------------------------------------------------------*/

/*  Cleanup()
 *
 *  Cleans up before quitting
 */

int
Cleanup(void)
{
  /* Delete xplanet files */
  if( isFlagSet(XPLANET_ENABLED) )
  {
    unlink( rc_data.config_file );
    unlink( rc_data.marker_file );
    unlink( rc_data.arc_file );
	return( system("pkill xplanet") );
  }
  return( 0 );
}

/*------------------------------------------------------------------------*/

/* Functions for testing and setting/clearing flags */

/* An int variable holding the single-bit flags */
static int Flags = 0;

int
isFlagSet(int flag)
{
  return(Flags & flag);
}

int
isFlagClear(int flag)
{
  return(~Flags & flag);
}

void
SetFlag(int flag)
{
  Flags |= flag;
}

void
ClearFlag(int flag)
{
  Flags &= ~flag;
}

void
ToggleFlag(int flag)
{
  Flags ^= flag;
}

/*------------------------------------------------------------------*/

/*  Usage()
 *
 *  Prints usage information
 */

  void
Usage( void )
{
  fprintf( stderr, "%s\n",
	  _("Usage: xgridloc [-hv]") );

  fprintf( stderr, "%s\n",
	  _("       -h: Print this usage information and exit"));

  fprintf( stderr, "%s\n",
	  _("       -v: Print version number and exit"));

} /* End of Usage() */

/*------------------------------------------------------------------------*/

/* Strlcpy()
 *
 * Copies n-1 chars from src string into dest string. Unlike other
 * such library fuctions, this makes sure that the dest string is
 * null terminated by copying only n-1 chars to leave room for the
 * terminating char. n would normally be the sizeof(dest) string but
 * copying will not go beyond the terminating null of src string
 */
  void
Strlcpy( char *dest, const char *src, size_t n )
{
  char ch = src[0];
  int idx = 0;

  /* Leave room for terminating null in dest */
  n--;
 
  /* Copy till terminating null of src or to n-1 */
  while( (ch != '\0') && (n > 0) )
  {
	dest[idx] = src[idx];
	idx++;
	ch = src[idx];
	n--;
  }

  /* Terminate dest string */
  dest[idx] = '\0';

} /* Strlcpy() */

/*------------------------------------------------------------------*/

/* Atof()
 *
 * Replaces atof() to take into account the
 * locale-dependent decimal point character
 */
double Atof( const char *nptr )
{
  int idx;
  size_t len;
  double d = 0.0;
  char *s = NULL;
  static gboolean first_call = TRUE;
  static char dp = '.';

  /* Find locale-dependent decimal point character */
  if( first_call )
  {
	struct lconv *lcnv;
	lcnv = localeconv();
	dp = *lcnv->decimal_point;
	first_call = FALSE;
  }

  /* Look for a . or , decimal point character
   * in the supplied number buffer (string) */
  len = strlen( nptr );
  for( idx = 0; idx < (int)len; idx++ )
	if( (nptr[idx] == ',') || (nptr[idx] == '.') )
	  break;

  /* Create temporary string to modify decimal point */
  s = malloc( len + 1 );
  if( s == NULL )
  {
	Error_Dialog(
		_("Failed to allocate memory for\n"\
		  "Atof() temporary string buffer.\n"\
		  "Please quit xgridloc and correct") );
	return( d );
  }
  strncpy( s, nptr, len );
  s[ len ] = '\0';

  /* If a decimal point character is found, replace */
  if( idx < (int)len ) s[idx] = dp;
  d = atof( s );
  free( s );
  
  return( d );
} /* End of Atof() */

/*------------------------------------------------------------------*/
