/*
 *  xgridloc: An X/GTK+ application to calculate the Maidenhead Grid
 *  Locator of a given position or vice-versa. Also to calculate
 *  bearing and distance to a remote position and plot the Great
 *  Circle arc to it on screen, using xplanet if installed.
 *
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */


/*  main.c
 *
 *  Main body of xgridloc application
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include "main.h"
#include "shared.h"

/* Signal handler */
static void Signal_Handler( int signal );

/* Longitude and Latitude */
extern double
  home_lon, home_lat,
  remt_lon, remt_lat;


int
main( int argc, char *argv[] )
{
  /* Command line option returned by getopt() */
  int option;

  /* New and old actions for sigaction() */
  struct sigaction sa_new, sa_old;

  /* Initialize new actions */
  sa_new.sa_handler = Signal_Handler;
  sigemptyset( &sa_new.sa_mask );
  sa_new.sa_flags = 0;

  /* Register function to handle signals */
  sigaction( SIGINT,  &sa_new, &sa_old );
  sigaction( SIGSEGV, &sa_new, 0 );
  sigaction( SIGFPE,  &sa_new, 0 );
  sigaction( SIGTERM, &sa_new, 0 );
  sigaction( SIGABRT, &sa_new, 0 );

  /* Process command line options */
  while( (option = getopt(argc, argv, "hv") ) != -1 )
	switch( option )
	{
	  case 'h' : /* Print usage and exit */
		Usage();
		return(0);

	  case 'v' : /* Print version */
		puts( PACKAGE_STRING );
		return(0);

	  default: /* Print usage and exit */
		Usage();
		exit(-1);

	} /* End of switch( option ) */

  gtk_set_locale();
  gtk_init(&argc, &argv);

  xgridloc_main_window = create_main_window();
  gtk_window_set_title( GTK_WINDOW(xgridloc_main_window), PACKAGE_STRING );

  /* Setup signal callbacks */
  {
	GtkWidget *entry;

	entry = lookup_widget(xgridloc_main_window, "home_lon_deg");
	g_signal_connect ((gpointer) entry, "activate",
		G_CALLBACK (on_home_pos_activate), "179");
	g_signal_connect ((gpointer) entry, "changed",
		G_CALLBACK (on_pos_changed), "179");

	entry = lookup_widget(xgridloc_main_window, "home_lon_min");
	g_signal_connect ((gpointer) entry, "activate",
		G_CALLBACK (on_home_pos_activate), "59");
	g_signal_connect ((gpointer) entry, "changed",
		G_CALLBACK (on_pos_changed), "59");

	entry = lookup_widget(xgridloc_main_window, "home_lon_sec");
	g_signal_connect ((gpointer) entry, "activate",
		G_CALLBACK (on_home_pos_activate), "59");
	g_signal_connect ((gpointer) entry, "changed",
		G_CALLBACK (on_pos_changed), "59");

	entry = lookup_widget(xgridloc_main_window, "home_lat_deg");
	g_signal_connect ((gpointer) entry, "activate",
		G_CALLBACK (on_home_pos_activate), "89");
	g_signal_connect ((gpointer) entry, "changed",
		G_CALLBACK (on_pos_changed), "89");

	entry = lookup_widget(xgridloc_main_window, "home_lat_min");
	g_signal_connect ((gpointer) entry, "activate",
		G_CALLBACK (on_home_pos_activate), "59");
	g_signal_connect ((gpointer) entry, "changed",
		G_CALLBACK (on_pos_changed), "59");

	entry = lookup_widget(xgridloc_main_window, "home_lat_sec");
	g_signal_connect ((gpointer) entry, "activate",
		G_CALLBACK (on_home_pos_activate), "59");
	g_signal_connect ((gpointer) entry, "changed",
		G_CALLBACK (on_pos_changed), "59");

	entry = lookup_widget(xgridloc_main_window, "remt_lon_deg");
	g_signal_connect ((gpointer) entry, "activate",
		G_CALLBACK (on_remt_pos_activate), "179");
	g_signal_connect ((gpointer) entry, "changed",
		G_CALLBACK (on_pos_changed), "179");

	entry = lookup_widget(xgridloc_main_window, "remt_lon_min");
	g_signal_connect ((gpointer) entry, "activate",
		G_CALLBACK (on_remt_pos_activate), "59");
	g_signal_connect ((gpointer) entry, "changed",
		G_CALLBACK (on_pos_changed), "59");

	entry = lookup_widget(xgridloc_main_window, "remt_lon_sec");
	g_signal_connect ((gpointer) entry, "activate",
		G_CALLBACK (on_remt_pos_activate), "59");
	g_signal_connect ((gpointer) entry, "changed",
		G_CALLBACK (on_pos_changed), "59");

	entry = lookup_widget(xgridloc_main_window, "remt_lat_deg");
	g_signal_connect ((gpointer) entry, "activate",
		G_CALLBACK (on_remt_pos_activate), "89");
	g_signal_connect ((gpointer) entry, "changed",
		G_CALLBACK (on_pos_changed), "89");

	entry = lookup_widget(xgridloc_main_window, "remt_lat_min");
	g_signal_connect ((gpointer) entry, "activate",
		G_CALLBACK (on_remt_pos_activate), "59");
	g_signal_connect ((gpointer) entry, "changed",
		G_CALLBACK (on_pos_changed), "59");

	entry = lookup_widget(xgridloc_main_window, "remt_lat_sec");
	g_signal_connect ((gpointer) entry, "activate",
		G_CALLBACK (on_remt_pos_activate), "59");
	g_signal_connect ((gpointer) entry, "changed",
		G_CALLBACK (on_pos_changed), "59");
  }

  gtk_widget_show(xgridloc_main_window);
  gtk_idle_add( Load_Config, "" );

  gtk_main();

  return 0;
}

/*------------------------------------------------------------------------*/

/*  Load_Config()
 *
 *  Loads the xgridloc configuration file
 */

  gboolean
Load_Config( gpointer data )
{
  char
	dist_unit[10],  /* Name of distance unit   */
	*rc_fpath,		/* File path to xgridlocrc */
	line[196];		/* Buffer for Load_Line    */

  int idx, len;

  /* Config file pointer */
  FILE *xgridlocrc;


  /* Setup file path to xgridlocrc */
  size_t s = strlen(getenv("HOME")) + strlen(".xgridlocrc") + 2;
  rc_fpath = (char *)malloc( s );
  if( rc_fpath == NULL )
  {
	Error_Dialog(
		_("Failed to allocate memory for\n"\
		  "Configuration file name string.\n"\
		  "Please quit xgridloc and correct") );
	return( FALSE );
  }
  snprintf( rc_fpath, s, "%s/%s", getenv("HOME"), ".xgridlocrc" );
  rc_fpath[s-1] = '\0';

  /* Open ~./xgridlocrc file */
  xgridlocrc = fopen( rc_fpath, "r" );
  if( xgridlocrc == NULL )
  {
	Error_Dialog(
		_("Error opening configuration file\n"\
		  "Please quit xgridloc and correct") );
	free(rc_fpath);
	return( FALSE );
  }

  free(rc_fpath);

  /*** Read runtime configuration data ***/
  /* Read 'Home' position, abort on EOF */
  if( Load_Line(line, xgridlocrc, _("Home Position") ) != SUCCESS )
	return( FALSE );
  Strlcpy( rc_data.home_posn, line, sizeof(rc_data.home_posn) );

  /* Read 'Home' position name, abort if EOF */
  if( Load_Line(line, xgridlocrc, _("Home Position Name") ) != SUCCESS )
	return( FALSE );
  Strlcpy( rc_data.home_name, line, sizeof(rc_data.home_name) );

  /* Read distance unit name and conversion factor, abort if EOF */
  if( Load_Line(line, xgridlocrc, _("Distance Unit Name") ) != SUCCESS )
	return( FALSE );
  len = (int)strlen( line );
  for( idx = 0; idx < len; idx++ )
	if( (line[idx] == ' ') || (idx == 9) )
	  break;
  Strlcpy( dist_unit, line, idx+1 );
  rc_data.unit_conv = Atof( &line[idx] );

  /* Set distance labels */
  gtk_label_set_text( GTK_LABEL(
		lookup_widget(xgridloc_main_window, "short_label")), dist_unit );
  gtk_label_set_text( GTK_LABEL(
		lookup_widget(xgridloc_main_window, "long_label")), dist_unit );

  /* Read xplanet command */
  if( Load_Line(rc_data.xplanet_cmnd, xgridlocrc, _("Xplanet Command") ) != SUCCESS )
	return( FALSE );

  /* Read xplanet config file */
  if( Load_Line(rc_data.config_file, xgridlocrc, _("Xplanet config file") ) != SUCCESS )
	return( FALSE );

  /* Read xplanet marker file */
  if( Load_Line(rc_data.marker_file, xgridlocrc, _("Xplanet marker file") ) != SUCCESS )
	return( FALSE );

  /* Read xplanet arc file */
  if( Load_Line(rc_data.arc_file, xgridlocrc, _("Xplanet arc file") ) != SUCCESS )
	return( FALSE );

  fclose( xgridlocrc );

  return( FALSE );

} /* End of Load_Config */

/*------------------------------------------------------------------*/

/*  Sig_Handler()
 *
 *  Signal Action Handler function
 */

static void Signal_Handler( int signal )
{
  Cleanup();

  switch( signal )
  {
    case SIGINT  : fprintf( stderr,  _("\nxgridloc: Exiting via User Interrupt\n") );
		   exit(0);
    case SIGSEGV : fprintf( stderr,  _("xgridloc: Segmentation Fault\n") );
		   break;
    case SIGFPE  : fprintf( stderr,  _("xgridloc: Floating Point Exception\n") );
		   break;
    case SIGABRT : fprintf( stderr,  _("xgridloc: Abort Signal received\n") );
		   break;
    case SIGTERM : fprintf( stderr,  _("xgridloc: Termination Request received\n") );
  }

  exit( -1 );

} /* End of Sig_Handler() */

/*------------------------------------------------------------------------*/

