/*
 *  xgridloc: A GTK+ application to calculate the Maidenhead Grid
 *  Locator of a given position or vice-versa. Also to calculate
 *  bearing and distance to a remote position and display a Great
 *  Circle arc between Home and Remote positions using 'xplanet'.
 *
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

/*  calculate.c
 *
 *  Functions to calculate a Maidenhead grid locator
 *  given longitude/latitude and vice versa. Also to
 *  calculate great circle bearing and distance to a
 *  second location given a grid locator or lon/lat
 */

#include "calculate.h"
#include "shared.h"

/* Short path bearing and distance A to B */
static double
bng_spath,
  dst_spath;

static FILE
  *config_fd,	/* xgridloc config file for xplanet */
  *marker_fd,   /* Marker file for xplanet with local/remote location */
  *greatarc_fd; /* Great circle arc co-ordinates for xplanet */


/*  Position_to_Gridloc()
 *
 *  Function to calculate a grid locator give lon/lat
 */

  void
Position_to_Gridloc( double lon, double lat, char *grid_loc )
{
  double temp;

  /* Adjust longitude and latitude to references for the   */
  /* Maidenhead grid locator system and roundup to 1/2 sec */
  lon += 180.0 + 1.3888888888E-4;
  lat +=  90.0 + 1.3888888888E-4;

  /* Calculate first letter of field */
  temp = lon / 20.0;
  grid_loc[0] = (char)temp + 'A';

  /* Calculate first number of square */
  lon -= floor(temp) * 20.0;
  temp = lon / 2.0;
  grid_loc[2] = (char)temp + '0';

  /* Calculate first letter of sub-square */
  lon -= floor(temp) * 2.0;
  temp = lon * 12.0;
  grid_loc[4] = (char)temp + 'A';

  /* Calculate second letter of field */
  temp = lat / 10.0;
  grid_loc[1] = (char)temp + 'A';

  /* Calculate second number of square */
  lat -= floor(temp) * 10.0;
  temp = lat / 1.0;
  grid_loc[3] = (char)temp + '0';

  /* Calculate second letter of sub-square */
  lat -= floor(temp) * 1.0;
  temp = lat * 24.0;
  grid_loc[5] = (char)temp + 'A';

  grid_loc[6] = '\0';

} /* End of Position_to_Gridloc() */

/*--------------------------------------------------------------------*/

/*  Gridloc_to_Position()
 *
 *  Function to calculate longitude/latitude
 *  given a position's grid locator
 */

  void
Gridloc_to_Position( char *grid_loc, double *lon, double *lat )
{
  *lon  = (grid_loc[0] - 'A') * 20.0;
  *lon += (grid_loc[2] - '0') * 2.0;
  *lon += (grid_loc[4] - 'A') / 12.0;
  *lon -= 180.0;

  *lat  = (grid_loc[1] - 'A') * 10.0;
  *lat += (grid_loc[3] - '0') * 1.0;
  *lat += (grid_loc[5] - 'A') / 24.0;
  *lat -= 90.0;

} /* End of Gridloc_to_Position() */

/*--------------------------------------------------------------------*/

/*  Degrees_to_DMS()
 *
 * Function to convert an angle in decimal degrees to D:M:S
 */

  void
Degrees_to_DMS( double deg, int *ideg, int *imin, int *isec )
{
  double temp;

  deg += 1.0/7200.0; /* Round-up to 0.5 sec */
  *ideg = (int)deg;
  temp = ( deg - (double)*ideg ) * 60.0;
  *imin = (int)temp;
  temp = ( temp - (double)*imin ) * 60.0;
  *isec = (int)(temp);

} /* End of Degrees_to_DMS() */

/*--------------------------------------------------------------------*/

/*  DMS_to_Degrees()
 *
 *  Function to convert an angle in D:M:S to decimal degrees
 */

  void
DMS_to_Degrees( int ideg, int imin, int isec, double *deg )
{
  *deg = (double)ideg + (double)imin/60.0 + (double)isec/3600.0;

} /* End of DMS_to_degrees() */

/*--------------------------------------------------------------------*/

/*  Bearing_Distance()
 *
 *  Function to calculate and show bearing
 *  and distance of Remote from Home
 */

  void
Bearing_Distance()
{
  double
	gc_arc, cos_gc_arc,       /* Great circle arc and cos A to B */
	cos_bearing, sin_bearing, /* cos/sin of bearing  from A to B */
	lon_diff,                 /* Difference in longitude of B from A    */
	bng_lpath, dst_lpath;     /* Long path bearing and distance A to B  */

  /* Temporary long/latd */
  double
	lon_a, lat_a,
	lon_b, lat_b;

  /* Buffer for text entries */
  char buff[6];

  GtkWidget *widget;

  /* Convert to radians */
  lat_a = home_lat * deg2rad;
  lon_a = home_lon * deg2rad;
  lat_b = remt_lat * deg2rad;
  lon_b = remt_lon * deg2rad;

  /* Longitude differnce of B from A */
  lon_diff = lon_b - lon_a;

  /* Calculate great circle distance A to B */
  cos_gc_arc = cos(lon_diff)*cos(lat_a)*cos(lat_b) + sin(lat_a)*sin(lat_b);
  gc_arc = acos( cos_gc_arc );

  /* Distance in km */
  dst_spath = eradius * gc_arc;

  /* Calculate bearing A to B */
  cos_bearing  = sin(lat_b) - sin(lat_a) * cos_gc_arc;
  sin_bearing  = sin(lon_diff) * cos(lat_a) * cos(lat_b);
  bng_spath = atan2(sin_bearing, cos_bearing);

  /* Correct negative (anticlockwise) bearings */
  if( bng_spath < 0.0 )
	bng_spath = twopi + bng_spath;

  /* Convert to degrees */
  bng_spath /= deg2rad;

  /* Calculate long-path data */
  bng_lpath = bng_spath + 180.0;
  if( bng_lpath > 360.0 )
	bng_lpath -= 360.0;
  dst_lpath = eperim - dst_spath;

  /* Update Short Path bearing entry field */
  snprintf( buff, 4, "%03d", (int)(bng_spath + 0.5) );
  widget = lookup_widget( xgridloc_main_window, "bng_spath" );
  gtk_entry_set_text(GTK_ENTRY(widget), buff );

  /* Update Short Path distance entry field */
  snprintf( buff, 6, "%5d", (int)(dst_spath * rc_data.unit_conv + 0.5) );
  widget = lookup_widget( xgridloc_main_window, "dist_spath" );
  gtk_entry_set_text(GTK_ENTRY(widget), buff );

  /* Update Long Path bearing entry field */
  snprintf( buff, 4, "%03d", (int)(bng_lpath + 0.5) );
  widget = lookup_widget( xgridloc_main_window, "bng_lpath" );
  gtk_entry_set_text(GTK_ENTRY(widget), buff );

  /* Update Long Path distance entry field */
  snprintf( buff, 6, "%5d", (int)(dst_lpath * rc_data.unit_conv + 0.5) );
  widget = lookup_widget( xgridloc_main_window, "dist_lpath" );
  gtk_entry_set_text(GTK_ENTRY(widget), buff );

  /* Display Home/Remote positions in xplanet */
  if( isFlagSet(XPLANET_ENABLED) )
	Display_Location();

} /* End of Bearing_Distance() */

/*--------------------------------------------------------------------*/

/*  Read_Home_Position()
 *
 *  Reads position fields (Lon/Lat) and calculates Lon/Lat
 */

  void
Read_Home_Position(void)
{
  /* Used for converting degrees to ddmmss */
  int ideg, imin, isec;

  char buff[4];

  GtkWidget *widget;


  /*** Calculate Home position and grid locator ***/

  /* Select longitude deg entry field */
  widget = lookup_widget( xgridloc_main_window, "home_lon_deg" );
  if( !Position_Valid(GTK_ENTRY(widget), "179") )
	return;
  ideg = atoi( gtk_entry_get_text(GTK_ENTRY(widget)) );

  /* Reformat entry and set to GtkEntry widget */
  snprintf( buff, 4, "%03d", ideg );
  gtk_entry_set_text( GTK_ENTRY(widget), buff );

  /* Select longitude min entry field */
  widget = lookup_widget( xgridloc_main_window, "home_lon_min" );
  if( !Position_Valid(GTK_ENTRY(widget), "59") )
	return;
  imin = atoi( gtk_entry_get_text(GTK_ENTRY(widget)) );

  /* Reformat entry and set to GtkEntry widget */
  snprintf( buff, 3, "%02d", imin );
  gtk_entry_set_text( GTK_ENTRY(widget), buff );

  /* Select longitude sec entry field */
  widget = lookup_widget( xgridloc_main_window, "home_lon_sec" );
  if( !Position_Valid(GTK_ENTRY(widget), "59") )
	return;
  isec = atoi( gtk_entry_get_text(GTK_ENTRY(widget)) );

  /* Reformat entry and set to GtkEntry widget */
  snprintf( buff, 3, "%02d", isec );
  gtk_entry_set_text( GTK_ENTRY(widget), buff );

  /* Convert d/m/s to double */
  DMS_to_Degrees( ideg, imin, isec, &home_lon );

  /* Negate west longitudes */
  if( isFlagSet(HOME_LONG_WEST) )
	home_lon = -home_lon;

  /* Select latitude deg entry field */
  widget = lookup_widget( xgridloc_main_window, "home_lat_deg" );
  if( !Position_Valid(GTK_ENTRY(widget), "89") )
	return;
  ideg = atoi( gtk_entry_get_text(GTK_ENTRY(widget)) );

  /* Reformat entry and set to GtkEntry widget */
  snprintf( buff, 3, "%02d", ideg );
  gtk_entry_set_text( GTK_ENTRY(widget), buff );

  /* Select latitude min entry field */
  widget = lookup_widget( xgridloc_main_window, "home_lat_min" );
  if( !Position_Valid(GTK_ENTRY(widget), "59") )
	return;
  imin = atoi( gtk_entry_get_text(GTK_ENTRY(widget)) );

  /* Reformat entry and set to GtkEntry widget */
  snprintf( buff, 3, "%02d", imin );
  gtk_entry_set_text( GTK_ENTRY(widget), buff );

  /* Select latitude sec entry field */
  widget = lookup_widget( xgridloc_main_window, "home_lat_sec" );
  if( !Position_Valid(GTK_ENTRY(widget), "59") )
	return;
  isec = atoi( gtk_entry_get_text(GTK_ENTRY(widget)) );

  /* Reformat entry and set to GtkEntry widget */
  snprintf( buff, 3, "%02d", isec );
  gtk_entry_set_text( GTK_ENTRY(widget), buff );

  /* Convert d/m/s to double */
  DMS_to_Degrees( ideg, imin, isec, &home_lat );

  /* Negate south latitudes */
  if( isFlagSet(HOME_LATD_SOUTH) )
	home_lat = -home_lat;

  /* Calculate grid locator and show */
  Position_to_Gridloc( home_lon, home_lat, home_loc );
  widget = lookup_widget( xgridloc_main_window, "home_loc" );
  gtk_entry_set_text( GTK_ENTRY(widget), home_loc );

  SetFlag( HOME_LOC_ENTERED );

  /* Display Home/Remote positions */
  if( isFlagSet(REMT_LOC_ENTERED) )
	Bearing_Distance();

} /* End of Read_Home_Position() */

/*------------------------------------------------------------------------*/

/*  Read_Remote_Position()
 *
 *  Reads position fields (Lon/Lat) and calculates Lon/Lat
 */

  void
Read_Remote_Position(void)
{
  /* Used for converting degrees to ddmmss */
  int ideg, imin, isec;

  char buff[4];

  GtkWidget *widget;


  /*** Calculate Remote position and grid locator ***/

  /* Select longitude deg entry field */
  widget = lookup_widget( xgridloc_main_window, "remt_lon_deg" );
  if( !Position_Valid(GTK_ENTRY(widget), "179") )
	return;
  ideg = atoi( gtk_entry_get_text(GTK_ENTRY(widget)) );

  /* Reformat entry and set to GtkEntry widget */
  snprintf( buff, 4, "%03d", ideg );
  gtk_entry_set_text( GTK_ENTRY(widget), buff );

  /* Select longitude min entry field */
  widget = lookup_widget( xgridloc_main_window, "remt_lon_min" );
  if( !Position_Valid(GTK_ENTRY(widget), "59") )
	return;
  imin = atoi( gtk_entry_get_text(GTK_ENTRY(widget)) );

  /* Reformat entry and set to GtkEntry widget */
  snprintf( buff, 3, "%02d", imin );
  gtk_entry_set_text( GTK_ENTRY(widget), buff );

  /* Select longitude sec entry field */
  widget = lookup_widget( xgridloc_main_window, "remt_lon_sec" );
  if( !Position_Valid(GTK_ENTRY(widget), "59") )
	return;
  isec = atoi( gtk_entry_get_text(GTK_ENTRY(widget)) );

  /* Reformat entry and set to GtkEntry widget */
  snprintf( buff, 3, "%02d", isec );
  gtk_entry_set_text( GTK_ENTRY(widget), buff );

  /* Convert d/m/s to double */
  DMS_to_Degrees( ideg, imin, isec, &remt_lon );

  /* Negate west longitudes */
  if( isFlagSet(REMT_LONG_WEST) )
	remt_lon = -remt_lon;

  /* Select latitude deg entry field */
  widget = lookup_widget( xgridloc_main_window, "remt_lat_deg" );
  if( !Position_Valid(GTK_ENTRY(widget), "89") )
	return;
  ideg = atoi( gtk_entry_get_text(GTK_ENTRY(widget)) );

  /* Reformat entry and set to GtkEntry widget */
  snprintf( buff, 3, "%02d", ideg );
  gtk_entry_set_text( GTK_ENTRY(widget), buff );

  /* Select latitude min entry field */
  widget = lookup_widget( xgridloc_main_window, "remt_lat_min" );
  if( !Position_Valid(GTK_ENTRY(widget), "59") )
	return;
  imin = atoi( gtk_entry_get_text(GTK_ENTRY(widget)) );

  /* Reformat entry and set to GtkEntry widget */
  snprintf( buff, 3, "%02d", imin );
  gtk_entry_set_text( GTK_ENTRY(widget), buff );

  /* Select latitude sec entry field */
  widget = lookup_widget( xgridloc_main_window, "remt_lat_sec" );
  if( !Position_Valid(GTK_ENTRY(widget), "59") )
	return;
  isec = atoi( gtk_entry_get_text(GTK_ENTRY(widget)) );

  /* Reformat entry and set to GtkEntry widget */
  snprintf( buff, 3, "%02d", isec );
  gtk_entry_set_text( GTK_ENTRY(widget), buff );

  /* Convert d/m/s to double */
  DMS_to_Degrees( ideg, imin, isec, &remt_lat );

  /* Negate south latitudes */
  if( isFlagSet(REMT_LATD_SOUTH) )
	remt_lat = -remt_lat;

  /* Calculate grid locator and show */
  Position_to_Gridloc( remt_lon, remt_lat, remt_loc );
  widget = lookup_widget( xgridloc_main_window, "remt_loc" );
  gtk_entry_set_text( GTK_ENTRY(widget), remt_loc );

  SetFlag( REMT_LOC_ENTERED );

  /* Display Home/Remote positions in xplanet */
  if( isFlagSet(HOME_LOC_ENTERED) )
	Bearing_Distance();

} /* End of Read_Remote_Position() */

/*------------------------------------------------------------------------*/

/*  Read_Home_Locator()
 *
 *  Reads a Locator field and calculates Lon/Lat
 */

  void
Read_Home_Locator(void)
{
  /* Used for converting degrees to ddmmss */
  int ideg, imin, isec;
  char buff[4];
  int idx;

  /* Temporary for displaying etc */
  double lng, ltd;

  GtkWidget *widget;


  /*** Read grid locator and calculate position ***/
  widget = lookup_widget( xgridloc_main_window, "home_loc" );
  Strlcpy( home_loc,
	  gtk_entry_get_text(GTK_ENTRY(widget)),
	  sizeof(home_loc) );

  /* Capitalize letters */
  for( idx = 0; idx < 6; idx++ )
	if( (home_loc[idx] > 0x60) && (home_loc[idx] < 0x7b) )
	  home_loc[idx] -= 0x20;

  /* Display formatted locator */
  gtk_entry_set_text( GTK_ENTRY(widget), home_loc );

  /* Convert to long/latd */
  Gridloc_to_Position( home_loc, &home_lon, &home_lat );

  /*** Enter longitude data to Home fields ***/

  /* Select longitude direction button */
  widget = lookup_widget( xgridloc_main_window, "home_lon_dir" );

  /* Negative longitude shown as West */
  if( home_lon < 0.0 )
  {
	lng = -home_lon;
	SetFlag( HOME_LONG_WEST );
	/* Change longitude button label to West */
	gtk_button_set_label( GTK_BUTTON(widget), _("West") );
  }
  else
  {
	lng = home_lon;
	ClearFlag( HOME_LONG_WEST );
	/* Change longitude button label to East */
	gtk_button_set_label( GTK_BUTTON(widget), _("East") );
  }

  /* Convert longitude to deg/min/sec */
  Degrees_to_DMS( lng, &ideg, &imin, &isec );

  /* Select longitude deg entry field */
  widget = lookup_widget( xgridloc_main_window, "home_lon_deg" );

  /* Convert number to string */
  snprintf( buff, 4, "%03d", ideg );

  /* Set as text in the GtkEntry widget */
  gtk_entry_set_text( GTK_ENTRY(widget), buff );

  /* Select longitude min entry field */
  widget = lookup_widget( xgridloc_main_window, "home_lon_min" );

  /* Convert number to string */
  snprintf( buff, 3, "%02d", imin );

  /* Set as text in the GtkEntry widget */
  gtk_entry_set_text( GTK_ENTRY(widget), buff );

  /* Select longitude sec entry field */
  widget = lookup_widget( xgridloc_main_window, "home_lon_sec" );

  /* Convert number to string */
  snprintf( buff, 3, "%02d", isec );

  /* Set as text in the GtkEntry widget */
  gtk_entry_set_text( GTK_ENTRY(widget), buff );

  /*** Enter latitude data to Home fields ***/

  /* Select latitude direction button */
  widget = lookup_widget( xgridloc_main_window, "home_lat_dir" );

  /* Negative latitude shown as South */
  if( home_lat < 0.0 )
  {
	ltd = -home_lat;
	SetFlag( HOME_LATD_SOUTH );
	/* Change latitude button label to South */
	gtk_button_set_label( GTK_BUTTON(widget), _("South") );
  }
  else
  {
	ltd = home_lat;
	ClearFlag( HOME_LATD_SOUTH );
	/* Change latitude button label to North */
	gtk_button_set_label( GTK_BUTTON(widget), _("North") );
  }

  /* Convert latitude to deg/min/sec */
  Degrees_to_DMS( ltd, &ideg, &imin, &isec );

  /* Select latitude deg entry field */
  widget = lookup_widget( xgridloc_main_window, "home_lat_deg" );

  /* Convert number to string */
  snprintf( buff, 3, "%02d", ideg );

  /* Set as text in the GtkEntry widget */
  gtk_entry_set_text( GTK_ENTRY(widget), buff );

  /* Select latitude min entry field */
  widget = lookup_widget( xgridloc_main_window, "home_lat_min" );

  /* Convert number to string */
  snprintf( buff, 3, "%02d", imin );

  /* Set as text in the GtkEntry widget */
  gtk_entry_set_text( GTK_ENTRY(widget), buff );

  /* Select latitude sec entry field */
  widget = lookup_widget( xgridloc_main_window, "home_lat_sec" );

  /* Convert number to string */
  snprintf( buff, 3, "%02d", isec );

  /* Set as text in the GtkEntry widget */
  gtk_entry_set_text( GTK_ENTRY(widget), buff );

  SetFlag( HOME_LOC_ENTERED );

  /* Display Home/Remote positions in xplanet */
  if( isFlagSet(REMT_LOC_ENTERED) )
	Bearing_Distance();

}/* End of Read_Home_Locator() */

/*------------------------------------------------------------------------*/

/*  Read_Remote_Locator()
 *
 *  Reads a Locator field and calculates Lon/Lat
 */

  void
Read_Remote_Locator(void)
{
  /* Used for converting degrees to ddmmss */
  int ideg, imin, isec;
  char buff[4];
  int idx;

  /* Temporary for displaying */
  double lng, ltd;

  GtkWidget *widget;


  /*** Read grid locator and calculate position ***/
  widget = lookup_widget( xgridloc_main_window, "remt_loc" );
  Strlcpy( remt_loc,
	  gtk_entry_get_text(GTK_ENTRY(widget)),
	  sizeof( remt_loc ) );

  /* Capitalize letters */
  for( idx = 0; idx < 6; idx++ )
	if( (remt_loc[idx] > 0x60) && (remt_loc[idx] < 0x7b) )
	  remt_loc[idx] -= 0x20;

  /* Display formatted locator */
  gtk_entry_set_text( GTK_ENTRY(widget), remt_loc );

  /* Convert to long/latd */
  Gridloc_to_Position( remt_loc, &remt_lon, &remt_lat );

  /*** Enter longitude data to Remote fields ***/

  /* Select longitude direction button */
  widget = lookup_widget( xgridloc_main_window, "remt_lon_dir" );

  /* Negative longitude shown as West */
  if( remt_lon < 0.0 )
  {
	lng = - remt_lon;
	SetFlag( REMT_LONG_WEST );
	/* Change long button label to West */
	gtk_button_set_label( GTK_BUTTON(widget), _("West") );
  }
  else
  {
	lng = remt_lon;
	ClearFlag( REMT_LONG_WEST );
	/* Change long button label to East */
	gtk_button_set_label( GTK_BUTTON(widget), _("East") );
  }

  /* Convert longitude to deg/min/sec */
  Degrees_to_DMS( lng, &ideg, &imin, &isec );

  /* Select longitude deg entry field */
  widget = lookup_widget( xgridloc_main_window, "remt_lon_deg" );

  /* Convert number to string */
  snprintf( buff, 4, "%03d", ideg );

  /* Set as text in the GtkEntry widget */
  gtk_entry_set_text( GTK_ENTRY(widget), buff );

  /* Select longitude min entry field */
  widget = lookup_widget( xgridloc_main_window, "remt_lon_min" );

  /* Convert number to string */
  snprintf( buff, 3, "%02d", imin );

  /* Set as text in the GtkEntry widget */
  gtk_entry_set_text( GTK_ENTRY(widget), buff );

  /* Select longitude sec entry field */
  widget = lookup_widget( xgridloc_main_window, "remt_lon_sec" );

  /* Convert number to string */
  snprintf( buff, 3, "%02d", isec );

  /* Set as text in the GtkEntry widget */
  gtk_entry_set_text( GTK_ENTRY(widget), buff );

  /*** Enter latitude data to Remote fields ***/

  /* Select latitude direction button */
  widget = lookup_widget( xgridloc_main_window, "remt_lat_dir" );

  /* Negative latitude shown as South */
  if( remt_lat < 0.0 )
  {
	ltd = - remt_lat;
	SetFlag( REMT_LATD_SOUTH );
	/* Change latd button label to South */
	gtk_button_set_label( GTK_BUTTON(widget), _("South") );
  }
  else
  {
	ltd = remt_lat;
	ClearFlag( REMT_LATD_SOUTH );
	/* Change long button label to North */
	gtk_button_set_label( GTK_BUTTON(widget), _("North") );
  }

  /* Convert latitude to deg/min/sec */
  Degrees_to_DMS( ltd, &ideg, &imin, &isec );

  /* Select latitude deg entry field */
  widget = lookup_widget( xgridloc_main_window, "remt_lat_deg" );

  /* Convert number to string */
  snprintf( buff, 3, "%02d", ideg );

  /* Set as text in the GtkEntry widget */
  gtk_entry_set_text( GTK_ENTRY(widget), buff );

  /* Select latitude min entry field */
  widget = lookup_widget( xgridloc_main_window, "remt_lat_min" );

  /* Convert number to string */
  snprintf( buff, 3, "%02d", imin );

  /* Set as text in the GtkEntry widget */
  gtk_entry_set_text( GTK_ENTRY(widget), buff );

  /* Select latitude sec entry field */
  widget = lookup_widget( xgridloc_main_window, "remt_lat_sec" );

  /* Convert number to string */
  snprintf( buff, 3, "%02d", isec );

  /* Set as text in the GtkEntry widget */
  gtk_entry_set_text( GTK_ENTRY(widget), buff );

  SetFlag( REMT_LOC_ENTERED );

  /* Display Home/Remote positions in xplanet */
  if( isFlagSet(HOME_LOC_ENTERED) )
	Bearing_Distance();

} /* End of Read_Remote_Locator() */

/*------------------------------------------------------------------------*/

/*  Default_Home_Position()
 *
 *  Enters default home position data
 */

  void
Default_Home_Position(void)
{
  /* Buffer for conversions */
  char buff[32];

  /* Enter default Home position */
  Strlcpy( buff, rc_data.home_posn, sizeof(buff) );

  /* Set terminators in between deg, min, sec */
  /* of home long and lat. Crude but it works */
  buff[4]  = '\0';
  buff[8]  = '\0';
  buff[11] = '\0';
  buff[14] = '\0';
  buff[20] = '\0';
  buff[23] = '\0';
  buff[26] = '\0';
  buff[29] = '\0';

  /* Display default home position */
  gtk_button_set_label( GTK_BUTTON(lookup_widget(xgridloc_main_window,
		  "home_lon_dir")), &buff[0] );
  gtk_entry_set_text( GTK_ENTRY(lookup_widget(xgridloc_main_window,
		  "home_lon_deg")), &buff[5] );
  gtk_entry_set_text( GTK_ENTRY(lookup_widget(xgridloc_main_window,
		  "home_lon_min")), &buff[9] );
  gtk_entry_set_text( GTK_ENTRY(lookup_widget(xgridloc_main_window,
		  "home_lon_sec")), &buff[12] );
  gtk_button_set_label( GTK_BUTTON(lookup_widget(xgridloc_main_window,
		  "home_lat_dir")), &buff[15] );
  gtk_entry_set_text( GTK_ENTRY(lookup_widget(xgridloc_main_window,
		  "home_lat_deg")), &buff[21] );
  gtk_entry_set_text( GTK_ENTRY(lookup_widget(xgridloc_main_window,
		  "home_lat_min")), &buff[24] );
  gtk_entry_set_text( GTK_ENTRY(lookup_widget(xgridloc_main_window,
		  "home_lat_sec")), &buff[27] );

  /* Set direction flags */
  if( strcmp(&buff[0], _("West")) == 0 )
	SetFlag( HOME_LONG_WEST );
  else
	ClearFlag( HOME_LONG_WEST );

  if( strcmp(&buff[15], _("South")) == 0 )
	SetFlag( HOME_LATD_SOUTH );
  else
	ClearFlag( HOME_LATD_SOUTH );

  /* Read Home position and convert to locator */
  Read_Home_Position();

} /* End of Default_Home_Position() */

/*------------------------------------------------------------------------*/

/*  Display_Location
 *
 *  Displays Home and Remote locations
 *  and Great Circle arc using xplanet
 */

  int
Display_Location( void )
{
  int error = 0;  /* Error flag for system calls */

  /* Open xplanet file descriptors */
  config_fd   = fopen( rc_data.config_file, "w" );
  marker_fd   = fopen( rc_data.marker_file, "w" );
  greatarc_fd = fopen( rc_data.arc_file, "w" );

  if( (config_fd   == NULL) ||
	  (marker_fd   == NULL) ||
	  (greatarc_fd == NULL) )
  {
	perror( "xgridloc: fopen()" );
	Cleanup();
	Error_Dialog( _("Cannot open xplanet config file") );
	return( error );
  }

  /* Setup config file */
  error = fprintf( config_fd,
	  "arc_file=%s\nmarker_file=%s\n", rc_data.arc_file, rc_data.marker_file );

  /* Enter "Home" position in marker file */
  fprintf( marker_fd, "%8.3f %8.3f \"%s\" %s\n",
	  home_lat, home_lon,
	  ( isFlagSet(DEFAULT_HOME) ? rc_data.home_name : home_loc),
	  "color=red" );

  /* Enter "Remote" position in marker file */
  fprintf( marker_fd, "%8.3f %8.3f \"%s %03dd/%dkm\" %s\n",
	  remt_lat, remt_lon, remt_loc, (int)bng_spath, (int)dst_spath,
	  "color=yellow" );

  /* Enter Home/Remote positions in greatarc file */
  fprintf( greatarc_fd, "%8.3f %8.3f %8.3f %8.3f %s\n",
	  home_lat, home_lon, remt_lat, remt_lon, "color=yellow" );

  fclose( config_fd );
  fclose( greatarc_fd );
  fclose( marker_fd );

  /* Invoke xplanet */
  error = system( "pkill xplanet" );
  error = system( rc_data.xplanet_cmnd );

  /* On error delete xplanet's files */
  if( error )
  {
	Error_Dialog( _("Cannot run xplanet command") );
	Cleanup();
	return( error );
  }

  return( 0 );

} /* End of Display_Location() */

/*------------------------------------------------------------------------*/

/*  Locator_Valid()
 *
 *  Validates a grid locator
 */

  gboolean
Locator_Valid( GtkEntry *entry )
{
  int
	len, /* Length of user-entered data  */
	idx;

  char buff[7];

  /* Get entry field */
  Strlcpy( buff, gtk_entry_get_text(entry), sizeof(buff) );
  len = (int)strlen(buff);

  /* Ignore blank fields */
  if( len == 0 ) return( FALSE );

  /* Capitalize letters */
  for( idx = 0; idx < len; idx++ )
	if( (buff[idx] > 0x60) && (buff[idx] < 0x7b) )
	{
	  buff[idx] -= 0x20;
	  gtk_entry_set_text( entry, buff );
	}

  /* Validate grid locator */
  for( idx = 0; idx < len; idx++ )
	switch( idx )
	{
	  case 0: case 1: /* First letters */
		if( (buff[idx] < 'A') || (buff[idx] > 'S') )
		{
		  Error_Dialog(
			  _("You have entered invalid data\n"\
				"Please make a correction\n"\
				"(Letters A to S inclusive)") );
		  return( FALSE );
		}
		break;

	  case 2: case 3: /* Middle numbers */
		if( (buff[idx] < '0') || (buff[idx] > '9') )
		{
		  Error_Dialog(
			  _("You have entered invalid data\n"\
				"Please make a correction\n"\
				"(Numbers only, 0-9 inclusive)") );
		  return( FALSE );
		}
		break;

	  case 4: case 5: /* Last letters */
		if( (buff[idx] < 'A') || (buff[idx] > 'X') )
		{
		  Error_Dialog(
			  _("You have entered invalid data\n"\
				"Please make a correction\n"\
				"(Letters A to X inclusive)") );
		  return( FALSE );
		}
	} /* switch( idx ) */

  if( len == 6 )
	return( TRUE );
  else
	return( FALSE );

} /* End of Locator_Valid()*/

/*------------------------------------------------------------------------*/

/*  Position_Valid()
 *
 *  Validates user's entry to a position field
 */

  gboolean
Position_Valid( GtkEntry *entry, gpointer user_data )
{
  int
	len,   /* Length of user-entered data  */
	value, /* Numeric value of entry field */
	limit, /* Verification's upper limit   */
	idx;

  char buff[4];

  /*** Validate and reformat user input with leading zero's ***/

  /* Get entry field */
  Strlcpy( buff, gtk_entry_get_text(entry), sizeof(buff) );
  len = (int)strlen(buff);

  /* Ignore empty fields */
  if( len == 0 ) return( FALSE );

  /* Reject non-numeric entries */
  for( idx = 0; idx < len; idx++ )
	if( ((buff[idx] < '0') || (buff[idx] > '9')) && (buff[idx] != '!') )
	{
	  buff[idx] = '!';
	  gtk_entry_set_text( entry, buff );
	  Error_Dialog(
		  _("You have entered invalid data\n"\
			"Please make a correction\n"\
			"(Numbers 0-9, inclusive)") );
	  return( FALSE );
	}

  /* Read user input */
  value = atoi( buff );

  /* Read limit */
  limit = atoi( user_data );

  /* Verify user input and select field on fail */
  if( (value < 0) || (value > limit) )
  {
	buff[len-1] = '!';
	gtk_entry_set_text( entry, buff );
	Error_Dialog(
		_("Value is out of range\n"\
		  "Please make a correction") );
	return( FALSE );
  }

  return( TRUE );

} /* End of Position_Valid() */

/*------------------------------------------------------------------------*/
