/*  utils.c
 *
 *  Utility functions of wxapt application
 */

/*
 *  wxapt: An application to decode APT signals from
 *  weather satellites and produce an image of the weather.
 *
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "utils.h"
#include "shared.h"

/*------------------------------------------------------------------*/

/*  Load_Config()
 *
 *  Loads the wxaptrc configuration file
 */

  void
Load_Config( void )
{
  char
	rc_fpath[64], /* File path to wxaptrc */
	line[81];     /* Buffer for Load_Line  */

  /* Config file pointer */
  FILE *wxaptrc;

  /* ALSA channel names and values */
  char *chan_name[] =
  {
	"FRONT_LEFT",
	"FRONT_RIGHT",
	"REAR_LEFT",
	"REAR_RIGHT",
	"SIDE_LEFT",
	"SIDE_RIGHT",
	"MONO",
	"NONE"
  };

  int chan_number[] =
  {
	SND_MIXER_SCHN_FRONT_LEFT,
	SND_MIXER_SCHN_FRONT_RIGHT,
	SND_MIXER_SCHN_REAR_LEFT,
	SND_MIXER_SCHN_REAR_RIGHT,
	SND_MIXER_SCHN_SIDE_LEFT,
	SND_MIXER_SCHN_SIDE_RIGHT,
	SND_MIXER_SCHN_MONO,
	-1
  };

  int idx, num_chan = 8;


  /* Setup file path to wxaptrc and working dir */
  snprintf( rc_fpath, sizeof(rc_fpath),
	  "%s/wxapt/wxaptrc", getenv("HOME") );
  snprintf( rc_data.wxapt_dir, sizeof(rc_data.wxapt_dir),
	  "%s/wxapt/", getenv("HOME") );

  /* Open wxaptrc file */
  wxaptrc = fopen( rc_fpath, "r" );
  if( wxaptrc == NULL )
  {
	perror( rc_fpath );
	fprintf( stderr, "Failed to open wxaptrc file\n" );
	exit( -1 );
  }

  /*** Read runtime configuration data ***/

  /* Read Receiver Type to use (Analogue via soundcard
   * and ALSA or a librtlsdr compatible SDR device) */
  if( Load_Line(line, wxaptrc, "Receiver Type") != SUCCESS )
	exit( -1 );
  if( strcmp(line, "SDR") == 0 )
	SetFlag( USE_RTLSDR_RX );
  else if( strcmp(line, "SND") == 0 )
	ClearFlag( USE_RTLSDR_RX );
  else
  {
	fprintf( stderr, "Invalid Radio Receiver type\n" );
	exit( -1 );
  }

  /* Read librtlsdr Device Index, abort if EOF */
  if( Load_Line(line, wxaptrc, "Device Index") != SUCCESS )
	exit( -1 );
  idx = atoi( line );
  if( (idx < 0) || (idx > 8) )
  {
	fprintf( stderr, "Invalid librtlsdr Device Index\n" );
	exit( -1 );
  }
  rc_data.rtlsdr_dev_index = idx;

  /* Read Frequency Correction Factor, abort if EOF */
  if( Load_Line(line, wxaptrc, "Frequency Correction Factor") != SUCCESS )
	exit( -1 );
  rc_data.rtlsdr_freq_corr = atoi( line );
  if( abs(rc_data.rtlsdr_freq_corr) > 100 )
  {
	fprintf( stderr, "Invalid Frequency Correction Factor\n" );
	exit( -1 );
  }

  /* Read Low Pass Filter Bandwidth, abort if EOF */
  if( Load_Line(line, wxaptrc, "Low Pass Filter Bandwidth") != SUCCESS )
	exit( -1 );
  if( !rc_data.rtlsdr_lpf_bw )
	rc_data.rtlsdr_lpf_bw = atoi( line );
  if( rc_data.rtlsdr_lpf_bw < 100 )
  {
	fprintf( stderr, "Invalid Low Pass Filter Bandwidth\n" );
	exit( -1 );
  }

  /* Read sound card name, abort if EOF */
  if( Load_Line(line, wxaptrc, "Sound Card Name") != SUCCESS )
	exit( -1 );
  Strlcpy( rc_data.pcm_dev, line, sizeof(rc_data.pcm_dev) );

  /* Read ALSA "channel", abort if EOF */
  if( Load_Line(line, wxaptrc, "ALSA Channel" ) != SUCCESS )
	exit( -1 );
  for( idx = 0; idx < num_chan; idx++ )
	if( strcmp(chan_name[idx], line) == 0 )
	  break;
  if( idx == num_chan )
  {
	rc_data.channel = -1;
	fclose( wxaptrc );
	fprintf( stderr, "Invalid ALSA channel name\n" );
	exit( -1 );
  }
  rc_data.channel = chan_number[idx];

  /* Set right or left channel buffer index */
  if( strstr(line, "LEFT") || strstr(line, "MONO") )
	rc_data.use_chn = 0;
  else
	rc_data.use_chn = 1;

  /* Set number of channels */
  if( strstr(line, "MONO") )
	rc_data.num_chn = 1;
  else
	rc_data.num_chn = 2;

  /* Read capture source, abort if EOF */
  if( Load_Line(line, wxaptrc, "Capture Source") != SUCCESS )
	exit( -1 );
  Strlcpy( rc_data.cap_src, line, sizeof(rc_data.cap_src) );

  /* Read Capture volume control, abort if EOF */
  if( Load_Line(line, wxaptrc, "Capture Volume Control") != SUCCESS )
	exit( -1 );
  Strlcpy( rc_data.cap_vol, line, sizeof(rc_data.cap_vol) );

  /* Read Capture volume, abort if EOF */
  if( Load_Line(line, wxaptrc, "Capture Volume") != SUCCESS )
	exit( -1 );
  rc_data.cap_lev = atoi( line );

  /* Read default decode duration, abort if EOF */
  if( Load_Line(line, wxaptrc, "Decoding Duration") != SUCCESS )
	exit( -1 );
  rc_data.default_dur = atoi( line );
  gbl_duration = rc_data.default_dur;

  /* Warn if decoding duration is too long */
  if( gbl_duration > 600 )
  {
	fprintf( stderr,
		"Default decoding duration specified\n"\
		"in wxaptrc (%d sec) seems excessive\n"\
		"Default decoding duration limited to 600 sec\n",
		gbl_duration );
	gbl_duration = 600;
  }

  /*** Do the global config variables ***/
  /* Load pseudo-colorization maps */
  Load_Colormap( wxaptrc, &rc_data.noaa_A_map );
  Load_Colormap(wxaptrc, &rc_data.meteor_map );

  fclose( wxaptrc );

} /* End of Load_Config() */

/*------------------------------------------------------------------*/

/* Load_Colormap()
 *
 * Loads the pseudo-colorization maps from rc file
 */

  void
Load_Colormap( FILE *wxaptrc, color_map_t *map )
{
  /* Buffer for Load_Line  */
  char line[81];
  int idx, match;

  /* Read number of ranges */
  if( Load_Line(line, wxaptrc, "Colormap Number of Ranges") != SUCCESS )
	exit( -1 );
  map->num_ranges = atoi(line);
  if( (map->num_ranges < 0) || (map->num_ranges > 6) )
  {
	fprintf( stderr, "Number of colorization ranges incorrect in wxaptrc\n" );
	exit( -1 );
  }

  /* Read color map */
  for( idx = 0; idx < map->num_ranges; idx++ )
  {
	if( Load_Line(line, wxaptrc, "Color Map") != SUCCESS )
	  exit( -1 );

	match = sscanf( line,
		"%3hhu %3hhu %3hhu %3hhu %3hhu %3hhu %3hhu %3hhu",
		&map->gray_from[idx],  &map->gray_to[idx],
		&map->red_from[idx],   &map->red_to[idx],
		&map->green_from[idx], &map->green_to[idx],
		&map->blue_from[idx],  &map->blue_to[idx] );

	if( match != 8 )
	{
	  perror( "wxapt: sscanf" );
	  fclose( wxaptrc );
	  fprintf( stderr, "Error scanning colorization ranges in wxaptrc\n" );
	  exit( -1 );
	}
  } /* for( idx = 0; idx < map->num_ranges; idx++ ) */

  /* Read white (cloud) threshold */
  if( Load_Line(line, wxaptrc, "White (cloud) Threshold") != SUCCESS )
	exit( -1 );
  map->white_thld = atoi(line);

} /* Load_Colormap() */

/*------------------------------------------------------------------*/

/*  Load_Line()
 *
 *  Loads a line from a file, aborts on failure. Lines beginning
 *  with a '#' are ignored as comments. At the end of file EOF is
 *  returned. Lines assumed maximum 80 characters long.
 */

  int
Load_Line( char *buff, FILE *pfile, char *messg )
{
  int
	num_chr, /* Number of characters read, excluding lf/cr */
	chr;     /* Character read by getc() */
  char error_mesg[MESG_SIZE];

  /* Prepare error message */
  snprintf( error_mesg, MESG_SIZE,
	  "Error reading %s\n"\
	  "Premature EOF (End Of File)", messg );

  /* Clear buffer at start */
  buff[0] = '\0';
  num_chr = 0;

  /* Get next character, return error if chr = EOF */
  if( (chr = fgetc(pfile)) == EOF )
  {
	fprintf( stderr, "wxapt: %s\n", error_mesg );
	fclose( pfile );
	return( EOF );
  }

  /* Ignore commented lines and eol/cr and tab */
  while(
	  (chr == '#') ||
	  (chr == HT ) ||
	  (chr == CR ) ||
	  (chr == LF ) )
  {
	/* Go to the end of line (look for LF or CR) */
	while( (chr != CR) && (chr != LF) )
	  /* Get next character, return error if chr = EOF */
	  if( (chr = fgetc(pfile)) == EOF )
	  {
		fprintf( stderr, "wxapt: %s\n", error_mesg );
		fclose( pfile );
		return( EOF );
	  }

	/* Dump any CR/LF remaining */
	while( (chr == CR) || (chr == LF) )
	  /* Get next character, return error if chr = EOF */
	  if( (chr = fgetc(pfile)) == EOF )
	  {
		fprintf( stderr, "wxapt: %s\n", error_mesg );
		fclose( pfile );
		return( EOF );
	  }

  } /* End of while( (chr == '#') || ... */

  /* Continue reading characters from file till
   * number of characters = 80 or EOF or CR/LF */
  while( num_chr < 80 )
  {
	/* If LF/CR reached before filling buffer, return line */
	if( (chr == LF) || (chr == CR) ) break;

	/* Enter new character to line buffer */
	buff[num_chr++] = (char)chr;

	/* Get next character */
	if( (chr = fgetc(pfile)) == EOF )
	{
	  /* Terminate buffer as a string if chr = EOF */
	  buff[num_chr] = '\0';
	  return( SUCCESS );
	}

	/* Abort if end of line not reached at 80 char. */
	if( (num_chr == 80) && (chr != LF) && (chr != CR) )
	{
	  /* Terminate buffer as a string */
	  buff[num_chr] = '\0';
	  fprintf( stderr,
		  "Error reading %s\n"\
		  "Line longer than 80 characters\n", messg );
	  fprintf( stderr, "wxapt: %s\n%s\n", error_mesg, buff );
	  fclose( pfile );
	  exit( -1 );
	}

  } /* End of while( num_chr < max_chr ) */

  /* Terminate buffer as a string */
  buff[num_chr] = '\0';

  return( SUCCESS );

} /* End of Load_Line() */

/*------------------------------------------------------------------------*/

/*  Start_Stop_Time()
 *
 *  Calculates sleep time and duration of recording or decoding.
 *  If optarg for the -s option is only 5 char, this is taken as
 *  the time to start recording or decoding and the number of
 *  sec for wxapt to sleep is calculated. If optarg is 11 char
 *  long then sleep time and duration of recording or decoding
 *  is also calculated. The expected format is -s<hh:mm[-hh:mm]>
 */

void
Start_Stop_Time( char *optarg,
	unsigned int *start_hrs, unsigned int *start_min,
	unsigned int *sleep_sec, unsigned int *duration )
{
  int
	hrs, min, sec,
	start_sec,  /* Start time in sec since 00:00 hrs */
	time_sec,   /* Time now in sec since 00:00 hrs   */
	optarg_len; /* Length of optarg  */

  /* Used to read real time */
  struct tm time_now;
  time_t t;

  /* Test optarg length (5 or 11 char) */
  optarg_len = (int)strlen( optarg );
  if( !((optarg_len == 5) || (optarg_len == 11)) )
  {
	fprintf( stderr, "Start-Stop time is not valid\n" );
	exit( -1 );
  }

  /* Extract start time and sleep duration. */
  /* (Expected format hh:mm in 24-hour UTC) */
  if( optarg[2] !=':' )
  {
	fprintf( stderr, "Start-Stop time is not valid\n" );
	exit( -1 );
  }

  /* Extract hours and minutes of start time */
  hrs = atoi( optarg );
  if( (hrs < 0) || (hrs > 23) )
  {
	fprintf( stderr, "Start-Stop time is not valid\n" );
	exit( -1 );
  }
  *start_hrs = (unsigned int)hrs;

  min = atoi( &optarg[3] );
  if( (min < 0) || (min > 59) )
  {
	fprintf( stderr, "Start-Stop time is not valid\n" );
	exit( -1 );
  }
  *start_min = (unsigned int)min;

  /* Time now */
  t = time( &t );
  time_now = *gmtime( &t );
  time_sec = time_now.tm_hour * 3600 +
	time_now.tm_min * 60 + time_now.tm_sec;

  start_sec  = hrs * 3600 + min * 60;
  sec = start_sec - time_sec;

  /* If stop time is also specified, extract */
  /* stop time and duration of record/decode */
  if( optarg_len == 11 )
  {
	int
	  stop_hrs,   /* Stop time hours   */
	  stop_min,   /* Stop time minutes */
	  stop_sec;   /* Stop time in sec since 00:00 hrs  */

	/* Check basic format ( Expected is hh:mm-hh:mm ) */
	if( (optarg[5] != '-') || (optarg[8] != ':') )
	{
	  fprintf( stderr, "Start-Stop time is not valid\n" );
	  exit( -1 );
	}

	stop_hrs = atoi( &optarg[6] );
	if( (stop_hrs < 0) || (stop_hrs > 23) )
	{
	  fprintf( stderr, "Start-Stop time is not valid\n" );
	  exit( -1 );
	}

	stop_min = atoi( &optarg[9] );
	if( (stop_min < 0) || (stop_min > 59) )
	{
	  fprintf( stderr, "Start-Stop time is not valid\n" );
	  exit( -1 );
	}

	stop_sec = stop_hrs * 3600 + stop_min * 60;
	stop_sec -= time_sec;

	if( (sec < 0) && (stop_sec > 0) )
	{
	  fprintf( stderr,
		  "wxapt: Start time has elapsed - starting now\n" );
	  *sleep_sec = 0;
	}

	if( stop_sec < 0 )
	  stop_sec += 86400; /* Next day */

	*duration = (unsigned int)(stop_sec - sec);

	if( *duration <= 0 )
	{
	  fprintf( stderr, "Start-Stop time is not valid\n" );
	  exit( -1 );
	}

  } /* if( optarg_len == 11 ) */

  if( sec < 0 )
	sec += 86400; /* Next day */
  *sleep_sec = (unsigned int)sec;

} /* End of Start_Stop_Time() */

/*------------------------------------------------------------------------*/


/*  File_Name()
 *
 *  Prepare a file name, use date and time if null argument
 */

  void
File_Name( char *file_name )
{
  int len; /* String length of file_name */

  /* If file_name is null, use date and time as file name */
  if( strlen(file_name) == 0 )
  {
	/* Variables for reading time (UTC) */
	time_t tp;
	struct tm utc;

	/* Prepare a file name as UTC date-time. */
	/* Default paths are images/ and record/ */
	time( &tp );
	utc = *gmtime( &tp );

	Strlcpy( file_name, rc_data.wxapt_dir, MAX_FILE_NAME );
	len = (int)strlen( file_name );

	if( isFlagSet(ACTION_PROCESS_DSP) )
	  strftime( &file_name[len], 24, "images/%d%b%Y-%H%M", &utc );
	else if( isFlagSet(ACTION_RECORD_APT) )
	  strftime( &file_name[len], 24, "record/%d%b%Y-%H%M", &utc );
  }
  else /* Remove leading spaces from file_name */
  {
	int idx = 0;
	len = (int)strlen( file_name );
	while( (file_name[idx] == ' ') && (idx < len) )
	{
	  file_name++;
	  idx++;
	}
  }

} /* End of File_Name() */

/*------------------------------------------------------------------------*/

/* Fname()
 *
 * Finds file name in a file path
 */

  char *
Fname( char *fpath )
{
  int idx;

  idx = (int)strlen( fpath );

  while( (--idx >= 0) && (fpath[idx] != '/') );

  return( &fpath[++idx] );

} /* Fname() */

/*------------------------------------------------------------------------*/

/*  Usage()
 *
 *  Prints usage information
 */

  void
Usage( void )
{
  fprintf( stderr, "%s\n",
	  "Usage: wxapt [-a<satellite-type>] [-b<LPF-cutoff>] [-f<tuner-frequency>]" );

  fprintf( stderr, "%s\n",
	  "             [-s<start-time[-<stop-time>]>] [-t<operation-duration>]" );

  fprintf( stderr, "%s\n\n",
	  "             [-i<file>] [-p<file>] [-r[<file>]] [-R[<file>]] [-lnohv]" );

  fprintf( stderr, "%s\n\n",
	  "       -a<satellite-type>: 0=NOAA-15 1=NOAA-18 2=NOAA-19 3=METEOR" );

  fprintf( stderr, "%s\n\n",
	  "       -b:<LPF-cutoff>: Specify Low Pass Filter Cutoff in Hz" );

  fprintf( stderr, "%s\n\n",
	  "       -f:<tuner-frequency>: Specify Tuner Center Frequency in Hz" );

  fprintf( stderr, "%s\n\n",
	  "       -s<start-time[-<stop-time>]>: Time (UTC) in format hh:mm at which\n"
	  "       to start and optionally to stop APT recording or processing" );

  fprintf( stderr, "%s\n\n",
	  "       -t<duration>: Duration of APT recording or processing. If duration\n"
	  "       is >= 60, it is taken as seconds, else as minutes" );

  fprintf( stderr, "%s\n\n",
	  "       -i<file>: Specify image file name. If this option is not used,\n"
	  "       use date and time as file name: ddMonyy-hhmm<extension>");

  fprintf( stderr, "%s\n\n",
	  "       -p<file>: Process images from samples recorded in <file>");

  fprintf( stderr, "%s\n\n",
	  "       -r[<file>]: Only record APT signal in <file>. If no file name is\n"
	  "       specified, use date and time as file name: ddMonyy-hhmm.bin");

  fprintf( stderr, "%s\n\n",
	  "       -R[<file>]: Record APT signal in <file> and process images\n"
	  "       If no file is specified, files will be named as above");

  fprintf( stderr, "%s\n\n",
	  "       -l: Start in audio input-level setup mode");

  fprintf( stderr, "%s\n\n",
	  "       -n: Suppress histogram normalization of images");

  fprintf( stderr, "%s\n\n",
	  "       -o: Rotate images by 180 degrees");

  fprintf( stderr, "%s\n\n",
	  "       -h: Print this usage information and exit");

  fprintf( stderr, "%s\n\n",
	  "       -v: Print version number and exit");

} /* End of Usage() */

/*------------------------------------------------------------------------*/

/***  Memory allocation/freeing utils ***/
void mem_alloc( void **ptr, size_t req )
{
  free_ptr( ptr );
  *ptr = malloc( req );
  if( *ptr == NULL )
  {
	perror( "wxapt: A memory allocation request failed" );
	fprintf( stderr, "A memory allocation request failed\n" );
	exit( -1 );
  }
} /* End of void mem_alloc() */

/*------------------------------------------------------------------------*/

void mem_realloc( void **ptr, size_t req )
{
  *ptr = realloc( *ptr, req );
  if( *ptr == NULL )
  {
	perror( "wxapt: A memory allocation request failed" );
	fprintf( stderr, "A memory allocation request failed\n" );
	exit( -1 );
  }
} /* End of void mem_realloc() */

/*------------------------------------------------------------------------*/

void free_ptr( void **ptr )
{
  if( *ptr != NULL )
	free( *ptr );
  *ptr = NULL;

} /* End of void free_ptr() */

/*------------------------------------------------------------------------*/

/* Open_File()
 *
 * Opens a file, aborts on error
 */

  void
Open_File( FILE **fp, char *fname, const char *mode )
{
  /* Open Channel A image file */
  *fp = fopen( fname, mode );
  if( *fp == NULL )
  {
	perror( fname );
	fprintf( stderr, "Failed to open file\n%s", Fname(fname) );
	exit( -1 );
  }

} /* End of Open_File() */

/*------------------------------------------------------------------------*/

/* File_Image()
 *
 * Write an image buffer to file
 */

  void
File_Image(
	FILE *fp, const char *type,
	int width, int height,
	int max_val, unsigned char *buffer)
{
  size_t size;

  /* Write header in Ch-A output PPM files */
  if( fprintf(fp, "%s\n%s\n%d %d\n%d\n",
		type, "# Created by wxapt", width, height, max_val) < 0 )
  {
	perror( "wxapt: Error writing image to file" );
	fprintf( stderr, "Error writing image to file\n" );
	exit( -1 );
  }

  /* P6 type (PPM) files are 3* size in pixels */
  if( strcmp(type, "P6") == 0 )
	size = (size_t)(3 * width * height);
  else
	size = (size_t)(width * height);

  /* Write image buffer to file, abort on error */
  if( fwrite(buffer, 1, size, fp) != size )
  {
	perror( "wxapt: Error writing image to file" );
	fprintf( stderr, "Error writing image to file\n" );
	exit( -1 );
  }

} /* File_Image() */

/*------------------------------------------------------------------------*/

/* Functions for testing and setting/clearing flags */

/* An int variable holding the single-bit flags */
static int Flags = 0;

  int
isFlagSet(int flag)
{
  return (Flags & flag);
}

  int
isFlagClear(int flag)
{
  return (~Flags & flag);
}

  void
SetFlag(int flag)
{
  Flags |= flag;
}

  void
ClearFlag(int flag)
{
  Flags &= ~flag;
}

  void
ToggleFlag(int flag)
{
  Flags ^= flag;
}

/*------------------------------------------------------------------*/

/* Strlcpy()
 *
 * Copies n-1 chars from src string into dest string. Unlike other
 * such library fuctions, this makes sure that the dest string is
 * null terminated by copying only n-1 chars to leave room for the
 * terminating char. n would normally be the sizeof(dest) string but
 * copying will not go beyond the terminating null of src string
 */
  void
Strlcpy( char *dest, const char *src, size_t n )
{
  char ch = src[0];
  int idx = 0;

  /* Leave room for terminating null in dest */
  n--;

  /* Copy till terminating null of src or to n-1 */
  while( (ch != '\0') && (n > 0) )
  {
	dest[idx] = src[idx];
	idx++;
	ch = src[idx];
	n--;
  }

  /* Terminate dest string */
  dest[idx] = '\0';

} /* Strlcpy() */

/*------------------------------------------------------------------*/

/* Strlcat()
 *
 * Concatenates at most n-1 chars from src string into dest string.
 * Unlike other such library fuctions, this makes sure that the dest
 * string is null terminated by copying only n-1 chars to leave room
 * for the terminating char. n would normally be the sizeof(dest)
 * string but copying will not go beyond the terminating null of src

 */
  void
Strlcat( char *dest, const char *src, size_t n )
{
  char ch = dest[0];
  int idd = 0; /* dest index */
  int ids = 0; /* src  index */

  /* Find terminating null of dest */
  while( (n > 0) && (ch != '\0') )
  {
	idd++;
	ch = dest[idd];
	n--; /* Count remaining char's in dest */
  }

  /* Copy n-1 chars to leave room for terminating null */
  n--;
  ch = src[ids];
  while( (n > 0) && (ch != '\0') )
  {
	dest[idd] = src[ids];
	ids++;
	ch = src[ids];
	idd++;
	n--;
  }

  /* Terminate dest string */
  dest[idd] = '\0';

} /* Strlcat() */

/*------------------------------------------------------------------*/

/* Cleanup()
 *
 * Close rtlsdr and sound devices
 */
  void
Cleanup( void )
{
  Close_RTL_Device();
  Close_Capture();
  if( gbl_samples_fp != NULL )
	fclose( gbl_samples_fp );
  if( gbl_snd_buffer != NULL )
	free( gbl_snd_buffer );

} /* Cleanup() */

/*------------------------------------------------------------------*/

