#include "rtlsdr.h"
#include <time.h>
#include "shared.h"

/* rtlsdr device handle */
static rtlsdr_dev_t *dev = NULL;

/* RTL Device Samples Buffer */
static char *rtl_buf = NULL;

/* Index into Samples Buffer */
static int rtl_buf_idx = ASYNC_BUF_LEN;

/*-------------------------------------------------------------------*/

// void SimFM( char *buf );

/* Rtl_Cb()
 *
 * Callback function for rtlsdr_read_async()
 */
  void
Rtl_Cb( unsigned char *buf, uint32_t len, void *ctx )
{
  /* Convert sample values to range of +/- 127 */
  int idx;
  for( idx = 0; idx < len; idx++ )
	rtl_buf[idx] = (char)( buf[idx] - 127 );

//  SimFM( rtl_buf );

  /* Low pass filter I and Q data */
  Butterworth_2Pole_LPF( rtl_buf, len );

  /* Wake up Read_RTL_Buffer() */
  rtl_buf_idx = 0;

} /* Rtl_Cb() */

/*-----------------------------------------------------------------------*/

/* RtlSdr_Read_Async()
 *
 * Pthread function for async reading of RTL I/Q samples
 */
  void *
RtlSdr_Read_Async( void *pid )
{
  rtlsdr_read_async(
	  dev, Rtl_Cb, pid, NUM_ASYNC_BUF, ASYNC_BUF_LEN );

  return( NULL );
} /* RtlSdr_Read_Async() */

/*-----------------------------------------------------------------------*/

/* Rtlsdr_Set_Center_Freq()
 *
 * Sets the Center Frequency of the RTL-SDR Tuner
 */
  void
Rtlsdr_Set_Center_Freq( uint32_t center_freq )
{
  int ret;

  /* Set the Center Frequency of the RTL_SDR Device */
  ret = rtlsdr_set_center_freq( dev, center_freq );
  if( ret != SUCCESS )
  {
	fprintf( stderr, "Failed to set Center Frequency\n" );
	exit( -1 );
  }

  /* Get the Center Frequency of the RTL_SDR Device */
  ret = rtlsdr_get_center_freq( dev );
  if( (ret != center_freq) || (ret == 0) )
  {
	fprintf( stderr, "Failed to set Center Frequency\n" );
	exit( -1 );
  }

} /* Rtlsdr_Set_Center_Freq() */

/*-----------------------------------------------------------------------*/

/* Set_Tuner_Gain_Mode()
 *
 * Sets the Tunaer Gain mode to Auto or Manual
 */
  void
Set_Tuner_Gain_Mode( int mode )
{
  /* Set Tuner Gain Mode */
  int ret = rtlsdr_set_tuner_gain_mode( dev, mode );
  if( ret != SUCCESS )
  {
	fprintf( stderr, "Failed to set Tuner Gain Mode\n" );
	exit( -1 );
  }

} /* Set_Tuner_Gain_Mode() */

/*-----------------------------------------------------------------------*/

/* Rtlsdr_Set_Agc_Mode()
 *
 * Sets the RTL-SDR Digital AGC Mode
 */
  void
Rtlsdr_Set_Agc_Mode( int mode )
{
  /* Set the internal digital AGC mode of RTL2832 */
  int ret = rtlsdr_set_agc_mode( dev, mode );
  if( ret != SUCCESS )
  {
	fprintf( stderr, "Failed to set RTL2832 AGC mode\n" );
	exit( -1 );
  }

} /* Rtlsdr_Set_Agc_Mode() */

/*-----------------------------------------------------------------------*/

/* Rtlsdr_Init()
 *
 * Initialize rtlsdr device status
 */
  void
Rtlsdr_Init( void )
{
  /* rtlsdr device sample rate */
  uint32_t sample_rate;

  /* Check function return values */
  int ret = -1;

  /* Device USB strings */
  char
	manufact[256],
	product[256],
	serial[256],
	mesg[1024];

  const char *tuner_types[] = { TUNER_TYPES };

  /* rtlsdr tuner handle */
  enum rtlsdr_tuner tuner;

  /* Device name */
  const char *dev_name = NULL;

  /* Thread ID for the newly created thread */
  pthread_t pthread_id;


  /* Abort if already init */
  if( isFlagSet(RTLSDR_INIT) ) return;

  /* Get RTL Device Name */
  printf( "Initializing RTLSDR Device\n" );
  printf( "RTLSDR Device Information:\n" );
  dev_name = rtlsdr_get_device_name( rc_data.rtlsdr_dev_index );

  /* Get device USB strings */
  ret = rtlsdr_get_device_usb_strings(
	  rc_data.rtlsdr_dev_index, manufact, product, serial );
  if( ret != SUCCESS )
  {
	fprintf( stderr, "Failed to get device usb strings\n" );
	exit( -1 );
  }

  /* Display device name and USB strings */
  snprintf( mesg, sizeof(mesg),
	  "Device Index: %d\nName: %s\n"\
	  "Manufacturer: %s\nProduct: %s\nSerial: %s\n",
	  rc_data.rtlsdr_dev_index, dev_name, manufact, product, serial );
  printf( "%s\n", mesg );

  /* Open RTL-SDR Device */
  ret = rtlsdr_open( &dev, rc_data.rtlsdr_dev_index);
  if( ret != SUCCESS )
  {
	fprintf( stderr, "Failed to open RTL-SDR device\n" );
	exit( -1 );
  }

  /* Set the Center Frequency of the RTL_SDR Device */
  snprintf( mesg, sizeof(mesg),
	  "Setting Center Frequency to %dHz", rc_data.rtlsdr_center_freq );
  printf( "%s\n", mesg );
  Rtlsdr_Set_Center_Freq( rc_data.rtlsdr_center_freq );

  /* Set the Frequency Correction factor for the device */
  ret = rtlsdr_set_freq_correction( dev, rc_data.rtlsdr_freq_corr );
  if( ret != SUCCESS )
  {
	fprintf( stderr, "Failed to set Frequency Correction factor\n" );
	exit( -1 );
  }

  /* Get the RTL_SDR Tuner type */
  tuner = rtlsdr_get_tuner_type( dev );
  if( tuner == RTLSDR_TUNER_UNKNOWN )
  {
	fprintf( stderr, "Failed to get Tuner type\n" );
	exit( -1 );
  }
  snprintf( mesg, sizeof(mesg), "Tuner Type: %s", tuner_types[tuner] );
  printf( "%s\n", mesg );

  /* Set Tuner Gain Mode to auto */
  ret = rtlsdr_set_tuner_gain_mode( dev, TUNER_GAIN_AUTO );
  if( ret != SUCCESS )
  {
	fprintf( stderr, "Failed to set Tuner Gain Mode\n" );
	exit( -1 );
  }

  /* Enable RTLSDR Digital AGC */
  Rtlsdr_Set_Agc_Mode(RTL_DAGC_ON);

  /* Set RTL Sample Rate */
  ret = rtlsdr_set_sample_rate( dev, RTL_DSP_RATE );
  if( (ret != SUCCESS) || (ret == -EINVAL) )
  {
	fprintf( stderr, "Failed to set ADC Sample Rate\n" );
	exit( -1 );
  }

  /* Get RTL Sample Rate */
  sample_rate = rtlsdr_get_sample_rate( dev );
  if( sample_rate == 0 )
  {
	fprintf( stderr, "Failed to get ADC Sample Rate\n" );
	exit( -1 );
  }
  snprintf( mesg, sizeof(mesg),
	  "ADC Sample Rate: %7d S/s\n", sample_rate );
  printf( mesg, "black" );

  /* Reset RTL data buffer */
  ret = rtlsdr_reset_buffer( dev );
  if( ret != SUCCESS )
  {
	fprintf( stderr, "Failed to Reset sampling Buffer\n" );
	exit( -1 );
  }

  /* Init Butterworth Low Pass Filter */
  Init_Butterworth_2Pole_LPF();

  /* Allocate RTLSDR buffer */
  mem_alloc( (void *)&rtl_buf, (size_t)ASYNC_BUF_LEN );

  /* Create a thread for async read from RTl device */
  ret = pthread_create( &pthread_id, NULL, RtlSdr_Read_Async, NULL );
  if( ret != SUCCESS )
  {
	fprintf( stderr, "Failed to create async read thread\n" );
	exit( -1 );
  }

  SetFlag( RTLSDR_INIT );
  printf( "RTLSDR Device Initialized OK\n" );

} /* Rtlsdr_Init() */

/*-----------------------------------------------------------------------*/

/* Read_RTL_Buffer()
 *
 * Reads the RTLSDR buffer
 */
  void
Read_RTL_Buffer( short *buffer, int buff_size )
{
  int cnt, idx, i, q;

  double
	phi1 = 0.0,	/* Phase angle of sampled IF signal */
	phi2 = 0.0,	/* Phase angle of sampled IF signal */
	dphi = 0.0,	/* Change in Phase angle of sampled IF signal */
	freq = 0.0,	/* Measured frequency of sampled IF signal */
	frqi = 0.0, /* Instantaneous frequency of IF signal */
	i1 = 0.0,	/* In-phase samples in float */
	q1 = 0.0,	/* Quadrature samples in float */
	i2 = 0.0,	/* In-phase samples in float */
	q2 = 0.0;	/* Quadrature samples in float */

  /* This ring buffer saves some rtlsdr I/Q samples so
   * that the estimate of instantaneous frequency is done
   * using I/Q samples spaced FMDET_RINGBUF_LEN apart */
  static char *ring_buf = NULL;
  static int
	ring_buf_idx = 0,
	ringbuf_len2 = (double)FMDET_RINGBUF_LEN / 2.0;


  /* Allocate and clear ring buffer */
  if( ring_buf == NULL )
  {
	mem_alloc( (void *)&ring_buf, FMDET_RINGBUF_LEN );
	bzero( ring_buf, FMDET_RINGBUF_LEN );
  }

  /* Transfer data from RTL buffer to buffer to be processed */
  if( buff_size >= BLOCK_BUFFER_SIZE )
	buff_size -= BLOCK_BUFFER_SIZE;

  /* Fill buffer from RTLSDR device. The 2.4kHz
   * sub-carrier amplitude is proportional to
   * the measured VHF carrier's freq. deviation
   */
  for( idx = 0; idx < buff_size; idx++ )
  {
	/* Use RTLSDR_OVERSAMPLING samples to produce
	 * one subcarrier amplitude sample */
	freq = 0.0;
	for( cnt = 0; cnt < RTLSDR_OVERSAMPLING; cnt++ )
	{
	  /* Wait for buffer to be refilled asynchronously */
	  while( rtl_buf_idx >= ASYNC_BUF_LEN )
		usleep( STANDBY_DELAY );

	  /* Past In-phase and Quadrature signal samples */
	  i  = (int)ring_buf[ring_buf_idx];
	  i1 = (double)i;
	  q  = (int)ring_buf[ring_buf_idx+1];
	  q1 = (double)q;

	  /* Current In-phase and Quadrature signal samples */
	  i  = (int)rtl_buf[rtl_buf_idx];
	  i2 = (double)i;
	  ring_buf[ring_buf_idx] = rtl_buf[rtl_buf_idx];
	  ring_buf_idx++; rtl_buf_idx++;

	  q  = (int)rtl_buf[rtl_buf_idx];
	  q2 = (double)q;
	  ring_buf[ring_buf_idx] = rtl_buf[rtl_buf_idx];
	  ring_buf_idx++; rtl_buf_idx++;

	  if( ring_buf_idx >= FMDET_RINGBUF_LEN )
		ring_buf_idx = 0;

	  /* Phase angle of Past sampled IF signal */
	  phi1 = atan2( q1, i1 );

	  /* Phase angle of Current sampled IF signal */
	  phi2 = atan2( q2, i2 );

	  /* Phase difference from past to current I/Q sample */
	  dphi  = phi2 - phi1;

	  /* Correct for discontinuity at +-180 deg phase angle */
	  if( dphi > M_PI )        dphi -= TWOPI;
	  else if(  dphi < -M_PI ) dphi += TWOPI;

	  /* Measured frequency deviation */
	  dphi /= ringbuf_len2;
	  frqi  = (double)RTL_DSP_RATE * dphi / TWOPI;
	  freq += frqi;

	} /* for( cnt = 0; cnt < RTLSDR_OVERSAMPLING; cnt++ ) */

	/* Fill the buffer used to produce image */
	freq /= (double)RTLSDR_OVERSAMPLING;
	buffer[idx] = (short)( freq * RTL_SCALE_FACTOR );

  } /* for( idx = 0; idx < buff_size; idx++ ) */

} /* Read_RTL_Buffer() */

/*-----------------------------------------------------------------------*/

/* Init_Butterworth_2Pole_LPF()
 *
 * Initializes coefficients of 2-pole Butterworth Low Pass Filter
 */
static double k, a1, a2, b1, b2; /* Coefficients */

  void
Init_Butterworth_2Pole_LPF( void )
{
  double sqrt2 = sqrt( 2.0 );

  /* Find cutoff frequency in [0..PI] */
  double Wd  =
	(TWOPI * (double)rc_data.rtlsdr_lpf_bw) / (double)RTL_DSP_RATE;

  /* Warp cutoff frequency */
  double a  = tan( Wd / 2.0 );
  double as = a * a;


  /* The gain of the filter */
  k = as / ( 1.0 + sqrt2 * a + as );

  /* Filter coefficients */
  a1 = 2.0 * (as - 1) / ( 1.0 + sqrt2 * a + as );
  a2 = ( 1.0 - sqrt2 * a + as ) / ( 1.0 + sqrt2 * a + as );
  b1 = 2.0 * k;
  b2 = 1.0 * k;

} /* Init_Butterworth_2Pole_LPF() */

/*-----------------------------------------------------------------------*/

/* Butterworth_2Pole_LPF()
 *
 * Butterworth 2-pole Low Pass Filter
 */
  void
Butterworth_2Pole_LPF( char *buf, int buf_size )
{
  int idx;

  /* Variables for I low pass filter */
  static double
	xi_0 = 0.0, xi_1 = 0.0, xi_2 = 0.0,
	yi_0 = 0.0, yi_1 = 0.0, yi_2 = 0.0;

  /* Variables for Q low pass filter */
  static double
	xq_0 = 0.0, xq_1 = 0.0, xq_2 = 0.0,
	yq_0 = 0.0, yq_1 = 0.0, yq_2 = 0.0;

  /* Low pass filter I samples */
  for( idx = 0; idx < buf_size; idx += 2 )
  {
	xi_2 = xi_1;
	xi_1 = xi_0;
	xi_0 = (double)buf[idx];

	yi_2 = yi_1;
	yi_1 = yi_0;
	yi_0  =  k * xi_0 + b1 * xi_1 + b2 * xi_2;
	yi_0 -= a1 * yi_1 + a2 * yi_2;

	buf[idx] = (char)yi_0;
  } /* for( idx = 0; idx < buf_size; idx += 2 ) */

  /* Low pass Q samples */
  for( idx = 1; idx < buf_size; idx += 2 )
  {
	/* Low pass filter Q samples */
	xq_2 = xq_1;
	xq_1 = xq_0;
	xq_0 = (double)buf[idx];

	yq_2 = yq_1;
	yq_1 = yq_0;
	yq_0  =  k * xq_0 + b1 * xq_1 + b2 * xq_2;
	yq_0 -= a1 * yq_1 + a2 * yq_2;

	buf[ idx ] = (char)yq_0;

  } /* for( idx = 1; idx < buf_size; idx += 2 ) */

} /* Butterworth_2Pole_LPF() */

/*-----------------------------------------------------------------------*/

/* Close_RTL_Device()
 *
 * Closes thr RTLSDR device, if open
 */
  void
Close_RTL_Device( void )
{
  if( dev != NULL )
  {
	rtlsdr_cancel_async( dev );
	rtlsdr_close( dev );
  }
  if( rtl_buf != NULL )
	free( rtl_buf );
} /* Close_RTL_Device() */

/*-----------------------------------------------------------------------*/

/* Simulates an FM carrier with 2.4 KHz FM modulation
  void
SimFM( char *buf )
{
  double
	fdev = 18000.0,
	mfrq = 2400.0,
	dphi = 0.0,
	dmph = 0.0,
	cfrq = 0.0,
	dtim = 1.0 / (double)RTL_DSP_RATE;

  static double phas = 0.0, mphs = 0.0;

  int scnt, bidx;


  bidx = 0;
  dmph = TWOPI * mfrq * dtim;
  for( scnt = 0; scnt < ASYNC_BUF_LEN/2; scnt++ )
  {
	buf[bidx] = (char)( 100.0 * cos(phas) );
	bidx++;
	buf[bidx] = (char)( 100.0 * sin(phas) );
	bidx++;

	cfrq = fdev * sin( mphs );
	mphs += dmph;
	if( mphs >=  TWOPI ) mphs -= TWOPI;

	dphi = TWOPI * cfrq * dtim;
	phas += dphi;
	if( phas >=  TWOPI ) phas -= TWOPI;
	if( phas <= -TWOPI ) phas += TWOPI;
  }
} */

/*-----------------------------------------------------------------------*/

