/*
 * twHamQTH:  A gui application that sends Morse Code 
 * Copyright (C) 2011->2014 Ted Williams - WA0EIR 
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 *
 * Version: 1.0 Nov 2011
 */


#include "twhamqth.h"

#define LOOKUP 0
#define SAVE   1
#define CLEAR  2
#define REMOVE 3
#define EXPORT 4
#define QRT    5

#define ABOUT  0
#define HELP   1

#if DEBUG == 1
   void printXML (struct allXML *pt);
#endif

#if (MAKE_ICON == 1 && (HAVE_X11_XPM_H == 1) && HAVE_LIBXPM == 1)
   extern Pixmap pixmap;
#endif


/*
 * fileMenu callback - calls the PB's callback
 * pb's rowcolumn parent's userData has the
 * pointer to the cbData
 */
void fileCB (Widget w, XtPointer client_data, XtPointer cbs)
{
   int btn = (long) client_data;
   struct CBdata *cbData;

   /* get the userData (*cbData) from the rowcolumn */
   XtVaGetValues (XtParent (w),
      XmNuserData, (struct CBdata *) &cbData,
      NULL);

   switch (btn)
   {
      case LOOKUP:
         lookupCB (shell, (XtPointer) cbData, NULL);
         break;

      case SAVE:
         saveCB (shell, (XtPointer) cbData, NULL);
         break;

      case CLEAR:
         clearCB (shell, (XtPointer) cbData, NULL);
         break;

      case REMOVE:
         removeCB (w, (XtPointer) cbData, NULL);
         break;

      case EXPORT:
         exportCB (shell, (XtPointer) cbData, NULL);
         break;
   
      case QRT:
         qrtCB (shell, NULL, NULL);
         break;

      default:
         fprintf (stderr, "fileCB: invalid button number\n");
   }
}


/*
 * helpMenu callback
 */
void helpCB (Widget w, XtPointer client_data, XtPointer cbs)
{
   long btn = (long) client_data;

   switch (btn)
   {
      case ABOUT:
         popup_aboutDiag ();
         break;

      case HELP:
         popup_helpDiag ();
         break;
   
      default:
         fprintf (stderr, "helpCB: invalid button number\n");
   }
}


/*
 * LookupCB
 * gets call sign data from Petr
 * client_data is a *CBdata struct
 * lookupTF, mainTX, and NULL
 */
void lookupCB (Widget w, XtPointer client_data, XtPointer cbs)
{
   char *call;
   struct CBdata *wids = (struct CBdata *) client_data;
   int rtn;
   char msg[50];
   char txt[] = " was not found in the database.\n\n";
   char errmsg[70] = "Connection error. Please report to ";
   static Cursor watch;

   if (watch == 0)                   /* create watch cursor the first time */ 
   {
      watch = XCreateFontCursor (XtDisplay (shell), XC_watch);
   }

   XtVaGetValues (wids->wid2,        /* get call from lookupTF */
      XmNvalue, &call,
      NULL);

   /* if no callsign, tell them about it in a dialog box */
   if (strlen (call) == 0)
   {
      errorDiag (wids->wid2, "Please enter a call", wids);
      return;
   }

   /* read the data. login is automatic if sid == NULL */
   /* change cursor to the watch. */
   XDefineCursor (XtDisplay (shell), XtWindow (shell), watch);
   /* got to traverse to see the new cursor, but I don't want to move */
   XmProcessTraversal (wids->wid3, XmTRAVERSE_CURRENT);
   XmProcessTraversal (wids->wid2, XmTRAVERSE_CURRENT);

   rtn = do_request (SEARCH, wids);
   if (rtn < 0)                        /* connection/read problems */
   {
      XUndefineCursor (XtDisplay (shell), XtWindow (shell));
      strcat (errmsg, PACKAGE_BUGREPORT);
      errorDiag (w, errmsg, wids);
      return;
   }

   /* check for any login problem */
   if (rtn == EXPIRED_LOGIN)
   {
      /* appRes user and pw should be good and sid was cleared */
      rtn = do_request (SEARCH, wids);  /* so just try again */
   }

   if (rtn == BAD_LOGIN)
   {
      appRes.password[0] = '\0';       /* pw=NULL - needed by getLogin */
      getLogin(w);

      /* set cursor back to normal */
      XUndefineCursor (XtDisplay (shell), XtWindow (shell));
      return;
   }

   if (rtn == SUCCESS)
   {
      XtVaSetValues (wids->wid2,       /* clear the lookupTF */
         XmNvalue, "",
         NULL);

   #if DEBUG == 1
      fprintf (stderr, "\n\nRESULTS XML\n");
      printXML(&results);
   #endif

      XUndefineCursor (XtDisplay (shell), XtWindow (shell));
      return;
   }
   
   if (rtn == UNKNOWN_CALL)
   {
      strcpy (msg, call);
      strcat (msg, txt);
      XtVaSetValues (wids->wid3,      /* msg to mainTX */
         XmNvalue, msg,
         NULL);

      /* put the call into results so there is something to export */
      strcpy (results.callsign, call);

      /* check for AG if enabled */
      if (appRes.doAG == 1)
      {
         getAGstatus (wids->wid3, call); /* Unknown to hamqth but AG at eQSL? */
      }

      XtVaSetValues (wids->wid2,      /* clear the lookupTF */
         XmNvalue, "",
         NULL);

   }
   else
   {
      errorDiag (w, "Unknown error", wids);
      return;
   }

   /* set cursor back to normal */
   XUndefineCursor (XtDisplay (shell), XtWindow (shell));
}


/*
 * mainCB - force focus to text field
 * client_data is *cbData struct - textfield and mainTX wids
 */
void mainCB (Widget w, XtPointer client_data, XtPointer cbs)
{
   struct CBdata *cbData = (struct CBdata *) client_data;

   XmProcessTraversal (cbData->wid2, XmTRAVERSE_CURRENT);
}


/*
 * saveCB - save the data in a new tab
 * client_data is *CBdata struct 
 * notebook and mainTX, and NULL
 */
void saveCB (Widget w, XtPointer client_data, XtPointer cbs)
{
   int i;
   char buffer[32];
   char *call;                   /* for mainTX */
   struct list *current, *new;
   Widget page, rc1, textWid, tab, removePB ;

   struct CBdata *wids = (struct CBdata *) client_data;

   Widget notebook = wids->wid1; /* notebook */
   Widget lookupTF = wids->wid2; /* lookupTF */
   Widget mainTX = wids->wid3;   /* mainTX */

   /* get text from mainTX */
   XtVaGetValues (mainTX,        /* get the data from the mainTX */
      XmNvalue, &call,
      NULL);

   if (strlen (call) == 0)       /* if no call to save, tell'un and return */
   {
      errorDiag (w, "There is nothing to save", wids);
      return;
   }

   /*
    * we got data, so add a null to the end of the call,
    * copy the callsign from data to buffer, and use as
    * tab label
    */
   i = 0;
   while (call[i] != '\n' && call[i] != ' ')  /* stop copy on nl or sp */
   {
      buffer[i] = call[i];
      /* watch out for stack smashing */
      if (i >= 15)
      {
         fprintf (stderr, "saveCB: buffer overflow - saveCB canceled\n");
         return;
      }
      i++;
   }
   buffer[i] = '\0';
   /* in case call was not found, copy call from tab to xml data */
   strcpy (results.callsign, buffer);

   /* create a new node */ 
   new = (struct list *) XtMalloc (sizeof (struct list));

   /* insert the new node into the list */
   if (head == NULL)               /* if the list is empty */
   {
      head = new;                  /* new is the head */
      head->next = NULL;           /* set the next to NULL/end of list */
   }
   else
   {
      /*
       * insert the new node in alphabetic order
       * Note: 0 calls follow 1 -> 9 calls because of slash zeros.
       * Special case - insert as new head of the list
       */
      if (strcmp (buffer, head->xmlData->callsign) <= 0)
      {
         if (strcmp (buffer, head->xmlData->callsign) == 0)
         {  /* duplicate call so force focus to lookupTF */
            /* free new and return */
            XmProcessTraversal (lookupTF, XmTRAVERSE_CURRENT);
            XtFree ((char *) new);
            return;
         }
         new->next = head;
         head = new;
      }
      else
      {  /* walk the list and find the node to append to */
         current = head;

         while ((current->next != NULL) &&
                (strcmp (current->next->xmlData->callsign, buffer) <= 0))
         {
            if (strcmp (current->next->xmlData->callsign, buffer) == 0)
            {  /* found a duplicate call, so free new and return */
               free (new);
               return;
            }
            current = current->next;
         }
         new->next = current->next;
         current->next = new;
      }
   }
   /* Create the form/page for the new node */
   new->page = XtVaCreateWidget ("page", xmFormWidgetClass, notebook,
      NULL);

   /* Create the tab - buffer is the callsign */
   new->tab = XtVaCreateWidget (buffer, xmPushButtonWidgetClass, notebook,
      /* hack for lessTif */
      XmNnotebookChildType, useMotif ? XmMAJOR_TAB : XmMINOR_TAB,
      NULL);

   /* Create the widgets for the new page */
   removePB = XtVaCreateManagedWidget ("Remove", xmPushButtonWidgetClass,
      new->page,
      XmNheight, 32,
      XmNtopOffset, 15,
      XmNbottomOffset, 15,
      XmNtopAttachment, XmATTACH_FORM,
      XmNbottomAttachment, XmATTACH_NONE,
      XmNrightAttachment, XmATTACH_NONE,
      XmNleftAttachment, XmATTACH_POSITION,
      XmNleftPosition, 40,
      XmNmnemonic, 'R',
      NULL);

   textWid = XtVaCreateWidget ("textWids", xmTextWidgetClass, new->page,
      XmNrows, TEXT_ROWS,
      XmNmarginWidth, 10,
      XmNmarginHeight, 10,
      XmNeditMode, XmMULTI_LINE_EDIT,
      XmNtopAttachment, XmATTACH_WIDGET,
      XmNtopWidget, removePB,
      XmNbottomAttachment, XmATTACH_FORM,
      XmNrightAttachment, XmATTACH_FORM,
      XmNleftAttachment, XmATTACH_FORM,
      XmNvalue, call,              /* mainTX data to new textWid */
      NULL);
   XtFree (call);                  /* free malloc'd space */

   XtAddCallback (removePB, XmNactivateCallback, removeCB,
      (XtPointer) wids);

   XtManageChild (textWid);
   XtManageChild (new->page);
   XtManageChild (new->tab);

   /* do a deep copy of last lookup - results to new->xmlData */
   new->xmlData = (struct allXML *) malloc (sizeof (struct allXML));
   deepcpy (new->xmlData, &results);

   /* clear the main tab text */
   XtVaSetValues (mainTX,
      XmNvalue, "",
      NULL);

   /* renumber the nodes */
   renumber ();

   /* force focus to lookupTF */
   XmProcessTraversal (lookupTF, XmTRAVERSE_CURRENT);

#if DEBUG == 1
   fprintf (stderr, "\n\nSAVE->XMLDATA\n");
   printXML (new->xmlData);
   printList();
#endif
}


/*
 * renumber the pages - calls are in order in the linked list
 * but pages need to be renumbered to put the tabs in order
 */
void renumber (void)
{
   int i = 2;
   struct list *current = head;

   while (current != NULL)
   {
      XtVaSetValues (current->tab,
         XmNpageNumber, i,
         NULL);

      XtVaSetValues (current->page,
         XmNpageNumber, i,
         NULL);

         i++;
         current = current->next;
   }
}


/*
 * clearCB - clear the text fields
 * client_data is a *CBdata struct
 */
void clearCB (Widget w, XtPointer client_data, XtPointer cbs)
{
   struct CBdata *cbData = (struct CBdata *) client_data;

   XtVaSetValues (cbData->wid2,     /* lookupTF */
      XmNvalue, "",
      NULL);

   XtVaSetValues (cbData->wid3,     /* main Text Widet */
      XmNvalue, "",
      NULL);

   /* force focus to lookupTF */
   XmProcessTraversal (cbData->wid2, XmTRAVERSE_CURRENT);
}


/*
 * export callback - put the xml data into shared memory.
 * main created the semaphore as locked.  So, the first time here
 * we don't need to lock it. Don't leave here until all xfers
 * are complete and sops are reset to inital values.
 */
void exportCB (Widget w, XtPointer client_data, XtPointer cbs)
{
   int i;
   int pg = 1, cpage;
   char *text;
   struct CBdata *wids = (struct CBdata *) client_data;
   Widget notebook = wids->wid1;
   Widget mainTX = wids->wid3;
   struct list *current; 
   struct allXML *src;   

   struct sembuf sops[] =  {
                 {0, 1, SEM_UNDO},   /* set server has data */
                 {1, 0, SEM_UNDO},   /* wait for count == 0 */
                 {0, -1, 0}          /* set server has no data */
   };

   /* get current page */
   XtVaGetValues (notebook,
      XmNcurrentPageNumber, &cpage,
      NULL);


   if (cpage == 1)   /* if we want to export page 1 */
   {
      /* make sure the main page isn't blank */
      /* if it is, just exit */
      text = XmTextGetString (mainTX);
      if (strlen (text) == 0)
      {
         XtFree (text);
         errorDiag (mainTX, "There is nothing to export", wids);
         return;
      }
      XtFree (text);
      src = &results;
   }
   else    /* walk the list to find the page number to save */
   { 
      current = head;
      do
      {
         XtVaGetValues (current->page,
            XmNpageNumber, &pg,
            NULL);

         if (pg == cpage)
         {
            src = current->xmlData;
            break;
         }
         else
         {
            current = current->next;      /* move to the next one */
         }
      }
      while (current != NULL);
   }

   /* use address in src to copy to xmlData */
   deepcpy (XMLdata, src);
   i = 0;
   while (XMLdata->callsign[i] != '\0')
   {
      if (XMLdata->callsign[i] == '\330')
      {
         XMLdata->callsign[i] = '0';
      }
      i++;
   }

   /* get the semaphore id */
   if ((semid = semget (SEM_KEY, 0, 0)) == -1)
   {
      /* semget failed */
      fprintf (stderr, "exportCB: semget failed\n");
      return;
   }

   /* set the sem to show that data is available */
   if ((semop (semid, &sops[0], 1)) == -1)
   {
      perror ("exportCB: semop error");
      exit (1);
   }

#if DEBUG == 1
   fprintf (stderr, "exportCB - current page = %d\n", pg);
   fprintf (stderr, "EXPORTED DATA\n");
   printXML (XMLdata);
#endif

   /* sleep so the clients can get hooked up */
   sleep (2);

   /* wait for it */
   semop (semid, &sops[1], 1);  /* wait */
  
   semop (semid, &sops[2], 1);
   return;
}


/*
 * removeCB - close a tab and hook the linked list around it
 * client data is a * to the CBdata struct
 */
void removeCB (Widget w, XtPointer client_data, XtPointer cbs)
{
   int pnum;
   //Widget notebook = (Widget) client_data;
   struct CBdata *cbData = (struct CBdata *) client_data;
   struct list *prev, *current, *next;

   current = head;
   XtVaGetValues (current->page,
      XmNpageNumber, &pnum,
      NULL);

   /* loop to find the page num*/
   while (pnum != selectedPageNum)
   {
      prev = current;                /* remember this one for the next loop */
      current = current->next;
      XtVaGetValues (current->page,
         XmNpageNumber, &pnum,
         NULL);
   }

   /* got it. now, where is it? head, middle or end? */
   if (pnum == 2)                    /* it's the first node */
   {
      if (current->next == NULL)     /* its the first and only node */
      {
         head = NULL;                /* so the list is empty */     
      }
      else                           /* it's the first and there are more */
      {
         head = current->next;
         next = current->next;       /* next is the new first node */
      }
   }
   else                              /* it's not the first one, so */
   {
      if (current->next == NULL)     /* if so, it's the last one */
      {
         prev->next = NULL;
      }
      else                           /* in the middle */
      {
         next =current->next;
         prev->next = current->next;
      }
   }

   /* do the housekeeping */
   XtDestroyWidget (current->page);
   XtDestroyWidget (current->tab);

   /* free struct allXML element */
   XtFree ((XtPointer) current);

   XtVaSetValues (cbData->wid1,          /* move to the main tab/page */
      XmNcurrentPageNumber, 1,
      NULL);

   /* renumber the tabs and pages */
   renumber ();

   /* force focus to the lookupTF */
   XmProcessTraversal (cbData->wid2, XmTRAVERSE_CURRENT);
}


/*
 * pageChanged callback - keeps track of the current page number
 * and controls which pushbuttons are in the main menu pulldown
 */
void pageChangeCB (Widget w, XtPointer client_data, XtPointer cbs)
{
   Widget fileMenu = (Widget) client_data;
   XmNotebookCallbackStruct *pt = (XmNotebookCallbackStruct *) cbs;

   selectedPageNum = pt->page_number;

   if (selectedPageNum == 1)
   {
      /* manage Lookup, Save and Clear. Unmanage Remove */
      XtManageChild (XtNameToWidget (fileMenu, LOOKUP_BTN));
      XtManageChild (XtNameToWidget (fileMenu, SAVE_BTN));
      XtManageChild (XtNameToWidget (fileMenu, CLEAR_BTN));
      XtUnmanageChild (XtNameToWidget (fileMenu, REMOVE_BTN));
   }
   else
   {
      /* unmanage Lookup, Save and Clear. manage Remove */
      XtUnmanageChild (XtNameToWidget (fileMenu, LOOKUP_BTN));
      XtUnmanageChild (XtNameToWidget (fileMenu, SAVE_BTN));
      XtUnmanageChild (XtNameToWidget (fileMenu, CLEAR_BTN));
      XtManageChild (XtNameToWidget (fileMenu, REMOVE_BTN));
   }
}


/*
 * exit the program when QRT is pressed
 */
void qrtCB (Widget w, XtPointer client_data, XtPointer cbs)
{
   shmctl (shmid2, IPC_RMID, NULL);
   /* rm all semaphores */
   semctl (semid, 0, IPC_RMID, 0);
   exit (0);
}


/*
 * modify/verify CB for call sign box - all letters to caps
 * and zero to slashed zero.  Also trim all pre/post spaces
 */
void mvCB (Widget w, XtPointer client_data, XtPointer cbs)
{
   int i, j;
   XmTextVerifyCallbackStruct *pt = (XmTextVerifyCallbackStruct *) cbs;

   /* if empty, just return */
   if (pt->text->length == 0)
   {
      /* leave doit true to handle backspace */
      return;
   }

   /* if a single char and alphanumeric or a /, do it and return */
   if (pt->text->length == 1)
   {
      if (isalnum (pt->text->ptr[0]) || pt->text->ptr[0] == '/')
      {
         /* change to uppercase */
         pt->text->ptr[0] = toupper (pt->text->ptr[0]);

         /* change 0 to slash zero */
         if (pt->text->ptr[i] == '0')
         {
            pt->text->ptr[i] = '\330';
         } 
         return;
      }
      else  /* not a valid character */
      {
         pt->doit = False;
         return;
      }
   }

   /* must be more than one char, so handle the cut/past string */
   /* trim white space off the the end of the string */
   for (i=pt->text->length-1; i >= 0; i--)
   {
      if (!isalnum(pt->text->ptr[i]))
      {
         pt->text->ptr[i] = '\0';
      }
      else
      {
         pt->text->length = strlen (pt->text->ptr);  /* fix length */
         break;     /* not a space, so done */
      }
   }

   /* count leading white spaces and non valid chars to trim */
   j = 0;
   for (i=0; i<pt->text->length; i++)
   {
      if (!isalnum(pt->text->ptr[i]))
      {
         j++;
      }
      else
      {
         break;      /* found first valid char */
      }
   }
   
   /* move valid characters to the start of the string */ 
   if (j > 0)        /* if any non valid chars were found */
   {
      for (i=0; j<pt->text->length; i++,j++)
      {                                         
         pt->text->ptr[i] = pt->text->ptr[j];
      }
      pt->text->ptr[i] = '\0';
      pt->text->length = strlen (pt->text->ptr);   /* fix length */
   }

   /* change chars to upper case - if zero, change to slashed zero */
   for (i=0; i<pt->text->length; i++)
   {
      pt->text->ptr[i] = toupper (pt->text->ptr[i]);
      if (pt->text->ptr[i] == '0')
      {
         pt->text->ptr[i] = '\330';
      } 
   }
}


/*
 * okCB - ok callback for login diag
 */
void okCB (Widget w, XtPointer client_data, XtPointer cbs)
{
   struct CBdata *wids = (struct CBdata *) client_data;

   /* pwordCB has set appRes.password so just do... */
   strcpy (appRes.username, XmTextGetString (wids->wid1));
   XtUnmanageChild (XtParent (w));
}


/*
 * pwordCB - modify/verify callback for password
 */
void pwordCB (Widget w, XtPointer client_data, XtPointer cbs)
{
   int i;
   XmTextVerifyCallbackStruct *pt = (XmTextVerifyCallbackStruct *) cbs;

   if (pt->startPos < pt->currInsert)            /* for del key */
   {
      pt->endPos = strlen (appRes.password);     /* reset endPos */
      appRes.password[pt->startPos] = 0;         /* delete pw's last ch */
      return;
   }

   strcat (appRes.password, pt->text->ptr);      /* append text to password */

   for (i=0; i<pt->text->length; i++)
   {
      pt->text->ptr[i] =  '*';                   /* just *'s in the TF */
   }
}


/*
 * initCB focus callback
 * Check for DIRNAME.  If no DIRNAME, popup a dialog and ask
 * if it should be created.  Finally do a logon request.
 */
void initCB (Widget w, XtPointer client_data, XtPointer cbs)
{
   int rtn;
   XmString xs;
   struct stat buf;
   Widget initDiag;
   char dirpath[PATH_LEN];

   char note[] = "You have been logged onto HamQTH.com\n"
                 "via the resource file.\n";
   char mess[] =
      "TwHamQTH needs to create the directory .twHamQTH in your home "
      "directory.\nClick OK to create it or Cancel to exit.\n";

   struct CBdata *wids = (struct CBdata *) client_data;

   /* remove this callback so it is only called once - at start up */
   XtRemoveCallback (w, XmNfocusCallback, initCB, client_data);

   /* do this only if doAG is requested */
   if (appRes.doAG == TRUE)
   {
      /* build string for DIRNAME */
      if (stat (getenv ("HOME"), &buf) == -1)
      {
         fprintf (stderr, "initCB: Can't find your \"$HOME\"\n");
         exit (1);
      }

      /* build the path to DIRNAME */
      strncpy (dirpath, getenv ("HOME"), PATH_LEN -1);  /* $HOME to dirpath */
                                                        /* then add DIRNAME */
      strncat (dirpath, DIRNAME, PATH_LEN - strlen (dirpath) -1);

      /* see if the DIRNAME exists */
      if ((stat (dirpath, &buf)) == 0)
      {
         /* $HOME/DIRNAME exists but is it a dir */
         if (S_ISDIR(buf.st_mode))
         {
            /* gota dir */
            readAG (wids);
         }
         else
         {
            /* Nope */
            fprintf (stderr, "initCB: twlogDir exists, but not a directory\n");
            exit (1);
         }
      }
      else
      {
         /*  DIRNAME does not exist, so popup initDiag and ask to create it */
         xs = XmStringCreateLtoR (mess, XmSTRING_DEFAULT_CHARSET);
         initDiag = XmCreateQuestionDialog (shell, "initDiag", NULL, 0);

         XtVaSetValues (XtParent (initDiag),
            XmNtitle, "CREATE .twHamQTH Directory",
            #if MAKE_ICON && HAVE_X11_XPM_H && HAVE_LIBXPM
            XmNiconPixmap, pixmap,
            #endif
            NULL);

         XtVaSetValues (initDiag,
            XmNdialogStyle, XmDIALOG_PRIMARY_APPLICATION_MODAL,
            XmNmessageString, xs,
            NULL);

         XmStringFree (xs);
         XtUnmanageChild (XmMessageBoxGetChild (initDiag,
                          XmDIALOG_HELP_BUTTON));

         /* add the callbacks */
         XtAddCallback (initDiag, XmNokCallback, initDiagCB,
                       (XtPointer) wids);
         XtAddCallback (initDiag, XmNcancelCallback, initDiagCB,
                       (XtPointer) NULL);
 
         XtManageChild (initDiag);
      }
   }

   /*
    * always try to login using username and password from resource file
    */
   rtn = do_request (LOGIN, wids);
   if (rtn == GOT_SID)
   {
      fprintf (stderr, "%s", note);
      XmTextInsert (wids->wid3, XmTextGetLastPosition (wids->wid3), note);
   }
   else
   {
   /* Login failed. Do nothing. We will ask for the
    * username/password on the first lookup request
    */
   }
}


/*
 * readAG - reads a list of AG callsigns from eQSL
 * called by form1's focus callback
 *
 * monitor file by:
 * wget http://www.eqsl.cc/qslcard/DownloadedFiles/AGMemberList.txt
 */
Boolean readAG (struct CBdata *wids)
{
   int flag = 0, localOK = 0;               /* initialize 0 */
   int i, j, rtn;
   FILE *fpFile = NULL, *fpEQSL = NULL, *fp = NULL;
   char *home, path[PATH_LEN];
   char AGhead[HEAD_LEN] = "\0";
   char EQSLhead[HEAD_LEN] = "\0";
   char *header;
   char info[256];
   char request[100] =
             "wget -t 1 -q -T 10 -O -"
             " http://www.eqsl.cc/qslcard/DownloadedFiles/AGMemberList.txt";
   struct AGcalls *cur, *last;

   /*
    * get the AGfile header from $HOME/.twHamQTHdir/AGcalls.
    * first, build the path to the AGcalls file
    */
   home = getenv ("HOME");
   if (home == NULL)
   {
      fprintf (stderr, "No $HOME environment variable!\n");  /* poor baby */
      exit (-1);      
   }
   strncpy (path, home, PATH_LEN - strlen (path) -1);
   strncat (path, DIRNAME, PATH_LEN - strlen (path) -1);
   strncat (path, AGNAME, PATH_LEN - strlen (path) -1);

   /* open AGcalls file for read */
   fpFile = fopen (path, "r");
   if (fpFile == NULL)
   {
      perror ("readAG error: open AGfile");
      /* can't use AGcalls. */
      /* Leave localOK = 0 */
   }
   else  /* open AGfile worked, now try to read it */
   {
      /* read the header line of AGcalls file to get date */
      if (fgets (AGhead, HEAD_LEN, fpFile) == NULL)
      {
         fprintf (stderr, "readAG error: read AGcalls header failed\n");
         /* read AGcalls failed so try to use eQSL */
         /* leave localOK = 0 */
      }
      else  /* set localOK, check it's age and set flag */
      {
         localOK = LOCAL_OK;            /* we can read local file */
         flag = getNewer (AGhead);      /* set flag if local isn't too old */
      }
   }

   /*
    * if USE_LOCAL is not set, we will need AG list from eQSL.
    * open pipe for AGmembers.txt file from www.eqsl.cc
    */
   if (flag != USE_LOCAL)
   {
      fpEQSL = popen (request, "r");
      if (fpEQSL < 0)
      {
         fprintf (stderr,  "readAG error: popen eQSL failed\n");
         errorDiag (shell, "readAG error: popen eQSL failed", wids);
         /* leave flag alone */
      }
      else
      {
         /* opened OK so read the header line from eqsl. */
         if ((fgets (EQSLhead, HEAD_LEN, fpEQSL)) == NULL)
         {
            fprintf (stderr,  "readAG error: read EQSL header failed\n");
            pclose (fpEQSL);
            /* leave flag alone */
         }
         else
         {
            /* we can read USE_EQSL so set its flag */
            flag = USE_EQSL;
         }
      }
   }

   /* ready to see what is working */
   switch (flag)
   {
      case BOTH_BAD:
         fprintf (stderr, "Both Local and eQSL failed\n");
         strcpy (info, "Both Local and eQSL failed\n");
         XmTextInsert (wids->wid3, XmTextGetLastPosition (wids->wid3), info);
         flag = USE_LOCAL;
         /*
          * No break;  let it fall through and use the old local file 
          *
          * NOTE: this will crash if the local file does not exist
          *       and eQSL failed too.  This might happen right after
          *       an install, but not once the local file is created.
          */

      case USE_LOCAL:
         fprintf (stderr, "Using local file.\n");
         /* copy this string to info */
         strcpy (info, "Using local file.\n");
         header = AGhead;               /* use local header\n */

         /* get rid of the header \n on local file */
         for (i=0; i<strlen (header); i++)
         {
            if (header[i] == '\n')
            {
               header[i] = '\0';
               break;
            }
         }

         fp = fpFile;
         break;
   
      case USE_EQSL:                    /* only gets set when local failed? */
         fprintf (stderr, "Using eQSL.cc\n");
         /* copy this string to info */
         strcpy (info, "Using eQSL.cc\n");
         header = EQSLhead;             /* need to write EQSLhead to local */
         /* get rid of the \r\n on header on eQSL dos format */
         for (i=0; i<strlen (header); i++)
         {
            if (header[i] == '\r' || header[i] == '\n')
            {
               header[i] = '\0';
            }
         }

         fclose (fpFile);                          /* close AGfile and */
         fpFile = fopen (path, "w");               /* reopen for write */
         rtn = fprintf (fpFile, "%s\n", header);   /* write the header */

         if (rtn < 46)
         {
            fprintf (stderr, "header is too short\n");
//TJW do the right thing
         }
         fp = fpEQSL;
         break;

      default:
         fprintf (stderr, "error - invalid flag\n");
         return True;
   }

   /*
    * build linked list of agCalls structs
    */
   headAG = (struct AGcalls *) XtMalloc (sizeof (struct AGcalls));
   cur = headAG;

   i = 0;                           /* counting the calls */
   /* fp now points to the file we want to read */
   while ((fgets (cur->call, EQSL_LEN, fp)) > 0)
   {
      for (j=0; j<strlen(cur->call); j++)
      {                             /* all to uppercase */
         cur->call[j] = toupper (cur->call[j]);
         if (cur->call[j] == '\r' || cur->call[j] == '\n')
         {                          /* convert to null string */
            cur->call[j] = '\0';    /* nothing happens with local file */
//            break;              
         }
      }

      /* if using EQSL, we also need to write it to local file */
      /* this also adds the \n - unix format now */
      if (flag == USE_EQSL)
      {
         fprintf (fpFile, "%s\n", cur->call);
      }
      i++;                          /* just counting the calls */
      cur->next = (struct AGcalls *) XtMalloc (sizeof (struct AGcalls));
      last = cur;                   /* remember the last ptr */
      cur = cur->next;              /* ready for the next read */
   }
   last->next = NULL;               /* terminate the list */
   XtFree ((char *) cur);           /* since we went one too far */

   fflush (fpFile);                 /* flush the results - needed! */
   if (flag == USE_LOCAL)
   {
      fclose (fpFile);              /* close things */
   }
   if (flag == USE_EQSL)
   {
      pclose (fpEQSL);
   }
   
   if (i == 0)
   {
      fprintf (stderr, "No AG calls read from eQSL.cc\n");
      sprintf (info, "No AG calls read from eQSL.cc\n");
   }
   else
   {
      fprintf (stderr, "%s.\nRead %d AG calls.\n\n", header, i);
      snprintf (&info[strlen (info)], sizeof (info),
               "%s.\nRead %d AG calls.\n\n", header, i);
   }

   /* put startup status in the text wid */
   XmTextInsert (wids->wid3, XmTextGetLastPosition (wids->wid3), info);
   return True;
}


/*
 * getNewer - pass string with the header line from the local AGcalls
 * file. Compare with current time.  If AGcalls is <= to n days old,
 * return USE_LOCAL (1).  Otherwise return BOTH_BAD (0).  
 */
int getNewer (char *AGhead)
{
   int i;
   time_t delta1, delta2;
   struct tm tmtime;
   struct tm *tmp;
   const char *fmt = "%d-%b-%Y %R UTC";
   char *pt, *tzpt;

   /*
    * find the first part of the AG date
    * a number for day of month
    */
   for (i=0; i<strlen(AGhead); i++)
   {
      if (isdigit (AGhead[i]) > 0)
      {
         break;
      } 
   }

   tzpt = getenv ("TZ");             /* save current TZ */
   setenv ("TZ", "UTC", 1);          /* breaks if TZ=(null) */

   memset (&tmtime, 0, sizeof (struct tm));
   pt = strptime (&AGhead[i], fmt, &tmtime);

   if (pt == NULL)
   {
      /* can't convert time, return BOTH_BAD */
      printf ("getNewer error: AGhead convert failed\n");
      return BOTH_BAD;
   }
   delta1 = mktime (&tmtime);        /* local file time */
   delta2 = time (NULL);             /* current time */

   setenv ("TZ", tzpt, 1);           /*restore TZ */

   /* see if local file is <= appRes.days old */
   if ((delta2 - delta1) <= (appRes.days * 86400))
   {
      return USE_LOCAL;              /* AGcalls should work */
   }
   else
   {
      return 0;                      /* Still no file good yet */
   }
}


/*
 * initDiagCB - create DIRNAME and AGNAME if they said OK
 */
void initDiagCB (Widget w, XtPointer client_data, XtPointer cbs)
{
   int modeDir = S_IRWXU | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH;
   int modeFile = S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH;

   
   struct CBdata *wids = (struct CBdata *) client_data;
   XmAnyCallbackStruct *pt = (XmAnyCallbackStruct *) cbs;
   char path[PATH_LEN];
   
   strcpy (path, getenv ("HOME"));
   strncat (path, DIRNAME, PATH_LEN - strlen (path) -1);

   switch (pt->reason)
   {
      case XmCR_CANCEL:
         exit (1);       /* Cancel button so quit */
         break;

      case XmCR_OK:
         mkdir (path, modeDir);             /* create DIRNAME directory */
         strncat (path, AGNAME, PATH_LEN - strlen (path) -1);
         creat (path, modeFile);            /* create AGNAME file */
         readAG (wids);                     /* read the calls from eqsl */
         break;

      default:
         fprintf (stderr, "initDiagCB: invalid CB reason\n");
         break;
   }
}


/*
 * error diag
 */
void errorDiag (Widget w, char *emess, struct CBdata *cbData)
{
   Widget errDiag;
   XmString xs;
   struct CBdata *wids = (struct CBdata *) cbData;

   xs = XmStringCreateLocalized (emess);

   errDiag = XmCreateErrorDialog (w, "errDiag", NULL, 0);

   XtVaSetValues (errDiag,
      XmNdialogStyle, XmDIALOG_PRIMARY_APPLICATION_MODAL,
      XmNmessageString, xs,
      NULL);
   XmStringFree (xs);

   XtVaSetValues (XtParent (errDiag),
      XmNtitle, "twHamQTH ERROR",
      NULL);

   XtUnmanageChild (XmMessageBoxGetChild (errDiag, XmDIALOG_HELP_BUTTON));
   XtUnmanageChild (XmMessageBoxGetChild (errDiag, XmDIALOG_CANCEL_BUTTON));
   XtManageChild (errDiag);

   /* force focus to the lookupTB */
   XmProcessTraversal (wids->wid2, XmTRAVERSE_CURRENT);
}


/* debug function */
void printlist (void)
{
   struct list *pt;

   if (head == NULL)
   {
      fprintf (stderr, "List is empty\n");
      return;
   }
   else
   {
      fprintf (stderr, "head=%p\n", head);
   }

   pt = head;

   do
   {
      fprintf (stderr, "pt=%p  page=%p  tab=%p  next=%p\n",
               pt, pt->page, pt->tab, pt->next);
      pt = pt->next;
   } while (pt != NULL);
}


/*
 * deepcpy - does a deep copy of struct allXML from src to dest
 */
void deepcpy (struct allXML *dest, struct allXML *src)
{
   int i = 0;

   strcpy (dest->callsign, src->callsign);
   strcpy (dest->nick, src->nick);
   strcpy (dest->qth, src->qth);
   strcpy (dest->country, src->country);
   strcpy (dest->adif, src->adif);
   strcpy (dest->itu, src->itu);
   strcpy (dest->cq, src->cq);
   strcpy (dest->grid, src->grid);
   strcpy (dest->adr_name, src->adr_name);
   strcpy (dest->adr_street1, src->adr_street1);
   strcpy (dest->adr_street2, src->adr_street2);
   strcpy (dest->adr_street3, src->adr_street3);
   strcpy (dest->adr_city, src->adr_city);
   strcpy (dest->adr_zip, src->adr_zip);
   strcpy (dest->adr_country, src->adr_country);
   strcpy (dest->adr_adif, src->adr_adif);
   strcpy (dest->district, src->district);
   strcpy (dest->us_state, src->us_state);
   strcpy (dest->us_county, src->us_county);
   strcpy (dest->oblast, src->oblast);
   strcpy (dest->dok, src->dok);
   strcpy (dest->iota, src->iota);
   strcpy (dest->qsl_via, src->qsl_via);
   strcpy (dest->lotw, src->lotw);
   strcpy (dest->eqsl, src->eqsl);
   strcpy (dest->qsl, src->qsl);
   strcpy (dest->email, src->email);
   strcpy (dest->jabber, src->jabber);
   strcpy (dest->icq, src->icq);
   strcpy (dest->msn, src->msn);
   strcpy (dest->skype, src->skype);
   strcpy (dest->birth_year, src->birth_year);
   strcpy (dest->lic_year, src->lic_year);
   strcpy (dest->picture, src->picture);
   strcpy (dest->web, src->web);
   strcpy (dest->latitude, src->latitude);
   strcpy (dest->longitude, src->longitude);
   strcpy (dest->continent, src->continent);
   strcpy (dest->utc_offset, src->utc_offset);
   strcpy (dest->facebook, src->facebook);
   strcpy (dest->twitter, src->twitter);
   strcpy (dest->gplus, src->gplus);
   strcpy (dest->youtube, src->youtube);
   strcpy (dest->linkedin, src->linkedin);
   strcpy (dest->flicker, src->flicker);
   strcpy (dest->vimeo, src->vimeo);

/*
 * add new lookup element copies above this
 */

   return;
}


/* print list utility */
void printList (void)
{
   struct list *current = head;
  
   while (current != NULL)
   {
      fprintf (stderr, "%p is %s next %p\n",
               current, current->xmlData->callsign, current->next);
      current = current->next;
   }
   fprintf (stderr, "\n");
}


#if DEBUG == 1
void printXML (struct allXML *pt)
{
   fprintf (stderr, "callsign %s\nnick %s\nqth %s\ncountry %s\n"
      "adif %s\nitu %s\ncq %s\ngrid %s\n"
      "adr_name %s\nadr_street1 %s\nadr_street2 %s\nadr_street3 %s\n"
      "adr_city %s\nadr_zip %s\nadr_country %s\nadr_adif %s\n"
      "district %s\nus_state %s\n"
      "us_county %s\noblast %s\ndok %s\niota %s\n"
      "qsl_via %s\nlotw %s\neqsl %s\nqsl %s\nqsldirect %s\n"
      "email %s\njabber %s\nicq %s\n"
      "msn %s\nskype %s\nbirth_year %s\nlic_year %s\n"
      "picture %s\nweb %s\n"  //web works but not on hamqth.com list
      "latitude %s\nlongitude %s\ncontinent %s\nutc_offset %s\n"
      "facebook %s\ntwitter %s\ngplus %s\nyoutube %s\n"
      "linkedin %s\nflicker %s\nvimeo %s\n", 
      pt->callsign, pt->nick, pt->qth, pt->country,
      pt->adif, pt->itu, pt->cq, pt->grid,
      pt->adr_name, pt->adr_street1, pt->adr_street2, pt->adr_street3,
      pt->adr_city, pt->adr_zip, pt->adr_country, pt->adr_adif,
      pt->district, pt->us_state,
      pt->us_county, pt->oblast, pt->dok, pt->iota,
      pt->qsl_via, pt->lotw, pt->eqsl, pt->qsl, pt->qsldirect,
      pt->email, pt->jabber, pt->icq,
      pt->msn, pt->skype, pt->birth_year, pt->lic_year,
      pt->picture, pt->web, //web???
      pt->latitude, pt->longitude, pt->continent, pt->utc_offset,
      pt->facebook, pt->twitter, pt->gplus, pt->youtube,
      pt->linkedin, pt->flicker, pt->vimeo);
/*
 * add new elements above here
 */

   return;
}
#endif
