/*
 * twHamQTH:  An online callsign lookup program
 * Copyright (C) 2011-2014 Ted Williams - WA0EIR
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have recieved a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge. MA 02139, USA.
 * See the COPYING file in this directory.
 *
 * Versions: 1.0 -  Nov 2011
 */

#include "twhamqth.h"

char sid[50] = "\0";
static int readXML (const char *request, Widget w);
static int processNode (xmlDocPtr doc, xmlNodePtr cur, struct allXML *pt);

/* struct for xml from URL */
struct memStruct
   {
      char *text;
      size_t size;
   };


/*
 * do_request - build the request string and
 * make the request - login or lookup
 */ 
int do_request (int type, struct CBdata *w)
{
   char request[200];

   /* for login */
   char host[]  = "https://www.hamqth.com";
   char login1[] = "/xml.php?u=";
   /* username goes here */
   char login2[] = "&p=";
   /* password goes here */

   /* for lookup */
   char lookup1[]  = "/xml.php?id=";
   /* sid goes here */
   char lookup2[]  = "&callsign=";
   char *call;
   char lookup3[]  = "&prg=twHamQTH";
   int rtn, i = 0;

   strcpy (request, host);                      /* for all requests */

   /*build the rest of the strings for the selected request */
   if (type == LOGIN)
   {
      strcat (request, login1);                 /* for log in */
      strcat (request, appRes.username);
      strcat (request, login2);
      strcat (request, appRes.password);
   }
   else
   {
      if (type == SEARCH)
      {
         if (sid[0] == '\0')                    /* do we need a sid */
         {
            rtn = do_request (LOGIN, w);        /* recursion is cool */
            if (rtn != GOT_SID)                 /* if that login failed */
            {
               sid[0] = '\0';
               return rtn;                      /* return BAD or EXPIRED */
            }
         }

         strcat (request, lookup1);             /* for call lookup */
         strcat (request, sid);
         strcat (request, lookup2);
         XtVaGetValues (w->wid2,
            XmNvalue, &call,
            NULL);

         /* convert slash zero's to zero for request */
         for (i=0; i<strlen (call); i++)
         {
            if (call[i] == '\330')
            {
               call[i] = '0';
            }
         }
         strcat (request, call);
         strcat (request, lookup3);
      }
      else
      {
         fprintf (stderr, "Invalid request type %d\n", type);
         return -3;
      }
   }

   /*
    * Macro to check that the libxml version is the same 
    * one used when we compiled
    */
   LIBXML_TEST_VERSION

   /*
    * call readXML to read the requested data 
    * pass the wid for the main text
    */
   rtn = readXML (request, w->wid3);

   if (rtn == SUCCESS)
   {
      return SUCCESS;
   }

   if (rtn == BAD_LOGIN)
   {
      return BAD_LOGIN;
   }

   /* for EXPIRED_LOGIN, just clear the sid and do the request again */
   if (rtn == EXPIRED_LOGIN)
   {
      sid[0] = '\0';
      return EXPIRED_LOGIN;
   }

   if (rtn < 0)                  /* something evil happened */
   {
      return rtn;
   }

   /*
    * Cleanup the XML library.
    */
   xmlCleanupParser ();

   /*
    * free xml2 allocated memory?
    */
   xmlMemoryDump ();
   return rtn;
}


/*
 * CURL Write Memory Callback 
 */
size_t WriteMemoryCB(void *contents, size_t size, size_t nmemb,
         void * userp)
{
  size_t realsize = size * nmemb;
  struct memStruct *mem = (struct memStruct *) userp;

  mem->text = realloc(mem->text, mem->size + realsize + 1);
  if(mem->text == NULL)
  {
    /* Out of memory. */
    fprintf(stderr, "Not enough memory (realloc returned NULL).\n");
    return 0;
  }

  memcpy (&(mem->text[mem->size]), contents, realsize);

  mem->size += realsize;
  mem->text[mem->size] = 0;
  return realsize;
}


/*
 * readXML: read the requested string into mem structure with libcurl
 * Parse the memory text with libxml
 */
static int readXML (const char *request, Widget w)
{

   int i, stat, rtn;
   char txt[SZ];
   struct memStruct mem;

   CURL *curl_handle;
   CURLcode res;

   xmlDocPtr doc = NULL;    /* doc tree */
   xmlNodePtr cur = NULL;   /* node pointer */
   xmlChar *tag, *value;

   /* clear the results struct */
   memset ((void *)&mem, '\0', sizeof (struct memStruct));

   /* request is URL for login or search */
#if DEBUG == 1
   fprintf (stderr, "request is %s\n", request);
#endif

   /* Initialize a libcurl handle */
   curl_global_init (CURL_GLOBAL_ALL);
   curl_handle = curl_easy_init();

   /* Set request URL and curl_easy options */
   rtn = curl_easy_setopt(curl_handle, CURLOPT_URL, request);
   curl_easy_setopt(curl_handle, CURLOPT_PROTOCOLS, CURLPROTO_HTTPS);
   curl_easy_setopt(curl_handle, CURLOPT_WRITEFUNCTION, WriteMemoryCB);
   curl_easy_setopt(curl_handle, CURLOPT_WRITEDATA, (void *)&mem);

   /* Perform the curl request */
   res = curl_easy_perform (curl_handle);

   if (res != CURLE_OK)
   {
   /* if any setopts above failed, perform will fail here
    * we should check each "easy" call above
    */
      fprintf (stderr, "curl_easy_perform() failed: %s\n",
         curl_easy_strerror (res));
      return;
   }

   /* parse reply from memory to doc */
   doc =xmlParseMemory (mem.text, mem.size);
   if (doc == NULL)
   {
      fprintf (stderr, "Document parse failed.\n");
      return;
   }

   /* get the root element and check for wrong kind or empty document */
   cur = xmlDocGetRootElement (doc);
   if (cur == NULL)
   {
      fprintf (stderr, "Document was empty.\n");
      xmlFreeDoc (doc);
      return;
   }

   /* is the root element "HamQTH"? */
   tag = (xmlChar *)cur->name;
   if (xmlStrcmp (cur->name, (const xmlChar *) "HamQTH"))
   {
      fprintf (stderr, "Wrong type of Document. - %s\n", tag);
      xmlFreeDoc (doc);
      return;
   }
   xmlFree (value);

   /* clear results struct */
   memset ((void *)&results, '\0', sizeof (struct allXML));
   /* process the node  - search or session */
   stat = processNode (doc, cur, &results);

   switch (stat)
   {
      case GOT_SID:
         strcpy (sid, results.sid);            /* save session_id */
         xmlFreeDoc (doc);
         return GOT_SID;

      case GOT_DATA:
         /* valid lookup data in results struct */
         xmlFreeDoc (doc);
         break;

      case BAD_DATA:                         
         xmlFreeDoc (doc);
         break;

      case BAD_LOGIN:
         xmlFreeDoc (doc);
         return BAD_LOGIN;

      case EXPIRED_LOGIN:
         xmlFreeDoc (doc);
         return EXPIRED_LOGIN;

      default:
         return UNKNOWN_CALL;
   }

   /* callsign to uppercase */
   for (i=0; i<strlen (results.callsign); i++)
   {
      results.callsign[i] = toupper (results.callsign[i]);
   }

   /* build the txt string for the mainTX */
    snprintf (txt, SZ, "%s\n%s\n%s\n%s %s %s\n%s\n",
        results.callsign,
        results.adr_name,
        results.adr_street1,
        results.adr_city, 
        strlen (results.us_state) == 2 ? results.us_state : results.district,
        results.adr_zip,
        results.country);

   XtVaSetValues (w,
      XmNvalue, txt,
      NULL);

   /* should we check tha AG status? */
   if (appRes.doAG == 1)
   {
      getAGstatus (w, results.callsign);
   }

   /* do curl housekeeping */
   curl_easy_cleanup(curl_handle);
   curl_global_cleanup();
}


/*
 * processNode:
 * Parse info in nodes
 * Some calls have returned invalid XML.  They contained & instead of &amp.
 */
static int processNode (xmlDocPtr doc, xmlNodePtr cur, struct allXML *pt)
{
   int i;
   char str[128];
   xmlChar *tag = NULL, *value = NULL;

   /* get next level - session or search */
   cur = cur->xmlChildrenNode;
   cur = cur->next;
   tag = (xmlChar *)cur->name;

   #if DEBUG == 1
      fprintf (stderr, "processNode next is: tag = %s\n", tag);
   #endif

   /* for session, get next - session_id and save sid */
   if (xmlStrcmp (tag, "session") == 0)
   {
      #if DEBUG == 1
         fprintf (stderr, "processNode got session tag\n");
      #endif
      /* get next level - session_id or error */
      cur = cur->xmlChildrenNode;
      cur = cur->next;
      tag = (xmlChar *)cur->name;
      value = xmlNodeListGetString (doc, cur->xmlChildrenNode, 1);

      /* if session_id, save value as sid */
      if (xmlStrcmp (tag, "session_id") == 0)
      {
         strcpy (pt->sid, value);
         #if DEBUG == 1
            fprintf (stderr, "processNode got sid: tag = %s value = %s\n",
                     tag, value);
         #endif
         xmlFree (value);
         return GOT_SID;
      }

      /* if error tag, return error type */
      if (xmlStrcmp (tag, "error") == 0)
      {
         if (xmlStrcmp (value, "Wrong user name or password") == 0)
         {
            xmlFree (value);
            return BAD_LOGIN;
         }

         if (xmlStrcmp (value, "Session does not exist or expired") == 0)
         {
            xmlFree (value);
            return EXPIRED_LOGIN;
         }

         if (xmlStrcmp (value, "Callsign not found") == 0)
         {
            xmlFree (value);
            return UNKNOWN_CALL;
         }
      }
   }

   /* it should be a search so, */
   /* read and save values */
   if (xmlStrcmp (tag, "search") != 0)
   {
      fprintf (stderr, "readXml error: Unknown tag\n");
      return BAD_DATA;
   }

   /* get next node  */
   cur = cur->xmlChildrenNode;
   cur = cur->next;

   while (cur != NULL)
   {
      tag = (xmlChar *)cur->name;
      value = xmlNodeListGetString (doc, cur->xmlChildrenNode, 1);

      /* if empty value so skip */
      if (value == NULL)
      {
         xmlFree (value);
         cur = cur->next;
         continue;
      }

      /* got some data, so what is it? */

      /* his callsign */
      if (xmlStrcmp (tag, "callsign") == 0)
      {
         strncpy (pt->callsign,
                  value,
                  sizeof (pt->callsign) - 1);
         #if DEBUG == 1
            printf ("tag = %s value = %s\n", tag, pt->callsign);
         #endif
         /* slash all zeros */
         i = 0;
         while (pt->callsign[i] != '\0')
         {
            if (pt->callsign[i] == '0')
            {
               pt->callsign[i] = '\330';
            }
            i++;
         }
      }

      /* nick - nick name */
      if (strcmp (tag, "nick") == 0)
      {
         strncpy (pt->nick,
                  value,
                  sizeof (pt->nick) - 1);
         #if DEBUG == 1
            printf ("tag = %s value = %s\n", tag, pt->nick);
         #endif
      }

      /* qth - qth name */
      if (strcmp (tag, "qth") == 0)
      {
         strncpy (pt->qth,
                  value,
                  sizeof (pt->qth) - 1);
         #if DEBUG == 1
            printf ("tag = %s value = %s\n", tag, pt->qth);
         #endif
      }

      /* country - country name */
      if (strcmp (tag, "country") == 0)
      {
         strncpy (pt->country,
                  value,
                  sizeof (pt->country) - 1);
         #if DEBUG == 1
            printf ("tag = %s value = %s\n", tag, pt->country);
         #endif
      }

      /* adif - adif id of country related to callsign */
      if (strcmp (tag, "adif") == 0)
      {
         strncpy (pt->adif,
                  value,
                  sizeof (pt->adif) - 1);
         #if DEBUG == 1
            printf ("tag = %s value = %s\n", tag, pt->adif);
         #endif
      }

      /* itu - itu zone */
      if (strcmp (tag, "itu") == 0)
      {
         strncpy (pt->itu,
                  value,
                  sizeof (pt->itu) - 1);
         #if DEBUG == 1
            printf ("tag = %s value = %s\n", tag, pt->itu);
         #endif
      }

      /* CQ - CQ (WAZ) zone */
      if (strcmp (tag, "cq") == 0)
      {
         strncpy (pt->cq,
                  value,
                  sizeof (pt->cq) - 1);
         #if DEBUG == 1
            printf ("tag = %s value = %s\n", tag, pt->cq);
         #endif
      }

      /* grid - grid code */
      if (strcmp (tag, "grid") == 0)
      {
         strncpy (pt->grid,
                  value,
                  sizeof (pt->grid) - 1);
      #if DEBUG == 1
         printf ("tag = %s value = %s\n", tag, pt->grid);
      #endif
      }

      /* adr_name - full name */
      if (strcmp (tag, "adr_name") == 0)
      {
         strncpy (pt->adr_name,
                  value,
                  sizeof (pt->adr_name) - 1);
         #if DEBUG == 1
            printf ("tag = %s value = %s\n", tag, pt->adr_name);
         #endif
      }

      /* adr_street1 address */
      if (strcmp (tag, "adr_street1") == 0)
      {
         strncpy (pt->adr_street1,
                  value,
                  sizeof (pt->adr_street1) - 1);
         #if DEBUG == 1
            printf ("tag = %s value = %s\n", tag, pt->adr_street1);
         #endif
      }

      /* adr_street2 address */
      if (strcmp (tag, "adr_street2") == 0)
      {
         strncpy (pt->adr_street2,
                  value,
                  sizeof (pt->adr_street2) - 1);
         #if DEBUG == 1
            printf ("tag = %s value = %s\n", tag, pt->adr_street2);
         #endif
      }

      /* adr_street3 address */
      if (strcmp (tag, "adr_street3") == 0)
      {
         strncpy (pt->adr_street3,
                  value,
                  sizeof (pt->adr_street3) - 1);
         #if DEBUG == 1
            printf ("tag = %s value = %s\n", tag, pt->adr_street3);
         #endif
      }

      /* adr_city */
      if (strcmp (tag, "adr_city") == 0)
      {
         strncpy (pt->adr_city,
                  value,
                  sizeof (pt->adr_city) - 1);
         #if DEBUG == 1
            printf ("name = %s value = %s\n", tag, pt->adr_city);
         #endif
      }

      /* adr_zip code */
      if (strcmp (tag, "adr_zip") == 0)
      {
         strncpy (pt->adr_zip,
                  value,
                  sizeof (pt->adr_zip) - 1);
         #if DEBUG == 1
            printf ("tag = %s value = %s\n", tag, pt->adr_zip);
         #endif
      }

      /* adr_country */
      if (strcmp (tag, "adr_country") == 0)
      {
         strncpy (pt->adr_country,
                  value,
                  sizeof (pt->adr_country) - 1);
         #if DEBUG == 1
            printf ("tag = %s value = %s\n", tag, pt->adr_country);
         #endif
      }

      /* adr_adif - adif id of address country */
      if (strcmp (tag, "adr_adif") == 0)
      {
         strncpy (pt->adr_adif,
                  value,
                  sizeof (pt->adr_adif) - 1);
         #if DEBUG == 1
            printf ("tag = %s value = %s\n", tag, pt->adr_adif);
         #endif
      }

      /* district */
      if (strcmp (tag, "district") == 0)
      {
         strncpy (pt->district,
                  value,
                  sizeof (pt->district) - 1);
         #if DEBUG == 1
            printf ("tag = %s value = %s\n", tag, pt->district);
         #endif
      }

      /* us_state */
      if (strcmp (tag, "us_state") == 0)
      {
         strncpy (pt->us_state,
                  value,
                  sizeof (pt->us_state) - 1);
         #if DEBUG == 1
            printf ("tag = %s value = %s\n", tag, pt->us_state);
         #endif
      }

      /* us_county */
      if (strcmp (tag, "us_county") == 0)
      {
         strncpy (pt->us_county,
                  value,
                  sizeof (pt->us_county) - 1);
         #if DEBUG == 1
            printf ("tag = %s value = %s\n", tag, pt->us_county);
         #endif
      }

      /* oblast */
      if (strcmp (tag, "oblast") == 0)
      {
         strncpy (pt->oblast,
                  value,
                  sizeof (pt->oblast) - 1);
         #if DEBUG == 1
            printf ("tag = %s value = %s\n", tag, pt->oblast);
         #endif
      }

      /* dok */
      if (strcmp (tag, "dok") == 0)
      {
         strncpy (pt->dok,
                  value,
                  sizeof (pt->dok) - 1);
         #if DEBUG == 1
            printf ("tag = %s value = %s\n", tag, pt->dok);
         #endif
      }

      /* iota - islands on the air */
      if (strcmp (tag, "iota") == 0)
      {
         strncpy (pt->iota,
                  value,
                  sizeof (pt->iota) - 1);
         #if DEBUG == 1
            printf ("tag = %s value = %s\n", tag, pt->iota);
         #endif
      }

      /* qsl_via - QSL information */
      if (strcmp (tag, "qsl_via") == 0)
      {
         strncpy (pt->qsl_via,
                  value,
                  sizeof (pt->qsl_via) - 1);
         #if DEBUG == 1
            printf ("tag = %s value = %s\n", tag, pt->qsl_via);
         #endif
      }

      /* lotw - Y = yes - N - no - ? don't know */
      if (strcmp (tag, "lotw") == 0)
      {
         strncpy (pt->lotw,
                  value,
                  sizeof (pt->lotw) - 1);
         #if DEBUG == 1
            printf ("tag = %s value = %s\n", tag, pt->lotw);
         #endif
      }

      /* eqsl - Y for eqsl, N no eqsl, ? don't know */
      if (strcmp (tag, "eqsl") == 0)
      {
         strncpy (pt->eqsl,
                  value,
                  sizeof (pt->eqsl) - 1);
         #if DEBUG == 1
            printf ("tag = %s value = %s\n", tag, pt->eqsl);
         #endif
      }

      /* qsl - Y for paper, N no paper, ? don't know */
      if (strcmp (tag, "qsl") == 0)
      {
         strncpy (pt->qsl,
                  value,
                  sizeof (pt->qsl) - 1);
         #if DEBUG == 1
            printf ("tag = %s value = %s\n", tag, pt->qsl);
         #endif
      }

      /* email */
      if (strcmp (tag, "email") == 0)
      {
         strncpy (pt->email,
                  value,
                  sizeof (pt->email) - 1);
         #if DEBUG == 1
            printf ("tag = %s value = %s\n", tag, pt->email);
         #endif
      }

      /* jabber */
      if (strcmp (tag, "jabber") == 0)
      {
         strncpy (pt->jabber,
                  value,
                  sizeof (pt->jabber) - 1);
         #if DEBUG == 1
            printf ("tag = %s value = %s\n", tag, pt->jabber);
         #endif
      }

      /* icq */
      if (strcmp (tag, "icq") == 0)
      {
         strncpy (pt->icq,
                  value,
                  sizeof (pt->icq) - 1);
         #if DEBUG == 1
            printf ("tag = %s value = %s\n", tag, pt->icq);
         #endif
      }

      /* msn */
      if (strcmp (tag, "msn") == 0)
      {
         strncpy (pt->msn,
                  value,
                  sizeof (pt->msn) - 1);
         #if DEBUG == 1
            printf ("tag = %s value = %s\n", tag, pt->msn);
         #endif
      }

      /* skype */
      if (strcmp (tag, "skype") == 0)
      {
         strncpy (pt->skype,
                  value,
                  sizeof (pt->skype) - 1);
         #if DEBUG == 1
            printf ("tag = %s value = %s\n", tag, pt->skype);
         #endif
      }

      /* birth_year */
      if (strcmp (tag, "birth_year") == 0)
      {
         strncpy (pt->birth_year,
                  value,
                  sizeof (pt->birth_year) - 1);
         #if DEBUG == 1
            printf ("tag = %s value = %s\n", tag, pt->birth_year);
         #endif
      }

      /* lic_year */
      if (strcmp (tag, "lic_year") == 0)
      {
         strncpy (pt->lic_year,
                  value,
                  sizeof (pt->lic_year) - 1);
         #if DEBUG == 1
            printf ("tag = %s value = %s\n", tag, pt->lic_year);
         #endif
      }

      /* web */
      if (strcmp (tag, "web") == 0)
      {
         strncpy (pt->web,
                  value,
                  sizeof (pt->web) - 1);
         #if DEBUG == 1
            printf ("tag = %s value = %s\n", tag, pt->web);
         #endif
      }

      /* picture */
      if (strcmp (tag, "picture") == 0)
      {
         strncpy (pt->picture,
                  value,
                  sizeof (pt->picture) - 1);
         #if DEBUG == 1
            printf ("tag = %s value = %s\n", tag, pt->picture);
         #endif
      }

      /* latitude */
      if (strcmp (tag, "latitude") == 0)
      {
         strncpy (pt->latitude,
                  value,
                  sizeof (pt->latitude) - 1);
         #if DEBUG == 1
            printf ("tag = %s value = %s\n", tag, pt->latitude);
         #endif
      }

      /* longitude */
      if (strcmp (tag, "longitude") == 0)
      {
         strncpy (pt->longitude,
                  value,
                  sizeof (pt->longitude) - 1);
         #if DEBUG == 1
            printf ("tag = %s value = %s\n", tag, pt->longitude);
         #endif
      }

      /* continent */
      if (strcmp (tag, "continent") == 0)
      {
         strncpy (pt->continent,
                  value,
                  sizeof (pt->continent) - 1);
         #if DEBUG == 1
            printf ("tag = %s value = %s\n", tag, pt->continent);
         #endif
      }

      /* utc_offset */
      if (strcmp (tag, "utc_offset") == 0)
      {
         strncpy (pt->utc_offset,
                  value,
                  sizeof (pt->utc_offset) - 1);
         #if DEBUG == 1
            printf ("tag = %s value = %s\n", tag, pt->utc_offset);
         #endif
      }

      /* facebook */
      if (strcmp (tag, "facebook") == 0)
      {
         strncpy (pt->facebook,
                  value,
                  sizeof (pt->facebook) - 1);
         #if DEBUG == 1
            printf ("tag = %s value = %s\n", tag, pt->facebook);
         #endif
      }

      /* twitter */
      if (strcmp (tag, "twitter") == 0)
      {
         strncpy (pt->twitter,
                  value,
                  sizeof (pt->twitter) - 1);
         #if DEBUG == 1
            printf ("tag = %s value = %s\n", tag, pt->twitter);
         #endif
      }

      /* gplus */
      if (strcmp (tag, "gplus") == 0)
      {
         strncpy (pt->gplus,
                  value,
                  sizeof (pt->gplus) - 1);
         #if DEBUG == 1
            printf ("tag = %s value = %s\n", tag, pt->gplus);
         #endif
      }

      /* youtube */
      if (strcmp (tag, "youtube") == 0)
      {
         strncpy (pt->youtube,
                  value,
                  sizeof (pt->youtube) - 1);
         #if DEBUG == 1
            printf ("tag = %s value = %s\n", tag, pt->youtube);
         #endif
      }

      /* linkedin */
      if (strcmp (tag, "linkedin") == 0)
      {
         strncpy (pt->linkedin,
                  value,
                  sizeof (pt->linkedin) - 1);
         #if DEBUG == 1
            printf ("tag = %s value = %s\n", tag, pt->linkedin);
         #endif
      }

      /* flicker */
      if (strcmp (tag, "flicker") == 0)
      {
         strncpy (pt->flicker,
                  value,
                  sizeof (pt->flicker) - 1);
         #if DEBUG == 1
            printf ("tag = %s value = %s\n", tag, pt->flicker);
         #endif
      }

      /* vimeo */
      if (strcmp (tag, "vimeo") == 0)
      {
         strncpy (pt->vimeo,
                  value,
                  sizeof (pt->vimeo) - 1);
         #if DEBUG == 1
            printf ("tag = %s value = %s\n", tag, pt->vimeo);
         #endif
      }

      /*
       * new fields get added above here
       */
      xmlFree (value);
      cur = cur->next;
   }  /* end while */
   return GOT_DATA;
}


/*
 * get eqsl status
 * passed w, mainTX widget id, and callsign
 */
void getAGstatus (Widget w, char *call)
{
   int   i, j, rtn,  hit = 0;
   struct AGcalls *pt;
   XmTextPosition l;
   char *txt[] = {
                    "\nNot AG\n",
                    "\nAuthenticity Guarenteed\n"
                 };

   /* replace slash zeros in call with zero for AG check */
   for (i=0; i<strlen (call); i++)
   {
      if (call[i] == '\330')
      {
         call[i] = '0';
      }
   }

   /* run throught the linked list of AG calls */
   pt = headAG;
   
   while (pt != NULL)
   {
      /* check for match */
      if (strncmp (pt->call, call, EQSL_LEN) == 0)
      {
         hit = 1;
         break;
      }
      pt = pt->next;
   }

   l = XmTextGetLastPosition (w);
   XmTextInsert (w, l, txt[hit]);
   return;
}
