/*
 *  gridloc: A ncurses application to calculate the Maidenhead Grid
 *  Locator of a given position or vice-versa. Also to calculate
 *  bearing and distance to a remote position.
 *
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */


/*  calculate.c
 *
 *  Functions to calculate a Maidenhead grid locator
 *  given longitude/latitude and vice versa. Also to
 *  calculate great circle bearing and distance to a
 *  second location given a grid locator or lon/lat
 */

#include "gridloc.h"

/*  Position_to_Gridloc()
 *
 *  Function to calculate a grid locator give lon/lat
 */

  void
Position_to_Gridloc( double lon, double lat, char *grid_loc )
{
  double temp;

  /* Adjust longitude and latitude to references for the   */
  /* Maidenhead grid locator system and roundup to 1/2 sec */
  lon += 180.0 + 1.3888888888E-4;
  lat +=  90.0 + 1.3888888888E-4;

  /* Calculate first letter of field */
  temp = lon / 20.0;
  grid_loc[0] = (char)temp + 'A';

  /* Calculate first number of square */
  lon -= floor(temp) * 20.0;
  temp = lon / 2.0;
  grid_loc[2] = (char)temp + '0';

  /* Calculate first letter of sub-square */
  lon -= floor(temp) * 2.0;
  temp = lon * 12.0;
  grid_loc[4] = (char)temp + 'A';

  /* Calculate second letter of field */
  temp = lat / 10.0;
  grid_loc[1] = (char)temp + 'A';

  /* Calculate second number of square */
  lat -= floor(temp) * 10.0;
  temp = lat / 1.0;
  grid_loc[3] = (char)temp + '0';

  /* Calculate second letter of sub-square */
  lat -= floor(temp) * 1.0;
  temp = lat * 24.0;
  grid_loc[5] = (char)temp + 'A';

  grid_loc[6] = '\0';

} /* End of Position_to_Gridloc() */

/*--------------------------------------------------------------------*/

/*  Gridloc_to_Position()
 *
 *  Function to calculate longitude/latitude
 *  given a position's grid locator
 */

  void
Gridloc_to_Position( char *grid_loc, double *lon, double *lat )
{
  *lon  = (grid_loc[0] - 'A') * 20.0;
  *lon += (grid_loc[2] - '0') * 2.0;
  *lon += (grid_loc[4] - 'A') / 12.0;
  *lon -= 180.0;

  *lat  = (grid_loc[1] - 'A') * 10.0;
  *lat += (grid_loc[3] - '0') * 1.0;
  *lat += (grid_loc[5] - 'A') / 24.0;
  *lat -= 90.0;

} /* End of Gridloc_to_Position() */

/*--------------------------------------------------------------------*/

/*  Degrees_to_DMS()
 *
 * Function to convert an angle in decimal degrees to D:M:S
 */

  void
Degrees_to_DMS( double deg, int *ideg, int *imin, int *isec )
{
  double temp;

  deg += 1.0/7200.0; /* Round-up to 0.5 sec */
  *ideg = (int)deg;
  temp = ( deg - (double)*ideg ) * 60.0;
  *imin = (int)temp;
  temp = ( temp - (double)*imin ) * 60.0;
  *isec = (int)(temp);

} /* End of Degrees_to_DMS() */

/*--------------------------------------------------------------------*/

/*  DMS_to_Degrees()
 *
 *  Function to convert an angle in D:M:S to decimal degrees
 */

  void
DMS_to_Degrees( int ideg, int imin, int isec, double *deg )
{
  *deg = (double)ideg + (double)imin/60.0 + (double)isec/3600.0;

} /* End of DMS_to_degrees() */

/*--------------------------------------------------------------------*/

/*  Bearing_Distance()
 *
 *  Function to calculate bearing and
 *  distance of location B from A
 */

void
Bearing_Distance(
	double lon_a, double lat_a, /* Lon/Lat of point A */
	double lon_b, double lat_b, /* Lon/Lat of point B */
	double *bearing, double *distance )/* From A to B */
{
  double
	gc_arc, cos_gc_arc,       /* Great circle arc   A to B */
	cos_bearing, sin_bearing, /* cos/sin of bearing A to B */
	lon_diff;                 /* Difference in longitude of B from A */


  /* Convert to radians */
  lat_a *= deg2rad;
  lon_a *= deg2rad;
  lat_b *= deg2rad;
  lon_b *= deg2rad;

  /* Longitude differnce of B from A */
  lon_diff = lon_b - lon_a;

  /* Calculate great circle distance A to B */
  cos_gc_arc = cos(lon_diff)*cos(lat_a)*cos(lat_b) + sin(lat_a)*sin(lat_b);
  gc_arc = acos( cos_gc_arc );

  /* Distance in km */
  *distance = eradius * gc_arc;

  /* Calculate bearing A to B */
  cos_bearing  = sin(lat_b) - sin(lat_a) * cos_gc_arc;
  sin_bearing  = sin(lon_diff) * cos(lat_a) * cos(lat_b);
  *bearing = atan2(sin_bearing, cos_bearing);

  /* Correct negative (anticlockwise) bearings */
  if( *bearing < 0.0 )
	*bearing = twopi + *bearing;

  /* Convert to degrees */
  *bearing /= deg2rad;

} /* End of Bearing_Distance() */

/*--------------------------------------------------------------------*/
