/***************************************************************************
 *   Copyright (C) 2005 by Thierry CHARLES   *
 *   thierry@les-charles.net   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#ifndef _FILE_PARSER_H_
#define _FILE_PARSER_H_

#include <wx/thread.h>
#include "lib/commons.h"



class FileInfo;

/**
 * parser g�n�raliste
 */
class FileParser
{
    public:
        FileParser() : state(Unknown), bStopRequested(false) {}
        virtual ~FileParser(){}
        /** les differents �tats du parser au cours de sa vie */
        enum ParserState { Prepared, Running, Paused, Finished, Unknown };

        /**
         * pr�pare le parser pour scanner un fichier dont on passe les infos actuelles en param�tre
         * @param info informations actuellement connues sur le fichier (surtout le nom, l'id et le type)
         * @return true si le parser peut g�rer le fichier demand� et que le parser n'est pas actuellement en fonction,
         *          false dans le cas contraire
         * @see FileParser::prepareParserImpl()
         */
        bool prepareParser(const FileInfo * info);

        /**
         * d�marre le parsing si le parser est en �tat Prepared. Rend la main lorsque le parsing est termin�.
         * @return false si le parser �choue ou si il n'�tait pas en �tat Prepared
         * @see FileParser::parse()
         */
        bool startParsing();
//        /**
//         * demande au parser de se mettre en pause (lib�ration du processeur ?)
//         * et rend la main lorsque le changement d'�tat est effectif.
//         * @return true si le parser a pu passer a l'�tat Paused (l'�tait d�j� ou l'est devenu)
//          * @see FileParser::checkPauseRequested()
//          */
//          bool pauseParsing();
//        /**
//          * reprend le parsing la ou il en �tait lors du dernier appel � pauseParsing().
//          * @return true si le parser a pu passer � l'�tat Running (l'�tait d�j� ou l'est devenu)
//          * @see FileParser::checkPauseRequested()
//          */
//         bool continueParsing();
        /**
         * demande au parser de s'arreter. Rend la main lorsque le parser s'est arret�
         * @see FileParser::checkStopRequested()
         */
        void stopParsing();

        /** renvoie l'�tat actuel du parser */
        ParserState getState();

        /**
         * r�cup�re les infos collect�es lors du dernier parsing (uniquement en �tat Finished)
         * @return NULL si il n'y a pas eu de parsing depuis la cr�ation de l'instance ou la derni�re pr�paration
         * @see FileParser::getParsedInfoImpl()
         */
        FileInfo * getParsedInfo();

    protected:
        /**
         * pr�pare le parser pour scanner un fichier dont on passe les infos actuelles en param�tre
         * @param info informations actuellement connues sur le fichier (surtout le nom, l'id et le type)
         * @return true si le parser peut g�rer le fichier demand� et que le parser n'est pas actuellement en fonction,
         *          false dans le cas contraire
         */
        virtual bool prepareParserImpl(const FileInfo * info) = 0;
        /**
         * effectue le parsing du fichier pr�par�. Cette m�thode est apell�e lors du passage dans startParsing
         * uniquement si le parser est en �tat Prepared.
         * @return true si le parsing s'est d�roul� correctement
         */
        virtual bool parse() = 0;
        /**
         * r�cup�re les infos collect�es lors du dernier parsing (uniquement en �tat Finished).
         * Cette m�thode est apell�e par getParsedInfo uniquement si le parser est en �tat Finished.
         * @return NULL si il n'y a pas eu de parsing depuis la cr�ation de l'instance ou la derni�re pr�paration
         */
        virtual FileInfo * getParsedInfoImpl() = 0;

        /**
         * v�rifie si une pause est demand�e.
         * Si c'est le cas, immobilise le thread jusqu'au prochain appel � continueParsing.
         * A utiliser r�guli�rement dans l'impl�mentation de la m�thode parse()
         * @return true si une pause a �t� faite
         */
        bool checkPauseRequested();

        /**
         * v�rifie si un arret d�finitif du parsing a �t� demand�.
         * A utiliser r�guli�rement dans l'impl�mentation de la m�thode parse()
         * @return true si un arret a �t� demand�
         */
        bool checkStopRequested();

    private:
        ParserState state;
        wxCriticalSection csState;
        wxCriticalSection csParsing;
        bool bStopRequested;
};

#endif // _FILE_PARSER_H_
