/***************************************************************************
 *   Copyright (C) 2005 by Thierry CHARLES   *
 *   thierry@les-charles.net   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "fileparser.h"

#include <stdexcept>
#include "lib/lib_logging.h"

/**
 * prpare le parser pour scanner un fichier dont on passe les infos actuelles en paramtre
 * @param info informations actuellement connues sur le fichier (surtout le nom, l'id et le type)
 * @return true si le parser peut grer le fichier demand et que le parser n'est pas actuellement en fonction,
 *          false dans le cas contraire
 * @see FileParser::prepareParserImpl()
 */
bool FileParser::prepareParser(const FileInfo * info)
{
    bool bRetVal = false;
    wxCriticalSectionLocker lock(this->csState);
    try
    {
        if(this->state == Prepared || this->state == Finished || this->state == Unknown)
            bRetVal = this->prepareParserImpl(info);
    }
    catch( std::exception &e )
    {
        LOG_MESSAGE(e.what(),logging::_ERROR_);
        bRetVal = false;
    }
    catch( ... )
    {
        LOG_MESSAGE("Exception occured while preparing parser",logging::_ERROR_);
        bRetVal = false;
    }
    this->state = (bRetVal) ? Prepared : Unknown;
    return bRetVal;
}

/**
 * dmarre le parsing si le parser est en tat Prepared. Rend la main lorsque le parsing est termin.
 * @return false si le parser choue ou si il n'tait pas en tat Prepared
 * @see FileParser::parse()
 */
bool FileParser::startParsing()
{
    wxCriticalSectionLocker lock(this->csState);
    try
    {
        if(this->state != Prepared)
            return false;

        this->state = Running;
        wxCriticalSectionLocker parsingLock(this->csParsing); // on indique que le parsing est en cours
        this->csState.Leave();
        bool bRetVal = this->parse(); // on effectue le parsing proprement dit
        this->csState.Enter();
        if(!this->checkStopRequested()) // on dtermine le nouvel tat en fonction du type d'arret
        {
            state = bRetVal ? Finished : Unknown;
        }
        else
            state = Unknown;
        return bRetVal;
    }
    catch( std::exception &e )
    {
        LOG_MESSAGE(e.what(),logging::_ERROR_);
        return false;
    }
    catch( ... )
    {
        LOG_MESSAGE("Exception occured while parsing",logging::_ERROR_);
        return false;
    }
}

/**
 * demande au parser de s'arreter. Rend la main lorsque le parser s'est arret
 * @see FileParser::checkStopRequested()
 */
void FileParser::stopParsing()
{
    try
    {
        wxCriticalSectionLocker lock(this->csState);

        if(this->state != Running)
            return; // le parser n'est pas en route : on quitte direct

        this->bStopRequested = true; // on installe la demande d'arret
        this->csState.Leave();

        wxCriticalSectionLocker parsingLock(this->csParsing); // on attend la fin du parsing
    }
    catch( std::exception &e )
    {
        LOG_MESSAGE(e.what(),logging::_ERROR_);
    }
    catch( ... )
    {
        LOG_MESSAGE("Exception occured while stoping parser",logging::_ERROR_);
    }
}

/** renvoie l'tat actuel du parser */
FileParser::ParserState FileParser::getState()
{
    try
    {
        wxCriticalSectionLocker lock(this->csState);
        return this->state;
    }
    catch( std::exception &e )
    {
        LOG_MESSAGE(e.what(),logging::_ERROR_);
        return Unknown;
    }
    catch( ... )
    {
        LOG_MESSAGE("Exception occured while getting parser state",logging::_ERROR_);
        return Unknown;
    }
}

/**
 * rcupre les infos collectes lors du dernier parsing (uniquement en tat Finished)
 * @return NULL si il n'y a pas eu de parsing depuis la cration de l'instance ou la dernire prparation
 * @see FileParser::getParsedInfoImpl()
 */
FileInfo * FileParser::getParsedInfo()
{
    try
    {
        wxCriticalSectionLocker lock(this->csState);
        if(this->state != Finished)
            return NULL;
        return this->getParsedInfoImpl();
    }
    catch( std::exception &e )
    {
        LOG_MESSAGE(e.what(),logging::_ERROR_);
        return NULL;
    }
    catch( ... )
    {
        LOG_MESSAGE("Exception occured while getting parsed infos",logging::_ERROR_);
        return NULL;
    }
}

/**
 * vrifie si un arret dfinitif du parsing a t demand.
 * A utiliser rgulirement dans l'implmentation de la mthode parse()
 * @return true si un arret a t demand
 */
bool FileParser::checkStopRequested()
{
    if((!wxThread::IsMain()) && wxThread::This()->TestDestroy())
    {
        this->bStopRequested = true;
    }
    return this->bStopRequested;
}

/**
 * vrifie si une pause est demande.
 * Si c'est le cas, immobilise le thread jusqu'au prochain appel  continueParsing.
 * A utiliser rgulirement dans l'implmentation de la mthode parse()
 * @return true si une pause a t faite
 */
bool FileParser::checkPauseRequested()
{
    if((!wxThread::IsMain()) && wxThread::This()->TestDestroy())
    {
        this->bStopRequested = true;
    }
    return false;
}
