/***************************************************************************
 *   Copyright (C) 2005 by Thierry CHARLES   *
 *   thierry@les-charles.net   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#ifndef _TAPPLICATION_H_
#define _TAPPLICATION_H_

/**
 * les 32768 premiers sont rservs par MS Windows
 * et les 1024 suivants sont rservs pour les menus statiques de l'application
 */
#define MENU_ID_STATIC_FIRST 32769

#include <wx/wx.h>
#include <wx/string.h>
#include <wx/mimetype.h>

#include <map>
#include <set>

#include "lib/commons.h"
#include "components/framework/xmlcfgcmp.h"
#include "components/framework/translation.h"

class TMainWindow;
class TApplicationWindow;
class TApplicationPanel;
class TPanelElement;
class TRessourceListener;
class TBitmap;

/**
 * classe de base des ressources d'application
 * cette classe ne fait rien mais permet la deletion via l'operateur "delete"
 */
class TApplicationRessource
{
    public:
        TApplicationRessource() {}
        virtual ~TApplicationRessource() {};
        /** decharge la ressource */
        virtual void prepareRessourceUnload();
        /** envoie un evenement */
        virtual void fireRessourceEvent(const int iEvent);
        /** ajoute un ecouteur a l'editeur */
        virtual bool addRessourceListener(TRessourceListener * l);
        /** enleve un ecouteur a l'editeur */
        virtual bool removeRessourceListener(TRessourceListener * l);

    private:
        typedef std::set<TRessourceListener *> TRessourceListenerList;
        /** liste des ecouteurs */
        TRessourceListenerList listeners;
};

/**
 * interface des ecouteurs de ressources
 */
class TRessourceListener
{
    public:
        virtual ~TRessourceListener(){}
        virtual void ressourceUnloading(TApplicationRessource * res){}
        /** indique qu'un evenement s'est declanche sur la ressource */
        virtual void ressourceEvent(TApplicationRessource * res, const int iEvent) = 0;
};

/**
 * @short Une application
 * @author Thierry CHARLES <thierry@les-charles.net>
 * @version 0.1
 */
class TApplication : public wxApp, public XmlConfiguredComponent, virtual public TranslatableCmp
{
    public:
        TApplication(wxString _sName) : wxApp(), sName(_sName), mainWindow(NULL), bMainWindowVisible(true), bInClosingPhase(false) {}
        virtual ~TApplication();

        TMainWindow * getMainWindow();

        /** d�fini le nom de l'application */
        void setName( const wxString & sName ) { this->sName = sName; }
        /** renvoie le nom de l'application */
        wxString getName() const { return this->sName; }
        /** renvoie le nom de l'application */
        const wxString & getName() { return this->sName; }

        /** d�fini si la fen�tre principale est visible ou non */
        void setMainWindowVisible( bool b );
        /** indique si la fen�tre principale est visible ou non */
        bool isMainWindowVisible() const { return this->bMainWindowVisible; }

        /** lis les parametres et les propage aux composants */
        virtual bool loadParameters();

        /** ecris les parametres de toute l'appli sur disque dans un rep nomme ~/.ApplName/config.xml */
        virtual void saveParameters();

        /** renvoie un pointeur sur une fenetre a partir de son ID */
        virtual TApplicationWindow * getWindowById(const wxString & sId) const;

        /** enregistre une fenetre dans l'application pour pouvoir la reservir via le getWindowById */
        virtual bool registerWindow(TApplicationWindow * window);

        /** supprime une r�f�rence a une fenetre */
        bool unregisterWindow(const wxString & sId);

        /** renvoie un pointeur sur une ressource publi�e de mani�re globale a l'applilcation */
        virtual TApplicationRessource * getRessource(int iResId) const;

        /**
         * publie une ressource dans l'application. Une ressource publi�e ne
         * doit plus etre delet�e avant la fin de l'application.
         * @returns true en cas de succes, false autrement (typiquement, l'Id existe d�j�)
         */
        virtual bool publishRessource(int iResId, TApplicationRessource * res);

        /**
         * cr�e une instance de l'�l�ment dont le type est pass� en parametre et l'ajoute au panneau parent. Le type doit correspondre au type renvoy� par TPanelElement::getType
         */
        virtual TPanelElement * builElement(const char * szType, TApplicationPanel * _owner, int iID) = 0;

        /** indique si le composant doit etre sauvegard� */
        virtual bool mightBeSaved() const { return true; }

        /** traduit un texte a partir des informations de traduction disponnibles */
        wxString translate(const wxString & sContext, const wxString & sToTranslate);

        /** defini le fichier de traduction a utiliser */
        void setTranslationFile(const wxString & sFile);

        /** renvoie le fichier de traduction utilise */
        const wxString & getTranslationFile() const { return this->sTranslationFile; }

        /** ajoute un bitmap a la liste des bitmaps permanents */
        bool addBitmap(int iBitmapID, TBitmap * bmp);

        /** renvoie un pointeur vers un bitmap permanent */
        const TBitmap * getBitmap(int iBitmapID) const;

        /** indique que l'application va se fermer */
        void setInClosingPhase(bool b) { this->bInClosingPhase = b; }
        /** indique si l'application va se fermer */
        bool isInClosingPhase() const { return this->bInClosingPhase; }

        /** renvoie le getionnaire de types mimes */
        wxMimeTypesManager & getMimeTypesManager() { return this->mimeTypesManager; }
        /** renvoie le getionnaire de types mimes */
        const wxMimeTypesManager & getMimeTypesManager() const { return this->mimeTypesManager; }

        /** renvoie le chemin a utiliser pour sauvegarder la config de l'appli, termine par un separateur */
        virtual wxString getApplConfigPath() const;
        
        /** renvoie le chemin a utiliser pour sauvegarder les donnes de l'appli, termine par un separateur */
        virtual wxString getApplDataPath() const;
        
        /** renvoie le chemin a utiliser pour sauvegarder le cache de l'appli, termine par un separateur */
        virtual wxString getApplCachePath() const;
        
        /** renvoie le chemin a utiliser pour sauvegarder les parametres de l'appli termine par un separateur */
        virtual wxString getApplPath() const;

        /** decharge les ressources (ne les supprime pas) */
        virtual void unloadAllRessources();

        /** dclare que l'init de l'application est termin */
        virtual void applicationStarted(){}

        /** declare que l'init de l'application est termine en erreur */
        virtual void applicationStartupFailed(){}

        /** gestion des exceptions inattendues */
        virtual void OnFatalException();

        /** renvoie un identifiant unique de menu */
        virtual int getUniqueMenuId();

#ifdef __WXMSW__
        /** mthode destine a forcer la fermeture de l'appli dans le cas de windows qui plante autrement */
        int forceExit() { return this->OnExit(); }
#endif

    protected:
        /**
         * Effectue les traitements d'init de l'application
         * @return true to continue processing, false to exit the application immediately.
         */
        virtual bool start() = 0;

        /**
         * Effectue les traitements de fermeture de l'application. N'est pas appel� si "bool start()" renvoie false
         */
        virtual void stop(){};

        /** construit la fenetre principale et en renvoie le pointeur */
        virtual TMainWindow * buildMainWindow() = 0;

        /** charge le fichier de traductions */
        void loadTranslations();

        /** ecris un fichier contenant les traductions n'ayant pas pu etre satisfaites */
        void dumpUnsatisfiedTranslationsQueries() const;

    private:
        /**
         * ne pas surcharger : surcharger "bool start()" � la place.
         *
         * This must be provided by the application, and will usually create the application's main window, optionally calling wxApp::SetTopWindow. You may use OnExit to clean up anything initialized here, provided that the function returns true.
         * Notice that if you want to to use the command line processing provided by wxWidgets you have to call the base class version in the derived class OnInit().
         * Return true to continue processing, false to exit the application immediately.
         */
        bool OnInit();

        /**
         * ne pas surcharger : surcharger "void stop()" � la place.
         * Override this member function for any processing which needs to be done as the application is about to exit. OnExit is called after destroying all application windows and controls, but before wxWidgets cleanup. Note that it is not called at all if OnInit failed.
         * The return value of this function is currently ignored, return the same value as returned by the base class method if you override it.
         */
        int OnExit();



        wxString                sName;
        TMainWindow *           mainWindow;
        bool                    bMainWindowVisible;
        bool                    bInClosingPhase;

        typedef std::map<wxString,TApplicationWindow *> TIdWindowMap;
        TIdWindowMap windowsMap;

        typedef std::map<int,TApplicationRessource *> TRessourcesMap;
        TRessourcesMap resMap;

        wxString                sTranslationFile;
        typedef std::map<wxString,wxString> TTranslationMap;
        typedef std::map<wxString,TTranslationMap> TTranslationContextMap;
        TTranslationContextMap translations;

        typedef std::set<wxString> TStringsSet;
        typedef std::map<wxString,TStringsSet> TTranslationQueries;
        TTranslationQueries unsatisfiedTranslationQueries;

        typedef std::map<int,TBitmap *> TBitmapsMap;
        TBitmapsMap bmpMap;

        wxMimeTypesManager mimeTypesManager;
};

TApplication * CurrentApplication();

#define GET_BMP(i) CurrentApplication()->getBitmap(i)

#endif // _TAPPLICATION_H_
