/*                    G E T _ V E R T E X . C
 * BRL-CAD
 *
 * Copyright (c) 1993-2021 United States Government as represented by
 * the U.S. Army Research Laboratory.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this file; see the file named COPYING for more
 * information.
 */
/** @file iges/get_vertex.c
 *
 */

#include "./iges_struct.h"
#include "./iges_extern.h"

struct vertex **
Get_vertex(struct iges_edge_use *edge_use)
{
    struct iges_edge_list *e_list;
    struct iges_vertex_list *v_list;
    int edge_index;
    int vert_index;
    int vert_de;

    if ((e_list = Get_edge_list(edge_use)) == NULL)
	return (struct vertex **)NULL;

    edge_index = edge_use->index-1;
    if (edge_use->orient) {
	vert_de = e_list->i_edge[edge_index].start_vert_de;
	vert_index = e_list->i_edge[edge_index].start_vert_index - 1;
    } else {
	vert_de = e_list->i_edge[edge_index].end_vert_de;
	vert_index = e_list->i_edge[edge_index].end_vert_index - 1;
    }

    if ((v_list = Get_vertex_list(vert_de)) == NULL)
	return (struct vertex **)NULL;

    return &v_list->i_verts[vert_index].v;
}


int
Put_vertex(struct vertex *v, struct iges_edge_use *edge)
{
    struct iges_edge_list *e_list;
    struct iges_edge_list *el;
    struct iges_vertex_list *v_list;
    int vert_index;
    int vert_de;

    if ((e_list = Get_edge_list(edge)) == NULL)
	return 0;


    el = e_list;
    while (el && el->edge_de != edge->edge_de)
	el = el->next;

    if (!el) {
	bu_exit(1, "Cannot find an edge list with edge_de = %d\n", edge->edge_de);
    }

    if (edge->orient) {
	vert_de = el->i_edge[edge->index-1].start_vert_de;
	vert_index = el->i_edge[edge->index-1].start_vert_index-1;
    } else {
	vert_de = el->i_edge[edge->index-1].end_vert_de;
	vert_index = el->i_edge[edge->index-1].end_vert_index-1;
    }


    if ((v_list = Get_vertex_list(vert_de)) == NULL)
	return 0;

    if (v_list->i_verts[vert_index].v) {
	bu_log("vertex already assigned %p, trying to assign %p\n",
	       (void *)v_list->i_verts[vert_index].v, (void *)v);
	bu_exit(1, "Multiple vertex assignments\n");
    }

    v_list->i_verts[vert_index].v = v;
    return 1;
}


struct iges_edge *
Get_edge(struct iges_edge_use *e_use)
{
    struct iges_edge_list *e_list;

    if ((e_list = Get_edge_list(e_use)) == NULL)
	return (struct iges_edge *)NULL;

    return &e_list->i_edge[e_use->index-1];
}


struct vertex *
Get_edge_start_vertex(struct iges_edge *edge)
{
    struct iges_vertex_list *v_list;

    if ((v_list = Get_vertex_list(edge->start_vert_de)) == NULL)
	return (struct vertex *)NULL;

    return v_list->i_verts[edge->start_vert_index-1].v;
}


struct vertex *
Get_edge_end_vertex(struct iges_edge *edge)
{
    struct iges_vertex_list *v_list;

    if ((v_list = Get_vertex_list(edge->end_vert_de)) == NULL)
	return (struct vertex *)NULL;

    return v_list->i_verts[edge->end_vert_index-1].v;
}


/*
 * Local Variables:
 * mode: C
 * tab-width: 8
 * indent-tabs-mode: t
 * c-file-style: "stroustrup"
 * End:
 * ex: shiftwidth=4 tabstop=8
 */
